#!/usr/bin/python
# -*- coding: utf-8 -*-


##################################################
#       BitLyClient Library
#-------------------------------------------------
#   Copyright:    (c) 2010 by Damjan Krstevski.
#   License:      GNU General Public License (GPL)
#   Feedback:     damjan[at]resume[dot]com
##################################################


"""
BitLyClient Library
This module will help you to cut the long link(s) via web service from the Bit.Ly
To use this module first you will need to register on http://bit.ly/
After your registration you need to get the API KEY http://bit.ly/a/your_api_key/
After this, you can use this module.

Also the BitLy will allow you to add your FaceBook and Twitter account.
This options you can find in the BitLy's accound settings.


Class variables:
    User        = Bit.Ly username
    ApiKey      = Bit.Ly API KEY
    LongUrl     = The long URL which you want to cut
    ShortUrl    = The short URL (Generated by Bit.Ly)
    Hash        = A bit.ly identifier for long_url which is unique to the given account.
    Global      = A bit.ly identifier for long_url which can be used to track aggregate
                stats across all matching bit.ly links.

Class methods:
    cut( long_url = None, flag = False )
    long_url    = The long URL which you want to cut
    flag        = If the flag is True then the cut() method will return list with the output
                    list["LongUrl"]     = LongUrl value
                    list["ShortUrl"]    = ShortUrl value
                    list["Hash"]        = Hash value
                    list["Global"]      = Global value

Exception:
    BitLyClientException as ex:
        print ex
        # This function will returns the exception
            

Usage:
    try:
        BitLy = BitLyClient( user, api, url )
        BitLy.cut()
    except BitLyClientException as ex:
        print ex
    print BitLy.ShortUrl
"""



# public symbols
__all__ 		= ["User", "ApiKey", "LongUrl", "ShortUrl", "Hash", "Global", "Cut"]

__version__             = "1.0.0"



import sys
from xml.dom.minidom import parseString



# The BitLyClientException Class
class BitLyClientException(Exception):
    """ The BitLyClient exception Message """
    pass


# The BitLyClient Class
class BitLyClient(object):
    """ Bit.Ly Client, cut the long link via Bit.Ly services """
    def __init__( self, user = None, api = None, url = None ):
        self.User       = user
        self.ApiKey     = api
        self.LongUrl    = url
        self.ShortUrl   = None
        self.Hash       = None
        self.Global     = None
        self.__xml      = None

        # XML tags
        self.__url_tag      = "url"
        self.__hash_tag     = "hash"
        self.__global_tag   = "global_hash"


    # Destructor
    def __del__( self ):
        """ Clear all data """
        self.User       = None
        self.ApiKey     = None
        self.LongUrl    = None
        self.ShortUrl   = None
        self.Hash       = None
        self.Global     = None
        self.__xml      = None
        

    # Fill the class information
    def __fill( self, data = None ):
        """ Fill the new link informations """
        if not data:
            raise BitLyClientException( "No data available." )

        try:
            doc = parseString( data )
            self.ShortUrl = doc.getElementsByTagName(self.__url_tag)[0].firstChild.data
            self.Hash = doc.getElementsByTagName(self.__hash_tag)[0].firstChild.data
            self.Global = doc.getElementsByTagName(self.__global_tag)[0].firstChild.data
        except Exception as ex:
            raise BitLyClientException( str(ex) )


    # Cut the long link
    def cut( self, long_url = None, flag = False ):
        """ Cut the long link via Bit.Ly service """
        if not self.User or not self.ApiKey:
            raise BitLyClientException( "The User and the ApiKey must have a value" )
        
        if not long_url:
            long_url = self.LongUrl

        if not long_url:
            raise BitLyClientException( "The LongUrl must have a value" )

        if sys.version[0] == "2":
            try:
                from urllib2 import urlopen
            except Exception as ex:
                raise BitLyClientException( "Failed to import urllib from urllib2\n" + str(ex) )
        else:
            try:
                from urllib2.request import urlopen
            except Exception as ex:
                raise BitLyClientException( "Failed to import urllib from urllib2.request\n" + str(ex) )

        try:
            url = "http://api.bit.ly/v3/shorten?login="
            url = url + self.User.lower() + "&apiKey=" + self.ApiKey + "&longUrl=" + long_url + "&format=xml"
            self.__xml = urlopen(url).read()
            if not self.__xml:
                raise BitLyClientException( "The XML document is empty" )
        except Exception as ex:
            raise BitLyClientException( "Failed to load XML document from Bit.Ly\n" + str(ex) )

        self.__fill( self.__xml )

        if flag:
            tmp = {"LongUrl":long_url, "ShortUrl":self.ShortUrl, "Hash":self.Hash, "Global":self.Global}
            return tmp
