# CheckableString - An hash-checkable string mixin.
#
# The MIT License (MIT)
# 
# Copyright (c) 2014 Roberto Reale
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.


import hashlib


class InvalidHexEncodedString(Exception):
    """ Odd-length hex-encoded digest """

class CheckableString(str):

    """ CheckableString adds checking against a hash to the basic string class.
    The hash can be automatically recognized.

    Example Usage:

        s = CheckableString("python")
        s.check("23eeeb4347bdd26bfc6b7ee9a3b755dd", hexdigest=True)
        > True
    """

    def _hex_decode(self, digest):
        if len(digest) % 2:
            raise InvalidHexEncodedString
        return digest.decode('hex')

    def _get_algorithm(self, digest, algorithm):
        if algorithm == "auto":
            digest_size = len(digest)
            for candidate_algorithm in hashlib.algorithms:
                hasher = getattr(hashlib, candidate_algorithm)
                if digest_size == hasher().digest_size:
                    return candidate_algorithm
        else:
            if algorithm in hashlib.algorithms:
                return algorithm
        return None

    def _check(self, digest, algorithm):
        candidate_algorithm = self._get_algorithm(digest, algorithm)
        if candidate_algorithm is None:
            return False
        hasher = getattr(hashlib, candidate_algorithm)
        calculated_digest = hasher(self.__str__()).digest()
        return (calculated_digest == digest)

    def get_algorithm(self, digest, algorithm="auto", hexdigest=False):
        if hexdigest:
            digest = self._hex_decode(digest)
        return self._get_algorithm(digest, algorithm)

    def check(self, digest, algorithm="auto", hexdigest=False):
        if hexdigest:
            digest = self._hex_decode(digest)
        return self._check(digest, algorithm)
