# -*- coding: utf-8 -*-

import os
import sys
import unittest
import codecs
import StringIO
import datetime
from decimal import Decimal
import json

CURRENT_DIR = os.path.dirname(__file__)
VERSION_INFO = sys.version_info


class LivreJournalTestCase(unittest.TestCase):
    def test_check_livre_journal(self):
        from colbert.livre_journal import check_livre_journal
        livre_journal = codecs.open(os.path.join(CURRENT_DIR, "livre-journal.txt"),
                                    mode="r", encoding="utf-8")
        result = check_livre_journal(livre_journal)
        self.assertEqual(
            result,
            [[u'18/03/2011 - Frais de constitution de la soci\xe9t\xe9 CFE Paris.',
              u'OK : d\xe9bit = cr\xe9dit (90.45).'],
             [u'18/03/2011 - Frais de constitution de la soci\xe9t\xe9 - Annonce l\xe9gale.',
              u'OK : d\xe9bit = cr\xe9dit (99.00).'],
             [u'31/03/2011 - Facture 2011-01 MyClient1 Prestation MyClient1 mars 2011',
              u'OK : d\xe9bit = cr\xe9dit (5980.00).'],
             [u'01/04/2011 - R\xe9sultat arr\xeat\xe9 compte',
              u'OK : d\xe9bit = cr\xe9dit (48.00).'],
             [u'02/04/2011 - Capital initial D\xe9p\xf4t de 1500 \u20ac par Stanislas Guerra au LCL Ledru Rollin',
              u'OK : d\xe9bit = cr\xe9dit (1500.00).'],
             [u'04/04/2011 - Facture 2011-02 MyClient2 Prestation MyClient2',
              u'OK : d\xe9bit = cr\xe9dit (1794.00).'],
             [u'28/04/2011 - Cotisation Option PRO  LCL',
              u'OK : d\xe9bit = cr\xe9dit (15.00).'],
             [u'02/05/2011 - Abonnement LCL Access', u'OK : d\xe9bit = cr\xe9dit (3.00).'],
             [u'11/06/2011 - BHV - Fournitures des bureau (Livres comptables).',
              u'OK : d\xe9bit = cr\xe9dit (25.65).'],
             [u'15/06/2011 - Remise ch\xe8que XXXXXXX r\xe8glement facture 2011-02',
              u'OK : d\xe9bit = cr\xe9dit (2088.00).'],
             [u'05/07/2011 - Traitement mois de juin g\xe9rant.',
              u'OK : d\xe9bit = cr\xe9dit (3000.00).'],
             [u'08/08/2011 - Ch\xe8que XXXXXXY Cotisation trimestrielle RSI/Pr\xe9vadi\xe8s.',
              u'OK : d\xe9bit = cr\xe9dit (393.00).'],
             [u'02/09/2011 - Virement MyClient1 ZZZZZZZZZZZ Facture 2011-01',
              u'OK : d\xe9bit = cr\xe9dit (6960.00).'],
             [u'03/09/2011 - Abonnement LCL Access', u'OK : d\xe9bit = cr\xe9dit (3.00).'],
             [u'28/09/2011 - Facture 2011-04 MyClient1 Prestation aout 2011',
              u'OK : d\xe9bit = cr\xe9dit (13156.00).'],
             [u'30/09/2011 - Solde des comptes de TVA du 01/03/2011 au 30/09/2011',
              u'OK : d\xe9bit = cr\xe9dit (1274.00).'],
             [u'06/10/2011 - Ch\xe8que WWWWWWW R\xe8glement de la TVA trimestrielle',
              u'OK : d\xe9bit = cr\xe9dit (1240.00).'],
             [u'01/11/2011 - Facture 2011-05 MyClient1 Prestation septembre 2011',
              u'OK : d\xe9bit = cr\xe9dit (5382.00).'],
             [u'17/11/2011 - Ch\xe8que ZZZZZZZ Cotisation sociales Urssaf 4\xe8me trimestre.',
              u'OK : d\xe9bit = cr\xe9dit (402.00).'],
             [u'01/12/2011 - Abonnement LCL Access', u'OK : d\xe9bit = cr\xe9dit (3.00).'],
             [u'01/12/2011 - Virement MyClient1 WWWWWWWWWW Facture 2011-04, 2011-05',
              u'OK : d\xe9bit = cr\xe9dit (21576.00).'],
             [u'01/12/2011 - Facture 2011-06 MyClient3',
              u'OK : d\xe9bit = cr\xe9dit (8372.00).'],
             [u'31/12/2011 - Solde des comptes de TVA du 01/10/2011 au 31/12/2011',
              u'OK : d\xe9bit = cr\xe9dit (3038.00).'],
             [u'31/12/2011 - Prestation MyClient1 d\xe9cembre 2011',
              u'OK : d\xe9bit = cr\xe9dit (13156.00).'],
             [u'01/01/2012 - Prestation MyClient1 d\xe9cembre 2011',
              u'OK : d\xe9bit = cr\xe9dit (13156.00).'],
             [u'03/01/2012 - Facture 2012-01 MyClient1 Prestation d\xe9cembre 2011',
              u'OK : d\xe9bit = cr\xe9dit (13156.00).'],
             [u"01/02/2012 - Restaurant La Tour d'argent D\xe9jeuner d'affaire avec Steve Jobs 0.88\u20ac TVA 19.6% ; 2.90\u20ac TVA 7.0%",
              u'OK : d\xe9bit = cr\xe9dit (49.80).']]
        )

    def test_ecritures_de_cloture(self):
        from colbert.livre_journal import ecritures_de_cloture
        balance_des_comptes = codecs.open(os.path.join(CURRENT_DIR, "balance_des_comptes-2011.json"),
                                          mode="r", encoding="utf-8")

        edc = ecritures_de_cloture(json.loads(balance_des_comptes.read()))
        # Uncomment to generate.
        # print json.dumps(edc, default=json_encoder, indent=4)
        self.maxDiff = None
        self.assertEqual(
            edc,
            [{'date': datetime.date(2011, 12, 31),
              'ecritures': [{'credit': Decimal('0.00'),
                             'debit': Decimal('40000.00'),
                             'nom_compte': u'Produits - prestations de services',
                             'numero_compte_credit': u'',
                             'numero_compte_debit': u'706'},
                            {'credit': Decimal('0.00'),
                             'debit': Decimal('0.34'),
                             'nom_compte': u'Produits divers de gestion courante',
                             'numero_compte_credit': u'',
                             'numero_compte_debit': u'758'},
                            {'credit': Decimal('40000.34'),
                             'debit': Decimal('0.0'),
                             'nom_compte': u'Regroupement des comptes de produits',
                             'numero_compte_credit': '127',
                             'numero_compte_debit': u''}],
              'intitule': u'Ecritures de cl\xf4ture des comptes.'},
             {'date': datetime.date(2011, 12, 31),
              'ecritures': [{'credit': Decimal('0.0'),
                             'debit': Decimal('4048.44'),
                             'nom_compte': u'Regroupement des comptes de charges',
                             'numero_compte_credit': u'',
                             'numero_compte_debit': '126'},
                            {'credit': Decimal('21.44'),
                             'debit': Decimal('0.00'),
                             'nom_compte': u'Achats - Fournitures de bureau',
                             'numero_compte_credit': u'60225',
                             'numero_compte_debit': u''},
                            {'credit': Decimal('160.00'),
                             'debit': Decimal('0.00'),
                             'nom_compte': u"Achats - Frais d'actes et de contentieux",
                             'numero_compte_credit': u'6227',
                             'numero_compte_debit': u''},
                            {'credit': Decimal('72.00'),
                             'debit': Decimal('0.00'),
                             'nom_compte': u'Autres frais de commission sur prestations de services',
                             'numero_compte_credit': u'6278-LCL',
                             'numero_compte_debit': u''},
                            {'credit': Decimal('3000.00'),
                             'debit': Decimal('0.00'),
                             'nom_compte': u'Charges - Salaires et appointements',
                             'numero_compte_credit': u'6411',
                             'numero_compte_debit': u''},
                            {'credit': Decimal('393.00'),
                             'debit': Decimal('0.00'),
                             'nom_compte': u'Charges - cotisations RSI',
                             'numero_compte_credit': u'6411-RSI',
                             'numero_compte_debit': u''},
                            {'credit': Decimal('161.80'),
                             'debit': Decimal('0.00'),
                             'nom_compte': u'Charges - cotisations URSSAF - Allocations familliales',
                             'numero_compte_credit': u'6411-URSF1',
                             'numero_compte_debit': u''},
                            {'credit': Decimal('153.31'),
                             'debit': Decimal('0.00'),
                             'nom_compte': u'Charges - cotisations URSSAF - CSG/RDS d\xe9ductible',
                             'numero_compte_credit': u'6411-URSF2',
                             'numero_compte_debit': u''},
                            {'credit': Decimal('86.89'),
                             'debit': Decimal('0.00'),
                             'nom_compte': u'Charges - cotisations URSSAF - CSG/RDS non-d\xe9ductible',
                             'numero_compte_credit': u'6411-URSF3',
                             'numero_compte_debit': u''}],
              'intitule': u'Ecritures de cl\xf4ture des comptes.'},
             {'date': datetime.date(2011, 12, 31),
              'ecritures': [{'credit': Decimal('0.0'),
                             'debit': Decimal('40000.34'),
                             'nom_compte': u'Regroupement des comptes de produits',
                             'numero_compte_credit': u'',
                             'numero_compte_debit': '127'},
                            {'credit': Decimal('4048.44'),
                             'debit': Decimal('0.0'),
                             'nom_compte': u'Regroupement des comptes de charges',
                             'numero_compte_credit': '126',
                             'numero_compte_debit': u''},
                            {'credit': Decimal('35951.90'),
                             'debit': Decimal('0.0'),
                             'nom_compte': u"r\xe9sultat de l'exercice (b\xe9n\xe9fice)",
                             'numero_compte_credit': '120',
                             'numero_compte_debit': u''}],
              'intitule': u"Enregistrement du r\xe9sultat net de l'exercice"}]
        )

    def test_ecritures_to_livre_journal(self):
        from colbert.livre_journal import ecritures_to_livre_journal
        ecritures = codecs.open(os.path.join(CURRENT_DIR, "ecritures_de_cloture-2011.json"),
                                mode="r", encoding="utf-8")
        output = StringIO.StringIO()
        ecritures_to_livre_journal(json.loads(ecritures.read()), output)
        self.maxDiff = None
        self.assertEqual(output.getvalue(),
u"""+----------------------------------------------------------------------------------------------------------------------------------------------------+
| Ecritures pour le Livre-journal                                                                                                                    |
+=============+=================+=================+==============================================================+=================+=================+
|| 31/12/2011 ||                ||                || Ecritures de clôture des comptes.                           ||                ||                |
||            || 706            ||                || Produits - prestations de services                          || 40000.00       ||                |
||            || 758            ||                || Produits divers de gestion courante                         || 0.34           ||                |
||            ||                || 127            ||     Regroupement des comptes de produits                    ||                || 40000.34       |
+-------------+-----------------+-----------------+--------------------------------------------------------------+-----------------+-----------------+
|| 31/12/2011 ||                ||                || Ecritures de clôture des comptes.                           ||                ||                |
||            || 126            ||                || Regroupement des comptes de charges                         || 4048.44        ||                |
||            ||                || 60225          ||     Achats - Fournitures de bureau                          ||                || 21.44          |
||            ||                || 6227           ||     Achats - Frais d'actes et de contentieux                ||                || 160.00         |
||            ||                || 6278-LCL       ||     Autres frais de commission sur prestations de services  ||                || 72.00          |
||            ||                || 6411           ||     Charges - Salaires et appointements                     ||                || 3000.00        |
||            ||                || 6411-RSI       ||     Charges - cotisations RSI                               ||                || 393.00         |
||            ||                || 6411-URSF1     ||     Charges - cotisations URSSAF - Allocations familliales  ||                || 161.80         |
||            ||                || 6411-URSF2     ||     Charges - cotisations URSSAF - CSG/RDS déductible       ||                || 153.31         |
||            ||                || 6411-URSF3     ||     Charges - cotisations URSSAF - CSG/RDS non-déductible   ||                || 86.89          |
+-------------+-----------------+-----------------+--------------------------------------------------------------+-----------------+-----------------+
|| 31/12/2011 ||                ||                || Enregistrement du résultat net de l'exercice                ||                ||                |
||            || 127            ||                || Regroupement des comptes de produits                        || 40000.34       ||                |
||            ||                || 126            ||     Regroupement des comptes de charges                     ||                || 4048.44        |
||            ||                || 120            ||     résultat de l'exercice (bénéfice)                       ||                || 35951.90       |
+-------------+-----------------+-----------------+--------------------------------------------------------------+-----------------+-----------------+

""")

    def test_get_solde_compte(self):
        from colbert.livre_journal import livre_journal_to_list
        from colbert.livre_journal import get_solde_compte

        livre_journal = codecs.open(os.path.join(CURRENT_DIR, "livre-journal.txt"),
                                    mode="r", encoding="utf-8")
        livre_journal_list = livre_journal_to_list(livre_journal)

        date_debut = datetime.date(2011, 1, 1)
        date_fin = datetime.date(2011, 12, 31)

        debit, credit = get_solde_compte(livre_journal_list, "512", date_debut, date_fin)
        # TODO verifier.
        self.assertEqual(debit, Decimal("22679.35"))
        self.assertEqual(credit, Decimal("0.00"))

    def test_livre_journal_to_list(self):
        from colbert.livre_journal import RX_DATE_INTITULE, RX_SUITE_INTITULE, RX_ECRITURE
        from colbert.livre_journal import livre_journal_to_list

        # Première ligne d'écriture.
        s = u"|| 31/03/2011 ||                ||                || Facture 2011-01 AdenClassifieds      ||          ||           |  "
        if VERSION_INFO >= (2, 7):
            self.assertRegexpMatches(s, RX_DATE_INTITULE)
        m = RX_DATE_INTITULE.match(s)
        self.assertEqual(m.groupdict(), {'intitule': u'Facture 2011-01 AdenClassifieds      ',
                                         'credit': u'           ',
                                         'debit': u'          ',
                                         'date': u'31/03/2011',
                                         'numero_compte_credit': u'                ',
                                         'numero_compte_debit': u'                '})

        # Ligne supplementaire d'intitulé.
        s = u"||   ||       ||        ||    suite et fin de l'intitulé     ||     ||   |"
        if VERSION_INFO >= (2, 7):
            self.assertRegexpMatches(s, RX_SUITE_INTITULE)
        m = RX_SUITE_INTITULE.match(s)

        # Ecriture.
        s = u"||   ||    4111-clie   ||        ||   Clients - ventes de biens ou prestations de services  ||    8 372   ||         |"
        if VERSION_INFO >= (2, 7):
            self.assertRegexpMatches(s, RX_ECRITURE)
        m = RX_ECRITURE.match(s)
        self.assertEqual(m.groupdict(), {'nom_compte': u'Clients - ventes de biens ou prestations de services  ',
                                         'credit': u'         ',
                                         'debit': u'    8 372   ',
                                         'date': u'   ',
                                         'numero_compte_credit': u'        ',
                                         'numero_compte_debit': u'    4111-clie   '})

        # Conversion du livre journal.
        livre_journal = codecs.open(os.path.join(CURRENT_DIR, "livre-journal.txt"),
                                    mode="r", encoding="utf-8")
        livre_journal_list = livre_journal_to_list(livre_journal)
        self.assertEqual(
            livre_journal_list,
            [{'date': datetime.date(2011, 3, 18),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('80.00'),
                              'nom_compte': u"Achats - Frais d'actes et de contentieux",
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6227'},
                             {u'credit': Decimal('0.00'),
                              u'debit': Decimal('10.45'),
                              'nom_compte': u'T.V.A. d\xe9ductible sur autres biens et services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'44566'},
                             {u'credit': Decimal('90.45'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Associ\xe9s - Comptes courants',
                              'numero_compte_credit': u'455',
                              'numero_compte_debit': u''}],
              'intitule': u'Frais de constitution de la soci\xe9t\xe9 CFE Paris.',
              u'numero_ligne_debut': 13,
              u'numero_ligne_fin': 17},
             {'date': datetime.date(2011, 3, 18),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('80.00'),
                              'nom_compte': u"Achats - Frais d'actes et de contentieux MONTANT \xe0 v\xe9rifier",
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6227'},
                             {u'credit': Decimal('0.00'),
                              u'debit': Decimal('19.00'),
                              'nom_compte': u'T.V.A. d\xe9ductible sur autres biens et services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'44566'},
                             {u'credit': Decimal('99.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Associ\xe9s - Comptes courants',
                              'numero_compte_credit': u'455',
                              'numero_compte_debit': u''}],
              'intitule': u'Frais de constitution de la soci\xe9t\xe9 - Annonce l\xe9gale.',
              u'numero_ligne_debut': 18,
              u'numero_ligne_fin': 22},
             {'date': datetime.date(2011, 3, 31),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('5980.00'),
                              'nom_compte': u'Clients - ventes de biens ou prestations de services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'4111-CL1'},
                             {u'credit': Decimal('5000.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Produits - prestations de services',
                              'numero_compte_credit': u'706',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('980.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Taxes sur le CA sur factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'44587',
                              'numero_compte_debit': u''}],
              'intitule': u'Facture 2011-01 MyClient1 Prestation MyClient1 mars 2011',
              u'numero_ligne_debut': 23,
              u'numero_ligne_fin': 28},
             {'date': datetime.date(2011, 4, 1),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('48.00'),
                              'nom_compte': u'Autres frais de commission sur prestations de services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6278-LCL'},
                             {u'credit': Decimal('48.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'512',
                              'numero_compte_debit': u''}],
              'intitule': u'R\xe9sultat arr\xeat\xe9 compte',
              u'numero_ligne_debut': 31,
              u'numero_ligne_fin': 34},
             {'date': datetime.date(2011, 4, 2),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('1500.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'512'},
                             {u'credit': Decimal('1500.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u"Capital et compte de l'exploitant",
                              'numero_compte_credit': u'100',
                              'numero_compte_debit': u''}],
              'intitule': u'Capital initial D\xe9p\xf4t de 1500 \u20ac par Stanislas Guerra au LCL Ledru Rollin',
              u'numero_ligne_debut': 35,
              u'numero_ligne_fin': 40},
             {'date': datetime.date(2011, 4, 4),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('1794.00'),
                              'nom_compte': u'Clients - ventes de biens ou prestations de services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'4111-CL2'},
                             {u'credit': Decimal('1500.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Produits - prestations de services',
                              'numero_compte_credit': u'706',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('294.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Taxes sur le CA sur factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'44587',
                              'numero_compte_debit': u''}],
              'intitule': u'Facture 2011-02 MyClient2 Prestation MyClient2',
              u'numero_ligne_debut': 41,
              u'numero_ligne_fin': 46},
             {'date': datetime.date(2011, 4, 28),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('15.00'),
                              'nom_compte': u'Autres frais de commission sur prestations de services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6278-LCL'},
                             {u'credit': Decimal('15.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'512',
                              'numero_compte_debit': u''}],
              'intitule': u'Cotisation Option PRO  LCL',
              u'numero_ligne_debut': 47,
              u'numero_ligne_fin': 50},
             {'date': datetime.date(2011, 5, 2),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('3.00'),
                              'nom_compte': u'Autres frais de commission sur prestations de services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6278-LCL'},
                             {u'credit': Decimal('3.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'512',
                              'numero_compte_debit': u''}],
              'intitule': u'Abonnement LCL Access',
              u'numero_ligne_debut': 53,
              u'numero_ligne_fin': 56},
             {'date': datetime.date(2011, 6, 11),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('21.44'),
                              'nom_compte': u'Achats - Fournitures de bureau',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'60225'},
                             {u'credit': Decimal('0.00'),
                              u'debit': Decimal('4.21'),
                              'nom_compte': u'T.V.A. d\xe9ductible sur autres biens et services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'44566'},
                             {u'credit': Decimal('25.65'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'512',
                              'numero_compte_debit': u''}],
              'intitule': u'BHV - Fournitures des bureau (Livres comptables).',
              u'numero_ligne_debut': 59,
              u'numero_ligne_fin': 63},
             {'date': datetime.date(2011, 6, 15),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('1794.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'512'},
                             {u'credit': Decimal('0.00'),
                              u'debit': Decimal('294.00'),
                              'nom_compte': u'Taxes sur le CA sur factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'44587'},
                             {u'credit': Decimal('1794.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Clients - ventes de biens ou prestations de services',
                              'numero_compte_credit': u'4111-CL2',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('294.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'T.V.A. Collect\xe9e',
                              'numero_compte_credit': u'44571',
                              'numero_compte_debit': u''}],
              'intitule': u'Remise ch\xe8que XXXXXXX r\xe8glement facture 2011-02',
              u'numero_ligne_debut': 64,
              u'numero_ligne_fin': 69},
             {'date': datetime.date(2011, 7, 5),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('3000.00'),
                              'nom_compte': u'Charges - Salaires et appointements',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6411'},
                             {u'credit': Decimal('3000.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'512',
                              'numero_compte_debit': u''}],
              'intitule': u'Traitement mois de juin g\xe9rant.',
              u'numero_ligne_debut': 72,
              u'numero_ligne_fin': 75},
             {'date': datetime.date(2011, 8, 8),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('393.00'),
                              'nom_compte': u'Charges - cotisations RSI',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6411-RSI'},
                             {u'credit': Decimal('393.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'512',
                              'numero_compte_debit': u''}],
              'intitule': u'Ch\xe8que XXXXXXY Cotisation trimestrielle RSI/Pr\xe9vadi\xe8s.',
              u'numero_ligne_debut': 78,
              u'numero_ligne_fin': 82},
             {'date': datetime.date(2011, 9, 2),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('5980.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'512'},
                             {u'credit': Decimal('0.00'),
                              u'debit': Decimal('980.00'),
                              'nom_compte': u'Taxes sur le CA sur factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'44587'},
                             {u'credit': Decimal('5980.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Clients - ventes de biens ou prestations de services',
                              'numero_compte_credit': u'4111-CL1',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('980.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'T.V.A. Collect\xe9e',
                              'numero_compte_credit': u'44571',
                              'numero_compte_debit': u''}],
              'intitule': u'Virement MyClient1 ZZZZZZZZZZZ Facture 2011-01',
              u'numero_ligne_debut': 85,
              u'numero_ligne_fin': 91},
             {'date': datetime.date(2011, 9, 3),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('3.00'),
                              'nom_compte': u'Autres frais de commission sur prestations de services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6278-LCL'},
                             {u'credit': Decimal('3.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'512',
                              'numero_compte_debit': u''}],
              'intitule': u'Abonnement LCL Access',
              u'numero_ligne_debut': 92,
              u'numero_ligne_fin': 95},
             {'date': datetime.date(2011, 9, 28),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('13156.00'),
                              'nom_compte': u'Clients - ventes de biens ou prestations de services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'4111-CL1'},
                             {u'credit': Decimal('11000.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Produits - prestations de services',
                              'numero_compte_credit': u'706',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('2156.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Taxes sur le CA sur factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'44587',
                              'numero_compte_debit': u''}],
              'intitule': u'Facture 2011-04 MyClient1 Prestation aout 2011',
              u'numero_ligne_debut': 96,
              u'numero_ligne_fin': 101},
             {'date': datetime.date(2011, 9, 30),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('1274.00'),
                              'nom_compte': u'TVA collect\xe9',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'44571'},
                             {u'credit': Decimal('33.66'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'TVA d\xe9ductible sur autres biens et services',
                              'numero_compte_credit': u'44566',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('1240.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'TVA \xe0 d\xe9caisser',
                              'numero_compte_credit': u'44551',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('0.34'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Produits divers de gestion courante',
                              'numero_compte_credit': u'758',
                              'numero_compte_debit': u''}],
              'intitule': u'Solde des comptes de TVA du 01/03/2011 au 30/09/2011',
              u'numero_ligne_debut': 104,
              u'numero_ligne_fin': 109},
             {'date': datetime.date(2011, 10, 6),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('1240.00'),
                              'nom_compte': u'TVA \xe0 d\xe9caisser',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'44551'},
                             {u'credit': Decimal('1240.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'512',
                              'numero_compte_debit': u''}],
              'intitule': u'Ch\xe8que WWWWWWW R\xe8glement de la TVA trimestrielle',
              u'numero_ligne_debut': 112,
              u'numero_ligne_fin': 116},
             {'date': datetime.date(2011, 11, 1),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('5382.00'),
                              'nom_compte': u'Clients - ventes de biens ou prestations de services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'4111-CL1'},
                             {u'credit': Decimal('4500.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Produits - prestations de services',
                              'numero_compte_credit': u'706',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('882.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Taxes sur le CA sur factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'44587',
                              'numero_compte_debit': u''}],
              'intitule': u'Facture 2011-05 MyClient1 Prestation septembre 2011',
              u'numero_ligne_debut': 119,
              u'numero_ligne_fin': 124},
             {'date': datetime.date(2011, 11, 17),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('161.80'),
                              'nom_compte': u'Charges - cotisations URSSAF - Allocations familliales',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6411-URSF1'},
                             {u'credit': Decimal('0.00'),
                              u'debit': Decimal('153.31'),
                              'nom_compte': u'Charges - cotisations URSSAF - CSG/RDS d\xe9ductible',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6411-URSF2'},
                             {u'credit': Decimal('0.00'),
                              u'debit': Decimal('86.89'),
                              'nom_compte': u'Charges - cotisations URSSAF - CSG/RDS non-d\xe9ductible',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6411-URSF3'},
                             {u'credit': Decimal('402.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'512',
                              'numero_compte_debit': u''}],
              'intitule': u'Ch\xe8que ZZZZZZZ Cotisation sociales Urssaf 4\xe8me trimestre.',
              u'numero_ligne_debut': 125,
              u'numero_ligne_fin': 131},
             {'date': datetime.date(2011, 12, 1),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('3.00'),
                              'nom_compte': u'Autres frais de commission sur prestations de services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6278-LCL'},
                             {u'credit': Decimal('3.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'512',
                              'numero_compte_debit': u''}],
              'intitule': u'Abonnement LCL Access',
              u'numero_ligne_debut': 134,
              u'numero_ligne_fin': 137},
             {'date': datetime.date(2011, 12, 1),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('18538.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'512'},
                             {u'credit': Decimal('0.00'),
                              u'debit': Decimal('2156.00'),
                              'nom_compte': u'Taxes sur le CA sur factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'44587'},
                             {u'credit': Decimal('0.00'),
                              u'debit': Decimal('882.00'),
                              'nom_compte': u'Taxes sur le CA sur factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'44587'},
                             {u'credit': Decimal('18538.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Clients - ventes de biens ou prestations de services',
                              'numero_compte_credit': u'4111-CL1',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('2156.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'T.V.A. Collect\xe9e',
                              'numero_compte_credit': u'44571',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('882.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'T.V.A. Collect\xe9e',
                              'numero_compte_credit': u'44571',
                              'numero_compte_debit': u''}],
              'intitule': u'Virement MyClient1 WWWWWWWWWW Facture 2011-04, 2011-05',
              u'numero_ligne_debut': 138,
              u'numero_ligne_fin': 146},
             {'date': datetime.date(2011, 12, 1),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('8372.00'),
                              'nom_compte': u'Clients - ventes de biens ou prestations de services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'4111-CL3'},
                             {u'credit': Decimal('7000.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Produits - prestations de services',
                              'numero_compte_credit': u'706',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('1372.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Taxes sur le CA sur factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'44587',
                              'numero_compte_debit': u''}],
              'intitule': u'Facture 2011-06 MyClient3',
              u'numero_ligne_debut': 147,
              u'numero_ligne_fin': 151},
             {'date': datetime.date(2011, 12, 31),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('3038.00'),
                              'nom_compte': u'TVA collect\xe9',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'44571'},
                             {u'credit': Decimal('3038.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'TVA \xe0 d\xe9caisser',
                              'numero_compte_credit': u'44551',
                              'numero_compte_debit': u''}],
              'intitule': u'Solde des comptes de TVA du 01/10/2011 au 31/12/2011',
              u'numero_ligne_debut': 154,
              u'numero_ligne_fin': 157},
             {'date': datetime.date(2011, 12, 31),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('13156.00'),
                              'nom_compte': u'Clients - Factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'4181'},
                             {u'credit': Decimal('11000.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Produits - prestations de services',
                              'numero_compte_credit': u'706',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('2156.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Taxes sur le CA sur factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'44587',
                              'numero_compte_debit': u''}],
              'intitule': u'Prestation MyClient1 d\xe9cembre 2011',
              u'numero_ligne_debut': 160,
              u'numero_ligne_fin': 164},
             {'date': datetime.date(2012, 1, 1),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('11000.00'),
                              'nom_compte': u'Produits - prestations de services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'706'},
                             {u'credit': Decimal('0.00'),
                              u'debit': Decimal('2156.00'),
                              'nom_compte': u'Taxes sur le CA sur factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'44587'},
                             {u'credit': Decimal('13156.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Clients - Factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'4181',
                              'numero_compte_debit': u''}],
              'intitule': u'Prestation MyClient1 d\xe9cembre 2011',
              u'numero_ligne_debut': 169,
              u'numero_ligne_fin': 173},
             {'date': datetime.date(2012, 1, 3),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('13156.00'),
                              'nom_compte': u'Clients - ventes de biens ou prestations de services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'4111-CL1'},
                             {u'credit': Decimal('11000.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Produits - prestations de services',
                              'numero_compte_credit': u'706',
                              'numero_compte_debit': u''},
                             {u'credit': Decimal('2156.00'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Taxes sur le CA sur factures \xe0 \xe9tablir',
                              'numero_compte_credit': u'44587',
                              'numero_compte_debit': u''}],
              'intitule': u'Facture 2012-01 MyClient1 Prestation d\xe9cembre 2011',
              u'numero_ligne_debut': 176,
              u'numero_ligne_fin': 181},
             {'date': datetime.date(2012, 2, 1),
              u'ecritures': [{u'credit': Decimal('0.00'),
                              u'debit': Decimal('46.02'),
                              'nom_compte': u'Charges - R\xe9ceptions',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'6257'},
                             {u'credit': Decimal('0.00'),
                              u'debit': Decimal('3.78'),
                              'nom_compte': u'T.V.A. d\xe9ductible sur autres biens et services',
                              'numero_compte_credit': u'',
                              'numero_compte_debit': u'44566'},
                             {u'credit': Decimal('49.80'),
                              u'debit': Decimal('0.00'),
                              'nom_compte': u'Banques',
                              'numero_compte_credit': u'512',
                              'numero_compte_debit': u''}],
              'intitule': u"Restaurant La Tour d'argent D\xe9jeuner d'affaire avec Steve Jobs 0.88\u20ac TVA 19.6% ; 2.90\u20ac TVA 7.0%",
              u'numero_ligne_debut': 184,
              u'numero_ligne_fin': 190}]
        )


def suite():
    suite = unittest.TestLoader().loadTestsFromTestCase(LivreJournalTestCase)
    return suite
