# -*- coding: utf-8 -*-

import os
import unittest
import codecs
import StringIO
import datetime
import json
from colbert.utils import decode_as_ecriture as as_ecriture

from decimal import Decimal

CURRENT_DIR = os.path.dirname(__file__)


class GrandLivreTestCase(unittest.TestCase):
    def test_grand_livre(self):
        from colbert.grand_livre import grand_livre
        livre_journal = codecs.open(os.path.join(CURRENT_DIR, "livre-journal.txt"),
                                    mode="r", encoding="utf-8")
        gl = grand_livre(livre_journal, "Grand-Livre 2011",
                         date_debut=datetime.date(2011, 3, 1),
                         date_fin=datetime.date(2011, 12, 31),
                         grand_livre_precedent=None)

        # Uncomment to generate the JSON test file.
        # from colbert.utils import json_encoder
        # output = codecs.open(os.path.join(CURRENT_DIR, "grand_livre-2011.json"),
        #                               mode="w+", encoding="utf-8")
        # output.write(json.dumps(gl, default=json_encoder, indent=4))
        # output.close
        # pprint(gl)
        self.maxDiff = None
        self.assertEqual(gl, {
            'comptes': {
                u'100': {'ecritures': [{'credit': Decimal('1500.00'),
                                        'date': datetime.date(2011, 4, 2),
                                        'intitule': u'Capital initial Dépôt de 1500 € par Stanislas Guerra au LCL Ledru Rollin'}],
                         'nom': u"Capital et compte de l'exploitant",
                         'solde_crediteur': Decimal('1500.00'),
                         'solde_debiteur': Decimal('0.00'),
                         'total_credit': Decimal('1500.00'),
                         'total_debit': Decimal('0.00')},
                u'4111-CL1': {'ecritures': [{'date': datetime.date(2011, 3, 31),
                                             'debit': Decimal('5980.00'),
                                             'intitule': u'Facture 2011-01 MyClient1 Prestation MyClient1 mars 2011'},
                                            {'credit': Decimal('5980.00'),
                                             'date': datetime.date(2011, 9, 2),
                                             'intitule': u'Virement MyClient1 ZZZZZZZZZZZ Facture 2011-01'},
                                            {'date': datetime.date(2011, 9, 28),
                                             'debit': Decimal('13156.00'),
                                             'intitule': u'Facture 2011-04 MyClient1 Prestation aout 2011'},
                                            {'date': datetime.date(2011, 11, 1),
                                             'debit': Decimal('5382.00'),
                                             'intitule': u'Facture 2011-05 MyClient1 Prestation septembre 2011'},
                                            {'credit': Decimal('18538.00'),
                                             'date': datetime.date(2011, 12, 1),
                                             'intitule': u'Virement MyClient1 WWWWWWWWWW Facture 2011-04, 2011-05'}],
                              'nom': u'Clients - ventes de biens ou prestations de services',
                              'solde_crediteur': Decimal('0.00'),
                              'solde_debiteur': Decimal('0.00'),
                              'total_credit': Decimal('24518.00'),
                              'total_debit': Decimal('24518.00')},
                u'4111-CL2': {'ecritures': [{'date': datetime.date(2011, 4, 4),
                                             'debit': Decimal('1794.00'),
                                             'intitule': u'Facture 2011-02 MyClient2 Prestation MyClient2'},
                                            {'credit': Decimal('1794.00'),
                                             'date': datetime.date(2011, 6, 15),
                                             'intitule': u'Remise chèque XXXXXXX règlement facture 2011-02'}],
                              'nom': u'Clients - ventes de biens ou prestations de services',
                              'solde_crediteur': Decimal('0.00'),
                              'solde_debiteur': Decimal('0.00'),
                              'total_credit': Decimal('1794.00'),
                              'total_debit': Decimal('1794.00')},
                u'4111-CL3': {'ecritures': [{'date': datetime.date(2011, 12, 1),
                                             'debit': Decimal('8372.00'),
                                             'intitule': u'Facture 2011-06 MyClient3'}],
                              'nom': u'Clients - ventes de biens ou prestations de services',
                              'solde_crediteur': Decimal('0.00'),
                              'solde_debiteur': Decimal('8372.00'),
                              'total_credit': Decimal('0.00'),
                              'total_debit': Decimal('8372.00')},
                u'4181': {'ecritures': [{'date': datetime.date(2011, 12, 31),
                                         'debit': Decimal('13156.00'),
                                         'intitule': u'Prestation MyClient1 décembre 2011'}],
                          'nom': u'Clients - Factures à établir',
                          'solde_crediteur': Decimal('0.00'),
                          'solde_debiteur': Decimal('13156.00'),
                          'total_credit': Decimal('0.00'),
                          'total_debit': Decimal('13156.00')},
                u'44551': {'ecritures': [{'credit': Decimal('1240.00'),
                                          'date': datetime.date(2011, 9, 30),
                                          'intitule': u'Solde des comptes de TVA du 01/03/2011 au 30/09/2011'},
                                         {'date': datetime.date(2011, 10, 6),
                                          'debit': Decimal('1240.00'),
                                          'intitule': u'Chèque WWWWWWW Règlement de la TVA trimestrielle'},
                                         {'credit': Decimal('3038.00'),
                                          'date': datetime.date(2011, 12, 31),
                                          'intitule': u'Solde des comptes de TVA du 01/10/2011 au 31/12/2011'}],
                           'nom': u'TVA à décaisser',
                           'solde_crediteur': Decimal('3038.00'),
                           'solde_debiteur': Decimal('0.00'),
                           'total_credit': Decimal('4278.00'),
                           'total_debit': Decimal('1240.00')},
                u'44566': {'ecritures': [{'date': datetime.date(2011, 3, 18),
                                          'debit': Decimal('10.45'),
                                          'intitule': u'Frais de constitution de la société CFE Paris.'},
                                         {'date': datetime.date(2011, 3, 18),
                                          'debit': Decimal('19.00'),
                                          'intitule': u'Frais de constitution de la société - Annonce légale.'},
                                         {'date': datetime.date(2011, 6, 11),
                                          'debit': Decimal('4.21'),
                                          'intitule': u'BHV - Fournitures des bureau (Livres comptables).'},
                                         {'credit': Decimal('33.66'),
                                          'date': datetime.date(2011, 9, 30),
                                          'intitule': u'Solde des comptes de TVA du 01/03/2011 au 30/09/2011'}],
                           'nom': u'T.V.A. déductible sur autres biens et services',
                           'solde_crediteur': Decimal('0.00'),
                           'solde_debiteur': Decimal('0.00'),
                           'total_credit': Decimal('33.66'),
                           'total_debit': Decimal('33.66')},
                u'44571': {'ecritures': [{'credit': Decimal('294.00'),
                                          'date': datetime.date(2011, 6, 15),
                                          'intitule': u'Remise chèque XXXXXXX règlement facture 2011-02'},
                                         {'credit': Decimal('980.00'),
                                          'date': datetime.date(2011, 9, 2),
                                          'intitule': u'Virement MyClient1 ZZZZZZZZZZZ Facture 2011-01'},
                                         {'date': datetime.date(2011, 9, 30),
                                          'debit': Decimal('1274.00'),
                                          'intitule': u'Solde des comptes de TVA du 01/03/2011 au 30/09/2011'},
                                         {'credit': Decimal('2156.00'),
                                          'date': datetime.date(2011, 12, 1),
                                          'intitule': u'Virement MyClient1 WWWWWWWWWW Facture 2011-04, 2011-05'},
                                         {'credit': Decimal('882.00'),
                                          'date': datetime.date(2011, 12, 1),
                                          'intitule': u'Virement MyClient1 WWWWWWWWWW Facture 2011-04, 2011-05'},
                                         {'date': datetime.date(2011, 12, 31),
                                          'debit': Decimal('3038.00'),
                                          'intitule': u'Solde des comptes de TVA du 01/10/2011 au 31/12/2011'}],
                           'nom': u'T.V.A. Collectée',
                           'solde_crediteur': Decimal('0.00'),
                           'solde_debiteur': Decimal('0.00'),
                           'total_credit': Decimal('4312.00'),
                           'total_debit': Decimal('4312.00')},
                u'44587': {'ecritures': [{'credit': Decimal('980.00'),
                                          'date': datetime.date(2011, 3, 31),
                                          'intitule': u'Facture 2011-01 MyClient1 Prestation MyClient1 mars 2011'},
                                         {'credit': Decimal('294.00'),
                                          'date': datetime.date(2011, 4, 4),
                                          'intitule': u'Facture 2011-02 MyClient2 Prestation MyClient2'},
                                         {'date': datetime.date(2011, 6, 15),
                                          'debit': Decimal('294.00'),
                                          'intitule': u'Remise chèque XXXXXXX règlement facture 2011-02'},
                                         {'date': datetime.date(2011, 9, 2),
                                          'debit': Decimal('980.00'),
                                          'intitule': u'Virement MyClient1 ZZZZZZZZZZZ Facture 2011-01'},
                                         {'credit': Decimal('2156.00'),
                                          'date': datetime.date(2011, 9, 28),
                                          'intitule': u'Facture 2011-04 MyClient1 Prestation aout 2011'},
                                         {'credit': Decimal('882.00'),
                                          'date': datetime.date(2011, 11, 1),
                                          'intitule': u'Facture 2011-05 MyClient1 Prestation septembre 2011'},
                                         {'date': datetime.date(2011, 12, 1),
                                          'debit': Decimal('2156.00'),
                                          'intitule': u'Virement MyClient1 WWWWWWWWWW Facture 2011-04, 2011-05'},
                                         {'date': datetime.date(2011, 12, 1),
                                          'debit': Decimal('882.00'),
                                          'intitule': u'Virement MyClient1 WWWWWWWWWW Facture 2011-04, 2011-05'},
                                         {'credit': Decimal('1372.00'),
                                          'date': datetime.date(2011, 12, 1),
                                          'intitule': u'Facture 2011-06 MyClient3'},
                                         {'credit': Decimal('2156.00'),
                                          'date': datetime.date(2011, 12, 31),
                                          'intitule': u'Prestation MyClient1 décembre 2011'}],
                           'nom': u'Taxes sur le CA sur factures à établir',
                           'solde_crediteur': Decimal('3528.00'),
                           'solde_debiteur': Decimal('0.00'),
                           'total_credit': Decimal('7840.00'),
                           'total_debit': Decimal('4312.00')},
                u'455': {'ecritures': [{'credit': Decimal('90.45'),
                                        'date': datetime.date(2011, 3, 18),
                                        'intitule': u'Frais de constitution de la société CFE Paris.'},
                                       {'credit': Decimal('99.00'),
                                        'date': datetime.date(2011, 3, 18),
                                        'intitule': u'Frais de constitution de la société - Annonce légale.'}],
                         'nom': u'Associés - Comptes courants',
                         'solde_crediteur': Decimal('189.45'),
                         'solde_debiteur': Decimal('0.00'),
                         'total_credit': Decimal('189.45'),
                         'total_debit': Decimal('0.00')},
                u'512': {'ecritures': [{'credit': Decimal('48.00'),
                                        'date': datetime.date(2011, 4, 1),
                                        'intitule': u'Résultat arr\xeaté compte'},
                                       {'date': datetime.date(2011, 4, 2),
                                        'debit': Decimal('1500.00'),
                                        'intitule': u'Capital initial Dépôt de 1500 € par Stanislas Guerra au LCL Ledru Rollin'},
                                       {'credit': Decimal('15.00'),
                                        'date': datetime.date(2011, 4, 28),
                                        'intitule': u'Cotisation Option PRO  LCL'},
                                       {'credit': Decimal('3.00'),
                                        'date': datetime.date(2011, 5, 2),
                                        'intitule': u'Abonnement LCL Access'},
                                       {'credit': Decimal('25.65'),
                                        'date': datetime.date(2011, 6, 11),
                                        'intitule': u'BHV - Fournitures des bureau (Livres comptables).'},
                                       {'date': datetime.date(2011, 6, 15),
                                        'debit': Decimal('1794.00'),
                                        'intitule': u'Remise chèque XXXXXXX règlement facture 2011-02'},
                                       {'credit': Decimal('3000.00'),
                                        'date': datetime.date(2011, 7, 5),
                                        'intitule': u'Traitement mois de juin gérant.'},
                                       {'credit': Decimal('393.00'),
                                        'date': datetime.date(2011, 8, 8),
                                        'intitule': u'Chèque XXXXXXY Cotisation trimestrielle RSI/Prévadiès.'},
                                       {'date': datetime.date(2011, 9, 2),
                                        'debit': Decimal('5980.00'),
                                        'intitule': u'Virement MyClient1 ZZZZZZZZZZZ Facture 2011-01'},
                                       {'credit': Decimal('3.00'),
                                        'date': datetime.date(2011, 9, 3),
                                        'intitule': u'Abonnement LCL Access'},
                                       {'credit': Decimal('1240.00'),
                                        'date': datetime.date(2011, 10, 6),
                                        'intitule': u'Chèque WWWWWWW Règlement de la TVA trimestrielle'},
                                       {'credit': Decimal('402.00'),
                                        'date': datetime.date(2011, 11, 17),
                                        'intitule': u'Chèque ZZZZZZZ Cotisation sociales Urssaf 4ème trimestre.'},
                                       {'credit': Decimal('3.00'),
                                        'date': datetime.date(2011, 12, 1),
                                        'intitule': u'Abonnement LCL Access'},
                                       {'date': datetime.date(2011, 12, 1),
                                        'debit': Decimal('18538.00'),
                                        'intitule': u'Virement MyClient1 WWWWWWWWWW Facture 2011-04, 2011-05'}],
                         'nom': u'Banques',
                         'solde_crediteur': Decimal('0.00'),
                         'solde_debiteur': Decimal('22679.35'),
                         'total_credit': Decimal('5132.65'),
                         'total_debit': Decimal('27812.00')},
                u'60225': {'ecritures': [{'date': datetime.date(2011, 6, 11),
                                          'debit': Decimal('21.44'),
                                          'intitule': u'BHV - Fournitures des bureau (Livres comptables).'}],
                           'nom': u'Achats - Fournitures de bureau',
                           'solde_crediteur': Decimal('0.00'),
                           'solde_debiteur': Decimal('21.44'),
                           'total_credit': Decimal('0.00'),
                           'total_debit': Decimal('21.44')},
                u'6227': {'ecritures': [{'date': datetime.date(2011, 3, 18),
                                         'debit': Decimal('80.00'),
                                         'intitule': u'Frais de constitution de la société CFE Paris.'},
                                        {'date': datetime.date(2011, 3, 18),
                                         'debit': Decimal('80.00'),
                                         'intitule': u'Frais de constitution de la société - Annonce légale.'}],
                          'nom': u"Achats - Frais d'actes et de contentieux",
                          'solde_crediteur': Decimal('0.00'),
                          'solde_debiteur': Decimal('160.00'),
                          'total_credit': Decimal('0.00'),
                          'total_debit': Decimal('160.00')},
                u'6278-LCL': {'ecritures': [{'date': datetime.date(2011, 4, 1),
                                             'debit': Decimal('48.00'),
                                             'intitule': u'Résultat arr\xeaté compte'},
                                            {'date': datetime.date(2011, 4, 28),
                                             'debit': Decimal('15.00'),
                                             'intitule': u'Cotisation Option PRO  LCL'},
                                            {'date': datetime.date(2011, 5, 2),
                                             'debit': Decimal('3.00'),
                                             'intitule': u'Abonnement LCL Access'},
                                            {'date': datetime.date(2011, 9, 3),
                                             'debit': Decimal('3.00'),
                                             'intitule': u'Abonnement LCL Access'},
                                            {'date': datetime.date(2011, 12, 1),
                                             'debit': Decimal('3.00'),
                                             'intitule': u'Abonnement LCL Access'}],
                              'nom': u'Autres frais de commission sur prestations de services',
                              'solde_crediteur': Decimal('0.00'),
                              'solde_debiteur': Decimal('72.00'),
                              'total_credit': Decimal('0.00'),
                              'total_debit': Decimal('72.00')},
                u'6411': {'ecritures': [{'date': datetime.date(2011, 7, 5),
                                         'debit': Decimal('3000.00'),
                                         'intitule': u'Traitement mois de juin gérant.'}],
                          'nom': u'Charges - Salaires et appointements',
                          'solde_crediteur': Decimal('0.00'),
                          'solde_debiteur': Decimal('3000.00'),
                          'total_credit': Decimal('0.00'),
                          'total_debit': Decimal('3000.00')},
                u'6411-RSI': {'ecritures': [{'date': datetime.date(2011, 8, 8),
                                             'debit': Decimal('393.00'),
                                             'intitule': u'Chèque XXXXXXY Cotisation trimestrielle RSI/Prévadiès.'}],
                              'nom': u'Charges - cotisations RSI',
                              'solde_crediteur': Decimal('0.00'),
                              'solde_debiteur': Decimal('393.00'),
                              'total_credit': Decimal('0.00'),
                              'total_debit': Decimal('393.00')},
                u'6411-URSF1': {'ecritures': [{'date': datetime.date(2011, 11, 17),
                                               'debit': Decimal('161.80'),
                                               'intitule': u'Chèque ZZZZZZZ Cotisation sociales Urssaf 4ème trimestre.'}],
                                'nom': u'Charges - cotisations URSSAF - Allocations familliales',
                                'solde_crediteur': Decimal('0.00'),
                                'solde_debiteur': Decimal('161.80'),
                                'total_credit': Decimal('0.00'),
                                'total_debit': Decimal('161.80')},
                u'6411-URSF2': {'ecritures': [{'date': datetime.date(2011, 11, 17),
                                               'debit': Decimal('153.31'),
                                               'intitule': u'Chèque ZZZZZZZ Cotisation sociales Urssaf 4ème trimestre.'}],
                                'nom': u'Charges - cotisations URSSAF - CSG/RDS déductible',
                                'solde_crediteur': Decimal('0.00'),
                                'solde_debiteur': Decimal('153.31'),
                                'total_credit': Decimal('0.00'),
                                'total_debit': Decimal('153.31')},
                u'6411-URSF3': {'ecritures': [{'date': datetime.date(2011, 11, 17),
                                               'debit': Decimal('86.89'),
                                               'intitule': u'Chèque ZZZZZZZ Cotisation sociales Urssaf 4ème trimestre.'}],
                                'nom': u'Charges - cotisations URSSAF - CSG/RDS non-déductible',
                                'solde_crediteur': Decimal('0.00'),
                                'solde_debiteur': Decimal('86.89'),
                                'total_credit': Decimal('0.00'),
                                'total_debit': Decimal('86.89')},
                u'706': {'ecritures': [{'credit': Decimal('5000.00'),
                                        'date': datetime.date(2011, 3, 31),
                                        'intitule': u'Facture 2011-01 MyClient1 Prestation MyClient1 mars 2011'},
                                       {'credit': Decimal('1500.00'),
                                        'date': datetime.date(2011, 4, 4),
                                        'intitule': u'Facture 2011-02 MyClient2 Prestation MyClient2'},
                                       {'credit': Decimal('11000.00'),
                                        'date': datetime.date(2011, 9, 28),
                                        'intitule': u'Facture 2011-04 MyClient1 Prestation aout 2011'},
                                       {'credit': Decimal('4500.00'),
                                        'date': datetime.date(2011, 11, 1),
                                        'intitule': u'Facture 2011-05 MyClient1 Prestation septembre 2011'},
                                       {'credit': Decimal('7000.00'),
                                        'date': datetime.date(2011, 12, 1),
                                        'intitule': u'Facture 2011-06 MyClient3'},
                                       {'credit': Decimal('11000.00'),
                                        'date': datetime.date(2011, 12, 31),
                                        'intitule': u'Prestation MyClient1 décembre 2011'}],
                         'nom': u'Produits - prestations de services',
                         'solde_crediteur': Decimal('40000.00'),
                         'solde_debiteur': Decimal('0.00'),
                         'total_credit': Decimal('40000.00'),
                         'total_debit': Decimal('0.00')},
                u'758': {'ecritures': [{'credit': Decimal('0.34'),
                                        'date': datetime.date(2011, 9, 30),
                                        'intitule': u'Solde des comptes de TVA du 01/03/2011 au 30/09/2011'}],
                         'nom': u'Produits divers de gestion courante',
                         'solde_crediteur': Decimal('0.34'),
                         'solde_debiteur': Decimal('0.00'),
                         'total_credit': Decimal('0.34'),
                         'total_debit': Decimal('0.00')}
            },
            'date_debut': datetime.date(2011, 3, 1),
            'date_fin': datetime.date(2011, 12, 31),
            'label': 'Grand-Livre 2011'
        })

    def test_grand_livre_N_plus_1(self):
        from colbert.grand_livre import grand_livre
        livre_journal = codecs.open(os.path.join(CURRENT_DIR, "livre-journal-N+1.txt"),
                                    mode="r", encoding="utf-8")
        grand_livre_precedent = codecs.open(os.path.join(CURRENT_DIR, "grand_livre-2011.json"),
                                            mode="r", encoding="utf-8")
        grand_livre_precedent = json.loads(grand_livre_precedent.read(), object_hook=as_ecriture)

        gl = grand_livre(livre_journal, "Grand-Livre 2012",
                         date_debut=datetime.date(2012, 1, 1),
                         date_fin=datetime.date(2012, 12, 31),
                         grand_livre_precedent=grand_livre_precedent)

        # Uncomment to generate the JSON test file.
        # output = codecs.open(os.path.join(CURRENT_DIR, "grand_livre-2012.json"),
        #                               mode="w+", encoding="utf-8")
        # output.write(json.dumps(gl, default=json_encoder, indent=4))
        # output.close
        # pprint(gl)
        self.maxDiff = None
        self.assertEqual(gl, {
            u'comptes': {
                u'100': {u'ecritures': [{u'credit': Decimal('1500.00'),
                                         'date': datetime.date(2012, 1, 1),
                                         'intitule': u'Report à nouveau'}],
                         u'nom': u"Capital et compte de l'exploitant",
                         u'solde_crediteur': Decimal('1500.00'),
                         u'solde_debiteur': Decimal('0.00'),
                         u'total_credit': Decimal('1500.00'),
                         u'total_debit': Decimal('0.00')},
                u'1061': {u'ecritures': [{u'credit': Decimal('150.00'),
                                          'date': datetime.date(2011, 12, 31),
                                          'intitule': u'Affectation du résultat'}],
                          u'nom': u'Réserve légale',
                          u'solde_crediteur': Decimal('150.00'),
                          u'solde_debiteur': Decimal('0.00'),
                          u'total_credit': Decimal('150.00'),
                          u'total_debit': Decimal('0.00')},
                u'110': {u'ecritures': [{u'credit': Decimal('10801.90'),
                                         'date': datetime.date(2011, 12, 31),
                                         'intitule': u'Affectation du résultat'}],
                         u'nom': u'Report à nouveau - Solde créditeur',
                         u'solde_crediteur': Decimal('10801.90'),
                         u'solde_debiteur': Decimal('0.00'),
                         u'total_credit': Decimal('10801.90'),
                         u'total_debit': Decimal('0.00')},
                u'120': {u'ecritures': [{u'credit': Decimal('35951.90'),
                                         'date': datetime.date(2011, 12, 31),
                                         'intitule': u"Enregistrement du résultat net de l'exercice"},
                                        {'date': datetime.date(2011, 12, 31),
                                         u'debit': Decimal('35951.90'),
                                         'intitule': u'Affectation du résultat'}],
                         u'nom': u"résultat de l'exercice (bénéfice)",
                         u'solde_crediteur': Decimal('0.00'),
                         u'solde_debiteur': Decimal('0.00'),
                         u'total_credit': Decimal('35951.90'),
                         u'total_debit': Decimal('35951.90')},
                u'126': {u'ecritures': [{'date': datetime.date(2011, 12, 31),
                                         u'debit': Decimal('4048.44'),
                                         'intitule': u'Ecritures de clôture des comptes.'},
                                        {u'credit': Decimal('4048.44'),
                                         'date': datetime.date(2011, 12, 31),
                                         'intitule': u"Enregistrement du résultat net de l'exercice"}],
                         u'nom': u'Regroupement des comptes de charges',
                         u'solde_crediteur': Decimal('0.00'),
                         u'solde_debiteur': Decimal('0.00'),
                         u'total_credit': Decimal('4048.44'),
                         u'total_debit': Decimal('4048.44')},
                u'127': {u'ecritures': [{u'credit': Decimal('40000.34'),
                                         'date': datetime.date(2011, 12, 31),
                                         'intitule': u'Ecritures de clôture des comptes.'},
                                        {'date': datetime.date(2011, 12, 31),
                                         u'debit': Decimal('40000.34'),
                                         'intitule': u"Enregistrement du résultat net de l'exercice"}],
                         u'nom': u'Regroupement des comptes de produits',
                         u'solde_crediteur': Decimal('0.00'),
                         u'solde_debiteur': Decimal('0.00'),
                         u'total_credit': Decimal('40000.34'),
                         u'total_debit': Decimal('40000.34')},
                u'4111-CL1': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                              u'debit': Decimal('0.00'),
                                              'intitule': u'Report à nouveau'},
                                             {'date': datetime.date(2012, 1, 3),
                                              u'debit': Decimal('13156.00'),
                                              'intitule': u'Facture 2012-01 MyClient1 Prestation décembre 2011'}],
                              u'nom': u'Clients - ventes de biens ou prestations de services',
                              u'solde_crediteur': Decimal('0.00'),
                              u'solde_debiteur': Decimal('13156.00'),
                              u'total_credit': Decimal('0.00'),
                              u'total_debit': Decimal('13156.00')},
                u'4111-CL2': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                              u'debit': Decimal('0.00'),
                                              'intitule': u'Report à nouveau'},
                                             {'date': datetime.date(2012, 3, 20),
                                              u'debit': Decimal('13156.00'),
                                              'intitule': u'Facture 2012-03 MyClient2 Prestation février 2012'}],
                              u'nom': u'Clients - ventes de biens ou prestations de services',
                              u'solde_crediteur': Decimal('0.00'),
                              u'solde_debiteur': Decimal('13156.00'),
                              u'total_credit': Decimal('0.00'),
                              u'total_debit': Decimal('13156.00')},
                u'4111-CL3': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                              u'debit': Decimal('8372.00'),
                                              'intitule': u'Report à nouveau'}],
                              u'nom': u'Clients - ventes de biens ou prestations de services',
                              u'solde_crediteur': Decimal('0.00'),
                              u'solde_debiteur': Decimal('8372.00'),
                              u'total_credit': Decimal('0.00'),
                              u'total_debit': Decimal('8372.00')},
                u'4181': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                          u'debit': Decimal('13156.00'),
                                          'intitule': u'Report à nouveau'},
                                         {u'credit': Decimal('13156.00'),
                                          'date': datetime.date(2012, 1, 1),
                                          'intitule': u'Prestation MyClient1 décembre 2011'}],
                          u'nom': u'Clients - Factures à établir',
                          u'solde_crediteur': Decimal('0.00'),
                          u'solde_debiteur': Decimal('0.00'),
                          u'total_credit': Decimal('13156.00'),
                          u'total_debit': Decimal('13156.00')},
                u'444': {u'ecritures': [{u'credit': Decimal('9800.00'),
                                         'date': datetime.date(2011, 12, 31),
                                         'intitule': u'Impôts sur les bénéfices.'}],
                         u'nom': u'État-Impôts sur les bénéfices',
                         u'solde_crediteur': Decimal('9800.00'),
                         u'solde_debiteur': Decimal('0.00'),
                         u'total_credit': Decimal('9800.00'),
                         u'total_debit': Decimal('0.00')},
                u'44551': {u'ecritures': [{u'credit': Decimal('3038.00'),
                                           'date': datetime.date(2012, 1, 1),
                                           'intitule': u'Report à nouveau'}],
                           u'nom': u'TVA à décaisser',
                           u'solde_crediteur': Decimal('3038.00'),
                           u'solde_debiteur': Decimal('0.00'),
                           u'total_credit': Decimal('3038.00'),
                           u'total_debit': Decimal('0.00')},
                u'44566': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                           u'debit': Decimal('0.00'),
                                           'intitule': u'Report à nouveau'},
                                          {'date': datetime.date(2012, 2, 1),
                                           u'debit': Decimal('3.78'),
                                           'intitule': u"Restaurant La Tour d'argent Déjeuner d'affaire avec Steve Jobs 0.88€ TVA 19.6% ; 2.90€ TVA 7.0%"}],
                           u'nom': u'T.V.A. déductible sur autres biens et services',
                           u'solde_crediteur': Decimal('0.00'),
                           u'solde_debiteur': Decimal('3.78'),
                           u'total_credit': Decimal('0.00'),
                           u'total_debit': Decimal('3.78')},
                u'44571': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                           u'debit': Decimal('0.00'),
                                           'intitule': u'Report à nouveau'}],
                           u'nom': u'T.V.A. Collectée',
                           u'solde_crediteur': Decimal('0.00'),
                           u'solde_debiteur': Decimal('0.00'),
                           u'total_credit': Decimal('0.00'),
                           u'total_debit': Decimal('0.00')},
                u'44587': {u'ecritures': [{u'credit': Decimal('3528.00'),
                                           'date': datetime.date(2012, 1, 1),
                                           'intitule': u'Report à nouveau'},
                                          {'date': datetime.date(2012, 1, 1),
                                           u'debit': Decimal('2156.00'),
                                           'intitule': u'Prestation MyClient1 décembre 2011'},
                                          {u'credit': Decimal('2156.00'),
                                           'date': datetime.date(2012, 1, 3),
                                           'intitule': u'Facture 2012-01 MyClient1 Prestation décembre 2011'},
                                          {u'credit': Decimal('2156.00'),
                                           'date': datetime.date(2012, 3, 20),
                                           'intitule': u'Facture 2012-03 MyClient2 Prestation février 2012'}],
                           u'nom': u'Taxes sur le CA sur factures à établir',
                           u'solde_crediteur': Decimal('5684.00'),
                           u'solde_debiteur': Decimal('0.00'),
                           u'total_credit': Decimal('7840.00'),
                           u'total_debit': Decimal('2156.00')},
                u'455': {u'ecritures': [{u'credit': Decimal('189.45'),
                                         'date': datetime.date(2012, 1, 1),
                                         'intitule': u'Report à nouveau'}],
                         u'nom': u'Associés - Comptes courants',
                         u'solde_crediteur': Decimal('189.45'),
                         u'solde_debiteur': Decimal('0.00'),
                         u'total_credit': Decimal('189.45'),
                         u'total_debit': Decimal('0.00')},
                u'457': {u'ecritures': [{u'credit': Decimal('25000.00'),
                                         'date': datetime.date(2011, 12, 31),
                                         'intitule': u'Affectation du résultat'}],
                         u'nom': u'Associés - Dividendes à payer',
                         u'solde_crediteur': Decimal('25000.00'),
                         u'solde_debiteur': Decimal('0.00'),
                         u'total_credit': Decimal('25000.00'),
                         u'total_debit': Decimal('0.00')},
                u'512': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                         u'debit': Decimal('22679.35'),
                                         'intitule': u'Report à nouveau'},
                                        {u'credit': Decimal('49.80'),
                                         'date': datetime.date(2012, 2, 1),
                                         'intitule': u"Restaurant La Tour d'argent Déjeuner d'affaire avec Steve Jobs 0.88€ TVA 19.6% ; 2.90€ TVA 7.0%"}],
                         u'nom': u'Banques',
                         u'solde_crediteur': Decimal('0.00'),
                         u'solde_debiteur': Decimal('22629.55'),
                         u'total_credit': Decimal('49.80'),
                         u'total_debit': Decimal('22679.35')},
                u'60225': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                           u'debit': Decimal('21.44'),
                                           'intitule': u'Report à nouveau'},
                                          {u'credit': Decimal('21.44'),
                                           'date': datetime.date(2011, 12, 31),
                                           'intitule': u'Ecritures de clôture des comptes.'}],
                           u'nom': u'Achats - Fournitures de bureau',
                           u'solde_crediteur': Decimal('0.00'),
                           u'solde_debiteur': Decimal('0.00'),
                           u'total_credit': Decimal('21.44'),
                           u'total_debit': Decimal('21.44')},
                u'6227': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                          u'debit': Decimal('160.00'),
                                          'intitule': u'Report à nouveau'},
                                         {u'credit': Decimal('160.00'),
                                          'date': datetime.date(2011, 12, 31),
                                          'intitule': u'Ecritures de clôture des comptes.'}],
                          u'nom': u"Achats - Frais d'actes et de contentieux",
                          u'solde_crediteur': Decimal('0.00'),
                          u'solde_debiteur': Decimal('0.00'),
                          u'total_credit': Decimal('160.00'),
                          u'total_debit': Decimal('160.00')},
                u'6257': {u'ecritures': [{'date': datetime.date(2012, 2, 1),
                                          u'debit': Decimal('46.02'),
                                          'intitule': u"Restaurant La Tour d'argent Déjeuner d'affaire avec Steve Jobs 0.88€ TVA 19.6% ; 2.90€ TVA 7.0%"}],
                          u'nom': u'Charges - Réceptions',
                          u'solde_crediteur': Decimal('0.00'),
                          u'solde_debiteur': Decimal('46.02'),
                          u'total_credit': Decimal('0.00'),
                          u'total_debit': Decimal('46.02')},
                u'6278-LCL': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                              u'debit': Decimal('72.00'),
                                              'intitule': u'Report à nouveau'},
                                             {u'credit': Decimal('72.00'),
                                              'date': datetime.date(2011, 12, 31),
                                              'intitule': u'Ecritures de clôture des comptes.'}],
                              u'nom': u'Autres frais de commission sur prestations de services',
                              u'solde_crediteur': Decimal('0.00'),
                              u'solde_debiteur': Decimal('0.00'),
                              u'total_credit': Decimal('72.00'),
                              u'total_debit': Decimal('72.00')},
                u'6411': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                          u'debit': Decimal('3000.00'),
                                          'intitule': u'Report à nouveau'},
                                         {u'credit': Decimal('3000.00'),
                                          'date': datetime.date(2011, 12, 31),
                                          'intitule': u'Ecritures de clôture des comptes.'}],
                          u'nom': u'Charges - Salaires et appointements',
                          u'solde_crediteur': Decimal('0.00'),
                          u'solde_debiteur': Decimal('0.00'),
                          u'total_credit': Decimal('3000.00'),
                          u'total_debit': Decimal('3000.00')},
                u'6411-RSI': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                              u'debit': Decimal('393.00'),
                                              'intitule': u'Report à nouveau'},
                                             {u'credit': Decimal('393.00'),
                                              'date': datetime.date(2011, 12, 31),
                                              'intitule': u'Ecritures de clôture des comptes.'}],
                              u'nom': u'Charges - cotisations RSI',
                              u'solde_crediteur': Decimal('0.00'),
                              u'solde_debiteur': Decimal('0.00'),
                              u'total_credit': Decimal('393.00'),
                              u'total_debit': Decimal('393.00')},
                u'6411-URSF1': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                                u'debit': Decimal('161.80'),
                                                'intitule': u'Report à nouveau'},
                                               {u'credit': Decimal('161.80'),
                                                'date': datetime.date(2011, 12, 31),
                                                'intitule': u'Ecritures de clôture des comptes.'}],
                                u'nom': u'Charges - cotisations URSSAF - Allocations familliales',
                                u'solde_crediteur': Decimal('0.00'),
                                u'solde_debiteur': Decimal('0.00'),
                                u'total_credit': Decimal('161.80'),
                                u'total_debit': Decimal('161.80')},
                u'6411-URSF2': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                                u'debit': Decimal('153.31'),
                                                'intitule': u'Report à nouveau'},
                                               {u'credit': Decimal('153.31'),
                                                'date': datetime.date(2011, 12, 31),
                                                'intitule': u'Ecritures de clôture des comptes.'}],
                                u'nom': u'Charges - cotisations URSSAF - CSG/RDS déductible',
                                u'solde_crediteur': Decimal('0.00'),
                                u'solde_debiteur': Decimal('0.00'),
                                u'total_credit': Decimal('153.31'),
                                u'total_debit': Decimal('153.31')},
                u'6411-URSF3': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                                u'debit': Decimal('86.89'),
                                                'intitule': u'Report à nouveau'},
                                               {u'credit': Decimal('86.89'),
                                                'date': datetime.date(2011, 12, 31),
                                                'intitule': u'Ecritures de clôture des comptes.'}],
                                u'nom': u'Charges - cotisations URSSAF - CSG/RDS non-déductible',
                                u'solde_crediteur': Decimal('0.00'),
                                u'solde_debiteur': Decimal('0.00'),
                                u'total_credit': Decimal('86.89'),
                                u'total_debit': Decimal('86.89')},
                u'695': {u'ecritures': [{'date': datetime.date(2011, 12, 31),
                                         u'debit': Decimal('9800.00'),
                                         'intitule': u'Impôts sur les bénéfices.'}],
                         u'nom': u'Impôts sur les bénéfices',
                         u'solde_crediteur': Decimal('0.00'),
                         u'solde_debiteur': Decimal('9800.00'),
                         u'total_credit': Decimal('0.00'),
                         u'total_debit': Decimal('9800.00')},
                u'706': {u'ecritures': [{u'credit': Decimal('40000.00'),
                                         'date': datetime.date(2012, 1, 1),
                                         'intitule': u'Report à nouveau'},
                                        {'date': datetime.date(2011, 12, 31),
                                         u'debit': Decimal('40000.00'),
                                         'intitule': u'Ecritures de clôture des comptes.'},
                                        {'date': datetime.date(2012, 1, 1),
                                         u'debit': Decimal('11000.00'),
                                         'intitule': u'Prestation MyClient1 décembre 2011'},
                                        {u'credit': Decimal('11000.00'),
                                         'date': datetime.date(2012, 1, 3),
                                         'intitule': u'Facture 2012-01 MyClient1 Prestation décembre 2011'},
                                        {u'credit': Decimal('11000.00'),
                                         'date': datetime.date(2012, 3, 20),
                                         'intitule': u'Facture 2012-03 MyClient2 Prestation février 2012'}],
                         u'nom': u'Produits - prestations de services',
                         u'solde_crediteur': Decimal('11000.00'),
                         u'solde_debiteur': Decimal('0.00'),
                         u'total_credit': Decimal('62000.00'),
                         u'total_debit': Decimal('51000.00')},
                u'758': {u'ecritures': [{u'credit': Decimal('0.34'),
                                         'date': datetime.date(2012, 1, 1),
                                         'intitule': u'Report à nouveau'},
                                        {'date': datetime.date(2011, 12, 31),
                                         u'debit': Decimal('0.34'),
                                         'intitule': u'Ecritures de clôture des comptes.'}],
                         u'nom': u'Produits divers de gestion courante',
                         u'solde_crediteur': Decimal('0.00'),
                         u'solde_debiteur': Decimal('0.00'),
                         u'total_credit': Decimal('0.34'),
                         u'total_debit': Decimal('0.34')}
            },
            u'date_debut': datetime.date(2012, 1, 1),
            u'date_fin': datetime.date(2012, 12, 31),
            u'label': 'Grand-Livre 2012'
        })

    def test_init_comptes_grand_livre_avec_precedent(self):
        from colbert.grand_livre import init_comptes_grand_livre_avec_precedent
        grand_livre_precedent = codecs.open(os.path.join(CURRENT_DIR, "grand_livre-2011.json"),
                                            mode="r", encoding="utf-8")
        comptes = init_comptes_grand_livre_avec_precedent(json.loads(grand_livre_precedent.read()),
                                                          datetime.date(2012, 1, 1))
        self.maxDiff = None
        self.assertEqual(
            comptes, {
                u'100': {u'ecritures': [{u'credit': Decimal('1500.00'),
                                         'date': datetime.date(2012, 1, 1),
                                         'intitule': u'Report à nouveau'}],
                         u'nom': u"Capital et compte de l'exploitant"},
                u'4111-CL1': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                              u'debit': Decimal('0.00'),
                                              'intitule': u'Report à nouveau'}],
                              u'nom': u'Clients - ventes de biens ou prestations de services'},
                u'4111-CL2': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                              u'debit': Decimal('0.00'),
                                              'intitule': u'Report à nouveau'}],
                              u'nom': u'Clients - ventes de biens ou prestations de services'},
                u'4111-CL3': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                              u'debit': Decimal('8372.00'),
                                              'intitule': u'Report à nouveau'}],
                              u'nom': u'Clients - ventes de biens ou prestations de services'},
                u'4181': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                          u'debit': Decimal('13156.00'),
                                          'intitule': u'Report à nouveau'}],
                          u'nom': u'Clients - Factures à établir'},
                u'44551': {u'ecritures': [{u'credit': Decimal('3038.00'),
                                           'date': datetime.date(2012, 1, 1),
                                           'intitule': u'Report à nouveau'}],
                           u'nom': u'TVA à décaisser'},
                u'44566': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                           u'debit': Decimal('0.00'),
                                           'intitule': u'Report à nouveau'}],
                           u'nom': u'T.V.A. déductible sur autres biens et services'},
                u'44571': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                           u'debit': Decimal('0.00'),
                                           'intitule': u'Report à nouveau'}],
                           u'nom': u'T.V.A. Collectée'},
                u'44587': {u'ecritures': [{u'credit': Decimal('3528.00'),
                                           'date': datetime.date(2012, 1, 1),
                                           'intitule': u'Report à nouveau'}],
                           u'nom': u'Taxes sur le CA sur factures à établir'},
                u'455': {u'ecritures': [{u'credit': Decimal('189.45'),
                                         'date': datetime.date(2012, 1, 1),
                                         'intitule': u'Report à nouveau'}],
                         u'nom': u'Associés - Comptes courants'},
                u'512': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                         u'debit': Decimal('22679.35'),
                                         'intitule': u'Report à nouveau'}],
                         u'nom': u'Banques'},
                u'60225': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                           u'debit': Decimal('21.44'),
                                           'intitule': u'Report à nouveau'}],
                           u'nom': u'Achats - Fournitures de bureau'},
                u'6227': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                          u'debit': Decimal('160.00'),
                                          'intitule': u'Report à nouveau'}],
                          u'nom': u"Achats - Frais d'actes et de contentieux"},
                u'6278-LCL': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                              u'debit': Decimal('72.00'),
                                              'intitule': u'Report à nouveau'}],
                              u'nom': u'Autres frais de commission sur prestations de services'},
                u'6411': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                          u'debit': Decimal('3000.00'),
                                          'intitule': u'Report à nouveau'}],
                          u'nom': u'Charges - Salaires et appointements'},
                u'6411-RSI': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                              u'debit': Decimal('393.00'),
                                              'intitule': u'Report à nouveau'}],
                              u'nom': u'Charges - cotisations RSI'},
                u'6411-URSF1': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                                u'debit': Decimal('161.80'),
                                                'intitule': u'Report à nouveau'}],
                                u'nom': u'Charges - cotisations URSSAF - Allocations familliales'},
                u'6411-URSF2': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                                u'debit': Decimal('153.31'),
                                                'intitule': u'Report à nouveau'}],
                                u'nom': u'Charges - cotisations URSSAF - CSG/RDS déductible'},
                u'6411-URSF3': {u'ecritures': [{'date': datetime.date(2012, 1, 1),
                                                u'debit': Decimal('86.89'),
                                                'intitule': u'Report à nouveau'}],
                                u'nom': u'Charges - cotisations URSSAF - CSG/RDS non-déductible'},
                u'706': {u'ecritures': [{u'credit': Decimal('40000.00'),
                                         'date': datetime.date(2012, 1, 1),
                                         'intitule': u'Report à nouveau'}],
                         u'nom': u'Produits - prestations de services'},
                u'758': {u'ecritures': [{u'credit': Decimal('0.34'),
                                         'date': datetime.date(2012, 1, 1),
                                         'intitule': u'Report à nouveau'}],
                         u'nom': u'Produits divers de gestion courante'}
            }
        )

    def test_ecriture_journal_in_grand_livre(self):
        from colbert.grand_livre import ecriture_journal_in_grand_livre
        grand_livre = codecs.open(os.path.join(CURRENT_DIR, "grand_livre-2011.json"),
                                  mode="r", encoding="utf-8")
        grand_livre = json.loads(grand_livre.read(), object_hook=as_ecriture)

        ecriture = {'credit': Decimal('11000'),
                    'debit': Decimal('0.00'),
                    'nom_compte': u'Produits - prestations de services',
                    'numero_compte_credit': u'706',
                    'numero_compte_debit': u''}
        intitule = u'Prestation MyClient1 décembre 2011'
        date = datetime.date(2011, 12, 31)
        self.assertTrue(ecriture_journal_in_grand_livre(ecriture, date, intitule, grand_livre))

        ecriture['credit'] = Decimal('9000')
        self.assertFalse(ecriture_journal_in_grand_livre(ecriture, date, intitule, grand_livre))

    def test_grand_livre_to_rst(self):
        from colbert.grand_livre import grand_livre_to_rst
        grand_livre = codecs.open(os.path.join(CURRENT_DIR, "grand_livre-2011.json"),
                                  mode="r", encoding="utf-8")
        output = StringIO.StringIO()
        # Uncomment to generate the file.
        # output = codecs.open(os.path.join(CURRENT_DIR, "grand_livre-2011.txt"),
        #                               mode="w+", encoding="utf-8")
        grand_livre_to_rst(json.loads(grand_livre.read()), output)
        grand_livre_txt = codecs.open(os.path.join(CURRENT_DIR, "grand_livre-2011.txt"),
                                      mode="r", encoding="utf-8")
        self.maxDiff = None
        self.assertEqual(output.getvalue(), grand_livre_txt.read())


def suite():
    suite = unittest.TestLoader().loadTestsFromTestCase(GrandLivreTestCase)
    return suite
