# coding: utf-8
"""pdfminer_xml_bindings.py:
Provides functions to convert the DOM tree generated by pdfminer to
proper Python objects.
"""

import operator
import unicodedata

from confopy.pdfextract import xml_util

# Constants

SVG_HEADER = u"""<?xml version="1.0" standalone="no"?>
<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN"
"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">

<svg width="800px" height="1000px" version="1.1"
 xmlns="http://www.w3.org/2000/svg">
<style type="text/css" >
<![CDATA[
ellipse, rect {
    stroke: #000000;
    fill:   #ffffff;
}
.textgroup {
    stroke: none;
    fill-opacity: 0.2;
}
]]>
</style>
"""

SVG_FOOTER = u"""</svg>
"""

SVG_TOP_PADDING = 16
SVG_LEFT_PADDING = 16


# Classes

class Box(object):
    """Super class for all objects that have a bounding box (and text).
    """

    def __init__(self, bbox):
        self.bbox = bbox

    def width(self):
        return self.bbox[2] - self.bbox[0]

    def height(self):
        return self.bbox[3] - self.bbox[1]

    def as_svg(self):
        return bbox2svg(self.bbox)

    def _print(self, pad=u""):
        pass

#    def as_textboxes(self):
#        return []
#
#    def as_lines(self):
#        return []

    def __repr__(self):
        return "Box(bbox=" + repr(self.bbox) + ")"


class TextBox(Box):
    def __init__(self, ID, bbox, lines, font, emph):
        """Constructor.
        Args:
            ID:       ID of the textbox.
            bbox:     Bounding box of the textbox.
            lines:    List of strings representing the lines in the textbox.
            font:     Tuple of dominant font and fontsize in the textbox.
            fontsize: Dominant font size in the textbox.
            emph:     List of words (string) showing highlighted words of
                      argument "lines".
        """
        super(TextBox, self).__init__(bbox)
        self.ID = ID
        self.lines = lines
        self.line_count = len(lines)
        self.font = font
        self.word_count = reduce(operator.add, map(lambda line: len(line.split()), lines), 0)
        self.character_count = reduce(operator.add, map(lambda line: len(line), lines), 0)
        self.emph = emph

#    def as_lines(self):
#        return self.lines

    def _print(self, pad=u""):
        if self.lines == []:
            print(pad + u"<TextBox ID=%s>" % (self.ID))
        else:
            print(pad + u"<TextBox ID=%s #w=%s %s>" % (self.ID, self.word_count, self.font))
            for line in self.lines:
                print(pad + u"  " + line)
                #for c in line:
                #    print repr(c)
            if len(self.emph) > 0:
                print(pad + u"  <emph=[" + reduce(lambda a,b: a+u", "+b, self.emph) + "]>")

    def __unicode__(self):
        if self.lines == []:
            return u"## Empty (Layout-)Textbox " + self.ID
        if self.ID != "":
            return u"## Textbox " + self.ID + u" (%s words, %s, %s) ##\n" % (self.word_count, self.font, self.fontsize) + u"\n".join(self.lines)
        return u"## Textbox (%s words, %s, %s) ##\n" % (self.word_count, self.font, self.fontsize) + u"\n".join(self.lines)
        #return "\n".join(unicodedata.normalize("NFKD", l) \
        #                 .encode("latin1", "replace") for l in self.lines)

    def __repr__(self):
        return "TextBox(super=%r,ID=%r,lines=%r,emph=%r)" \
                % (super(TextBox, self).__repr__(), \
                   self.ID, self.lines, self.emph)


class Page(Box):
    """A page is a collection of textboxes with hierarchical layout
    information (textgroups).
    Attributes:
        ID: Usually integer ID of the page.
        textboxes: List of Textbox objects.
        layout:    Root textgroup having multiple textgroups/dummy textboxes
                   as children.
        prim_font: Triple of primary font, primary fontsize and #words using it.
                   Primary font is the font used for most words on the page
                   Types: (unicode, unicode, int)
        word_count: Number of words on the page/in the textboxes.
    """
    def __init__(self, ID, bbox, textboxes, layout=None):
        """Constructor.
        """
        super(Page, self).__init__(bbox)
        self.ID = ID
        self.textboxes = textboxes
        self._textboxes_by_ID = dict()
        for tb in textboxes:
            self._textboxes_by_ID[tb.ID] = tb
        self.layout = layout
        self.prim_font = find_primary_font(textboxes=textboxes)
        self.word_count = reduce(operator.add, map(lambda tb: tb.word_count, textboxes), 0)
        #print unicode(self).encode("utf-8")

    def get_textbox(self, tbID):
        try:
            return self._textboxes_by_ID[tbID]
        except KeyError:
            print u"Page %s does not contain textbox with ID %s" % (self.ID, tbID)
            return None

    def get_siblings(self, tb):
        raise NotImplementedError("get textbox siblings not yet implemented")

    def is_sibling(self, tb, mb_sib, aunteded_by=None):
        """is_sibling
        Args:
            tb:        A TextBox.
            mb_sib:    Another TextBox. The mission is to check whether mb_sib
                       is a sibling of tb (= is in the same TextGroup).
            aunted_by: Additional TextBox constraining the check.
                       mb_sib must not only be a sibling of tb but must also be
                       in a TextGroup that is a sibling to aunted_by.
        Return:
            True if mb_sib is a sibling of tb. If aunted_by parameter is used
            mb_sib's and tb's parent TextGroup must be a sibling of aunted_by.
            False otherwise.
        """
        tg = self.get_textgroup(tb)
        if tg and tg.has_child(mb_sib):
            if not aunteded_by:
                return True
            elif aunteded_by and tg.parent:
                return tg.parent.has_child(aunteded_by)
            else:
                return False
        return False

    def get_textgroup(self, tb, group=None):
        """Gets the TextGroup that contains the passed TextBox.
        Args:
            tb: TextBox for which the TextGroup should be returned.
            group: TextGroup for recursion.
        Return:
            TextGroup
        """
        group = group or self.layout
        if group:
            for c in group.children:
                if isinstance(c, TextBox):
                    if c.ID == tb.ID:
                        return group
                elif isinstance(c, TextGroup):
                    rec = self.get_textgroup(tb, c)
                    if rec:
                        return rec
        return None


    def as_svg(self):
        svg = [SVG_HEADER]
        str_fmt = '<g transform="translate(%s, %s) scale(1, -1)">\n'
        svg.append(str_fmt % (SVG_LEFT_PADDING, self.height() + SVG_TOP_PADDING))
        svg.append(bbox2svg(self.bbox) + '\n')
        for box in self.textboxes:
            svg.append(box.as_svg() + '\n')
            for group in self.textgroups:
                svg.append(group.as_svg() + '\n')
                svg.append('</g>\n')
                svg.append(SVG_FOOTER)
                svg = ''.join(svg)
                return svg

#    def as_textboxes(self):
#        return self.textboxes

    def _print(self, pad=u""):
        print(pad + u"<Page ID=%s #w=%s prim_font=%s>" % (self.ID, self.word_count, self.prim_font))
        for tb in self.textboxes:
            tb._print(pad + u"  ")
        if self.layout:
            self.layout._print(pad + u"  ")


    def __unicode__(self):
        buf = list()
        buf.append((u"# Page %s (%s words, %s, %s)#" % (self.ID, self.word_count, self.font, self.fontsize)))
        for tb in self.textboxes:
            tb_str = unicode(tb)
            if tb_str != u"":
                buf.append(tb_str)
                return u"\n".join(buf)

    def __repr__(self):
        return "Page(super=%r,ID=%r,textboxes=%r,textgroups=%r)" \
                % (super(Page, self).__repr__(),
                   self.ID, self.textboxes, self.textgroups)


class TextGroup(Box):
    def __init__(self, bbox, children=[]):
        super(TextGroup, self).__init__(bbox)
        self.parent = None
        self.children = list(children)
        for c in self.children:
            c.parent = self

    def has_child(self, tb):
        for c in self.children:
            if isinstance(c, TextBox) and c.ID == tb.ID:
                return True
        return False

    def as_svg(self):
        import random
        rnd = random.random()
        rnd_color = int(999999.0 * rnd)
        return bbox2svg(self.bbox, rnd_color)

    def _print(self, pad=u""):
        print(pad + u"<TextGroup>")
        for c in self.children:
            c._print(pad + u"  ")

    def __repr__(self):
        return "TextGroup(super=%r,children=%r)" \
                % (super(TextGroup, self).__repr__(), self.children)


########################################################################
# Converter functions
########################################################################

## File, string and SVG functions

def DOM2pages(dom_document):
    dom_pages = dom_document.getElementsByTagName("page")
    # Forget about multiprocessing here.
    # DOM objects seems to have some side effects not allowing this.
    return map(DOM2page, dom_pages)


def str2bbox(bbstr):
    """Converts a bounding box string to the 4 corresponding float values
    Example:
        String: 23.000,42.000,25.000,50.000
        BBox: (x1, y1, x2, y2) = (23.0, 42.0, 25.0, 50.0)
    """
    return tuple(map(float, bbstr.split(",")))


def bbox2svg(bbox, fill_color=""):
    x = bbox[0]
    y = bbox[1]
    width = bbox[2] - x
    height = bbox[3] - y
    if fill_color == "":
        str_fmt = '<rect x="%s" y="%s" width="%s" height="%s" />'
        return str_fmt % (x, y, width, height)
    else:
        str_fmt = '<rect x="%s" y="%s" width="%s" height="%s" style="fill:#%s"/>'
        return str_fmt % (x, y, width, height, fill_color)


## DOM functions

#import re
#PDFMINER_CID = r"\(cid:\d+\)"

def DOM2textline(dom_textline):
    """Converts a DOM textline element to a tuple.
    First entry of the tuple represents the textline as a string.
    The second entry is a list of strings marking highlighted words in
    the textline.
    """
    dom_letters = dom_textline.getElementsByTagName("text")
    emph = list()
    letters = list()
    fonts = list()
    sizes = list()
    font = u""
    size = u""
    next_letter_uml = False
    for dom_letter in dom_letters:
        letter = u""
        if dom_letter.firstChild:
            letter = dom_letter.firstChild.nodeValue

        if dom_letter.hasAttributes() or letter.strip() != "":
            font = unicode(dom_letter.getAttribute("font"))
            size = unicode(dom_letter.getAttribute("size"))
            #if re.match(PDFMINER_CID, letter):
            #    print "FOUND CID: %s" % letter
            letter = _escape_pdfminer_cid(letter)
            if letter == u"(cid:127)":
                next_letter_uml = True
            else:
                if next_letter_uml:
                    letter = _convert2uml(letter)
                    next_letter_uml = False
                for c in letter:
                    letters.append(unicode(c))
                    fonts.append(font)
                    sizes.append(size)
        else:
            letters.append(u" ")
            fonts.append(font)
            sizes.append(size)
    line = u"".join(letters)
    #emph = find_emphasis(line, fonts, sizes)
    return (line, fonts, sizes)

_CID_MAP = { \
      u"(cid:12)": u"fi" \
    , u"(cid:13)": u"fl" \
    , u"(cid:25)": u"ß"  \
}
def _escape_pdfminer_cid(letter):
    return _CID_MAP.get(letter, letter)

def _convert2uml(letter):
    if letter == u"a":
        return u"ä"
    elif letter == u"o":
        return u"ö"
    elif letter == u"u":
        return u"ü"
    return letter

def DOM2textbox(dom_textbox):
    ID = unicode(dom_textbox.getAttribute("id"))
    bbox = str2bbox(dom_textbox.getAttribute("bbox"))
    dom_lines = dom_textbox.getElementsByTagName("textline")
    lines_fonts_sizes = map(DOM2textline, dom_lines)
    lines = map(lambda t: t[0], lines_fonts_sizes)
    fonts = map(lambda t: t[1], lines_fonts_sizes)
    sizes = map(lambda t: t[2], lines_fonts_sizes)
    def join_str(a, b):
        return a + u"\n" + b
    def join_lst(a, b):
        return a + [u"\n"] + b
    all_lines = reduce(join_str, lines, u"")
    all_fonts = reduce(join_lst, fonts, [])
    all_sizes = reduce(join_lst, sizes, [])
    (primary_font, primary_size, emph) = find_emphasis(all_lines, all_fonts, all_sizes)
    lines = filter(lambda l: l.strip() != u"", lines)
    return TextBox(ID, bbox, lines, (primary_font, primary_size), emph)

def find_emphasis(line, fonts, sizes):
    """Helper function for DOM2textbox.
    Returns the primary font, font size and all words
    not using those primary attributes
    """
    (primary_font, font_emph) = emph_words(line, fonts)
    (primary_size, size_emph) = emph_words(line, sizes)

    emph = list(set(font_emph + size_emph))
    return (primary_font, primary_size, emph)

def emph_words(line, attributes):
    """Helper function for find_emphasis.
    Returns the words having uncommon attributes.
    Args:
        line:       A string (character/word sequence).
        attributes: A list of not necessarily unique values.
                    len(line) == len(attributes) is required!
    Returns:
        A list of words that are special (having rare attributes).
    """
    global SLASHES
    unique_attrs = list(set(attributes))
    if "\n" in unique_attrs:
        unique_attrs.remove("\n")

    primary_attr = ""
    emph = list()
    if len(unique_attrs) == 1:
        primary_attr = unique_attrs[0]
    elif len(unique_attrs) > 1:
        occurrences = map(lambda f: attributes.count(f), unique_attrs)
        primary_attr = unique_attrs[occurrences.index(max(occurrences))]
        unique_attrs.remove(primary_attr)
        for attr in unique_attrs:
            buf = list()
            for i in range(0, len(line)):
                char = line[i]
                if char.strip() == u"":
                    buf.append(u" ")
                elif attributes[i] == attr:
                    buf.append(char)
            emph_ws = u"".join(buf)
            emph = emph + filter(lambda w: w != u"", emph_ws.split(u" "))

    return (primary_attr, emph)


def DOM2textgroup(dom_textgroup):
    bbox = str2bbox(dom_textgroup.getAttribute("bbox"))
    textboxes = map( DOM2textbox
                   , xml_util.getChildElementsByTagName(dom_textgroup, "textbox"))
    children = map( DOM2textgroup
                  , xml_util.getChildElementsByTagName(dom_textgroup, "textgroup"))
    return TextGroup(bbox, textboxes + children)


def DOM2page(dom_page):
    page_id = unicode(dom_page.getAttribute("id"))
    page_bbox = str2bbox(dom_page.getAttribute("bbox"))

    #dom_textboxes = dom_page.getElementsByTagName("textbox")
    dom_textboxes = xml_util.getChildElementsByTagName(dom_page, "textbox")
    textboxes = map(DOM2textbox, dom_textboxes)

    dom_layout = dom_page.getElementsByTagName("layout")
    layout = None
    if len(dom_layout):
        dom_textgroups = xml_util.getChildElementsByTagName(dom_layout[0], "textgroup")
        textgroups = map(DOM2textgroup, dom_textgroups)
        if len(textgroups):
            layout = textgroups[0]

    return Page(page_id, page_bbox, textboxes, layout)


## Analysis functions

def find_primary_font(textboxes=[], pages=[]):
    """Gets the primary (most used) font across multiple textboxes or pages.
    If both arguments are passed, only primary font of textboxes is
    determined!
    Args:
        textboxes: TextBoxes to extract primary font from.
        pages:     Pages to extract primary font from.
    Returns:
        3-Tuple (unicode, unicode, int) of primary font, primary fontsize
        and number of characters using this font.
        Returns (u"", u"", 0) if no argument is passed.
    """

    # TextBoxes
    fonts = dict()
    for tb in textboxes:
        fonts[tb.font] = fonts.get(tb.font, 0) + tb.character_count
    sorted_fonts = sorted(fonts.iteritems(), key=operator.itemgetter(1))
    if len(sorted_fonts) > 0:
        prim_font = sorted_fonts.pop()[0]
        return (prim_font[0], prim_font[1], fonts[prim_font])

    # Pages
    fonts = dict()
    for page in pages:
        font_key = (page.prim_font[0], page.prim_font[1])
        fonts[font_key] = fonts.get(font_key, 0) + page.prim_font[2]
    sorted_fonts = sorted(fonts.iteritems(), key=operator.itemgetter(1))
    if len(sorted_fonts) > 0:
        prim_font = sorted_fonts.pop()[0]
        return (prim_font[0], prim_font[1], fonts[prim_font])

    # Default
    return (u"", u"", 0)


""" TEST
"""
if __name__ == '__main__':
    import sys
    from xml.dom.minidom import parse
    args = sys.argv[1:]

    dom_doc = parse("SEUH_Kompetenzerwerb.xml")
    pages = DOM2pages(dom_doc)

    # Single page to SVG
    if len(args) > 0:
        pageNr = int(args[0])
        if pageNr >= 0 and pageNr < len(pages):
            print pages[pageNr].as_svg()

    # Debug: print all pages as text
    for page in pages:
        print unicode(page)
        print ""


