# dialects.py
#
# Copyright (c) 2007 Stephen Day
#
# This module is part of Creoleparser and is released under
# the MIT License: http://www.opensource.org/licenses/mit-license.php
#

from elements import *

class Creole10(object):

    """This class contains most of the logic and specification of the markup."""

    def __init__(self,wiki_links_base_url='http://',interwiki_links_base_urls={}):
        """Most attributes of new Creole objects are derived from the WikiElement
        class. Please see the constructor of that class and other specific element
        classes for details.

        :parameters:
          wiki_link_base_url
            self explanitory
          interwiki_links_base_urls
            Dictionary of urls for interwiki links.
        """
        self.br = LineBreak('br', r'\\')
        self.http_link = RawLink('a')
        self.interwiki_link = InterWikiLink(delimiter=':',
                                            base_urls=interwiki_links_base_urls,
                                            space_char='_')
        self.wiki_link = WikiLink(base_url=wiki_links_base_url,space_char='_')
        self.img = Image('img',('{{','}}'),[],delimiter='|')
        self.link = Link('a',('[[',']]'),[],delimiter='|',
                        link_types=[self.http_link,self.interwiki_link,self.wiki_link])
        self.strong = InlineElement('strong', '**',[])
        self.em = InlineElement('em', '//',[self.strong,self.br,self.link,self.img,self.http_link])
        self.no_wiki = NoWikiElement('span',['{{{','}}}'],[])
        self.strong.child_tags = [self.em,self.br,self.link,self.img,self.http_link]
        self.link.child_tags = [(self.strong, self.em), self.img]
                
        self.hr = LoneElement('hr','----',[])
        self.lone_br = LoneElement('br',r'\\',[])
        self.blank_line = BlankLine()

        header_children = [self.no_wiki,(self.strong, self.em), self.br,self.link,self.img,self.http_link]
        self.h1 = Heading('h1','=',header_children)
        self.h2 = Heading('h2','==',header_children)
        self.h3 = Heading('h3','===',header_children)
        self.h4 = Heading('h4','====',header_children)
        self.h5 = Heading('h5','=====',header_children)
        self.h6 = Heading('h6','======',header_children)

        headings = [self.h1,self.h2,self.h3,self.h4,self.h5,self.h6]
        
        self.td = TableCell('td','|',[self.no_wiki,(self.strong, self.em),self.br,self.link,self.img,self.http_link])
        self.th = TableCell('th','|=',[self.no_wiki,(self.strong, self.em),self.br,self.link,self.img,self.http_link])
        self.tr = TableRow('tr','|',[self.th,self.td])
        self.table = Table('table','|',[self.tr])

        self.p = Paragraph('p',[self.no_wiki,(self.strong, self.em),self.br,self.link,self.img,self.http_link])

        self.li = ListItem('li',child_tags=[],list_tokens='*#')
        self.ol = List('ol','#',[self.li],other_token='*')
        self.ul = List('ul','*',[self.li],other_token='#')
        self.nested_ol = NestedList('ol','#',[self.li])
        self.nested_ul = NestedList('ul','*',[self.li])
        self.li.child_tags = [(self.nested_ol,self.nested_ul),self.no_wiki,(self.strong,self.em),self.br,self.link,self.img,self.http_link]

        self.pre = PreBlock('pre',['{{{','}}}'])

        self.parse_order = [self.pre,self.blank_line,self.table]+ headings\
                           + [self.hr,self.lone_br,self.ul,self.ol,self.p]
        """These are the wiki elements that are searched at the top level of text to be
        processed. The order matters because elements later in the list need not have any
        knowledge of those before (as those were parsed out already). This makes the
        regular expression patterns for later elements very simple.
        """
