#include <cstdlib>
#include <cstring>
#include <string>

#include <Python.h>
#include <structmember.h>
#include <numpy/arrayobject.h>

#include "parser_defs.hpp"

using namespace std;

PyObject * 
parsed_to_string(const parsed_t & t)
{
#if IS_PY3K
    PyObject * const p = PyBytes_FromStringAndSize(t.first, distance(t.first, t.second));
#else
    PyObject * const p = PyString_FromStringAndSize(t.first, distance(t.first, t.second));
#endif // #if IS_PY3K

    if (p == NULL)
        PyErr_NoMemory();
    return p;
}

long 
parsed_to_long(const parsed_t & t, bool & err)
{
    err = false;
    char * e;
    const long l = strtol(t.first, &e, 10);
    if (e != t.second) {
        PyErr_Format(PyExc_TypeError, ("Cannot format " + string(t.first, t.second) + " to int").c_str());
        err = true;
    }    
    return l;
}

PyObject * 
parsed_to_int(const parsed_t & t)
{
    bool err;
    PyObject * const p = PyInt_FromLong(parsed_to_long(t, err));
    if (p == NULL)
        PyErr_NoMemory();
    return p;
}

double 
parsed_to_double(const parsed_t & t, bool & err)
{
    err = false;
    char * e;
    const double d = strtod(t.first, &e);
    if (e != t.second) {
        err = true;
        PyErr_Format(PyExc_TypeError, ("Cannot format " + string(t.first, t.second) + " to float").c_str());
    }    
    return d;
}

PyObject * 
parsed_to_float(const parsed_t & t)
{
    bool err;
    PyObject * const p = PyFloat_FromDouble(parsed_to_double(t, err));
    if (p == NULL)
        PyErr_NoMemory();
    return p;
}

char * 
pystring_as_string(PyObject * o, long & len)
{
#if IS_PY3K
    len = PyBytes_Size(o);
    return PyBytes_AsString(o);
#else // #if IS_PY3K
    len = PyString_Size(o);
    return PyString_AsString(o);
#endif // #if IS_PY3K
}

long *
parse_longs(PyObject * iterator, long & num, bool & err)
{
    num = 0;
    err = true;

    PyObject * const iter = PyObject_GetIter(iterator);
    if (iter == NULL) 
        return NULL;

    PyObject * obj;
    long ls[max_num_cols];

    while ((obj = PyIter_Next(iter)) != NULL) {
        if (num == max_num_cols) {
            PyErr_Format(PyExc_IndexError, "max num indices exceeded %d", max_num_cols);
            return NULL;
        }

#if IS_PY3K
        ls[num++] = PyLong_AsLong(obj);
#else // #if IS_PY3K
        ls[num++] = PyInt_AsLong(obj);
#endif // #if IS_PY3K
    }

    err = false;

    if (num == 0) {
        Py_DECREF(iter);       
        return NULL;
    }

    long * const ret = static_cast<long *>(PyMem_Malloc(num * sizeof(long)));    
    if (ret == NULL) {
        err = true;
        PyErr_NoMemory();
        Py_DECREF(iter);
        return NULL;
    }   

    memcpy(ret, ls, num * sizeof(long));

    Py_DECREF(iter);

    return ret;
}

char * *
parse_strings(PyObject * iterator, long & num, bool & err)
{
    num = 0;
    err = true;

    PyObject * const iter = PyObject_GetIter(iterator);
    if (iter == NULL)       
        return NULL;

    PyObject * obj;
    char * ss[max_num_cols];

    while ((obj = PyIter_Next(iter)) != NULL) {
        if (num == max_num_cols) {
            PyErr_Format(PyExc_IndexError, "max num indices exceeded %d", max_num_cols);
            return NULL;
        }

        long len;
        char * s = pystring_as_string(obj, len);
        char * save_s = static_cast<char *>(PyMem_Malloc((len + 1) * sizeof(char)));
        copy(s, s + len, save_s);
        save_s[len] = 0;
        // TRACE("parsed " << s << " " << len << " " << save_s);
        ss[num++] = save_s;
    }

    err = false;

    char * * const ret = static_cast<char * *>(PyMem_Malloc(num * sizeof(char *)));    
    if (ret == NULL) {
        PyErr_NoMemory();
        err = true;
        Py_DECREF(iter);
        return NULL;
    }

    memcpy(ret, ss, num * sizeof(char *));

    Py_DECREF(iter);

    return ret;
}

extern "C" PyObject *
parser_max_field_len(PyObject *module, PyObject *args, PyObject *keyword_args)
{
    return PyInt_FromLong(max_field_len);
}

