#include <iostream>

#include <Python.h>
#include <structmember.h>
#include <numpy/arrayobject.h>

#include "defs.hpp"
#include "correlator.hpp"

using namespace std;

struct Correlator
{
    PyObject_HEAD
    
    double sx, sy, sxx, sxy, syy;
    unsigned long n;
};

extern PyTypeObject CorrelatorType;

extern "C" int
correlator_traverse(Correlator * self, visitproc visit, void *arg)
{
    return 0;
}

extern "C" void
correlator_dealloc(Correlator * self)
{
    PyObject_GC_UnTrack(self);

    PyObject_GC_Del(self);
}

extern "C" PyObject *
correlator(PyObject *module, PyObject *args, PyObject *keyword_args)
{
    Correlator * const self = PyObject_GC_New(Correlator, &CorrelatorType);
    if (self == NULL) {
        PyErr_NoMemory();
        return NULL;
    }

    self->sx = self->sy = self->sxx = self->sxy = self->syy = self->n = 0;

    PyObject_GC_Track(self);
    return (PyObject *)self;
}

extern "C" int
correlator_clear(Correlator * self)
{
    self->sx = self->sy = self->sxx = self->sxy = self->syy = self->n = 0;

    return 0;
}

extern "C" PyObject *
correlator_push(PyObject *module, PyObject *args, PyObject *keyword_args)
{    
    Correlator * self;
    double x, y;
    if (!PyArg_ParseTuple(
            args, 
            "Odd",
            reinterpret_cast<PyObject **>(&self), &x, &y) || self == NULL) {
        PyErr_SetString(PyExc_TypeError, "Failed to parse stuff");
        return NULL;
    }
    
    self->sx += x;
    self->sy += y;
    self->sxx += x * x;
    self->sxy += x * y;
    self->syy += y * y;
    ++self->n;
    
    Py_RETURN_NONE;
}

extern "C" PyObject *
correlator_corr(PyObject *module, PyObject *args, PyObject *keyword_args)
{
    Correlator * self;
    if (!PyArg_ParseTuple(
            args, 
            "O",
            reinterpret_cast<PyObject **>(&self)) || self == NULL) {
        PyErr_SetString(PyExc_TypeError, "Failed to parse stuff");
        return NULL;
    }
    
    const double corr = (self->n * self->sxy - self->sx * self->sy) / 
        sqrt(self->n * self->sxx - self->sx * self->sx) / 
        sqrt(self->n * self->syy - self->sy * self->sy);
        
    return PyFloat_FromDouble(corr);    
}

static PyMethodDef correlator_methods[] = {
    { NULL}
};

static PyMemberDef correlator_memberlist[] = {
    { NULL }
};

PyDoc_STRVAR(CorrelatorType_doc, "");

PyTypeObject CorrelatorType = {
    PyVarObject_HEAD_INIT(NULL, 0)
    "dagpype_c.correlator",               /*tp_name*/
    sizeof(Correlator),                   /*tp_basicsize*/
    0,                                      /*tp_itemsize*/
    /* methods */
    (destructor)correlator_dealloc,          /*tp_dealloc*/
    (printfunc)0,                           /*tp_print*/
    (getattrfunc)0,                         /*tp_getattr*/
    (setattrfunc)0,                         /*tp_setattr*/
    (cmpfunc)0,                             /*tp_compare*/
    (reprfunc)0,                            /*tp_repr*/
    0,                                      /*tp_as_number*/
    0,                                      /*tp_as_sequence*/
    0,                                      /*tp_as_mapping*/
    (hashfunc)0,                            /*tp_hash*/
    (ternaryfunc)0,                         /*tp_call*/
    (reprfunc)0,                            /*tp_str*/
    0,                                      /*tp_getattro*/
    0,                                      /*tp_setattro*/
    0,                                      /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | 
    Py_TPFLAGS_HAVE_GC,                     /*tp_flags*/
    CorrelatorType_doc,                     /*tp_doc*/
    (traverseproc)correlator_traverse,       /*tp_traverse*/
    (inquiry)correlator_clear,               /*tp_clear*/
    0,                                      /*tp_richcompare*/
    0,                                      /*tp_weaklistoffset*/
    0,                      /*tp_iter*/
    0,                              /*tp_iternext*/
    correlator_methods,                      /*tp_methods*/
    correlator_memberlist,                   /*tp_members*/
    0,                                      /*tp_getset*/
};

