"""
Django dependency injection application powered by PyBinder.
"""

from inspect import getmembers, isclass
from importlib import import_module
from django.apps import AppConfig, apps
from django.utils.module_loading import module_has_submodule
from pybinder import Container
from pybinder.catalogs import Catalog, NAMESPACE_SEPARATOR
from pybinder.decorators import factory, singleton, requires, requires_provider
from pybinder.providers import Object


CATALOGS_MODULE_NAME = 'catalogs'
container = Container()


class DjPyBinderConfig(AppConfig):
    name = 'djpybinder'
    verbose_name = 'Django PyBinder Application'

    def ready(self):
        for config in apps.get_app_configs():
            catalog = Catalog(namespace=NAMESPACE_SEPARATOR.join((config.name,
                                                                  'models')))
            for model in config.get_models():
                catalog.bind(model._meta.model_name, Object(model))
            container.merge(catalog)

            if module_has_submodule(config.module, CATALOGS_MODULE_NAME):
                catalog_module_name = '%s.%s' % (config.module.__name__,
                                                 CATALOGS_MODULE_NAME)
                catalogs_module = import_module(catalog_module_name)

                for cls_info in getmembers(catalogs_module, isclass):
                    cls = cls_info[1]
                    if not issubclass(cls, Catalog) or cls is Catalog:
                        continue
                    container.merge(cls(namespace=config.name))
        container.assemble()


default_app_config = 'djpybinder.DjPyBinderConfig'

inject = container.inject
inject_provider = container.inject_provider


def inject_model(model, from_app, with_name=None):
    namespace = NAMESPACE_SEPARATOR.join((from_app, 'models'))
    return container.inject(model, from_namespace=namespace,
                            with_name=with_name)


__all__ = ['inject', 'inject_provider', 'inject_model', 'factory', 'singleton',
           'requires', 'requires_provider']
