"""
Django dependency injection application powered by PyBinder.
"""

import re
from inspect import getmembers, isclass
from importlib import import_module

from django.apps import AppConfig, apps
from django.conf import settings
from django.forms import Form, ModelForm
from django.utils.module_loading import module_has_submodule

from pybinder import Container, Namespace
from pybinder.catalogs import Catalog, NAMESPACE_SEPARATOR, GLOBAL_NAMESPACE
from pybinder.decorators import factory, singleton, requires, requires_provider
from pybinder.providers import Object, Value


CATALOGS_MODULE_NAME = 'catalogs'
MODELS_MODULE_NAME = 'models'
FORMS_MODULE_NAME = 'forms'
container = Container()


class DjPyBinderConfig(AppConfig):
    name = 'djpybinder'
    verbose_name = 'Django PyBinder Application'

    def ready(self):
        container.merge(self._get_settings_catalog())
        for app_config in apps.get_app_configs():
            container.merge(self._get_models_catalog(app_config))
            container.merge(self._get_forms_catalog(app_config))
            container.merge(*self._get_catalogs(app_config))
        container.assemble()

    def _get_settings_catalog(self):
        catalog = Catalog(namespace='settings')
        for setting_name in dir(settings):
            if not setting_name.isupper():
                continue
            catalog.bind(setting_name.lower(),
                         Value(getattr(settings, setting_name)))
        return catalog

    def _get_models_catalog(self, app_config):
        catalog = Catalog(namespace=NAMESPACE_SEPARATOR.join((
            app_config.name, MODELS_MODULE_NAME)))
        for model in app_config.get_models():
            catalog.bind(self._to_underscored(model.__name__), Object(model))
        return catalog

    def _get_forms_catalog(self, app_config):
        catalog = Catalog(namespace=NAMESPACE_SEPARATOR.join((
            app_config.name, FORMS_MODULE_NAME)))

        if not module_has_submodule(app_config.module, FORMS_MODULE_NAME):
            return catalog

        forms_module_name = '%s.%s' % (app_config.module.__name__,
                                         FORMS_MODULE_NAME)
        forms_module = import_module(forms_module_name)

        for name, cls in getmembers(forms_module, isclass):
            if not issubclass(cls, Form) or cls in (Form, ModelForm):
                continue
            catalog.bind(self._to_underscored(name), Object(cls))
        return catalog

    def _get_catalogs(self, app_config):
        catalogs = []
        if not module_has_submodule(app_config.module, CATALOGS_MODULE_NAME):
            return catalogs
        catalog_module_name = '%s.%s' % (app_config.module.__name__,
                                         CATALOGS_MODULE_NAME)
        catalogs_module = import_module(catalog_module_name)

        for cls_info in getmembers(catalogs_module, isclass):
            cls = cls_info[1]
            if not issubclass(cls, Catalog) or cls is Catalog:
                continue
            catalogs.append(cls(namespace=app_config.name))
        return catalogs

    def _to_underscored(self, name):
        return re.sub('([a-z0-9])([A-Z])', r'\1_\2',
                      re.sub('(.)([A-Z][a-z]+)', r'\1_\2', name)).lower()


default_app_config = 'djpybinder.DjPyBinderConfig'


def _namespace_for(app, module):
    return NAMESPACE_SEPARATOR.join((app, module))


# Settings API
def requires_setting(name, with_name=None):
    """
    Requires setting decorator.

    :param name:
    :param with_name:
    :return:
    """
    return requires(name, with_name=with_name, from_namespace='settings')


def inject_setting(name, with_name=None):
    """
    Inject setting decorator.

    :param name:
    :param with_name:
    :return:
    """
    return container.inject(name, with_name=with_name,
                            from_namespace='settings')


def provide_setting(name):
    """
    Provides setting.

    :param name:
    :return:
    """
    return container.provide(name)


# Models API
def requires_model(name, from_app, with_name=None):
    """
    Requires model decorator.

    :param name:
    :param from_app:
    :param with_name:
    :return:
    """
    return requires(name, with_name=with_name,
                    from_namespace=_namespace_for(from_app,
                                                  MODELS_MODULE_NAME))


def inject_model(name, from_app, with_name=None):
    """
    Inject model decorator.

    :param name:
    :param from_app:
    :param with_name:
    :return:
    """
    return container.inject(name, with_name=with_name,
                            from_namespace=_namespace_for(from_app,
                                                          MODELS_MODULE_NAME))


def provide_model(name, from_app):
    """
    Provides app model.

    :param name:
    :param from_app:
    :return:
    """
    app_models_namespace = _namespace_for(from_app, MODELS_MODULE_NAME)
    return container.namespace(app_models_namespace).provide(name)


# Forms API
def requires_form(name, from_app, with_name=None):
    """
    Requires form decorator.

    :param name:
    :param from_app:
    :param with_name:
    :return:
    """
    return requires(name, with_name=with_name,
                    from_namespace=_namespace_for(from_app,
                                                  FORMS_MODULE_NAME))


def inject_form(name, from_app, with_name=None):
    """
    Inject form decorator.

    :param name:
    :param from_app:
    :param with_name:
    :return:
    """
    return container.inject(name, with_name=with_name,
                            from_namespace=_namespace_for(from_app,
                                                          FORMS_MODULE_NAME))


def provide_form(name, from_app):
    """
    Provides app form.

    :param name:
    :param from_app:
    :return:
    """
    app_forms_namespace = _namespace_for(from_app, FORMS_MODULE_NAME)
    return container.namespace(app_forms_namespace).provide(name)


# DjPyBinder API
inject = container.inject
inject_provider = container.inject_provider
provide = container.provide
get_provider = container.get_provider


class App(Namespace):
    """
    App namespace.
    """

    def __init__(self, namespace):
        """
        Constructor.

        :param namespace:
        :return:
        """
        super(App, self).__init__(container, namespace)

    def inject_model(self, name, with_name=None):
        """
        Inject model decorator.

        :param name:
        :param with_name:
        :return:
        """
        return inject_model(name, self.namespace, with_name=with_name)

    def provide_model(self, name):
        """
        Provides model.

        :param name:
        :return:
        """
        return provide_model(name, self.namespace)

    def inject_form(self, name, with_name=None):
        """
        Inject from decorator.

        :param name:
        :param with_name:
        :return:
        """
        return inject_form(name, self.namespace, with_name=with_name)

    def provide_form(self, name):
        """
        Provides form.

        :param name:
        :return:
        """
        return provide_form(name, self.namespace)


__all__ = [
    # Settings API
    'requires_setting', 'inject_setting', 'provide_setting',

    # Models API
    'requires_model', 'inject_model', 'provide_model',

    # Forms API
    'requires_form', 'inject_form', 'provide_form',

    # DjPyBinder API
    'inject', 'inject_provider', 'provide', 'get_provider', 'App',

    # PyBinder API
    'factory', 'singleton', 'requires', 'requires_provider', 'GLOBAL_NAMESPACE'
]
