"""
Django PyBinder application.
"""

from inspect import getmembers, isclass

from django.apps import AppConfig, apps
from django.conf import settings
from django.utils.module_loading import module_has_submodule
from django.utils.importlib import import_module

from pybinder import Container
from pybinder.catalogs import Catalog
from pybinder.providers import Value


CATALOGS_MODULE_NAME = 'catalogs'
SETTINGS_CATALOGS_ATTR = 'DJPYBINDER_CATALOGS'


container = Container()


class DjPyBinderConfig(AppConfig):
    name = 'djpybinder'
    verbose_name = 'Django PyBinder Application'

    def ready(self):
        container.merge(_get_settings_catalog())

        for module_name in getattr(settings, SETTINGS_CATALOGS_ATTR, []):
            module = import_module(module_name)
            container.merge(*(cls() for cls in _get_module_catalogs(module)))

        for app_config in apps.get_app_configs():
            container.merge(*_get_catalogs(app_config))
        container.assemble()


default_app_config = 'djpybinder.DjPyBinderConfig'


def _get_settings_catalog():
    catalog = Catalog(namespace='settings')
    for setting_name in dir(settings):
        if not setting_name.isupper():
            continue
        catalog.bind(setting_name.lower(),
                     Value(getattr(settings, setting_name)))
    return catalog


def _get_catalogs(app_config):
    catalogs = []
    if not module_has_submodule(app_config.module, CATALOGS_MODULE_NAME):
        return catalogs
    catalog_module_name = '%s.%s' % (app_config.module.__name__,
                                     CATALOGS_MODULE_NAME)
    catalogs_module = import_module(catalog_module_name)

    for cls_info in getmembers(catalogs_module, isclass):
        cls = cls_info[1]
        if not issubclass(cls, Catalog) or cls is Catalog:
            continue
        catalogs.append(cls(namespace=app_config.name))
    return catalogs


def _get_module_catalogs(module):
    catalogs = []
    for cls_info in getmembers(module, isclass):
        cls = cls_info[1]
        if not issubclass(cls, Catalog) or cls is Catalog:
            continue
        catalogs.append(cls)
    return catalogs


inject = container.inject
inject_provider = container.inject_provider


__all__ = ['inject', 'inject_provider']
