#!/usr/bin/env python

"""
    6809 instruction set data
    ~~~~~~~~~~~~~~~~~~~~~~~~~

    merged data from:
        * http://www.maddes.net/m6809pm/sections.htm#sec4_4
        * http://www.burgins.com/m6809.html
        * http://www.maddes.net/m6809pm/appendix_a.htm#appA

    Note:
    * read_from_memory: it's "excluded" the address modes routines.
        So if the address mode will fetch the memory to get the
        effective address, but the content of the memory is not needed in
        the instruction them self, the read_from_memory must be set to False.

    Generated data is online here:
    https://docs.google.com/spreadsheet/ccc?key=0Alhtym6D6yKjdFBtNmF0UVR5OW05S3psaURnUTNtSFE

    :copyleft: 2013 by Jens Diemer
    :license: GNU GPL v3 or above, see LICENSE for more details.
"""

from __future__ import absolute_import, division, print_function


# this file was generated with MC6809_data2.py

BYTE = "8"
WORD = "16"


# Address modes:
DIRECT = "DIRECT"
EXTENDED = "EXTENDED"
IMMEDIATE = "IMMEDIATE"
IMMEDIATE_WORD = "IMMEDIATE_WORD"
INDEXED = "INDEXED"
INHERENT = "INHERENT"
RELATIVE = "RELATIVE"
RELATIVE_WORD = "RELATIVE_WORD"


# Registers:
REG_A = "A"
REG_B = "B"
REG_CC = "CC"
REG_D = "D"
REG_DP = "DP"
REG_PC = "PC"
REG_S = "S"
REG_U = "U"
REG_X = "X"
REG_Y = "Y"


# Instructions:
ABX = "ABX"
ADC = "ADC"
ADD = "ADD"
AND = "AND"
ASR = "ASR"
BEQ = "BEQ"
BGE = "BGE"
BGT = "BGT"
BHI = "BHI"
BHS = "BHS"
BIT = "BIT"
BLE = "BLE"
BLO = "BLO"
BLS = "BLS"
BLT = "BLT"
BMI = "BMI"
BNE = "BNE"
BPL = "BPL"
BRA = "BRA"
BRN = "BRN"
BSR = "BSR"
BVC = "BVC"
BVS = "BVS"
CLR = "CLR"
CMP = "CMP"
COM = "COM"
CWAI = "CWAI"
DAA = "DAA"
DEC = "DEC"
EOR = "EOR"
EXG = "EXG"
FIRQ = "FIRQ"
INC = "INC"
IRQ = "IRQ"
JMP = "JMP"
JSR = "JSR"
LD = "LD"
LEA = "LEA"
LSL = "LSL"
LSR = "LSR"
MUL = "MUL"
NEG = "NEG"
NMI = "NMI"
NOP = "NOP"
OR = "OR"
PAGE = "PAGE"
PSHS = "PSHS"
PSHU = "PSHU"
PULS = "PULS"
PULU = "PULU"
RESET = "RESET"
ROL = "ROL"
ROR = "ROR"
RTI = "RTI"
RTS = "RTS"
SBC = "SBC"
SEX = "SEX"
ST = "ST"
SUB = "SUB"
SWI = "SWI"
SWI2 = "SWI2"
SWI3 = "SWI3"
SYNC = "SYNC"
TFR = "TFR"
TST = "TST"




OP_DATA = {ABX: {'condition code': 'Not affected.',
     'description': 'Add the 8-bit unsigned value in accumulator B into index register X.',
     'instr_desc': 'Add B accumulator to X (unsigned)',
     'mnemonic': {ABX: {'HNZVC': '-----',
                      'desc': 'X = B+X (Unsigned)',
                      'needs_ea': False,
                      'ops': {0x3a: {'addr_mode': INHERENT,
                                   'bytes': 1,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None}},
     'operation': "IX' = IX + ACCB",
     'source form': ABX},
ADC: {'condition code': 'H - Set if a half-carry is generated; cleared otherwise.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Set if an overflow is generated; cleared otherwise.\nC - Set if a carry is generated; cleared otherwise.',
     'description': 'Adds the contents of the C (carry) bit and the memory byte into an 8-bit accumulator.',
     'instr_desc': 'Add memory to accumulator with carry',
     'mnemonic': {'ADCA': {'HNZVC': 'aaaaa',
                         'desc': 'A = A+M+C',
                         'needs_ea': False,
                         'ops': {0x89: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0x99: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xa9: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xb9: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_A,
                         'write_to_memory': None},
                 'ADCB': {'HNZVC': 'aaaaa',
                         'desc': 'B = B+M+C',
                         'needs_ea': False,
                         'ops': {0xc9: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0xd9: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xe9: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xf9: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': "R' = R + M + C",
     'source form': 'ADCA P; ADCB P'},
ADD: {'condition code': 'H - Set if a half-carry is generated; cleared otherwise.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Set if an overflow is generated; cleared otherwise.\nC - Set if a carry is generated; cleared otherwise.',
     'description': 'Adds the memory byte into an 8-bit accumulator.',
     'instr_desc': 'Add memory to accumulator',
     'mnemonic': {'ADDA': {'HNZVC': 'aaaaa',
                         'desc': 'A = A+M',
                         'needs_ea': False,
                         'ops': {0x8b: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0x9b: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xab: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xbb: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_A,
                         'write_to_memory': None},
                 'ADDB': {'HNZVC': 'aaaaa',
                         'desc': 'B = B+M',
                         'needs_ea': False,
                         'ops': {0xcb: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0xdb: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xeb: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xfb: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_B,
                         'write_to_memory': None},
                 'ADDD': {'HNZVC': '-aaaa',
                         'desc': 'D = D+M:M+1',
                         'needs_ea': False,
                         'ops': {0xc3: {'addr_mode': IMMEDIATE_WORD,
                                      'bytes': 3,
                                      'cycles': 4},
                                0xd3: {'addr_mode': 'DIRECT_WORD',
                                      'bytes': 2,
                                      'cycles': 6},
                                0xe3: {'addr_mode': 'INDEXED_WORD',
                                      'bytes': 2,
                                      'cycles': 6},
                                0xf3: {'addr_mode': 'EXTENDED_WORD',
                                      'bytes': 3,
                                      'cycles': 7}},
                         'read_from_memory': WORD,
                         'register': REG_D,
                         'write_to_memory': None}},
     'operation': "R' = R + M",
     'source form': 'ADDA P; ADDB P'},
AND: {'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Always cleared.\nC - Not affected.',
     'description': 'Performs the logical AND operation between the contents of an accumulator and the contents of memory location M and the result is stored in the accumulator.',
     'instr_desc': 'AND memory with accumulator',
     'mnemonic': {'ANDA': {'HNZVC': '-aa0-',
                         'desc': 'A = A && M',
                         'needs_ea': False,
                         'ops': {0x84: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0x94: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xa4: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xb4: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_A,
                         'write_to_memory': None},
                 'ANDB': {'HNZVC': '-aa0-',
                         'desc': 'B = B && M',
                         'needs_ea': False,
                         'ops': {0xc4: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0xd4: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xe4: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xf4: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_B,
                         'write_to_memory': None},
                 'ANDCC': {'HNZVC': 'ddddd',
                          'desc': 'C = CC && IMM',
                          'needs_ea': False,
                          'ops': {0x1c: {'addr_mode': IMMEDIATE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': BYTE,
                          'register': REG_CC,
                          'write_to_memory': None}},
     'operation': "R' = R AND M",
     'source form': 'ANDA P; ANDB P'},
ASR: {'condition code': 'H - Undefined.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Not affected.\nC - Loaded with bit zero of the original operand.',
     'description': 'Shifts all bits of the operand one place to the right. Bit seven is held constant. Bit zero is shifted into the C (carry) bit.',
     'instr_desc': 'Arithmetic shift of accumulator or memory right',
     'mnemonic': {ASR: {'HNZVC': 'uaa-s',
                      'desc': 'M = Arithmetic shift M right',
                      'needs_ea': True,
                      'ops': {0x7: {'addr_mode': DIRECT,
                                  'bytes': 2,
                                  'cycles': 6},
                             0x67: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 6},
                             0x77: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 7}},
                      'read_from_memory': BYTE,
                      'register': None,
                      'write_to_memory': BYTE},
                 'ASRA': {'HNZVC': 'uaa-s',
                         'desc': 'A = Arithmetic shift A right',
                         'needs_ea': False,
                         'ops': {0x47: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_A,
                         'write_to_memory': None},
                 'ASRB': {'HNZVC': 'uaa-s',
                         'desc': 'B = Arithmetic shift B right',
                         'needs_ea': False,
                         'ops': {0x57: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': 'b7 -> -> C\nb7 -> b0',
     'source form': 'ASR Q; ASRA; ASRB'},
BEQ: {'condition code': 'Not affected.',
     'description': 'Tests the state of the Z (zero) bit and causes a branch if it is set.\nWhen used after a subtract or compare operation, this instruction will branch if the compared values, signed or unsigned, were exactly the same.',
     'instr_desc': 'Branch if equal',
     'mnemonic': {BEQ: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x27: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBEQ': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x1027: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF Z = 1 then PC' = PC + TEMP",
     'source form': 'BEQ dd; LBEQ DDDD'},
BGE: {'condition code': 'Not affected.',
     'description': 'Causes a branch if the N (negative) bit and the V (overflow) bit are either both set or both clear.\nThat is, branch if the sign of a valid twos complement result is, or would be, positive.\nWhen used after a subtract or compare operation on twos complement values, this instruction will branch if the register was greater than or equal to the memory operand.',
     'instr_desc': 'Branch if greater than or equal (signed)',
     'mnemonic': {BGE: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x2c: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBGE': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x102c: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF [N XOR V] = 0 then PC' = PC + TEMP",
     'source form': 'BGE dd; LBGE DDDD'},
BGT: {'condition code': 'Not affected.',
     'description': 'Causes a branch if the N (negative) bit and V (overflow) bit are either both set or both clear and the Z (zero) bit is clear.\nIn other words, branch if the sign of a valid twos complement result is, or would be, positive and not zero.\nWhen used after a subtract or compare operation on twos complement values, this instruction will branch if the register was greater than the memory operand.',
     'instr_desc': 'Branch if greater (signed)',
     'mnemonic': {BGT: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x2e: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBGT': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x102e: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF Z AND [N XOR V] = 0 then PC' = PC + TEMP",
     'source form': 'BGT dd; LBGT DDDD'},
BHI: {'comment': 'Generally not useful after INC/DEC, LD/TST, and TST/CLR/COM instructions.',
     'condition code': 'Not affected.',
     'description': 'Causes a branch if the previous operation caused neither a carry nor a zero result.\nWhen used after a subtract or compare operation on unsigned binary values, this instruction will branch if the register was higher than the memory operand.',
     'instr_desc': 'Branch if higher (unsigned)',
     'mnemonic': {BHI: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x22: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBHI': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x1022: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF [ C OR Z ] = 0 then PC' = PC + TEMP",
     'source form': 'BHI dd; LBHI DDDD'},
BHS: {'comment': 'This is a duplicate assembly-language mnemonic for the single machine instruction BCC.\nGenerally not useful after INC/DEC, LD/ST, and TST/CLR/COM instructions.',
     'condition code': 'Not affected.',
     'description': 'Tests the state of the C (carry) bit and causes a branch if it is clear.\nWhen used after a subtract or compare on unsigned binary values, this instruction will branch if the register was higher than or the same as the memory operand.',
     'instr_desc': 'Branch if higher or same (unsigned)',
     'mnemonic': {'BCC': {'HNZVC': '-----',
                        'desc': None,
                        'needs_ea': True,
                        'ops': {0x24: {'addr_mode': RELATIVE,
                                     'bytes': 2,
                                     'cycles': 3}},
                        'read_from_memory': None,
                        'register': None,
                        'write_to_memory': None},
                 'LBCC': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x1024: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF C = 0 then PC' = PC + MI",
     'source form': 'BHS dd; LBHS DDDD'},
BIT: {'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Always cleared.\nC - Not affected.',
     'description': 'Performs the logical AND of the contents of accumulator A or B and the contents of memory location M and modifies the condition codes accordingly.\nThe contents of accumulator A or B and memory location M are not affected.',
     'instr_desc': 'Bit test memory with accumulator',
     'mnemonic': {'BITA': {'HNZVC': '-aa0-',
                         'desc': 'Bit Test A (M&&A)',
                         'needs_ea': False,
                         'ops': {0x85: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0x95: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xa5: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xb5: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_A,
                         'write_to_memory': None},
                 'BITB': {'HNZVC': '-aa0-',
                         'desc': 'Bit Test B (M&&B)',
                         'needs_ea': False,
                         'ops': {0xc5: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0xd5: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xe5: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xf5: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': 'TEMP = R AND M',
     'source form': 'BITA P; BITB P'},
BLE: {'condition code': 'Not affected.',
     'description': 'Causes a branch if the exclusive OR of the N (negative) and V (overflow) bits is 1 or if the Z (zero) bit is set.\nThat is, branch if the sign of a valid twos complement result is, or would be, negative.\nWhen used after a subtract or compare operation on twos complement values, this instruction will branch if the register was less than or equal to the memory operand.',
     'instr_desc': 'Branch if less than or equal (signed)',
     'mnemonic': {BLE: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x2f: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBLE': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x102f: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF Z OR [ N XOR V ] = 1 then PC' = PC + TEMP",
     'source form': 'BLE dd; LBLE DDDD'},
BLO: {'comment': 'This is a duplicate assembly-language mnemonic for the single machine instruction BCS.\nGenerally not useful after INC/DEC, LD/ST, and TST/CLR/COM instructions.',
     'condition code': 'Not affected.',
     'description': 'Tests the state of the C (carry) bit and causes a branch if it is set.\nWhen used after a subtract or compare on unsigned binary values, this instruction will branch if the register was lower than the memory operand.',
     'instr_desc': 'Branch if lower (unsigned)',
     'mnemonic': {BLO: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x25: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBCS': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x1025: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF C = 1 then PC' = PC + TEMP",
     'source form': 'BLO dd; LBLO DDDD'},
BLS: {'comment': 'Generally not useful after INC/DEC, LD/ST, and TST/CLR/COM instructions.',
     'condition code': 'Not affected.',
     'description': 'Causes a branch if the previous operation caused either a carry or a zero result.\nWhen used after a subtract or compare operation on unsigned binary values, this instruction will branch if the register was lower than or the same as the memory operand.',
     'instr_desc': 'Branch if lower or same (unsigned)',
     'mnemonic': {BLS: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x23: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBLS': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x1023: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF (C OR Z) = 1 then PC' = PC + TEMP",
     'source form': 'BLS dd; LBLS DDDD'},
BLT: {'condition code': 'Not affected.',
     'description': 'Causes a branch if either, but not both, of the N (negative) or V (overflow) bits is set.\nThat is, branch if the sign of a valid twos complement result is, or would be, negative.\nWhen used after a subtract or compare operation on twos complement binary values, this instruction will branch if the register was less than the memory operand.',
     'instr_desc': 'Branch if less than (signed)',
     'mnemonic': {BLT: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x2d: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBLT': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x102d: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF [ N XOR V ] = 1 then PC' = PC + TEMP",
     'source form': 'BLT dd; LBLT DDDD'},
BMI: {'comment': 'When used after an operation on signed binary values, this instruction will branch if the result is minus.\nIt is generally preferred to use the LBLT instruction after signed operations.',
     'condition code': 'Not affected.',
     'description': 'Tests the state of the N (negative) bit and causes a branch if set.\nThat is, branch if the sign of the twos complement result is negative.',
     'instr_desc': 'Branch if minus',
     'mnemonic': {BMI: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x2b: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBMI': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x102b: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF N = 1 then PC' = PC + TEMP",
     'source form': 'BMI dd; LBMI DDDD'},
BNE: {'condition code': 'Not affected.',
     'description': 'Tests the state of the Z (zero) bit and causes a branch if it is clear.\nWhen used after a subtract or compare operation on any binary values, this instruction will branch if the register is, or would be, not equal to the memory operand.',
     'instr_desc': 'Branch if not equal',
     'mnemonic': {BNE: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x26: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBNE': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x1026: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF Z = 0 then PC' = PC + TEMP",
     'source form': 'BNE dd; LBNE DDDD'},
BPL: {'comment': 'When used after an operation on signed binary values, this instruction will branch if the result (possibly invalid) is positive.\nIt is generally preferred to use the BGE instruction after signed operations.',
     'condition code': 'Not affected.',
     'description': 'Tests the state of the N (negative) bit and causes a branch if it is clear.\nThat is, branch if the sign of the twos complement result is positive.',
     'instr_desc': 'Branch if plus',
     'mnemonic': {BPL: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x2a: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBPL': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x102a: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF N = 0 then PC' = PC + TEMP",
     'source form': 'BPL dd; LBPL DDDD'},
BRA: {'condition code': 'Not affected.',
     'description': 'Causes an unconditional branch.',
     'instr_desc': 'Branch always',
     'mnemonic': {BRA: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x20: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBRA': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x16: {'addr_mode': RELATIVE_WORD,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI PC' = PC + TEMP",
     'source form': 'BRA dd; LBRA DDDD'},
BRN: {'condition code': 'Not affected.',
     'description': 'Does not cause a branch.\nThis instruction is essentially a no operation, but has a bit pattern logically related to branch always.',
     'instr_desc': 'Branch never',
     'mnemonic': {BRN: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x21: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBRN': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x1021: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': 'TEMP = MI',
     'source form': 'BRN dd; LBRN DDDD'},
BSR: {'comment': 'A return from subroutine (RTS) instruction is used to reverse this process and must be the last instruction executed in a subroutine.',
     'condition code': 'Not affected.',
     'description': 'The program counter is pushed onto the stack.\nThe program counter is then loaded with the sum of the program counter and the offset.',
     'instr_desc': 'Branch to subroutine',
     'mnemonic': {BSR: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x8d: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 7}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBSR': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x17: {'addr_mode': RELATIVE_WORD,
                                      'bytes': 3,
                                      'cycles': 9}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI SP' = SP-1, (SP) = PCL SP' = SP-1, (SP) = PCH PC' = PC + TEMP",
     'source form': 'BSR dd; LBSR DDDD'},
BVC: {'condition code': 'Not affected.',
     'description': 'Tests the state of the V (overflow) bit and causes a branch if it is clear.\nThat is, branch if the twos complement result was valid.\nWhen used after an operation on twos complement binary values, this instruction will branch if there was no overflow.',
     'instr_desc': 'Branch if valid twos complement result',
     'mnemonic': {BVC: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x28: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBVC': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x1028: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP = MI IFF V = 0 then PC' = PC + TEMP",
     'source form': 'BVC dd; LBVC DDDD'},
BVS: {'condition code': 'Not affected.',
     'description': 'Tests the state of the V (overflow) bit and causes a branch if it is set.\nThat is, branch if the twos complement result was invalid.\nWhen used after an operation on twos complement binary values, this instruction will branch if there was an overflow.',
     'instr_desc': 'Branch if invalid twos complement result',
     'mnemonic': {BVS: {'HNZVC': '-----',
                      'desc': None,
                      'needs_ea': True,
                      'ops': {0x29: {'addr_mode': RELATIVE,
                                   'bytes': 2,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 'LBVS': {'HNZVC': '-----',
                         'desc': None,
                         'needs_ea': True,
                         'ops': {0x1029: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 4,
                                        'cycles': 5}},
                         'read_from_memory': None,
                         'register': None,
                         'write_to_memory': None}},
     'operation': "TEMP' = MI IFF V = 1 then PC' = PC + TEMP",
     'source form': 'BVS dd; LBVS DDDD'},
CLR: {'condition code': 'H - Not affected.\nN - Always cleared.\nZ - Always set.\nV - Always cleared.\nC - Always cleared.',
     'description': 'Accumulator A or B or memory location M is loaded with 00000000 2 .\nNote that the EA is read during this operation.',
     'instr_desc': 'Clear accumulator or memory location',
     'mnemonic': {CLR: {'HNZVC': '-0100',
                      'desc': 'M = 0',
                      'needs_ea': True,
                      'ops': {0xf: {'addr_mode': DIRECT,
                                  'bytes': 2,
                                  'cycles': 6},
                             0x6f: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 6},
                             0x7f: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 7}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': BYTE},
                 'CLRA': {'HNZVC': '-0100',
                         'desc': 'A = 0',
                         'needs_ea': False,
                         'ops': {0x4f: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_A,
                         'write_to_memory': None},
                 'CLRB': {'HNZVC': '-0100',
                         'desc': 'B = 0',
                         'needs_ea': False,
                         'ops': {0x5f: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': 'TEMP = M M = 00 16',
     'source form': 'CLR Q'},
CMP: {'condition code': 'H - Undefined.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Set if an overflow is generated; cleared otherwise.\nC - Set if a borrow is generated; cleared otherwise.',
     'description': 'Compares the contents of memory location to the contents of the specified register and sets the appropriate condition codes.\nNeither memory location M nor the specified register is modified.\nThe carry flag represents a borrow and is set to the inverse of the resulting binary carry.',
     'instr_desc': 'Compare memory from accumulator',
     'mnemonic': {'CMPA': {'HNZVC': 'uaaaa',
                         'desc': 'Compare M from A',
                         'needs_ea': False,
                         'ops': {0x81: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0x91: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xa1: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xb1: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_A,
                         'write_to_memory': None},
                 'CMPB': {'HNZVC': 'uaaaa',
                         'desc': 'Compare M from B',
                         'needs_ea': False,
                         'ops': {0xc1: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0xd1: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xe1: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xf1: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_B,
                         'write_to_memory': None},
                 'CMPD': {'HNZVC': '-aaaa',
                         'desc': 'Compare M:M+1 from D',
                         'needs_ea': False,
                         'ops': {0x1083: {'addr_mode': IMMEDIATE_WORD,
                                        'bytes': 4,
                                        'cycles': 5},
                                0x1093: {'addr_mode': 'DIRECT_WORD',
                                        'bytes': 3,
                                        'cycles': 7},
                                0x10a3: {'addr_mode': 'INDEXED_WORD',
                                        'bytes': 3,
                                        'cycles': 7},
                                0x10b3: {'addr_mode': 'EXTENDED_WORD',
                                        'bytes': 4,
                                        'cycles': 8}},
                         'read_from_memory': WORD,
                         'register': REG_D,
                         'write_to_memory': None},
                 'CMPS': {'HNZVC': '-aaaa',
                         'desc': 'Compare M:M+1 from S',
                         'needs_ea': False,
                         'ops': {0x118c: {'addr_mode': IMMEDIATE_WORD,
                                        'bytes': 4,
                                        'cycles': 5},
                                0x119c: {'addr_mode': 'DIRECT_WORD',
                                        'bytes': 3,
                                        'cycles': 7},
                                0x11ac: {'addr_mode': 'INDEXED_WORD',
                                        'bytes': 3,
                                        'cycles': 7},
                                0x11bc: {'addr_mode': 'EXTENDED_WORD',
                                        'bytes': 4,
                                        'cycles': 8}},
                         'read_from_memory': WORD,
                         'register': REG_S,
                         'write_to_memory': None},
                 'CMPU': {'HNZVC': '-aaaa',
                         'desc': 'Compare M:M+1 from U',
                         'needs_ea': False,
                         'ops': {0x1183: {'addr_mode': IMMEDIATE_WORD,
                                        'bytes': 4,
                                        'cycles': 5},
                                0x1193: {'addr_mode': 'DIRECT_WORD',
                                        'bytes': 3,
                                        'cycles': 7},
                                0x11a3: {'addr_mode': 'INDEXED_WORD',
                                        'bytes': 3,
                                        'cycles': 7},
                                0x11b3: {'addr_mode': 'EXTENDED_WORD',
                                        'bytes': 4,
                                        'cycles': 8}},
                         'read_from_memory': WORD,
                         'register': REG_U,
                         'write_to_memory': None},
                 'CMPX': {'HNZVC': '-aaaa',
                         'desc': 'Compare M:M+1 from X',
                         'needs_ea': False,
                         'ops': {0x8c: {'addr_mode': IMMEDIATE_WORD,
                                      'bytes': 3,
                                      'cycles': 4},
                                0x9c: {'addr_mode': 'DIRECT_WORD',
                                      'bytes': 2,
                                      'cycles': 6},
                                0xac: {'addr_mode': 'INDEXED_WORD',
                                      'bytes': 2,
                                      'cycles': 6},
                                0xbc: {'addr_mode': 'EXTENDED_WORD',
                                      'bytes': 3,
                                      'cycles': 7}},
                         'read_from_memory': WORD,
                         'register': REG_X,
                         'write_to_memory': None},
                 'CMPY': {'HNZVC': '-aaaa',
                         'desc': 'Compare M:M+1 from Y',
                         'needs_ea': False,
                         'ops': {0x108c: {'addr_mode': IMMEDIATE_WORD,
                                        'bytes': 4,
                                        'cycles': 5},
                                0x109c: {'addr_mode': 'DIRECT_WORD',
                                        'bytes': 3,
                                        'cycles': 7},
                                0x10ac: {'addr_mode': 'INDEXED_WORD',
                                        'bytes': 3,
                                        'cycles': 7},
                                0x10bc: {'addr_mode': 'EXTENDED_WORD',
                                        'bytes': 4,
                                        'cycles': 8}},
                         'read_from_memory': WORD,
                         'register': REG_Y,
                         'write_to_memory': None}},
     'operation': 'TEMP = R - M',
     'source form': 'CMPA P; CMPB P'},
COM: {'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Always cleared.\nC - Always set.',
     'description': 'Replaces the contents of memory location M or accumulator A or B with its logical complement.\nWhen operating on unsigned values, only BEQ and BNE branches can be expected to behave properly following a COM instruction.\nWhen operating on twos complement values, all signed branches are available.',
     'instr_desc': 'Complement accumulator or memory location',
     'mnemonic': {COM: {'HNZVC': '-aa01',
                      'desc': 'M = complement(M)',
                      'needs_ea': True,
                      'ops': {0x3: {'addr_mode': DIRECT,
                                  'bytes': 2,
                                  'cycles': 6},
                             0x63: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 6},
                             0x73: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 7}},
                      'read_from_memory': BYTE,
                      'register': None,
                      'write_to_memory': BYTE},
                 'COMA': {'HNZVC': '-aa01',
                         'desc': 'A = complement(A)',
                         'needs_ea': False,
                         'ops': {0x43: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_A,
                         'write_to_memory': None},
                 'COMB': {'HNZVC': '-aa01',
                         'desc': 'B = complement(B)',
                         'needs_ea': False,
                         'ops': {0x53: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': "M' = 0 + M",
     'source form': 'COM Q; COMA; COMB'},
CWAI: {'comment': 'The following immediate values will have the following results: FF = enable neither EF = enable IRQ BF = enable FIRQ AF = enable both',
      'condition code': 'Affected according to the operation.',
      'description': 'This instruction ANDs an immediate byte with the condition code register which may clear the interrupt mask bits I and F, stacks the entire machine state on the hardware stack and then looks for an interrupt.\nWhen a non-masked interrupt occurs, no further machine state information need be saved before vectoring to the interrupt handling routine.\nThis instruction replaced the MC6800 CLI WAI sequence, but does not place the buses in a high-impedance state.\nA FIRQ (fast interrupt request) may enter its interrupt handler with its entire machine state saved.\nThe RTI (return from interrupt) instruction will automatically return the entire machine state after testing the E (entire) bit of the recovered condition code register.',
      'instr_desc': 'AND condition code register, then wait for interrupt',
      'mnemonic': {CWAI: {'HNZVC': 'ddddd',
                        'desc': 'CC = CC ^ IMM; (Wait for Interrupt)',
                        'needs_ea': False,
                        'ops': {0x3c: {'addr_mode': IMMEDIATE,
                                     'bytes': 2,
                                     'cycles': 21}},
                        'read_from_memory': BYTE,
                        'register': None,
                        'write_to_memory': None}},
      'operation': "CCR = CCR AND MI (Possibly clear masks) Set E (entire state saved) SP' = SP-1, (SP) = PCL SP' = SP-1, (SP) = PCH SP' = SP-1, (SP) = USL SP' = SP-1, (SP) = USH SP' = SP-1, (SP) = IYL SP' = SP-1, (SP) = IYH SP' = SP-1, (SP) = IXL SP' = SP-1, (SP) = IXH SP' = SP-1, (SP) = DPR SP' = SP-1, (SP) = ACCB SP' = SP-1, (SP) = ACCA SP' = SP-1, (SP) = CCR",
      'source form': 'CWAI #$XX E F H I N Z V C'},
DAA: {'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Undefined.\nC - Set if a carry is generated or if the carry bit was set before the operation; cleared otherwise.',
     'description': 'The sequence of a single-byte add instruction on accumulator A (either ADDA or ADCA) and a following decimal addition adjust instruction results in a BCD addition with an appropriate carry bit.\nBoth values to be added must be in proper BCD form (each nibble such that: 0 <= nibble <= 9).\nMultiple-precision addition must add the carry generated by this decimal addition adjust into the next higher digit during the add operation (ADCA) immediately prior to the next decimal addition adjust.',
     'instr_desc': 'Decimal adjust A accumulator',
     'mnemonic': {DAA: {'HNZVC': '-aa0a',
                      'desc': 'Decimal Adjust A',
                      'needs_ea': False,
                      'ops': {0x19: {'addr_mode': INHERENT,
                                   'bytes': 1,
                                   'cycles': 2}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None}},
     'operation': 'Decimal Adjust A',
     'source form': DAA},
DEC: {'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Set if the original operand was 10000000 2 ; cleared otherwise.\nC - Not affected.',
     'description': 'Subtract one from the operand.\nThe carry bit is not affected, thus allowing this instruction to be used as a loop counter in multiple-precision computations.\nWhen operating on unsigned values, only BEQ and BNE branches can be expected to behave consistently.\nWhen operating on twos complement values, all signed branches are available.',
     'instr_desc': 'Decrement accumulator or memory location',
     'mnemonic': {DEC: {'HNZVC': '-aaa-',
                      'desc': 'M = M - 1',
                      'needs_ea': True,
                      'ops': {0xa: {'addr_mode': DIRECT,
                                  'bytes': 2,
                                  'cycles': 6},
                             0x6a: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 6},
                             0x7a: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 7}},
                      'read_from_memory': BYTE,
                      'register': None,
                      'write_to_memory': BYTE},
                 'DECA': {'HNZVC': '-aaa-',
                         'desc': 'A = A - 1',
                         'needs_ea': False,
                         'ops': {0x4a: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_A,
                         'write_to_memory': None},
                 'DECB': {'HNZVC': '-aaa-',
                         'desc': 'B = B - 1',
                         'needs_ea': False,
                         'ops': {0x5a: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': "M' = M - 1",
     'source form': 'DEC Q; DECA; DECB'},
EOR: {'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Always cleared.\nC - Not affected.',
     'description': 'The contents of memory location M is exclusive ORed into an 8-bit register.',
     'instr_desc': 'Exclusive OR memory with accumulator',
     'mnemonic': {'EORA': {'HNZVC': '-aa0-',
                         'desc': 'A = A XOR M',
                         'needs_ea': False,
                         'ops': {0x88: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0x98: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xa8: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xb8: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_A,
                         'write_to_memory': None},
                 'EORB': {'HNZVC': '-aa0-',
                         'desc': 'B = M XOR B',
                         'needs_ea': False,
                         'ops': {0xc8: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0xd8: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xe8: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xf8: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': "R' = R XOR M",
     'source form': 'EORA P; EORB P'},
EXG: {'condition code': 'Not affected (unless one of the registers is the condition code\nregister).',
     'description': '0000 = A:B 1000 = A\n0001 = X 1001 = B\n0010 = Y 1010 = CCR\n0011 = US 1011 = DPR\n0100 = SP 1100 = Undefined\n0101 = PC 1101 = Undefined\n0110 = Undefined 1110 = Undefined\n0111 = Undefined 1111 = Undefined',
     'instr_desc': 'Exchange Rl with R2',
     'mnemonic': {EXG: {'HNZVC': 'ccccc',
                      'desc': 'exchange R1,R2',
                      'needs_ea': False,
                      'ops': {0x1e: {'addr_mode': IMMEDIATE,
                                   'bytes': 2,
                                   'cycles': 8}},
                      'read_from_memory': BYTE,
                      'register': None,
                      'write_to_memory': None}},
     'operation': 'R1 <-> R2',
     'source form': 'EXG R1,R2'},
INC: {'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Set if the original operand was 01111111 2 ; cleared otherwise.\nC - Not affected.',
     'description': 'Adds to the operand.\nThe carry bit is not affected, thus allowing this instruction to be used as a loop counter in multiple-precision computations.\nWhen operating on unsigned values, only the BEQ and BNE branches can be expected to behave consistently.\nWhen operating on twos complement values, all signed branches are correctly available.',
     'instr_desc': 'Increment accumulator or memory location',
     'mnemonic': {INC: {'HNZVC': '-aaa-',
                      'desc': 'M = M + 1',
                      'needs_ea': True,
                      'ops': {0xc: {'addr_mode': DIRECT,
                                  'bytes': 2,
                                  'cycles': 6},
                             0x6c: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 6},
                             0x7c: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 7}},
                      'read_from_memory': BYTE,
                      'register': None,
                      'write_to_memory': BYTE},
                 'INCA': {'HNZVC': '-aaa-',
                         'desc': 'A = A + 1',
                         'needs_ea': False,
                         'ops': {0x4c: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_A,
                         'write_to_memory': None},
                 'INCB': {'HNZVC': '-aaa-',
                         'desc': 'B = B + 1',
                         'needs_ea': False,
                         'ops': {0x5c: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': "M' = M + 1",
     'source form': 'INC Q; INCA; INCB'},
JMP: {'condition code': 'Not affected.',
     'description': 'Program control is transferred to the effective address.',
     'instr_desc': 'Jump',
     'mnemonic': {JMP: {'HNZVC': '-----',
                      'desc': 'pc = EA',
                      'needs_ea': True,
                      'ops': {0xe: {'addr_mode': DIRECT,
                                  'bytes': 2,
                                  'cycles': 3},
                             0x6e: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 3},
                             0x7e: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 3}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None}},
     'operation': "PC' = EA",
     'source form': 'JMP EA'},
JSR: {'condition code': 'Not affected.',
     'description': 'Program control is transferred to the effective address after storing the return address on the hardware stack.\nA RTS instruction should be the last executed instruction of the subroutine.',
     'instr_desc': 'Jump to subroutine',
     'mnemonic': {JSR: {'HNZVC': '-----',
                      'desc': 'jump to subroutine',
                      'needs_ea': True,
                      'ops': {0x9d: {'addr_mode': DIRECT,
                                   'bytes': 2,
                                   'cycles': 7},
                             0xad: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 7},
                             0xbd: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 8}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None}},
     'operation': "SP' = SP-1, (SP) = PCL SP' = SP-1, (SP) = PCH PC' =EA",
     'source form': 'JSR EA'},
LD: {'condition code': 'H - Not affected.\nN - Set if the loaded data is negative; cleared otherwise.\nZ - Set if the loaded data is zero; cleared otherwise.\nV - Always cleared.\nC - Not affected.',
    'description': 'Loads the contents of memory location M into the designated register.',
    'instr_desc': 'Load accumulator from memory',
    'mnemonic': {'LDA': {'HNZVC': '-aa0-',
                       'desc': 'A = M',
                       'needs_ea': False,
                       'ops': {0x86: {'addr_mode': IMMEDIATE,
                                    'bytes': 2,
                                    'cycles': 2},
                              0x96: {'addr_mode': DIRECT,
                                    'bytes': 2,
                                    'cycles': 4},
                              0xa6: {'addr_mode': INDEXED,
                                    'bytes': 2,
                                    'cycles': 4},
                              0xb6: {'addr_mode': EXTENDED,
                                    'bytes': 3,
                                    'cycles': 5}},
                       'read_from_memory': BYTE,
                       'register': REG_A,
                       'write_to_memory': None},
                'LDB': {'HNZVC': '-aa0-',
                       'desc': 'B = M',
                       'needs_ea': False,
                       'ops': {0xc6: {'addr_mode': IMMEDIATE,
                                    'bytes': 2,
                                    'cycles': 2},
                              0xd6: {'addr_mode': DIRECT,
                                    'bytes': 2,
                                    'cycles': 4},
                              0xe6: {'addr_mode': INDEXED,
                                    'bytes': 2,
                                    'cycles': 4},
                              0xf6: {'addr_mode': EXTENDED,
                                    'bytes': 3,
                                    'cycles': 5}},
                       'read_from_memory': BYTE,
                       'register': REG_B,
                       'write_to_memory': None},
                'LDD': {'HNZVC': '-aa0-',
                       'desc': 'D = M:M+1',
                       'needs_ea': False,
                       'ops': {0xcc: {'addr_mode': IMMEDIATE_WORD,
                                    'bytes': 3,
                                    'cycles': 3},
                              0xdc: {'addr_mode': 'DIRECT_WORD',
                                    'bytes': 2,
                                    'cycles': 5},
                              0xec: {'addr_mode': 'INDEXED_WORD',
                                    'bytes': 2,
                                    'cycles': 5},
                              0xfc: {'addr_mode': 'EXTENDED_WORD',
                                    'bytes': 3,
                                    'cycles': 6}},
                       'read_from_memory': WORD,
                       'register': REG_D,
                       'write_to_memory': None},
                'LDS': {'HNZVC': '-aa0-',
                       'desc': 'S = M:M+1',
                       'needs_ea': False,
                       'ops': {0x10ce: {'addr_mode': IMMEDIATE_WORD,
                                      'bytes': 4,
                                      'cycles': 4},
                              0x10de: {'addr_mode': 'DIRECT_WORD',
                                      'bytes': 3,
                                      'cycles': 6},
                              0x10ee: {'addr_mode': 'INDEXED_WORD',
                                      'bytes': 3,
                                      'cycles': 6},
                              0x10fe: {'addr_mode': 'EXTENDED_WORD',
                                      'bytes': 4,
                                      'cycles': 7}},
                       'read_from_memory': WORD,
                       'register': REG_S,
                       'write_to_memory': None},
                'LDU': {'HNZVC': '-aa0-',
                       'desc': 'U = M:M+1',
                       'needs_ea': False,
                       'ops': {0xce: {'addr_mode': IMMEDIATE_WORD,
                                    'bytes': 3,
                                    'cycles': 3},
                              0xde: {'addr_mode': 'DIRECT_WORD',
                                    'bytes': 2,
                                    'cycles': 5},
                              0xee: {'addr_mode': 'INDEXED_WORD',
                                    'bytes': 2,
                                    'cycles': 5},
                              0xfe: {'addr_mode': 'EXTENDED_WORD',
                                    'bytes': 3,
                                    'cycles': 6}},
                       'read_from_memory': WORD,
                       'register': REG_U,
                       'write_to_memory': None},
                'LDX': {'HNZVC': '-aa0-',
                       'desc': 'X = M:M+1',
                       'needs_ea': False,
                       'ops': {0x8e: {'addr_mode': IMMEDIATE_WORD,
                                    'bytes': 3,
                                    'cycles': 3},
                              0x9e: {'addr_mode': 'DIRECT_WORD',
                                    'bytes': 2,
                                    'cycles': 5},
                              0xae: {'addr_mode': 'INDEXED_WORD',
                                    'bytes': 2,
                                    'cycles': 5},
                              0xbe: {'addr_mode': 'EXTENDED_WORD',
                                    'bytes': 3,
                                    'cycles': 6}},
                       'read_from_memory': WORD,
                       'register': REG_X,
                       'write_to_memory': None},
                'LDY': {'HNZVC': '-aa0-',
                       'desc': 'Y = M:M+1',
                       'needs_ea': False,
                       'ops': {0x108e: {'addr_mode': IMMEDIATE_WORD,
                                      'bytes': 4,
                                      'cycles': 4},
                              0x109e: {'addr_mode': 'DIRECT_WORD',
                                      'bytes': 3,
                                      'cycles': 6},
                              0x10ae: {'addr_mode': 'INDEXED_WORD',
                                      'bytes': 3,
                                      'cycles': 6},
                              0x10be: {'addr_mode': 'EXTENDED_WORD',
                                      'bytes': 4,
                                      'cycles': 7}},
                       'read_from_memory': WORD,
                       'register': REG_Y,
                       'write_to_memory': None}},
    'operation': "R' = M",
    'source form': 'LDA P; LDB P'},
LEA: {'comment': "Instruction Operation Comment\nInstruction\n\nOperation\n\nComment\nLEAX 10,X X+10 -> X Adds 5-bit constant 10 to X\nLEAX 500,X X+500 -> X Adds 16-bit constant 500 to X\nLEAY A,Y Y+A -> Y Adds 8-bit accumulator to Y\nLEAY D,Y Y+D -> Y Adds 16-bit D accumulator to Y\nLEAU -10,U U-10 -> U Subtracts 10 from U\nLEAS -10,S S-10 -> S Used to reserve area on stack\nLEAS 10,S S+10 -> S Used to 'clean up' stack\nLEAX 5,S S+5 -> X Transfers as well as adds",
     'condition code': 'H - Not affected.\nN - Not affected.\nZ - LEAX, LEAY: Set if the result is zero; cleared otherwise. LEAS, LEAU: Not affected.\nV - Not affected.\nC - Not affected.',
     'description': 'Calculates the effective address from the indexed addressing mode and places the address in an indexable register. LEAX and LEAY affect the Z (zero) bit to allow use of these registers as counters and for MC6800 INX/DEX compatibility. LEAU and LEAS do not affect the Z bit to allow cleaning up the stack while returning the Z bit as a parameter to a calling routine, and also for MC6800 INS/DES compatibility.',
     'instr_desc': 'Load effective address into stack pointer',
     'mnemonic': {'LEAS': {'HNZVC': '-----',
                         'desc': 'S = EA',
                         'needs_ea': True,
                         'ops': {0x32: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4}},
                         'read_from_memory': None,
                         'register': REG_S,
                         'write_to_memory': None},
                 'LEAU': {'HNZVC': '-----',
                         'desc': 'U = EA',
                         'needs_ea': True,
                         'ops': {0x33: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4}},
                         'read_from_memory': None,
                         'register': REG_U,
                         'write_to_memory': None},
                 'LEAX': {'HNZVC': '--a--',
                         'desc': 'X = EA',
                         'needs_ea': True,
                         'ops': {0x30: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4}},
                         'read_from_memory': None,
                         'register': REG_X,
                         'write_to_memory': None},
                 'LEAY': {'HNZVC': '--a--',
                         'desc': 'Y = EA',
                         'needs_ea': True,
                         'ops': {0x31: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4}},
                         'read_from_memory': None,
                         'register': REG_Y,
                         'write_to_memory': None}},
     'operation': "R' = EA",
     'source form': 'LEAX, LEAY, LEAS, LEAU'},
LSL: {'comment': 'This is a duplicate assembly-language mnemonic for the single machine instruction ASL.',
     'condition code': 'H - Undefined.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Loaded with the result of the exclusive OR of bits six and seven of the original operand.\nC - Loaded with bit seven of the original operand.',
     'description': 'Shifts all bits of accumulator A or B or memory location M one place to the left.\nBit zero is loaded with a zero.\nBit seven of accumulator A or B or memory location M is shifted into the C (carry) bit.',
     'instr_desc': 'Logical shift left accumulator or memory location',
     'mnemonic': {LSL: {'HNZVC': 'naaas',
                      'desc': 'M = Logical shift M left',
                      'needs_ea': True,
                      'ops': {0x8: {'addr_mode': DIRECT,
                                  'bytes': 2,
                                  'cycles': 6},
                             0x68: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 6},
                             0x78: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 7}},
                      'read_from_memory': BYTE,
                      'register': None,
                      'write_to_memory': BYTE},
                 'LSLA': {'HNZVC': 'naaas',
                         'desc': 'A = Logical shift A left',
                         'needs_ea': False,
                         'ops': {0x48: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_A,
                         'write_to_memory': None},
                 'LSLB': {'HNZVC': 'naaas',
                         'desc': 'B = Logical shift B left',
                         'needs_ea': False,
                         'ops': {0x58: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': 'C = = 0\nb7 = b0',
     'source form': 'LSL Q; LSLA; LSLB'},
LSR: {'condition code': 'H - Not affected.\nN - Always cleared.\nZ - Set if the result is zero; cleared otherwise.\nV - Not affected.\nC - Loaded with bit zero of the original operand.',
     'description': 'Performs a logical shift right on the operand.\nShifts a zero into bit seven and bit zero into the C (carry) bit.',
     'instr_desc': 'Logical shift right accumulator or memory location',
     'mnemonic': {LSR: {'HNZVC': '-0a-s',
                      'desc': 'M = Logical shift M right',
                      'needs_ea': True,
                      'ops': {0x4: {'addr_mode': DIRECT,
                                  'bytes': 2,
                                  'cycles': 6},
                             0x64: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 6},
                             0x74: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 7}},
                      'read_from_memory': BYTE,
                      'register': None,
                      'write_to_memory': BYTE},
                 'LSRA': {'HNZVC': '-0a-s',
                         'desc': 'A = Logical shift A right',
                         'needs_ea': False,
                         'ops': {0x44: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_A,
                         'write_to_memory': None},
                 'LSRB': {'HNZVC': '-0a-s',
                         'desc': 'B = Logical shift B right',
                         'needs_ea': False,
                         'ops': {0x54: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': '0 -> -> C\nb7 -> b0',
     'source form': 'LSR Q; LSRA; LSRB'},
MUL: {'comment': 'The C (carry) bit allows rounding the most-significant byte through the sequence: MUL, ADCA #0.',
     'condition code': 'H - Not affected.\nN - Not affected.\nZ - Set if the result is zero; cleared otherwise.\nV - Not affected.\nC - Set if ACCB bit 7 of result is set; cleared otherwise.',
     'description': 'Multiply the unsigned binary numbers in the accumulators and place the result in both accumulators (ACCA contains the most-significant byte of the result).\nUnsigned multiply allows multiple-precision operations.',
     'instr_desc': 'Unsigned multiply (A * B ? D)',
     'mnemonic': {MUL: {'HNZVC': '--a-a',
                      'desc': 'D = A*B (Unsigned)',
                      'needs_ea': False,
                      'ops': {0x3d: {'addr_mode': INHERENT,
                                   'bytes': 1,
                                   'cycles': 11}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None}},
     'operation': "ACCA':ACCB' = ACCA * ACCB",
     'source form': MUL},
NEG: {'condition code': 'H - Undefined.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Set if the original operand was 10000000 2 .\nC - Set if a borrow is generated; cleared otherwise.',
     'description': 'Replaces the operand with its twos complement.\nThe C (carry) bit represents a borrow and is set to the inverse of the resulting binary carry.\nNote that 80 16 is replaced by itself and only in this case is the V (overflow) bit set.\nThe value 00 16 is also replaced by itself, and only in this case is the C (carry) bit cleared.',
     'instr_desc': 'Negate accumulator or memory',
     'mnemonic': {NEG: {'HNZVC': 'uaaaa',
                      'desc': 'M = !M + 1',
                      'needs_ea': True,
                      'ops': {0x0: {'addr_mode': DIRECT,
                                  'bytes': 2,
                                  'cycles': 6},
                             0x60: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 6},
                             0x70: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 7}},
                      'read_from_memory': BYTE,
                      'register': None,
                      'write_to_memory': BYTE},
                 'NEGA': {'HNZVC': 'uaaaa',
                         'desc': 'A = !A + 1',
                         'needs_ea': False,
                         'ops': {0x40: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_A,
                         'write_to_memory': None},
                 'NEGB': {'HNZVC': 'uaaaa',
                         'desc': 'B = !B + 1',
                         'needs_ea': False,
                         'ops': {0x50: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': "M' = 0 - M",
     'source form': 'NEG Q; NEGA; NEG B'},
NOP: {'condition code': 'This instruction causes only the program counter to be incremented.\nNo other registers or memory locations are affected.',
     'description': '',
     'instr_desc': 'No operation',
     'mnemonic': {NOP: {'HNZVC': '-----',
                      'desc': 'No Operation',
                      'needs_ea': False,
                      'ops': {0x12: {'addr_mode': INHERENT,
                                   'bytes': 1,
                                   'cycles': 2}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None}},
     'operation': 'Not affected.',
     'source form': NOP},
OR: {'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Always cleared.\nC - Not affected.',
    'description': 'Performs an inclusive OR operation between the contents of accumulator A or B and the contents of memory location M and the result is stored in accumulator A or B.',
    'instr_desc': 'OR memory with accumulator',
    'mnemonic': {'ORA': {'HNZVC': '-aa0-',
                       'desc': 'A = A || M',
                       'needs_ea': False,
                       'ops': {0x8a: {'addr_mode': IMMEDIATE,
                                    'bytes': 2,
                                    'cycles': 2},
                              0x9a: {'addr_mode': DIRECT,
                                    'bytes': 2,
                                    'cycles': 4},
                              0xaa: {'addr_mode': INDEXED,
                                    'bytes': 2,
                                    'cycles': 4},
                              0xba: {'addr_mode': EXTENDED,
                                    'bytes': 3,
                                    'cycles': 5}},
                       'read_from_memory': BYTE,
                       'register': REG_A,
                       'write_to_memory': None},
                'ORB': {'HNZVC': '-aa0-',
                       'desc': 'B = B || M',
                       'needs_ea': False,
                       'ops': {0xca: {'addr_mode': IMMEDIATE,
                                    'bytes': 2,
                                    'cycles': 2},
                              0xda: {'addr_mode': DIRECT,
                                    'bytes': 2,
                                    'cycles': 4},
                              0xea: {'addr_mode': INDEXED,
                                    'bytes': 2,
                                    'cycles': 4},
                              0xfa: {'addr_mode': EXTENDED,
                                    'bytes': 3,
                                    'cycles': 5}},
                       'read_from_memory': BYTE,
                       'register': REG_B,
                       'write_to_memory': None},
                'ORCC': {'HNZVC': 'ddddd',
                        'desc': 'C = CC || IMM',
                        'needs_ea': False,
                        'ops': {0x1a: {'addr_mode': IMMEDIATE,
                                     'bytes': 2,
                                     'cycles': 3}},
                        'read_from_memory': BYTE,
                        'register': REG_CC,
                        'write_to_memory': None}},
    'operation': "R' = R OR M",
    'source form': 'ORA P; ORB P'},
PAGE: {'description': 'Page 1/2 instructions',
      'instr_desc': 'Page 2 Instructions prefix',
      'mnemonic': {'PAGE 1': {'HNZVC': '+++++',
                            'desc': 'Page 1 Instructions prefix',
                            'needs_ea': False,
                            'ops': {0x10: {'addr_mode': None,
                                         'bytes': 1,
                                         'cycles': 1}},
                            'read_from_memory': None,
                            'register': None,
                            'write_to_memory': None},
                  'PAGE 2': {'HNZVC': '+++++',
                            'desc': 'Page 2 Instructions prefix',
                            'needs_ea': False,
                            'ops': {0x11: {'addr_mode': None,
                                         'bytes': 1,
                                         'cycles': 1}},
                            'read_from_memory': None,
                            'register': None,
                            'write_to_memory': None}}},
'PSH': {'comment': 'A single register may be placed on the stack with the condition codes set by doing an autodecrement store onto the stack (example: STX ,--S).',
       'condition code': 'Not affected.',
       'description': 'All, some, or none of the processor registers are pushed onto the hardware stack (with the exception of the hardware stack pointer itself).',
       'instr_desc': 'Push A, B, CC, DP, D, X, Y, U, or PC onto hardware stack',
       'mnemonic': {PSHS: {'HNZVC': '-----',
                         'desc': 'S -= 1: MEM(S) = R; Push Register on S Stack',
                         'needs_ea': False,
                         'ops': {0x34: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_S,
                         'write_to_memory': None},
                   PSHU: {'HNZVC': '-----',
                         'desc': 'U -= 1: MEM(U) = R; Push Register on U Stack',
                         'needs_ea': False,
                         'ops': {0x36: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_U,
                         'write_to_memory': None}},
       'operation': 'Push Registers on S Stack: S -= 1: MEM(S) = Reg.',
       'source form': 'b7 b6 b5 b4 b3 b2 b1 b0\nPC U Y X DP B A CC\npush order ->'},
'PUL': {'comment': 'A single register may be pulled from the stack with condition codes set by doing an autoincrement load from the stack (example: LDX ,S++).',
       'condition code': 'May be pulled from stack; not affected otherwise.',
       'description': 'All, some, or none of the processor registers are pulled from the hardware stack (with the exception of the hardware stack pointer itself).',
       'instr_desc': 'Pull A, B, CC, DP, D, X, Y, U, or PC from hardware stack',
       'mnemonic': {PULS: {'HNZVC': 'ccccc',
                         'desc': 'R=MEM(S) : S += 1; Pull register from S Stack',
                         'needs_ea': False,
                         'ops': {0x35: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_S,
                         'write_to_memory': None},
                   PULU: {'HNZVC': 'ccccc',
                         'desc': 'R=MEM(U) : U += 1; Pull register from U Stack',
                         'needs_ea': False,
                         'ops': {0x37: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_U,
                         'write_to_memory': None}},
       'operation': 'Pull Registers from S Stack: Reg. = MEM(S): S += 1',
       'source form': 'b7 b6 b5 b4 b3 b2 b1 b0\nPC U Y X DP B A CC\n= pull order'},
RESET: {'description': 'Build the ASSIST09 vector table and setup monitor defaults, then invoke the monitor startup routine.',
       'instr_desc': '',
       'mnemonic': {RESET: {'HNZVC': '*****',
                          'desc': 'Undocumented opcode',
                          'needs_ea': False,
                          'ops': {0x3e: {'addr_mode': None,
                                       'bytes': 1,
                                       'cycles': -1}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None}}},
ROL: {'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Loaded with the result of the exclusive OR of bits six and seven of the original operand.\nC - Loaded with bit seven of the original operand.',
     'description': 'Rotates all bits of the operand one place left through the C (carry) bit.\nThis is a 9-bit rotation.',
     'instr_desc': 'Rotate accumulator or memory left',
     'mnemonic': {ROL: {'HNZVC': '-aaas',
                      'desc': 'M = Rotate M left thru carry',
                      'needs_ea': True,
                      'ops': {0x9: {'addr_mode': DIRECT,
                                  'bytes': 2,
                                  'cycles': 6},
                             0x69: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 6},
                             0x79: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 7}},
                      'read_from_memory': BYTE,
                      'register': None,
                      'write_to_memory': BYTE},
                 'ROLA': {'HNZVC': '-aaas',
                         'desc': 'A = Rotate A left thru carry',
                         'needs_ea': False,
                         'ops': {0x49: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_A,
                         'write_to_memory': None},
                 'ROLB': {'HNZVC': '-aaas',
                         'desc': 'B = Rotate B left thru carry',
                         'needs_ea': False,
                         'ops': {0x59: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': 'C = = C\nb7 = b0',
     'source form': 'ROL Q; ROLA; ROLB'},
ROR: {'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Not affected.\nC - Loaded with bit zero of the previous operand.',
     'description': 'Rotates all bits of the operand one place right through the C (carry) bit.\nThis is a 9-bit rotation.',
     'instr_desc': 'Rotate accumulator or memory right',
     'mnemonic': {ROR: {'HNZVC': '-aa-s',
                      'desc': 'M = Rotate M Right thru carry',
                      'needs_ea': True,
                      'ops': {0x6: {'addr_mode': DIRECT,
                                  'bytes': 2,
                                  'cycles': 6},
                             0x66: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 6},
                             0x76: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 7}},
                      'read_from_memory': BYTE,
                      'register': None,
                      'write_to_memory': BYTE},
                 'RORA': {'HNZVC': '-aa-s',
                         'desc': 'A = Rotate A Right thru carry',
                         'needs_ea': False,
                         'ops': {0x46: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_A,
                         'write_to_memory': None},
                 'RORB': {'HNZVC': '-aa-s',
                         'desc': 'B = Rotate B Right thru carry',
                         'needs_ea': False,
                         'ops': {0x56: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': 'C -> -> C\nb7 -> b0',
     'source form': 'ROR Q; RORA; RORB'},
RTI: {'condition code': 'Recovered from the stack.',
     'description': 'The saved machine state is recovered from the hardware stack and control is returned to the interrupted program.\nIf the recovered E (entire) bit is clear, it indicates that only a subset of the machine state was saved (return address and condition codes) and only that subset is recovered.',
     'instr_desc': 'Return from interrupt',
     'mnemonic': {RTI: {'HNZVC': '-----',
                      'desc': 'Return from Interrupt',
                      'needs_ea': False,
                      'ops': {0x3b: {'addr_mode': INHERENT,
                                   'bytes': 1,
                                   'cycles': 6}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None}},
     'operation': "IFF CCR bit E is set, then: ACCA' ACCB' DPR' IXH' IXL' IYH' IYL' USH' USL' PCH' PCL' = (SP), SP' = SP+1 = (SP), SP' = SP+1 = (SP), SP' = SP+1 = (SP), SP' = SP+1 = (SP), SP' = SP+1 = (SP), SP' = SP+1 = (SP), SP' = SP+1 = (SP), SP' = SP+1 = (SP), SP' = SP+1 = (SP), SP' = SP+1 = (SP), SP' = SP+1\nIFF CCR bit E is clear, then: PCH' PCL' = (SP), SP' = SP+1 = (SP), SP' = SP+1",
     'source form': RTI},
RTS: {'condition code': 'Not affected.',
     'description': 'Program control is returned from the subroutine to the calling program.\nThe return address is pulled from the stack.',
     'instr_desc': 'Return from subroutine',
     'mnemonic': {RTS: {'HNZVC': '-----',
                      'desc': 'Return from subroutine',
                      'needs_ea': False,
                      'ops': {0x39: {'addr_mode': INHERENT,
                                   'bytes': 1,
                                   'cycles': 5}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None}},
     'operation': "PCH' = (SP), SP' = SP+1 PCL' = (SP), SP' = SP+1",
     'source form': RTS},
SBC: {'condition code': 'H - Undefined.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Set if an overflow is generated; cleared otherwise.\nC - Set if a borrow is generated; cleared otherwise.',
     'description': 'Subtracts the contents of memory location M and the borrow (in the C (carry) bit) from the contents of the designated 8-bit register, and places the result in that register.\nThe C bit represents a borrow and is set to the inverse of the resulting binary carry.',
     'instr_desc': 'Subtract memory from accumulator with borrow',
     'mnemonic': {'SBCA': {'HNZVC': 'uaaaa',
                         'desc': 'A = A - M - C',
                         'needs_ea': False,
                         'ops': {0x82: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0x92: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xa2: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xb2: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_A,
                         'write_to_memory': None},
                 'SBCB': {'HNZVC': 'uaaaa',
                         'desc': 'B = B - M - C',
                         'needs_ea': False,
                         'ops': {0xc2: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0xd2: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xe2: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xf2: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': "R' = R - M - C",
     'source form': 'SBCA P; SBCB P'},
SEX: {'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Not affected.\nC - Not affected.',
     'description': 'This instruction transforms a twos complement 8-bit value in accumulator B into a twos complement 16-bit value in the D accumulator.',
     'instr_desc': 'Sign Extend B accumulator into A accumulator',
     'mnemonic': {SEX: {'HNZVC': '-aa0-',
                      'desc': 'Sign extend B into A',
                      'needs_ea': False,
                      'ops': {0x1d: {'addr_mode': INHERENT,
                                   'bytes': 1,
                                   'cycles': 2}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None}},
     'operation': "If bit seven of ACCB is set then ACCA' = FF 16 else ACCA' = 00 16",
     'source form': SEX},
ST: {'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Always cleared.\nC - Not affected.',
    'description': 'Writes the contents of an 8-bit register into a memory location.',
    'instr_desc': 'Store accumulator to memroy',
    'mnemonic': {'STA': {'HNZVC': '-aa0-',
                       'desc': 'M = A',
                       'needs_ea': True,
                       'ops': {0x97: {'addr_mode': DIRECT,
                                    'bytes': 2,
                                    'cycles': 4},
                              0xa7: {'addr_mode': INDEXED,
                                    'bytes': 2,
                                    'cycles': 4},
                              0xb7: {'addr_mode': EXTENDED,
                                    'bytes': 3,
                                    'cycles': 5}},
                       'read_from_memory': None,
                       'register': REG_A,
                       'write_to_memory': BYTE},
                'STB': {'HNZVC': '-aa0-',
                       'desc': 'M = B',
                       'needs_ea': True,
                       'ops': {0xd7: {'addr_mode': DIRECT,
                                    'bytes': 2,
                                    'cycles': 4},
                              0xe7: {'addr_mode': INDEXED,
                                    'bytes': 2,
                                    'cycles': 4},
                              0xf7: {'addr_mode': EXTENDED,
                                    'bytes': 3,
                                    'cycles': 5}},
                       'read_from_memory': None,
                       'register': REG_B,
                       'write_to_memory': BYTE},
                'STD': {'HNZVC': '-aa0-',
                       'desc': 'M:M+1 = D',
                       'needs_ea': True,
                       'ops': {0xdd: {'addr_mode': DIRECT,
                                    'bytes': 2,
                                    'cycles': 5},
                              0xed: {'addr_mode': INDEXED,
                                    'bytes': 2,
                                    'cycles': 5},
                              0xfd: {'addr_mode': EXTENDED,
                                    'bytes': 3,
                                    'cycles': 6}},
                       'read_from_memory': None,
                       'register': REG_D,
                       'write_to_memory': WORD},
                'STS': {'HNZVC': '-aa0-',
                       'desc': 'M:M+1 = S',
                       'needs_ea': True,
                       'ops': {0x10df: {'addr_mode': DIRECT,
                                      'bytes': 3,
                                      'cycles': 6},
                              0x10ef: {'addr_mode': INDEXED,
                                      'bytes': 3,
                                      'cycles': 6},
                              0x10ff: {'addr_mode': EXTENDED,
                                      'bytes': 4,
                                      'cycles': 7}},
                       'read_from_memory': None,
                       'register': REG_S,
                       'write_to_memory': WORD},
                'STU': {'HNZVC': '-aa0-',
                       'desc': 'M:M+1 = U',
                       'needs_ea': True,
                       'ops': {0xdf: {'addr_mode': DIRECT,
                                    'bytes': 2,
                                    'cycles': 5},
                              0xef: {'addr_mode': INDEXED,
                                    'bytes': 2,
                                    'cycles': 5},
                              0xff: {'addr_mode': EXTENDED,
                                    'bytes': 3,
                                    'cycles': 6}},
                       'read_from_memory': None,
                       'register': REG_U,
                       'write_to_memory': WORD},
                'STX': {'HNZVC': '-aa0-',
                       'desc': 'M:M+1 = X',
                       'needs_ea': True,
                       'ops': {0x9f: {'addr_mode': DIRECT,
                                    'bytes': 2,
                                    'cycles': 5},
                              0xaf: {'addr_mode': INDEXED,
                                    'bytes': 2,
                                    'cycles': 5},
                              0xbf: {'addr_mode': EXTENDED,
                                    'bytes': 3,
                                    'cycles': 6}},
                       'read_from_memory': None,
                       'register': REG_X,
                       'write_to_memory': WORD},
                'STY': {'HNZVC': '-aa0-',
                       'desc': 'M:M+1 = Y',
                       'needs_ea': True,
                       'ops': {0x109f: {'addr_mode': DIRECT,
                                      'bytes': 3,
                                      'cycles': 6},
                              0x10af: {'addr_mode': INDEXED,
                                      'bytes': 3,
                                      'cycles': 6},
                              0x10bf: {'addr_mode': EXTENDED,
                                      'bytes': 4,
                                      'cycles': 7}},
                       'read_from_memory': None,
                       'register': REG_Y,
                       'write_to_memory': WORD}},
    'operation': "M' = R",
    'source form': 'STA P; STB P'},
SUB: {'condition code': 'H - Undefined.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Set if the overflow is generated; cleared otherwise.\nC - Set if a borrow is generated; cleared otherwise.',
     'description': 'Subtracts the value in memory location M from the contents of a designated 8-bit register.\nThe C (carry) bit represents a borrow and is set to the inverse of the resulting binary carry.',
     'instr_desc': 'Subtract memory from accumulator',
     'mnemonic': {'SUBA': {'HNZVC': 'uaaaa',
                         'desc': 'A = A - M',
                         'needs_ea': False,
                         'ops': {0x80: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0x90: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xa0: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xb0: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_A,
                         'write_to_memory': None},
                 'SUBB': {'HNZVC': 'uaaaa',
                         'desc': 'B = B - M',
                         'needs_ea': False,
                         'ops': {0xc0: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0xd0: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xe0: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xf0: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_B,
                         'write_to_memory': None},
                 'SUBD': {'HNZVC': '-aaaa',
                         'desc': 'D = D - M:M+1',
                         'needs_ea': False,
                         'ops': {0x83: {'addr_mode': IMMEDIATE_WORD,
                                      'bytes': 3,
                                      'cycles': 4},
                                0x93: {'addr_mode': 'DIRECT_WORD',
                                      'bytes': 2,
                                      'cycles': 6},
                                0xa3: {'addr_mode': 'INDEXED_WORD',
                                      'bytes': 2,
                                      'cycles': 6},
                                0xb3: {'addr_mode': 'EXTENDED_WORD',
                                      'bytes': 3,
                                      'cycles': 7}},
                         'read_from_memory': WORD,
                         'register': REG_D,
                         'write_to_memory': None}},
     'operation': "R' = R - M",
     'source form': 'SUBA P; SUBB P'},
SWI: {'condition code': 'Not affected.',
     'description': 'All of the processor registers are pushed onto the hardware stack (with the exception of the hardware stack pointer itself), and control is transferred through the software interrupt vector.\nBoth the normal and fast interrupts are masked (disabled).',
     'instr_desc': 'Software interrupt (absolute indirect)',
     'mnemonic': {SWI: {'HNZVC': '-----',
                      'desc': 'Software interrupt 1',
                      'needs_ea': False,
                      'ops': {0x3f: {'addr_mode': INHERENT,
                                   'bytes': 1,
                                   'cycles': 19}},
                      'read_from_memory': None,
                      'register': None,
                      'write_to_memory': None},
                 SWI2: {'HNZVC': '-----',
                       'desc': 'Software interrupt 2',
                       'needs_ea': False,
                       'ops': {0x103f: {'addr_mode': INHERENT,
                                      'bytes': 2,
                                      'cycles': 20}},
                       'read_from_memory': None,
                       'register': None,
                       'write_to_memory': None},
                 SWI3: {'HNZVC': '-----',
                       'desc': 'Software interrupt 3',
                       'needs_ea': False,
                       'ops': {0x113f: {'addr_mode': INHERENT,
                                      'bytes': 2,
                                      'cycles': 20}},
                       'read_from_memory': None,
                       'register': None,
                       'write_to_memory': None}},
     'operation': "Set E (entire state will be saved) SP' = SP-1, (SP) = PCL SP' = SP-1, (SP) = PCH SP' = SP-1, (SP) = USL SP' = SP-1, (SP) = USH SP' = SP-1, (SP) = IYL SP' = SP-1, (SP) = IYH SP' = SP-1, (SP) = IXL SP' = SP-1, (SP) = IXH SP' = SP-1, (SP) = DPR SP' = SP-1, (SP) = ACCB SP' = SP-1, (SP) = ACCA SP' = SP-1, (SP) = CCR Set I, F (mask interrupts) PC' = (FFFA):(FFFB)",
     'source form': SWI},
SYNC: {'condition code': 'Not affected.',
      'description': 'FAST SYNC WAIT FOR DATA\nInterrupt!\nLDA DISC DATA FROM DISC AND CLEAR INTERRUPT\nSTA ,X+ PUT IN BUFFER\nDECB COUNT IT, DONE?\nBNE FAST GO AGAIN IF NOT.',
      'instr_desc': 'Synchronize with interrupt line',
      'mnemonic': {SYNC: {'HNZVC': '-----',
                        'desc': 'Synchronize to Interrupt',
                        'needs_ea': False,
                        'ops': {0x13: {'addr_mode': INHERENT,
                                     'bytes': 1,
                                     'cycles': 2}},
                        'read_from_memory': None,
                        'register': None,
                        'write_to_memory': None}},
      'operation': 'Stop processing instructions',
      'source form': SYNC},
TFR: {'condition code': 'Not affected unless R2 is the condition code register.',
     'description': '0000 = A:B 1000 = A\n0001 = X 1001 = B\n0010 = Y 1010 = CCR\n0011 = US 1011 = DPR\n0100 = SP 1100 = Undefined\n0101 = PC 1101 = Undefined\n0110 = Undefined 1110 = Undefined\n0111 = Undefined 1111 = Undefined',
     'instr_desc': 'Transfer R1 to R2',
     'mnemonic': {TFR: {'HNZVC': 'ccccc',
                      'desc': None,
                      'needs_ea': False,
                      'ops': {0x1f: {'addr_mode': IMMEDIATE,
                                   'bytes': 2,
                                   'cycles': 7}},
                      'read_from_memory': BYTE,
                      'register': None,
                      'write_to_memory': None}},
     'operation': 'R1 -> R2',
     'source form': 'TFR R1, R2'},
TST: {'comment': 'The MC6800 processor clears the C (carry) bit.',
     'condition code': 'H - Not affected.\nN - Set if the result is negative; cleared otherwise.\nZ - Set if the result is zero; cleared otherwise.\nV - Always cleared.\nC - Not affected.',
     'description': 'Set the N (negative) and Z (zero) bits according to the contents of memory location M, and clear the V (overflow) bit.\nThe TST instruction provides only minimum information when testing unsigned values; since no unsigned value is less than zero, BLO and BLS have no utility.\nWhile BHI could be used after TST, it provides exactly the same control as BNE, which is preferred.\nThe signed branches are available.',
     'instr_desc': 'Test accumulator or memory location',
     'mnemonic': {TST: {'HNZVC': '-aa0-',
                      'desc': 'Test M',
                      'needs_ea': False,
                      'ops': {0xd: {'addr_mode': DIRECT,
                                  'bytes': 2,
                                  'cycles': 6},
                             0x6d: {'addr_mode': INDEXED,
                                   'bytes': 2,
                                   'cycles': 6},
                             0x7d: {'addr_mode': EXTENDED,
                                   'bytes': 3,
                                   'cycles': 7}},
                      'read_from_memory': BYTE,
                      'register': None,
                      'write_to_memory': None},
                 'TSTA': {'HNZVC': '-aa0-',
                         'desc': 'Test A',
                         'needs_ea': False,
                         'ops': {0x4d: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_A,
                         'write_to_memory': None},
                 'TSTB': {'HNZVC': '-aa0-',
                         'desc': 'Test B',
                         'needs_ea': False,
                         'ops': {0x5d: {'addr_mode': INHERENT,
                                      'bytes': 1,
                                      'cycles': 2}},
                         'read_from_memory': None,
                         'register': REG_B,
                         'write_to_memory': None}},
     'operation': 'TEMP = M - 0',
     'source form': 'TST Q; TSTA; TSTB'}}
