#!/usr/bin/env python

"""
    6809 instruction set data
    ~~~~~~~~~~~~~~~~~~~~~~~~~

    merged data from:
        * http://www.maddes.net/m6809pm/sections.htm#sec4_4
        * http://www.burgins.com/m6809.html
        * http://www.maddes.net/m6809pm/appendix_a.htm#appA

    Note:
    * read_from_memory: it's "excluded" the address modes routines.
        So if the address mode will fetch the memory to get the
        effective address, but the content of the memory is not needed in
        the instruction them self, the read_from_memory must be set to False.

    Generated data is online here:
    https://docs.google.com/spreadsheet/ccc?key=0Alhtym6D6yKjdFBtNmF0UVR5OW05S3psaURnUTNtSFE

    :copyleft: 2013-2014 by Jens Diemer
    :license: GNU GPL v3 or above, see LICENSE for more details.
"""




BYTE = "8"
WORD = "16"


# Address modes:
DIRECT = "DIRECT"
DIRECT_WORD = "DIRECT_WORD"
EXTENDED = "EXTENDED"
EXTENDED_WORD = "EXTENDED_WORD"
IMMEDIATE = "IMMEDIATE"
IMMEDIATE_WORD = "IMMEDIATE_WORD"
INDEXED = "INDEXED"
INDEXED_WORD = "INDEXED_WORD"
INHERENT = "INHERENT"
RELATIVE = "RELATIVE"
RELATIVE_WORD = "RELATIVE_WORD"


# Registers:
REG_A = "A"
REG_B = "B"
REG_CC = "CC"
REG_D = "D"
REG_DP = "DP"
REG_PC = "PC"
REG_S = "S"
REG_U = "U"
REG_X = "X"
REG_Y = "Y"


OP_DATA={'ABX': {'mnemonic': {'ABX': {'needs_ea': False,
                          'ops': {0x3a: {'addr_mode': INHERENT,
                                       'bytes': 1,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None}}},
'ADC': {'mnemonic': {'ADCA': {'needs_ea': False,
                           'ops': {0x89: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0x99: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xa9: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xb9: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_A,
                           'write_to_memory': None},
                   'ADCB': {'needs_ea': False,
                           'ops': {0xc9: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0xd9: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xe9: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xf9: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_B,
                           'write_to_memory': None}}},
'ADD': {'mnemonic': {'ADDA': {'needs_ea': False,
                           'ops': {0x8b: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0x9b: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xab: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xbb: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_A,
                           'write_to_memory': None},
                   'ADDB': {'needs_ea': False,
                           'ops': {0xcb: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0xdb: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xeb: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xfb: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_B,
                           'write_to_memory': None},
                   'ADDD': {'needs_ea': False,
                           'ops': {0xc3: {'addr_mode': IMMEDIATE_WORD,
                                        'bytes': 3,
                                        'cycles': 4},
                                  0xd3: {'addr_mode': DIRECT_WORD,
                                        'bytes': 2,
                                        'cycles': 6},
                                  0xe3: {'addr_mode': INDEXED_WORD,
                                        'bytes': 2,
                                        'cycles': 6},
                                  0xf3: {'addr_mode': EXTENDED_WORD,
                                        'bytes': 3,
                                        'cycles': 7}},
                           'read_from_memory': WORD,
                           'register': REG_D,
                           'write_to_memory': None}}},
'AND': {'mnemonic': {'ANDA': {'needs_ea': False,
                           'ops': {0x84: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0x94: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xa4: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xb4: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_A,
                           'write_to_memory': None},
                   'ANDB': {'needs_ea': False,
                           'ops': {0xc4: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0xd4: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xe4: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xf4: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_B,
                           'write_to_memory': None},
                   'ANDCC': {'needs_ea': False,
                            'ops': {0x1c: {'addr_mode': IMMEDIATE,
                                         'bytes': 2,
                                         'cycles': 3}},
                            'read_from_memory': BYTE,
                            'register': REG_CC,
                            'write_to_memory': None}}},
'ASR': {'mnemonic': {'ASR': {'needs_ea': True,
                          'ops': {0x07: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x67: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x77: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 7}},
                          'read_from_memory': BYTE,
                          'register': None,
                          'write_to_memory': BYTE},
                   'ASRA': {'needs_ea': False,
                           'ops': {0x47: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_A,
                           'write_to_memory': None},
                   'ASRB': {'needs_ea': False,
                           'ops': {0x57: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_B,
                           'write_to_memory': None}}},
'BEQ': {'mnemonic': {'BEQ': {'needs_ea': True,
                          'ops': {0x27: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBEQ': {'needs_ea': True,
                           'ops': {0x1027: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BGE': {'mnemonic': {'BGE': {'needs_ea': True,
                          'ops': {0x2c: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBGE': {'needs_ea': True,
                           'ops': {0x102c: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BGT': {'mnemonic': {'BGT': {'needs_ea': True,
                          'ops': {0x2e: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBGT': {'needs_ea': True,
                           'ops': {0x102e: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BHI': {'mnemonic': {'BHI': {'needs_ea': True,
                          'ops': {0x22: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBHI': {'needs_ea': True,
                           'ops': {0x1022: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BHS': {'mnemonic': {'BCC': {'needs_ea': True,
                          'ops': {0x24: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBCC': {'needs_ea': True,
                           'ops': {0x1024: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BIT': {'mnemonic': {'BITA': {'needs_ea': False,
                           'ops': {0x85: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0x95: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xa5: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xb5: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_A,
                           'write_to_memory': None},
                   'BITB': {'needs_ea': False,
                           'ops': {0xc5: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0xd5: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xe5: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xf5: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_B,
                           'write_to_memory': None}}},
'BLE': {'mnemonic': {'BLE': {'needs_ea': True,
                          'ops': {0x2f: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBLE': {'needs_ea': True,
                           'ops': {0x102f: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BLO': {'mnemonic': {'BLO': {'needs_ea': True,
                          'ops': {0x25: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBCS': {'needs_ea': True,
                           'ops': {0x1025: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BLS': {'mnemonic': {'BLS': {'needs_ea': True,
                          'ops': {0x23: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBLS': {'needs_ea': True,
                           'ops': {0x1023: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BLT': {'mnemonic': {'BLT': {'needs_ea': True,
                          'ops': {0x2d: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBLT': {'needs_ea': True,
                           'ops': {0x102d: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BMI': {'mnemonic': {'BMI': {'needs_ea': True,
                          'ops': {0x2b: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBMI': {'needs_ea': True,
                           'ops': {0x102b: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BNE': {'mnemonic': {'BNE': {'needs_ea': True,
                          'ops': {0x26: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBNE': {'needs_ea': True,
                           'ops': {0x1026: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BPL': {'mnemonic': {'BPL': {'needs_ea': True,
                          'ops': {0x2a: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBPL': {'needs_ea': True,
                           'ops': {0x102a: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BRA': {'mnemonic': {'BRA': {'needs_ea': True,
                          'ops': {0x20: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBRA': {'needs_ea': True,
                           'ops': {0x16: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BRN': {'mnemonic': {'BRN': {'needs_ea': True,
                          'ops': {0x21: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBRN': {'needs_ea': True,
                           'ops': {0x1021: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BSR': {'mnemonic': {'BSR': {'needs_ea': True,
                          'ops': {0x8d: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 7}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBSR': {'needs_ea': True,
                           'ops': {0x17: {'addr_mode': RELATIVE_WORD,
                                        'bytes': 3,
                                        'cycles': 9}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BVC': {'mnemonic': {'BVC': {'needs_ea': True,
                          'ops': {0x28: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBVC': {'needs_ea': True,
                           'ops': {0x1028: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'BVS': {'mnemonic': {'BVS': {'needs_ea': True,
                          'ops': {0x29: {'addr_mode': RELATIVE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'LBVS': {'needs_ea': True,
                           'ops': {0x1029: {'addr_mode': RELATIVE_WORD,
                                          'bytes': 4,
                                          'cycles': 5}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'CLR': {'mnemonic': {'CLR': {'needs_ea': True,
                          'ops': {0x0f: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x6f: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x7f: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 7}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': BYTE},
                   'CLRA': {'needs_ea': False,
                           'ops': {0x4f: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_A,
                           'write_to_memory': None},
                   'CLRB': {'needs_ea': False,
                           'ops': {0x5f: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_B,
                           'write_to_memory': None}}},
'CMP': {'mnemonic': {'CMPA': {'needs_ea': False,
                           'ops': {0x81: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0x91: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xa1: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xb1: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_A,
                           'write_to_memory': None},
                   'CMPB': {'needs_ea': False,
                           'ops': {0xc1: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0xd1: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xe1: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xf1: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_B,
                           'write_to_memory': None},
                   'CMPD': {'needs_ea': False,
                           'ops': {0x1083: {'addr_mode': IMMEDIATE_WORD,
                                          'bytes': 4,
                                          'cycles': 5},
                                  0x1093: {'addr_mode': DIRECT_WORD,
                                          'bytes': 3,
                                          'cycles': 7},
                                  0x10a3: {'addr_mode': INDEXED_WORD,
                                          'bytes': 3,
                                          'cycles': 7},
                                  0x10b3: {'addr_mode': EXTENDED_WORD,
                                          'bytes': 4,
                                          'cycles': 8}},
                           'read_from_memory': WORD,
                           'register': REG_D,
                           'write_to_memory': None},
                   'CMPS': {'needs_ea': False,
                           'ops': {0x118c: {'addr_mode': IMMEDIATE_WORD,
                                          'bytes': 4,
                                          'cycles': 5},
                                  0x119c: {'addr_mode': DIRECT_WORD,
                                          'bytes': 3,
                                          'cycles': 7},
                                  0x11ac: {'addr_mode': INDEXED_WORD,
                                          'bytes': 3,
                                          'cycles': 7},
                                  0x11bc: {'addr_mode': EXTENDED_WORD,
                                          'bytes': 4,
                                          'cycles': 8}},
                           'read_from_memory': WORD,
                           'register': REG_S,
                           'write_to_memory': None},
                   'CMPU': {'needs_ea': False,
                           'ops': {0x1183: {'addr_mode': IMMEDIATE_WORD,
                                          'bytes': 4,
                                          'cycles': 5},
                                  0x1193: {'addr_mode': DIRECT_WORD,
                                          'bytes': 3,
                                          'cycles': 7},
                                  0x11a3: {'addr_mode': INDEXED_WORD,
                                          'bytes': 3,
                                          'cycles': 7},
                                  0x11b3: {'addr_mode': EXTENDED_WORD,
                                          'bytes': 4,
                                          'cycles': 8}},
                           'read_from_memory': WORD,
                           'register': REG_U,
                           'write_to_memory': None},
                   'CMPX': {'needs_ea': False,
                           'ops': {0x8c: {'addr_mode': IMMEDIATE_WORD,
                                        'bytes': 3,
                                        'cycles': 4},
                                  0x9c: {'addr_mode': DIRECT_WORD,
                                        'bytes': 2,
                                        'cycles': 6},
                                  0xac: {'addr_mode': INDEXED_WORD,
                                        'bytes': 2,
                                        'cycles': 6},
                                  0xbc: {'addr_mode': EXTENDED_WORD,
                                        'bytes': 3,
                                        'cycles': 7}},
                           'read_from_memory': WORD,
                           'register': REG_X,
                           'write_to_memory': None},
                   'CMPY': {'needs_ea': False,
                           'ops': {0x108c: {'addr_mode': IMMEDIATE_WORD,
                                          'bytes': 4,
                                          'cycles': 5},
                                  0x109c: {'addr_mode': DIRECT_WORD,
                                          'bytes': 3,
                                          'cycles': 7},
                                  0x10ac: {'addr_mode': INDEXED_WORD,
                                          'bytes': 3,
                                          'cycles': 7},
                                  0x10bc: {'addr_mode': EXTENDED_WORD,
                                          'bytes': 4,
                                          'cycles': 8}},
                           'read_from_memory': WORD,
                           'register': REG_Y,
                           'write_to_memory': None}}},
'COM': {'mnemonic': {'COM': {'needs_ea': True,
                          'ops': {0x03: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x63: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x73: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 7}},
                          'read_from_memory': BYTE,
                          'register': None,
                          'write_to_memory': BYTE},
                   'COMA': {'needs_ea': False,
                           'ops': {0x43: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_A,
                           'write_to_memory': None},
                   'COMB': {'needs_ea': False,
                           'ops': {0x53: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_B,
                           'write_to_memory': None}}},
'CWAI': {'mnemonic': {'CWAI': {'needs_ea': False,
                            'ops': {0x3c: {'addr_mode': IMMEDIATE,
                                         'bytes': 2,
                                         'cycles': 21}},
                            'read_from_memory': BYTE,
                            'register': None,
                            'write_to_memory': None}}},
'DAA': {'mnemonic': {'DAA': {'needs_ea': False,
                          'ops': {0x19: {'addr_mode': INHERENT,
                                       'bytes': 1,
                                       'cycles': 2}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None}}},
'DEC': {'mnemonic': {'DEC': {'needs_ea': True,
                          'ops': {0x0a: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x6a: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x7a: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 7}},
                          'read_from_memory': BYTE,
                          'register': None,
                          'write_to_memory': BYTE},
                   'DECA': {'needs_ea': False,
                           'ops': {0x4a: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_A,
                           'write_to_memory': None},
                   'DECB': {'needs_ea': False,
                           'ops': {0x5a: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_B,
                           'write_to_memory': None}}},
'EOR': {'mnemonic': {'EORA': {'needs_ea': False,
                           'ops': {0x88: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0x98: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xa8: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xb8: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_A,
                           'write_to_memory': None},
                   'EORB': {'needs_ea': False,
                           'ops': {0xc8: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0xd8: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xe8: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xf8: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_B,
                           'write_to_memory': None}}},
'EXG': {'mnemonic': {'EXG': {'needs_ea': False,
                          'ops': {0x1e: {'addr_mode': IMMEDIATE,
                                       'bytes': 2,
                                       'cycles': 8}},
                          'read_from_memory': BYTE,
                          'register': None,
                          'write_to_memory': None}}},
'INC': {'mnemonic': {'INC': {'needs_ea': True,
                          'ops': {0x0c: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x6c: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x7c: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 7}},
                          'read_from_memory': BYTE,
                          'register': None,
                          'write_to_memory': BYTE},
                   'INCA': {'needs_ea': False,
                           'ops': {0x4c: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_A,
                           'write_to_memory': None},
                   'INCB': {'needs_ea': False,
                           'ops': {0x5c: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_B,
                           'write_to_memory': None}}},
'JMP': {'mnemonic': {'JMP': {'needs_ea': True,
                          'ops': {0x0e: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 3},
                                 0x6e: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 3},
                                 0x7e: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 3}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None}}},
'JSR': {'mnemonic': {'JSR': {'needs_ea': True,
                          'ops': {0x9d: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 7},
                                 0xad: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 7},
                                 0xbd: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 8}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None}}},
'LD': {'mnemonic': {'LDA': {'needs_ea': False,
                         'ops': {0x86: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0x96: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xa6: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xb6: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_A,
                         'write_to_memory': None},
                  'LDB': {'needs_ea': False,
                         'ops': {0xc6: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0xd6: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xe6: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xf6: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_B,
                         'write_to_memory': None},
                  'LDD': {'needs_ea': False,
                         'ops': {0xcc: {'addr_mode': IMMEDIATE_WORD,
                                      'bytes': 3,
                                      'cycles': 3},
                                0xdc: {'addr_mode': DIRECT_WORD,
                                      'bytes': 2,
                                      'cycles': 5},
                                0xec: {'addr_mode': INDEXED_WORD,
                                      'bytes': 2,
                                      'cycles': 5},
                                0xfc: {'addr_mode': EXTENDED_WORD,
                                      'bytes': 3,
                                      'cycles': 6}},
                         'read_from_memory': WORD,
                         'register': REG_D,
                         'write_to_memory': None},
                  'LDS': {'needs_ea': False,
                         'ops': {0x10ce: {'addr_mode': IMMEDIATE_WORD,
                                        'bytes': 4,
                                        'cycles': 4},
                                0x10de: {'addr_mode': DIRECT_WORD,
                                        'bytes': 3,
                                        'cycles': 6},
                                0x10ee: {'addr_mode': INDEXED_WORD,
                                        'bytes': 3,
                                        'cycles': 6},
                                0x10fe: {'addr_mode': EXTENDED_WORD,
                                        'bytes': 4,
                                        'cycles': 7}},
                         'read_from_memory': WORD,
                         'register': REG_S,
                         'write_to_memory': None},
                  'LDU': {'needs_ea': False,
                         'ops': {0xce: {'addr_mode': IMMEDIATE_WORD,
                                      'bytes': 3,
                                      'cycles': 3},
                                0xde: {'addr_mode': DIRECT_WORD,
                                      'bytes': 2,
                                      'cycles': 5},
                                0xee: {'addr_mode': INDEXED_WORD,
                                      'bytes': 2,
                                      'cycles': 5},
                                0xfe: {'addr_mode': EXTENDED_WORD,
                                      'bytes': 3,
                                      'cycles': 6}},
                         'read_from_memory': WORD,
                         'register': REG_U,
                         'write_to_memory': None},
                  'LDX': {'needs_ea': False,
                         'ops': {0x8e: {'addr_mode': IMMEDIATE_WORD,
                                      'bytes': 3,
                                      'cycles': 3},
                                0x9e: {'addr_mode': DIRECT_WORD,
                                      'bytes': 2,
                                      'cycles': 5},
                                0xae: {'addr_mode': INDEXED_WORD,
                                      'bytes': 2,
                                      'cycles': 5},
                                0xbe: {'addr_mode': EXTENDED_WORD,
                                      'bytes': 3,
                                      'cycles': 6}},
                         'read_from_memory': WORD,
                         'register': REG_X,
                         'write_to_memory': None},
                  'LDY': {'needs_ea': False,
                         'ops': {0x108e: {'addr_mode': IMMEDIATE_WORD,
                                        'bytes': 4,
                                        'cycles': 4},
                                0x109e: {'addr_mode': DIRECT_WORD,
                                        'bytes': 3,
                                        'cycles': 6},
                                0x10ae: {'addr_mode': INDEXED_WORD,
                                        'bytes': 3,
                                        'cycles': 6},
                                0x10be: {'addr_mode': EXTENDED_WORD,
                                        'bytes': 4,
                                        'cycles': 7}},
                         'read_from_memory': WORD,
                         'register': REG_Y,
                         'write_to_memory': None}}},
'LEA': {'mnemonic': {'LEAS': {'needs_ea': True,
                           'ops': {0x32: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4}},
                           'read_from_memory': None,
                           'register': REG_S,
                           'write_to_memory': None},
                   'LEAU': {'needs_ea': True,
                           'ops': {0x33: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4}},
                           'read_from_memory': None,
                           'register': REG_U,
                           'write_to_memory': None},
                   'LEAX': {'needs_ea': True,
                           'ops': {0x30: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4}},
                           'read_from_memory': None,
                           'register': REG_X,
                           'write_to_memory': None},
                   'LEAY': {'needs_ea': True,
                           'ops': {0x31: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4}},
                           'read_from_memory': None,
                           'register': REG_Y,
                           'write_to_memory': None}}},
'LSL': {'mnemonic': {'LSL': {'needs_ea': True,
                          'ops': {0x08: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x68: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x78: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 7}},
                          'read_from_memory': BYTE,
                          'register': None,
                          'write_to_memory': BYTE},
                   'LSLA': {'needs_ea': False,
                           'ops': {0x48: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_A,
                           'write_to_memory': None},
                   'LSLB': {'needs_ea': False,
                           'ops': {0x58: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_B,
                           'write_to_memory': None}}},
'LSR': {'mnemonic': {'LSR': {'needs_ea': True,
                          'ops': {0x04: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x64: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x74: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 7}},
                          'read_from_memory': BYTE,
                          'register': None,
                          'write_to_memory': BYTE},
                   'LSRA': {'needs_ea': False,
                           'ops': {0x44: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_A,
                           'write_to_memory': None},
                   'LSRB': {'needs_ea': False,
                           'ops': {0x54: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_B,
                           'write_to_memory': None}}},
'MUL': {'mnemonic': {'MUL': {'needs_ea': False,
                          'ops': {0x3d: {'addr_mode': INHERENT,
                                       'bytes': 1,
                                       'cycles': 11}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None}}},
'NEG': {'mnemonic': {'NEG': {'needs_ea': True,
                          'ops': {0x00: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x60: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x70: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 7}},
                          'read_from_memory': BYTE,
                          'register': None,
                          'write_to_memory': BYTE},
                   'NEGA': {'needs_ea': False,
                           'ops': {0x40: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_A,
                           'write_to_memory': None},
                   'NEGB': {'needs_ea': False,
                           'ops': {0x50: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_B,
                           'write_to_memory': None}}},
'NOP': {'mnemonic': {'NOP': {'needs_ea': False,
                          'ops': {0x12: {'addr_mode': INHERENT,
                                       'bytes': 1,
                                       'cycles': 2}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None}}},
'OR': {'mnemonic': {'ORA': {'needs_ea': False,
                         'ops': {0x8a: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0x9a: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xaa: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xba: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_A,
                         'write_to_memory': None},
                  'ORB': {'needs_ea': False,
                         'ops': {0xca: {'addr_mode': IMMEDIATE,
                                      'bytes': 2,
                                      'cycles': 2},
                                0xda: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xea: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xfa: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': BYTE,
                         'register': REG_B,
                         'write_to_memory': None},
                  'ORCC': {'needs_ea': False,
                          'ops': {0x1a: {'addr_mode': IMMEDIATE,
                                       'bytes': 2,
                                       'cycles': 3}},
                          'read_from_memory': BYTE,
                          'register': REG_CC,
                          'write_to_memory': None}}},
'PAGE': {'mnemonic': {'PAGE 1': {'needs_ea': False,
                              'ops': {0x10: {'addr_mode': None,
                                           'bytes': 1,
                                           'cycles': 1}},
                              'read_from_memory': None,
                              'register': None,
                              'write_to_memory': None},
                    'PAGE 2': {'needs_ea': False,
                              'ops': {0x11: {'addr_mode': None,
                                           'bytes': 1,
                                           'cycles': 1}},
                              'read_from_memory': None,
                              'register': None,
                              'write_to_memory': None}}},
'PSH': {'mnemonic': {'PSHS': {'needs_ea': False,
                           'ops': {0x34: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_S,
                           'write_to_memory': None},
                   'PSHU': {'needs_ea': False,
                           'ops': {0x36: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_U,
                           'write_to_memory': None}}},
'PUL': {'mnemonic': {'PULS': {'needs_ea': False,
                           'ops': {0x35: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_S,
                           'write_to_memory': None},
                   'PULU': {'needs_ea': False,
                           'ops': {0x37: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_U,
                           'write_to_memory': None}}},
'RESET': {'mnemonic': {'RESET': {'needs_ea': False,
                              'ops': {0x3e: {'addr_mode': None,
                                           'bytes': 1,
                                           'cycles': -1}},
                              'read_from_memory': None,
                              'register': None,
                              'write_to_memory': None}}},
'ROL': {'mnemonic': {'ROL': {'needs_ea': True,
                          'ops': {0x09: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x69: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x79: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 7}},
                          'read_from_memory': BYTE,
                          'register': None,
                          'write_to_memory': BYTE},
                   'ROLA': {'needs_ea': False,
                           'ops': {0x49: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_A,
                           'write_to_memory': None},
                   'ROLB': {'needs_ea': False,
                           'ops': {0x59: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_B,
                           'write_to_memory': None}}},
'ROR': {'mnemonic': {'ROR': {'needs_ea': True,
                          'ops': {0x06: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x66: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x76: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 7}},
                          'read_from_memory': BYTE,
                          'register': None,
                          'write_to_memory': BYTE},
                   'RORA': {'needs_ea': False,
                           'ops': {0x46: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_A,
                           'write_to_memory': None},
                   'RORB': {'needs_ea': False,
                           'ops': {0x56: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_B,
                           'write_to_memory': None}}},
'RTI': {'mnemonic': {'RTI': {'needs_ea': False,
                          'ops': {0x3b: {'addr_mode': INHERENT,
                                       'bytes': 1,
                                       'cycles': 6}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None}}},
'RTS': {'mnemonic': {'RTS': {'needs_ea': False,
                          'ops': {0x39: {'addr_mode': INHERENT,
                                       'bytes': 1,
                                       'cycles': 5}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None}}},
'SBC': {'mnemonic': {'SBCA': {'needs_ea': False,
                           'ops': {0x82: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0x92: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xa2: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xb2: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_A,
                           'write_to_memory': None},
                   'SBCB': {'needs_ea': False,
                           'ops': {0xc2: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0xd2: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xe2: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xf2: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_B,
                           'write_to_memory': None}}},
'SEX': {'mnemonic': {'SEX': {'needs_ea': False,
                          'ops': {0x1d: {'addr_mode': INHERENT,
                                       'bytes': 1,
                                       'cycles': 2}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None}}},
'ST': {'mnemonic': {'STA': {'needs_ea': True,
                         'ops': {0x97: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xa7: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xb7: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': None,
                         'register': REG_A,
                         'write_to_memory': BYTE},
                  'STB': {'needs_ea': True,
                         'ops': {0xd7: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xe7: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 4},
                                0xf7: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 5}},
                         'read_from_memory': None,
                         'register': REG_B,
                         'write_to_memory': BYTE},
                  'STD': {'needs_ea': True,
                         'ops': {0xdd: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 5},
                                0xed: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 5},
                                0xfd: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 6}},
                         'read_from_memory': None,
                         'register': REG_D,
                         'write_to_memory': WORD},
                  'STS': {'needs_ea': True,
                         'ops': {0x10df: {'addr_mode': DIRECT,
                                        'bytes': 3,
                                        'cycles': 6},
                                0x10ef: {'addr_mode': INDEXED,
                                        'bytes': 3,
                                        'cycles': 6},
                                0x10ff: {'addr_mode': EXTENDED,
                                        'bytes': 4,
                                        'cycles': 7}},
                         'read_from_memory': None,
                         'register': REG_S,
                         'write_to_memory': WORD},
                  'STU': {'needs_ea': True,
                         'ops': {0xdf: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 5},
                                0xef: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 5},
                                0xff: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 6}},
                         'read_from_memory': None,
                         'register': REG_U,
                         'write_to_memory': WORD},
                  'STX': {'needs_ea': True,
                         'ops': {0x9f: {'addr_mode': DIRECT,
                                      'bytes': 2,
                                      'cycles': 5},
                                0xaf: {'addr_mode': INDEXED,
                                      'bytes': 2,
                                      'cycles': 5},
                                0xbf: {'addr_mode': EXTENDED,
                                      'bytes': 3,
                                      'cycles': 6}},
                         'read_from_memory': None,
                         'register': REG_X,
                         'write_to_memory': WORD},
                  'STY': {'needs_ea': True,
                         'ops': {0x109f: {'addr_mode': DIRECT,
                                        'bytes': 3,
                                        'cycles': 6},
                                0x10af: {'addr_mode': INDEXED,
                                        'bytes': 3,
                                        'cycles': 6},
                                0x10bf: {'addr_mode': EXTENDED,
                                        'bytes': 4,
                                        'cycles': 7}},
                         'read_from_memory': None,
                         'register': REG_Y,
                         'write_to_memory': WORD}}},
'SUB': {'mnemonic': {'SUBA': {'needs_ea': False,
                           'ops': {0x80: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0x90: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xa0: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xb0: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_A,
                           'write_to_memory': None},
                   'SUBB': {'needs_ea': False,
                           'ops': {0xc0: {'addr_mode': IMMEDIATE,
                                        'bytes': 2,
                                        'cycles': 2},
                                  0xd0: {'addr_mode': DIRECT,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xe0: {'addr_mode': INDEXED,
                                        'bytes': 2,
                                        'cycles': 4},
                                  0xf0: {'addr_mode': EXTENDED,
                                        'bytes': 3,
                                        'cycles': 5}},
                           'read_from_memory': BYTE,
                           'register': REG_B,
                           'write_to_memory': None},
                   'SUBD': {'needs_ea': False,
                           'ops': {0x83: {'addr_mode': IMMEDIATE_WORD,
                                        'bytes': 3,
                                        'cycles': 4},
                                  0x93: {'addr_mode': DIRECT_WORD,
                                        'bytes': 2,
                                        'cycles': 6},
                                  0xa3: {'addr_mode': INDEXED_WORD,
                                        'bytes': 2,
                                        'cycles': 6},
                                  0xb3: {'addr_mode': EXTENDED_WORD,
                                        'bytes': 3,
                                        'cycles': 7}},
                           'read_from_memory': WORD,
                           'register': REG_D,
                           'write_to_memory': None}}},
'SWI': {'mnemonic': {'SWI': {'needs_ea': False,
                          'ops': {0x3f: {'addr_mode': INHERENT,
                                       'bytes': 1,
                                       'cycles': 19}},
                          'read_from_memory': None,
                          'register': None,
                          'write_to_memory': None},
                   'SWI2': {'needs_ea': False,
                           'ops': {0x103f: {'addr_mode': INHERENT,
                                          'bytes': 2,
                                          'cycles': 20}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None},
                   'SWI3': {'needs_ea': False,
                           'ops': {0x113f: {'addr_mode': INHERENT,
                                          'bytes': 2,
                                          'cycles': 20}},
                           'read_from_memory': None,
                           'register': None,
                           'write_to_memory': None}}},
'SYNC': {'mnemonic': {'SYNC': {'needs_ea': False,
                            'ops': {0x13: {'addr_mode': INHERENT,
                                         'bytes': 1,
                                         'cycles': 2}},
                            'read_from_memory': None,
                            'register': None,
                            'write_to_memory': None}}},
'TFR': {'mnemonic': {'TFR': {'needs_ea': False,
                          'ops': {0x1f: {'addr_mode': IMMEDIATE,
                                       'bytes': 2,
                                       'cycles': 7}},
                          'read_from_memory': BYTE,
                          'register': None,
                          'write_to_memory': None}}},
'TST': {'mnemonic': {'TST': {'needs_ea': False,
                          'ops': {0x0d: {'addr_mode': DIRECT,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x6d: {'addr_mode': INDEXED,
                                       'bytes': 2,
                                       'cycles': 6},
                                 0x7d: {'addr_mode': EXTENDED,
                                       'bytes': 3,
                                       'cycles': 7}},
                          'read_from_memory': BYTE,
                          'register': None,
                          'write_to_memory': None},
                   'TSTA': {'needs_ea': False,
                           'ops': {0x4d: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_A,
                           'write_to_memory': None},
                   'TSTB': {'needs_ea': False,
                           'ops': {0x5d: {'addr_mode': INHERENT,
                                        'bytes': 1,
                                        'cycles': 2}},
                           'read_from_memory': None,
                           'register': REG_B,
                           'write_to_memory': None}}}}
