#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Copyright (C) 2011 by Michael Sarahan

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""

import numpy as np
import msaplots
from matplotlib import pyplot as plt
import utils.msa_rpy as msa

plt.ion()

def run(data,wFacs=None,numFacs=None,ncpus=1,nruns=5):

    print "Data dimensions: "
    print data.shape
    
    if not wFacs:
        print "Running PCA to estimate dimensionality"
        vec, sc, vals = msa.pca( data )
    
        valplot=msaplots.eigenvalueplot(vals)
        valplot.savefig('scree.png')
        if vals.shape[0]>50:
            wFacs=msa.linfit(vals,50,windowsize=5)
            print "\n\nEstimated number of factors as %i."%wFacs
            print "Please check the scree plot window and verify that this is reasonable."
        else:
            wFacs=msa.linfit(vals,vals.shape[0]-10,windowsize=5)
            print "\n\nEstimated number of factors as %i."%wFacs
            print "Please check the scree plot window and verify that this is reasonable."
        cnt=None
        isValid=0
        '''
        while isValid!=1:
            try:
                cnt = raw_input("""
Press Enter to continue, or input an integer 
number of factors to use in ICA instead: """)
                if cnt not in ('','\n'):
                    wFacs=int(cnt)
                    isValid=1
                else:
                    wFacs=wFacs
                    isValid=1      
            except:
                print 'Invalid input.  You must input an integer.'
            
        plt.close('all')
        '''

    print "Starting ICA using %i factors."%wFacs

    # Derive the ICA factors and scores
    iv,isc=msa.diff_ica(data,wFacs)
    
    if not numFacs:
        numFacs=wFacs

    rcon=msa.reconstruct(iv,isc)
    # make the background positive (flip over the spectra)
    if rcon[3,4]<0:
        rcon=rcon*-1
    # offset any negative values
    rcon=rcon-np.min(rcon)
    rcon=rcon[3:-3,:]
    nmf_fac,nmf_sc=msa.nmf(rcon,numFacs,nruns=nruns,ncpus=ncpus)
    return nmf_fac,nmf_sc
    
if __name__=='__main__':
    import getopt
    import sys
    
    def usage():
        print """
        Run this script in any folder with a txt file containing a
        dumped spectrum image.  It will pop up a scree plot and ask you to
        confirm its guess at the correct number of factors to return.  
        It outputs 3 files, which Bernhard Schaffer\'s PCA script reads.
        
        The command line accepts up to four arguments.  Two arguments are for
        specifying energy range (-b and -e), the beginning and end of the 
        energy ranges input to MSA.  Two other arguments, -p and -n, are used
        to specify the number of processors and number of iterations of NMF
        to run.  The default number of processors is one, and the default 
        number of iterations is 5.  This is suitable only for testing the 
        system requirements - verify the amount of RAM used by the single NMF
        process, then re-run this script with more runs (50 is good), and as
        many processes as you have RAM & CPUs for.
        
        Usage Examples:
            run PCA & ICA on entire energy range:
            siRead.py
            run PCA & ICA only between 300 and 600 eV:
            siRead.py -b 300 -e 600
            run 50 NMF runs on 4 processors:
            siRead.py -p 4 -n 50\n"""
    
    try:
        opts, args = getopt.getopt(sys.argv[1:], "hp:n:f:b:e:w:", 
                                   ["help", "ncpus=","nruns=","wFacs=",
                                    "numFacs=","beg=","end="])
    except getopt.GetoptError, err:
        # print help information and exit:
        print str(err) # will print something like "option -a not recognized"
        usage()
        sys.exit(2)

    for o, a in opts:
        if o in ("-h", "--help"):
            usage()
            sys.exit()
        if o in ("-p", "--ncpus"):
            ncpus=int(a)
        if o in ("-n", "--nruns"):
            nruns=int(a)
        if o in ("-f", "--numFacs"):
            numFacs=int(a)
        if o in ("-w", "--wFacs"):
            wFacs=int(a)
        if o in ("-b","--beg"):
            beg = float(a)
        if o in ("-e","--end"):
            end = float(a)
    
    run()
