# -*- coding: iso-8859-1 -*-
"""
Multivariate Statistical Analysis model module.

Copyright (C) 2011 by Michael Sarahan

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""

from sqlalchemy import *
from sqlalchemy.orm import mapper, relation
from sqlalchemy import Table, ForeignKey, Column
from sqlalchemy.types import Integer, Unicode, Float, PickleType
from sqlalchemy.orm import relation, backref
from sqlalchemy.ext.declarative import declarative_base
from zope.sqlalchemy import ZopeTransactionExtension
from sqlalchemy.orm import scoped_session, sessionmaker

import numpy as np

DeclarativeBase=declarative_base()
DBSession = scoped_session(sessionmaker(
                             extension=ZopeTransactionExtension()))

__all__ = ['Data','Data_EELS','Data_image','Data_subimage','Data_peaks','Data_SI']

class Data(DeclarativeBase):
    __tablename__ = 'data'
    
    #{ Columns
    id    = Column(Integer, autoincrement=True, primary_key=True)
    name  = Column(Unicode)
    description = Column(Unicode)
    arr   = Column(PickleType)
    
    def __init__(self,name=None,description=None,arr=None):
        DeclarativeBase.__init__()
        self.name=name
        self.description=description
        self.arr=arr
        
    def plot(self):
        pass
        
class Data_EELS(Data):
    __tablename__ = 'data_EELS'
    __mapper_args__ = {'polymorphic_identity': 'data_EELS'}
    id = Column('id', Integer, ForeignKey('data.id'), 
                          primary_key=True)
    energy_offset = Column('energy_offset',Float)
    energy_slope  = Column('energy_slope',Float)
    
    def __init__(self,name=None, description=None, arr=None, 
                 energy_offset=None, energy_slope=None, infile=None, skiprows=None):
        Data.__init__(name,description,arr) 
        self.energy_offset=energy_offset
        self.energy_slope=energy_slope
        if infile:
            loadtxt(infile,skiprows)
        
    def loadtxt(self,infile,skiprows=None):
        d=np.loadtxt(infile,skiprows=skiprows)
        e=d[:,0]
        d=d[:,1:]
        self.arr=d
        self.energy_offset=e[0]
        self.energy_slope=e[1]-e[0]
        
class Data_image(Data):
    __tablename__ = 'data_image'
    __mapper_args__ = {'polymorphic_identity': 'data_image'}
    id = Column('id', Integer, ForeignKey('data.id'), 
                          primary_key=True)
    dim_per_pixel = Column('dim_per_pixel', Float)
    
    def __init__(self,name=None,description=None,arr=None, dim_per_pixel=None):
        Data.__init__(name,description,arr)
        self.dim_per_pixel=dim_per_pixel
    
class Data_subimage(Data_image):
    __tablename__ = 'data_subimage'
    __mapper_args__ = {'polymorphic_identity': 'data_subimage'}
    id = Column('id', Integer, ForeignKey('data.id'), 
                          primary_key=True)
    dim_per_pixel = Column('dim_per_pixel', Float)
    parent = Column('parent_id', Integer, 
                    ForeignKey('data_image.id'), 
                          primary_key=True)
    x_coord = Column('x',Integer, primary_key=True)
    y_coord = Column('y',Integer, primary_key=True)
    
    def __init__(self,name=None, description=None, arr=None, 
                 dim_per_pixel=None, parent=None, x_coord=None,
                 y_coord=None):
        Data_image.__init__(name,description,arr,dim_per_pixel)
        self.parent=parent
        self.x_coord=x_coord
        self.y_coord=y_coord
         
    
class Data_peaks(Data_image):
    __tablename__ = 'data_peaks'
    __mapper_args__ = {'polymorphic_identity': 'data_peaks'}
    id = Column('id', Integer, ForeignKey('data.id'), 
                          primary_key=True)
    parent = Column('parent_id', Integer, 
                    ForeignKey('data_image.id'), 
                          primary_key=True)
    def __init__(self,name=None, description=None, arr=None, 
                 dim_per_pixel=None, parent=None):
        Data_image.__init__(name,description,arr,dim_per_pixel)
        self.parent=parent
    
class Data_SI(Data_image):
    __tablename__ = 'data_SI'
    __mapper_args__ = {'polymorphic_identity': 'data_SI'}
    id = Column('id', Integer, ForeignKey('data.id'), 
                          primary_key=True)
    energy_offset = Column('energy_offset',Float)
    energy_slope  = Column('energy_slope',Float)
    
    def __init__(self,name=None, description=None, arr=None, 
                 dim_per_pixel=None, energy_offset=None, energy_slope=None):
        Data_image.__init__(name,description,arr,dim_per_pixel)
        self.energy_offset=energy_offset
        self.energy_slope=energy_slope
