# -*- coding: iso-8859-1 -*-
"""
Multivariate Statistical Analysis model module.

Copyright (C) 2011 by Michael Sarahan

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""

from sqlalchemy import *
from sqlalchemy.orm import mapper, relation
from sqlalchemy import Table, ForeignKey, Column
from sqlalchemy.types import Integer, Unicode, Float, PickleType
from sqlalchemy.orm import relation, backref
from sqlalchemy.ext.declarative import declarative_base
from zope.sqlalchemy import ZopeTransactionExtension
from sqlalchemy.orm import scoped_session, sessionmaker

import numpy as np
import utils.msa_rpy as msa

DeclarativeBase=declarative_base()
DBSession = scoped_session(sessionmaker(
                             extension=ZopeTransactionExtension()))

__all__ = ['Project']

"""
group_project_table = Table('tg_group_project', metadata,
    Column('group_id', Integer, ForeignKey('tg_group.group_id',
        onupdate="CASCADE", ondelete="CASCADE"), primary_key=True),
    Column('project_id', Integer, ForeignKey('projects.id',
        onupdate="CASCADE", ondelete="CASCADE"), primary_key=True)
        )

project_data_table = Table('tg_project_data', metadata,
    Column('project_id', Integer, ForeignKey('projects.id',
        onupdate="CASCADE", ondelete="CASCADE"), primary_key=True),
    Column('data_id', Integer, ForeignKey('data.id',
        onupdate="CASCADE", ondelete="CASCADE"), primary_key=True)
        )

project_result_table = Table('tg_project_result', metadata,
    Column('project_id', Integer, ForeignKey('projects.id',
        onupdate="CASCADE", ondelete="CASCADE"), primary_key=True),
    Column('result_id', Integer, ForeignKey('results.id',
        onupdate="CASCADE", ondelete="CASCADE"), primary_key=True)
        )
"""

from data_model import *
from result_model import Result, PCA_Result 

class Project(DeclarativeBase):
    __tablename__ = 'projects'
    
    #{ Columns
    id = Column(Integer, primary_key=True)
    name = Column(Unicode)
    datatype = Column(Unicode)
    description = Column(Unicode)
    
    #{ Relations
    groups  = relation('Group', secondary=group_project_table, backref='projects')
    data    = relation('Data', secondary=project_data_table, backref='data')
    results = relation('Result', secondary=project_result_table, backref='results')
    
    def __init__(self,name,datatype,groups,description=None,data=None,results=None):
        DeclarativeBase.__init__()
        self.name=name
        self.datatype=datatype
        self.groups=groups
        self.description=description
        self.data=data
        self.results=results
    
    def compile_data(self,start,end):
        if start:
            eranges=[np.arange(d.arr.shape[0])*d.energy_slope+d.energy_offset for d in self.data]
            starts = [np.argmin(eranges[i]-start)]
            if end:
                ends   = starts = [np.argmin(eranges[i]-end)] 
                arrs   = [self.data[i].arr[starts[i]:ends[i],
                                       :] for i in xrange(len(self.data))]
            else:
                arrs   = [self.data[i].arr[starts[i]:,
                                       :] for i in xrange(len(self.data))]
        else:
            arrs=[d.arr for d in self.data]
        # assert that all input data is the same shape
        #assert 
        arr=np.vstack((arrs))
        return arr
        
    def run_pca(self,d=None,weight=True):
        if not d:
            d=compile_data(self)
        factors,scores,eigenvalues=msa.pca(d,weight=weight)
        res=PCA_Result()
        self.results+=res
        return {"Status":"Complete"}
    
    def run_ica(self,d=None,num_factors=None):
        if not d:
            d=compile_data(self)
        factors,scores=msa.ica(d,)
        
    def run_nmf(self,d=None,num_factors=None):
        if not d:
            d=compile_data(self)
        factors,scores=msa.nmf(d,)
