# -*- coding: utf-8 -*-
"""
Functions useful for image analysis.  Mostly focused around pattern recognition.
Created on Fri Sep 10 12:03:06 2010

Copyright (C) 2011 by Michael Sarahan

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""

import numpy as np
from cvFuncs import cv2array
import cv

def xcorr(templateImage,exptImage,savename=None):
    tmp=cv.CloneImage(templateImage)
    padImage=cv.CloneImage(exptImage)
    rltWidth = padImage.width - tmp.width + 1
    rltHeight = padImage.height - tmp.height + 1
    rlt = cv.CreateImage((rltWidth,rltHeight),cv.IPL_DEPTH_32F,1)
    cv.MatchTemplate(padImage,tmp,rlt,cv.CV_TM_CCOEFF_NORMED)
    if savename:
        # if savename is specified, then save a file.  Make it 8-bit, so most
        # system tools can be used to view it.  At the time of coding, 16-bit
        # and above images are not widely supported.
        rlt8bit=norm(rlt)
        cv.SaveImage(savename,rlt8bit)
    return rlt

def findpeaks(arr,subpixel=False,boxsize=10):
    # Code translated from Dan Masiel's matlab functions
    def lmax(seq):
        # code from http://software.allfaq.org/forums/t/178601.aspx
        m = None             # not even initialized
        candidate = False    # cannot be candidate
        for elem in xrange(len(seq)):     # loop through any iterable
            if m is None:        
                m = seq[elem]         # the first is the initial value 
                candidate = True # and also is the candidate
            elif seq[elem] > m:       # sharply greater
                m = seq[elem]         # better candidate
                candidate = True
            elif seq[elem] == m:      # if equal, local maximum was lost
                candidate = False
            elif seq[elem] < m:       # if sharply lower then possible candidate to output
                if candidate and m>0:
                    yield (elem,m)
                m = seq[elem]           # start again...
                candidate = False  # being smaller it cannot be candidate
        if candidate:        # if the peak at the very end
            yield (len(seq),m)          # the yield the last one
        
    def oneDimFind(OneDarray):
        peaks = np.array([i for i in lmax(list(OneDarray))],dtype=int)
        fmap = np.zeros(OneDarray.shape[0])
        fmap[peaks[:,0]] = 1
        return fmap
        
    mapX = oneDimFind(arr.reshape((-1))).reshape(arr.shape)
    arr=arr.T
    mapY = oneDimFind(arr.reshape((-1))).reshape(arr.shape)
    # Dan's comment from Matlab code, left in for curiosity:
    #% wow! lame!
    Fmap = mapX*mapY.T
    nonzeros=np.nonzero(Fmap)
    coords=np.vstack((nonzeros[1],nonzeros[0])).T
    if subpixel:
        coords=subpix_locate(arr,coords,boxsize)
    return coords
    
def norm(input,to8bit=True):
    if type(input).__name__ is 'numpy.ndarray':
        rlt=input-input.min()
        rlt=rlt/float(max(rlt))
        if to8bit:
            rlt=np.array(rlt*255,dtype=np.uint8)
    elif type(input).__name__ is 'cv.iplimage':
        rlt=cv.CreateImage((input.width,input.height),cv.IPL_DEPTH_8U,input.nChannels)
        cv.ConvertScaleAbs(input,rlt,255)
    return rlt

def subpix_locate(data,points,boxsize,scale=None):
    """
    Calculates the subpixel centers of mass of several rectangular regions.
    """
    from scipy.ndimage.measurements import center_of_mass as CofM
    top=left=boxsize/2
    centers=np.array(points,dtype=np.float32)
    for i in xrange(points.shape[0]):
        pt=points[i]
        center=np.array(CofM(data[(pt[0]-left):(pt[0]+left),(pt[1]-top):(pt[1]+top)]))
        center=center[0]-boxsize/2,center[1]-boxsize/2
        centers[i]=np.array([pt[0]+center[0],pt[1]+center[1]])
    if scale:
        centers=centers*scale
    return centers
