# -*- coding: utf-8 -*-
"""
Copyright (C) 2011 by Michael Sarahan

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""

import os
from glob import glob
import numpy as np
import calibrations

def getFiles(dir=None):
    if dir:
        os.chdir(dir)
    fnames=sorted(glob('*.txt'))
    [fnames.remove(f) for f in fnames if f[-10:]=='result.txt']
    [fnames.remove(f) for f in fnames if f[-10:]=='result.txt']
    return fnames
    
def aggregate(fnames,beg=None,end=None):
    fs=[]
    for fname in fnames:
        dims,d,ecal=read_CSV(fname,beg=beg,end=end)  
        fs.append([fname,dims,d,ecal])  
    dsize=fs[0][2].shape[0],np.sum([f[2].shape[1] for f in fs])
    d=np.zeros(dsize)
    idx=0
    for f in fs:
       d[:,idx:idx+f[2].shape[1]]=f[2]
       idx=idx+f[2].shape[1]
    return fs,d
    
def split_export_to_DM(flist,factors,scores,atype='NMF'):
    import DMexport
    idx=0
    for f in flist:
        # dump the files to the raw format.  Suitable for streaming into DM.
        # f[0] is filename
        # f[1] is SI dimensions
        # f[2] is data array
        # f[3] is energy calibration info
        DMexport.MSAspectrumImage(f[0], f[1], factors, 
                               scores[idx:idx+f[2].shape[1]],
                               f[3],atype=atype)
        idx=idx+f[2].shape[1]

def split_plot_maps(flist,factors,scores,atype='NMF'):
    idx=0
    for f in flist:
        # dump the files to the raw format.  Suitable for streaming into DM.
        # f[0] is filename
        # f[1] is SI dimensions
        # f[2] is data array
        # f[3] is energy calibration info
        plot_maps(f[0],f[1], factors, scores[:,idx:idx+f[2].shape[1]], 
                  ecal=f[3],atype=atype)
        idx=idx+f[2].shape[1]

def plot_maps(filename,dims,factors,scores,ecal=None,scal=None,atype='NMF'):
    import msaplots
    import matplotlib.pyplot as plt
    for fac in xrange(factors.shape[1]):
        p=msaplots.spectrumImageScoreMap(factors[:,fac],
                                         scores[fac],
                                         dims, energyCal=ecal, 
                                         spatialCal=scal)
        p.savefig('%s_%s_%03i.png'%(filename,atype,fac))
        plt.close('all')    
    

def read_CSV(filename,beg=None,end=None):
    f=open(filename,'r')
    dims=f.readlines()[1].split(",")
    f.close()
    dims=[float(dim) for dim in dims]
    d = np.loadtxt(filename,skiprows=3)
    e = d[:,0] 
    d = d[:,1:]
    if beg:
        beg = np.argmin(np.abs(e-float(beg)))
    if end:
        end = np.argmin(np.abs(e-float(end)))
    e = e[beg:end]
    d = d[beg:end]
    d=d-np.min(d)
    ecal=calibrations.EnergyCal(offset=e[0], increment=e[1]-e[0], 
                                    units='eV')   
    return (dims,d,ecal)
