"""
Checkout the sources corresponding to the projects a specified egg or enthought
project depends on.

Copyright (c) 2007 by Enthought, Inc.
License: BSD Style.

"""

# Standard library imports
import optparse
import sys

# Project library imports
from project_set import ProjectMapNotFound, ProjectNotFound, ProjectSet
from enthought.proxy.api import setup_proxy


# Constants
ENTHOUGHT_REPO = 'https://svn.enthought.com/svn/enthought'


def main():
    """
    Entry point for the setuptools installed script.

    """

    # Parse the user's command line.
    parser = optparse.OptionParser(
        version = '0.2',
        description = 'Checkout the sources for the specified projects and '
            'their dependencies.  At least one project must be specified.  '
            'The location of the sources, and the project dependencies, is '
            'found by consulting a project map.  That map may be explicitly '
            'provided or generated by inspecting a set of repositories.',
        usage = '%prog [options] project [project] ...'
        )
    parser.add_option('-d', '--dry-run',
        action='store_true',
        default=False,
        dest='dry_run',
        help='Do not actually check anything out',
        )
    parser.add_option('-i', '--ignore-repo-map',
        action='store_true',
        default=False,
        dest='ignore_repo_map',
        help='Ignore any pre-built project maps stored in a repository. '
            'always build our own map instead.',
        )
    parser.add_option('-m', '--use-map',
        dest='local_map_file',
        help='Use the specified project map instead of inspecting any '
            'repositories.',
        )
    parser.add_option('-N', '--no-deps',
        action='store_true',
        default=False,
        dest='nodeps',
        help='Do not analyze the explicitly specified projects for their '
            'dependencies. Thus limiting the checked-out source to what '
            'the user explicitly requested.',
        )
    parser.add_option('-o', '--output-dir',
        dest='path',
        help='Specify the path to the top-level directory for the sources '
            'to be checked-out.  If the directory does not exist, it is '
            'created.',
        )
    parser.add_option('-p', '--print-map',
        action='store_true',
        default=False,
        dest='print_project_map',
        help='Print the contents of the project map to the console.',
        )
    parser.add_option('-P', '--write-map',
        dest='map_file',
        help='Write the project map to the specified file, creating the file '
            'if it does not already exist.',
        type='string',
        )
    parser.add_option('--proxy',
        default='',
        dest='proxy',
        help='Specify user:password@proxy.server:portnum to use a proxy for '
            'accessing repositories. (user, password, and portnum are '
            'optional.)  If you provide a user but not a password, you will '
            'be prompted for the password.  Note that you will also need to '
            'configure svn itself to use the proxy!',
        type='string',
        )
    parser.add_option('-r', '--repo',
        action='append',
        default=[ENTHOUGHT_REPO],
        dest='repos',
        help='Specify the URL of an additional svn repository '
            'to be searched for source.  You can provide multiple instances '
            'of this option.',
        )
    parser.add_option('-v', '--verbose',
        action='store_true',
        dest='verbose',
        help='Show all output from the build commands',
        )
    options, args = parser.parse_args()


    # Install a proxy handler if either PROXY_<HOST, PORT, USER, PASS>
    # environment variables are set or the proxy option is set.
    try:
        setup_proxy(getattr(options, 'proxy', ''))
    except ValueError, e:
        print >>sys.stderr, 'Proxy configuration error: %s' % e
        return 2

    # Create a project database.  If the user specified a project map to use,
    # then use it instead of inspecting the repositories.
    project_set = ProjectSet(options.verbose)
    if options.local_map_file is None:
        project_set.inspect_repositories(options.repos, options.ignore_repo_map)
    else:
        try:
            project_set.load_project_map(options.local_map_file)
        except ProjectMapNotFound, e:
            print >>sys.stderr, 'ERROR: %s' % str(e)
            return 3

    # If the user request printing or saving the project map, then do so now.
    if options.print_project_map:
        project_set.print_project_map()
    if options.map_file is not None:
        project_set.save_map_file(options.map_file)

    # Add all the user-specified projects.
    try:
        for project in args:
            project_set.add(project)
    except ProjectNotFound, e:
        print >>sys.stderr, str(e)
        return 2

    # Add all the dependent projects if asked to do so.
    if not options.nodeps:
        project_set.add_dependencies()

    # Only persue checking anything out if the user specified at least one
    # project.
    if len(args) > 0:

        # Determine the output directory.  If none was specified, use the name
        # and version of the first project/egg.
        if options.path is None or options.path == '':
            name, version = project_set.get_root_project_info()
            options.path = '%s_%s' % (name, version)

        # Do the checkout
        project_set.checkout(options.path, dry_run=options.dry_run)

        # Generate a warning about any missing project sources.
        if len(project_set.missing) > 0:
            msg = ('\nWARNING: The SOURCE for the projects listed below could '
                'not be located.\nIf the source that was checked out is '
                'sufficient for your purposes, then you can\nsafely ignore '
                'this warning.\n'
                'NOTE: This script doesn\'t attempt to locate BINARY '
                'installations on your system.\n')
            for k,v in project_set.missing.items():
                msg += '\t%s\t%s\n' % (k,v)
            msg +='If you really need the SOURCES for these projects, please '
                'try running again\nand providing additional repositories by '
                'adding  more "-r" options.\n'

            print >>sys.stderr, msg

    return


if __name__ == '__main__':
    main()

