
import os
from distutils.errors import DistutilsError

# We use the same separator as examples_writer.
from examples_writer import SEPARATOR, join
from hierarchical_dict import HierarchicalDict


def strip_path(prefix, path):
    """ Truncates **path** based on removing **prefix** from its beginning.
    Uses os.sep as the separator.
    """
    prefix = prefix.split(os.sep)
    new_path = path.split(os.sep)
    while len(prefix) > 0 and len(new_path) > 0 and new_path[0] == prefix[0]:
        new_path = new_path[1:]
        prefix = prefix[1:]
    return os.sep.join(new_path)


class TutorialSpec(object):
    """ Defines a tutorial item """

    # Class attribute 
    commonprefix = None

    def __init__(self, filespec):
        self.pathglob = None
        self._name = None
        self._name_autogenerated = True
        self._destpath = None
        self._destpath_autogenerated = True

        if isinstance(filespec, tuple):
            if len(filespec) == 1 and isinstance(filespec[0], str):
                self.pathglob = filespec[0]
            elif len(filespec) == 2:
                self.name = filespec[0]
                self.pathglob = filespec[1]
            elif len(filespec) == 3:
                self.name = filespec[0]
                self.pathglob = filespec[1]
                self.destpath = filespec[2]
            else:
                raise DistutilsError("Invalid filespec passed to find_packages: '%s'" % \
                                     filespec)
        elif isinstance(filespec, str):
            self.pathglob = filespec
        else:
            raise DistutilsError("Invalid filespec passed to find_packages: '%s'" % \
                                 filespec)
        return


    def _get_name(self):
        if self._name is None or self._name_autogenerated:
            self._name = strip_path(self.commonprefix, self.pathglob)
            self._name_autogenerated = True
            return self._name
        else:
            return self._name
    def _set_name(self, name):
        self._name = name
        self._name_autogenerated = False
    name = property(_get_name, _set_name)

    def _get_destpath(self):
        if self._destpath is None or self._destpath_autogenerated:
            self._destpath = strip_path(self.commonprefix, os.path.split(self.pathglob))
            self._destpath_autogenerated = True
            return self._destpath
        else:
            return self._destpath
    def _set_destpath(self, path):
        self._destpath = path
        self._destpath_autogenerated = False
    destpath = property(_get_destpath, _set_destpath)


def find_examples(*filespecs, **kwargs):
    """ Returns a list of examples from a series of specification strings or
    tuples.

    This function's signature is:
    find_examples(*filespecs, destdir=None, commonprefix=None, auto_nest_names=True)

    Parameters
    ----------
    filespecs : list
        Each filespec can be a string, a 2-tuple, or a 3-tuple.
        
        string: path or glob (e.g. "foo/bar/baz*.py")
        2-tuple: ("Tutorial Name", path_or_glob)
        3-tuple: ("Tutorial Name", path_or_glob, dest_path)
        
        The path or glob specifies the files that comprise a single example.
        By default, the name of the tutorial will be the path.  If a glob
        is provided, then it will be sanitized by stripping the first wildcard
        character and everything after.

        "Tutorial Name" will be used by installers on target systems which
        provide a user-friendly place to provide links to examples.
        
        **dest_path** is the desired location of the example file or files
        when it is installed.  It should be a relative path, and the 
        platform-specific installer will determine the appropriate root.
        Note that the path elements common to all the example files (across
        all examples) will be stripped automatically.  (See **commonprefix**
        below to customize or override this behavior.)

    destdir : string, optional
        The root destination directory into which to install the examples.
        Usually the name of the package or something to help distinguish this
        package's files from other packages.

    commonprefix : string, optional
        If a destination filename or directory is not provided for an example,
        this common prefix will be stripped from the example's list of files to
        generate the output directory.  By default, this value will be derived
        from calling os.path.commonprefix() on all the filespecs.  If this
        behavior is not desired, pass in an empty string for this parameter.

    auto_nest_names : bool, default=True
        Configures the name generator for filespecs that do not specify an
        explicit name.  When **auto_nest_names** is True, if an example's
        filespec is located in a subdirectory relative to the **commonprefix**,
        then its name is placed into a hierarchical structure that mirrors the
        relative path.
        
        For instance, for the filespec "foo/bar/demo.py", if auto_nest_names
        if False, then the following is placed into the example config:
                [foo/bar/demo.py]
                files = foo/bar/demo.py
        
        If auto_nest_names is True, then the following nested section is
        created:
                [foo]
                sourcedir = foo
                    
                    [[bar]]
                    sourcedir = bar

                        [[[demo.py]]]
                        files = demo.py
        
        If other examples have filespecs that are also nested, they will
        be integrated into this nested structure.


    Returns
    -------
    config : dict
        A ConfigObj-compatible dict representing the files that comprises the
        examples for a package.  This config is compatible with the
        examples_writer in examples_writer.py, and thus all the paths use
        forward slashes.
        
        Each section in the dict is an example specification consistent with
        the format of examples_map files.  This dict also contains a special
        top-level key __basedir__ that serves as the root directory of all the
        pathglobs in the dict.
        
    """
    # Sort through the kwargs... 
    # TODO: Remove this when PEP 3102 (keyword-only arguments) finally 
    # takes effect!
    destdir = kwargs.get('destdir', None)
    commonprefix = kwargs.get('commonprefix', None)
    auto_nest_names = kwargs.get('auto_nest_names', True)

    # Transform the input filespecs into FileSpec objects
    input_filespecs = filespecs
    filespecs = map(TutorialSpec, input_filespecs)

    if commonprefix is None:
        commonprefix = os.path.commonprefix([x.pathglob for x in filespecs])

    TutorialSpec.commonprefix = commonprefix

    config = HierarchicalDict(nested = auto_nest_names)
    if destdir:
        config["global/destdir"] = destdir

    for filespec in filespecs:
        path = filespec.pathglob
        name = filespec.name
        if SEPARATOR != os.sep:
            path = path.replace(os.sep, SEPARATOR)
        config[name] = {}
        config[name]["files"] = path
    
    config["__basedir__"] = commonprefix
    return config


def _generate_name(filespec):
    """ Converts a valid filespec for find_packages into a "nice" name
    designed for human consumption, e.g. in a menu or list of examples.
    """
    if filespec.endswith(".py"):
        return filespec[:-3]
    else:
        return filespec

