"""
Create a new checkout.

A checkout is a directory that contains sub-directories representing the
sources for a set of projects that the user requested the source for.  This
usually includes the sources for the declared dependencies as well.

Copyright (c) 2007 by Enthought, Inc.
License: BSD Style.

"""

# Standard library imports
import sys

# Project library imports
from enthought.ets.base_subcommand import BaseSubcommand


class Checkout(BaseSubcommand):
    """
    The ets checkout command.

    """

    def __init__(self, subparsers):
        """
        Constructor.

        Overloaded to customize our parser configuration.

        """

        desc = ('Checkout the sources for the specified projects and '
            'their dependencies.  At least one project must be specified.  '
            'The location of the sources, and the project dependencies, is '
            'found by consulting a project map.  That map may be explicitly '
            'provided or generated by inspecting a set of repositories.')
        parser = subparsers.add_parser('co',
            description = desc,
            help = '. . . %s' % desc,
            )

        # Add arguments
        parser.add_argument('project',
            nargs = "+",
            help = 'Specifications of project(s) to retrieve.  These are of '
                'the same form as setuptools\' version specifications.  For '
                'example "ets==2.7.0" or "ets >=2.7, <3.0a"',
            )

        # Add the options
        self.dry_run(parser)
        self.use_project_map(parser)
        self.no_deps(parser)
        parser.add_argument('-o', '--output-dir',
            dest = 'path',
            help = 'Specify the path to the top-level directory for the sources '
                'to be checked-out.  If the directory does not exist, it is '
                'created.',
            )
        self.proxy(parser)
        parser.add_argument('-r', '--repo',
            action = 'append',
            default = [],
            dest = 'repos',
            help = 'Specify the URL of an additional svn repository '
                'to be searched for source.  You can provide multiple instances '
                'of this option.',
            )
        parser.add_argument('--version',
            action = 'store_true',
            default = False,
            dest = 'use_version',
            help = 'Add the version number as a suffix to the names of the '
                'sub-project directories.  i.e. checkout "Traits" as '
                '"Traits_3.1.2" rather than "Traits"',
            )

        parser.set_defaults(func = self.main)

        return


    def main(self, args, cfg):
        """
        Execute the ets checkout command.

        """

        # Build a project set to reflect the user's interests and add all the
        # dependent projects.
        project_set = self.build_project_set(args, cfg)
        if not args.nodeps:
            project_set.add_dependencies()

        # Only persue checking anything out if the user specified at least one
        # project.
        if len(args.project) > 0:

            # Determine the output directory.  If none was specified, use the
            # name and version of the first project/egg.
            if args.path is None or args.path == '':
                name, version = project_set.get_root_project_info()
                args.path = '%s_%s' % (name, version)

            # Do the checkout
            project_set.checkout(args.path, dry_run=args.dry_run,
                use_version=args.use_version)

            # Generate a warning about any missing project sources.
            if len(project_set.missing) > 0:
                msg = ('\nTHE SOURCE FOR THE FOLLOWING PROJECTS COULD NOT BE '
                    'LOCATED:\n')
                for k,v in project_set.missing.items():
                    msg += '\t%s\t%s\n' % (k,v)
                msg +='HINT: Perhaps try again with more "-r" options?\n'

                print >>sys.stderr, msg

        return


