from unittest import TestCase
from datetime import date, time

import formencode as fe
from formencode import validators as fev
from BeautifulSoup import BeautifulSoup

import ew
import ew.kajiki_ew
import ew.jinja2_ew

def soup(text):
    soup = BeautifulSoup(text)
    for n in soup.findAll():
        n.attrs = sorted(n.attrs)
    return soup

class _TestFields(TestCase):

    def setUp(self):
        ew.TemplateEngine.initialize({})
        ew.widget_context.set_up({})

    def test_text_field(self):
        w = self.ew.InputField(name='foo')
        text = w.display(value='bar')
        assert soup(text)==soup('<input name="foo" value="bar">')
        text = w.display()
        assert text=='<input name="foo">'

    def test_hidden_field(self):
        w = self.ew.HiddenField(name='foo')
        text = w.display(value='bar')
        assert soup(text)==soup('<input name="foo" type="hidden" value="bar">'), soup(text)
        text = w.display()
        assert text=='<input name="foo" type="hidden">', text

    def test_validate(self):
        w = self.ew.InputField(
            name='foo',
            validator=fev.UnicodeString(min=2))
        self.assertRaises(fe.Invalid, w.to_python, 'x', None)
        assert isinstance(ew.widget_context.validation_error, fe.Invalid)

class _TestFieldSet(TestCase):

    def setUp(self):
        ew.TemplateEngine.initialize({})
        ew.widget_context.set_up({})
        self.w = self.ew.FieldSet(
            fields=[
                self.ew.InputField(
                    name='foo',
                    validator=fev.UnicodeString(min=2)),
                self.ew.InputField(
                    name='bar',
                    validator=fev.UnicodeString(min=2)),
                ])
        self.nested_w = self.ew.FieldSet(
            fields=[self.ew.FieldSet(
                    name='fs',
                    fields=[
                        self.ew.InputField(
                            name='foo',
                            validator=fev.UnicodeString(min=2)),
                        self.ew.InputField(
                            name='bar',
                            validator=fev.UnicodeString(min=2)),
                        ])
                    ])

    def test_display(self):
        text = self.w.display()
        assert '<label for="foo"' in text
        assert '<input name="foo"' in text
        text = self.w.display(value=dict(foo='bar', bar='baz'))
        assert '<label for="foo"' in text
        assert '<input name="foo" value="bar"' in text
        assert '<label for="bar"' in text
        assert '<input name="bar" value="baz"' in text

    def test_validation(self):
        self.assertRaises(fe.Invalid, self.w.to_python, dict(foo='x', bar='xxy'), None)
        assert isinstance(ew.widget_context.validation_error, fe.Invalid)
        text = self.w.display()
        assert '<label for="foo"' in text
        assert '<span class="fielderror">Enter' in text, text
        assert '<input name="foo" value="x"' in text
        assert '<input name="bar" value="xxy"' in text

    def test_nested_display(self):
        text = self.nested_w.display()
        assert '<label for="fs.foo"' in text
        assert '<input name="fs.foo"' in text
        text = self.nested_w.display(value=dict(fs=dict(foo='bar')))
        assert '<label for="fs.foo"' in text, text
        assert '<input name="fs.foo" value="bar"' in text
        
    def test_nested_validation(self):
        self.assertRaises(fe.Invalid, self.nested_w.to_python,
                          dict(fs=dict(foo='x')), None)
        assert isinstance(ew.widget_context.validation_error, fe.Invalid)
        text = self.nested_w.display()
        assert '<label for="fs.foo"' in text
        assert '<span class="fielderror"' in text
        assert '<input name="fs.foo" value="x"' in text

class _TestRowField(TestCase):

    def setUp(self):
        ew.TemplateEngine.initialize({})
        ew.widget_context.set_up({})
        self.w = self.ew.RowField(
            fields=[
                self.ew.InputField(
                    name='a',
                    validator=fev.UnicodeString(min=2)),
                self.ew.InputField(
                    name='b',
                    validator=fev.UnicodeString(min=2)),
                ])
    
    def test_display(self):
        text = self.w.display()
        assert '<input name="a"' in text
        text = self.w.display(value=dict(a='bar'))
        assert '<input name="a" value="bar"' in text

    def test_validation(self):
        self.assertRaises(fe.Invalid, self.w.to_python, dict(a='x', b='xx'), None)
        assert isinstance(ew.widget_context.validation_error, fe.Invalid)
        text = self.w.display()
        assert '<span class="fielderror"' in text
        assert '<input name="a" value="x"' in text, text

class _TestRepeatedField(TestCase):

    def setUp(self):
        ew.TemplateEngine.initialize({})
        ew.widget_context.set_up({})
        self.w = self.ew.RepeatedField(
            field=self.ew.InputField(
                    name='a',
                    validator=fev.UnicodeString(min=2)))

    def test_display(self):
        text = self.w.display()
        assert '<input name="a-0"' in text
        assert '<input name="a-1"' in text
        assert '<input name="a-2"' in text
        text = self.w.display(value=['bar'])
        assert '<input name="a-0" value="bar"' in text
        assert '<input name="a-1"' not in text
        
    def test_validation(self):
        self.assertRaises(fe.Invalid, self.w.to_python, ['x', 'xx'], None)
        assert isinstance(ew.widget_context.validation_error, fe.Invalid)
        text = self.w.display(value=['xxx', 'xxx'])
        assert '<span class="fielderror"' in text
        assert '<input name="a-0" value="x"' in text, text
        

class _TestTableField(TestCase):

    def setUp(self):
        ew.TemplateEngine.initialize({})
        ew.widget_context.set_up({})
        self.w = self.ew.TableField(
            name='foo',
            fields=[
                self.ew.InputField(
                    name='a',
                    validator=fev.UnicodeString(min=2)),
                self.ew.InputField(
                    name='b',
                    validator=fev.UnicodeString(min=2))
                ])

    def test_display(self):
        text = self.w.display()
        assert '<input name="foo-0.a" value="">' in text
        text = self.w.display(value=[dict(a='xx',b='yy')])
        assert '<input name="foo-0.a" value="xx">' in text
        
    def test_validation(self):
        self.assertRaises(
            fe.Invalid, self.w.to_python,
            [ dict(a='x', b='xx'),
              dict(a='yy', b='yy')],
            None)
        assert isinstance(ew.widget_context.validation_error, fe.Invalid)
        text = self.w.display()
        assert '<span class="fielderror"' in text
        assert '<input name="foo-0.a" value="x"' in text, text

class _TestInputFields(TestCase):

    def setUp(self):
        ew.TemplateEngine.initialize({})
        ew.widget_context.set_up({})

    def test_text_field(self):
        t = self.ew.TextField(name='foo')
        assert t.display() == '<input name="foo" type="text">'
    
    def test_email_field(self):
        t = self.ew.EmailField(name='foo')
        assert t.display() == '<input name="foo" type="text">'
        self.assertRaises(fe.Invalid, t.to_python, 'asdf', None)
        self.assertRaises(fe.Invalid, t.to_python, 'asdf@bar', None)
        self.assertRaises(fe.Invalid, t.to_python, 'asdf@bar@bar.com', None)
        t.to_python('foo@bar.com', None)
        
    def test_int_field(self):
        t = self.ew.IntField(name='foo')
        assert t.display() == '<input name="foo" type="text">'
        self.assertRaises(fe.Invalid, t.to_python, 'asdf', None)
        self.assertRaises(fe.Invalid, t.to_python, '42.5', None)
        assert 42 == t.to_python('42', None)
        
    def test_date_field(self):
        t = self.ew.DateField(name='foo')
        text = t.display()
        assert soup(text) == soup('<input name="foo" type="text" style="width:6em">'), text
        self.assertRaises(fe.Invalid, t.to_python, 'asdf', None)
        self.assertRaises(fe.Invalid, t.to_python, '1990.01.32', None)
        assert date(1990, 1, 1) == t.to_python('01/01/1990', None)

    def test_time_field(self):
        t = self.ew.TimeField(name='foo')
        text = t.display()
        assert soup(text) == soup('<input name="foo" style="width:5em" type="text">'), soup(text)
        self.assertRaises(fe.Invalid, t.to_python, 'asdf', None)
        self.assertRaises(fe.Invalid, t.to_python, '01:67', None)
        assert time(1,34) == t.to_python('1:34', None)
        assert time(13,34) == t.to_python('1:34 pm', None)

    def test_textarea(self):
        t = self.ew.TextArea(name='foo')
        text = t.display()
        assert text == '<textarea name="foo"></textarea>', text

    def test_checkbox(self):
        t = self.ew.Checkbox(name='foo')
        text = t.display()
        assert text.startswith('<input'), text
        assert 'type="checkbox"' in text, text
        assert 'name="foo"' in text, text
        assert '<label for="foo">Foo</label>' in text, text
        text = t.display(value=True)
        assert 'type="checkbox"' in text, text
        assert 'CHECKED' in text, text

    def test_submit_button(self):
        t = self.ew.SubmitButton(name='foo')
        text = t.display()
        assert (soup(text)==soup('<input name="foo" '
                'value="Foo" type="submit" class="submit">')), soup(text)
        text = t.display(label='Bar')
        assert (soup(text)==soup('<input name="foo" '
                'value="Bar" type="submit" class="submit">')), text

class _TestSelectFields(TestCase):

    def setUp(self):
        ew.TemplateEngine.initialize({})
        ew.widget_context.set_up({})

    def test_single_select(self):
        for optlist in [
            'abc',
            lambda:'abc',
            [self.ew.Option(label='A', html_value='a'),
             self.ew.Option(label='B', html_value='b'),
             self.ew.Option(label='C', html_value='c') ]
            ]:
            t = self.ew.SingleSelectField(name='foo', options=optlist)
            text = t.display()
            assert '<select name="foo">' in text, text
            assert '<option value="a">' in text, text
            assert '<option value="b">' in text, text
            assert '<option value="c">' in text, text
            text = t.display(value="b")
            assert '<option value="a">' in text, text
            assert '<option SELECTED value="b">' in text, text
            assert '<option value="c">' in text, text

    def test_multi_select(self):
        for optlist in [
            'abc',
            lambda:'abc',
            [self.ew.Option(label='A', html_value='a'),
             self.ew.Option(label='B', html_value='b'),
             self.ew.Option(label='C', html_value='c') ]
            ]:
            t = self.ew.MultiSelectField(name='foo', options=optlist)
            text = t.display()
            assert '<select MULTIPLE name="foo">' in text, text
            assert '<option value="a">' in text, text
            assert '<option value="b">' in text, text
            assert '<option value="c">' in text, text
            text = t.display(value=["b","c"])
            assert '<option value="a">' in text, text
            assert '<option SELECTED value="b">' in text, text
            assert '<option SELECTED value="c">' in text, text
            
    def test_checkbox_set(self):
        for optlist in [
            'abc',
            lambda:'abc',
            [self.ew.Option(label='A', html_value='a'),
             self.ew.Option(label='B', html_value='b'),
             self.ew.Option(label='C', html_value='c') ]
            ]:
            t = self.ew.CheckboxSet(name='foo', options=optlist)
            text = t.display()
            assert '<fieldset' in text, text
            assert '<legend>Foo</legend>' in text, text
            assert '<input type="checkbox" name="foo" value="a">' in text, text
            assert '<input type="checkbox" name="foo" value="b">' in text, text
            assert '<input type="checkbox" name="foo" value="c">' in text, text
            text = t.display(value=["b","c"])
            assert '<input type="checkbox" name="foo" value="a">' in text, text
            assert '<input type="checkbox" CHECKED name="foo" value="b">' in text, text
            assert '<input type="checkbox" CHECKED name="foo" value="c">' in text, text

    def test_html_field(self):
        fld = self.ew.HTMLField()
        assert '9' == fld.display(text='${value+4}', value=5)
        fld = self.ew.HTMLField()
        assert '<h1>Hi</h1>' == fld.display(value='<h1>Hi</h1>')
            
    def test_link_field(self):
        fld = self.ew.LinkField(href='/foo/${value}/', label='Bar')
        text = fld.display(value=5)
        assert soup('<a href="/foo/5/">Bar</a>') == soup(text), text
            
class _TestForms(TestCase):

    def setUp(self):
        ew.TemplateEngine.initialize({})
        ew.widget_context.set_up({})
        self.w = self.ew.SimpleForm(
            fields=[
                self.ew.InputField(
                    name='foo',
                    validator=fev.UnicodeString(min=2)),
                self.ew.InputField(
                    name='bar',
                    validator=fev.UnicodeString(min=2)),
                ])

    def test_display(self):
        text = self.w.display()
        assert '<label for="foo"' in text
        assert '<input name="foo"' in text
        text = self.w.display(value=dict(foo='bar', bar='baz'))
        assert '<label for="foo"' in text
        assert '<input name="foo" value="bar"' in text
        assert '<label for="bar"' in text
        assert '<input name="bar" value="baz"' in text

    def test_validation(self):
        self.assertRaises(fe.Invalid, self.w.to_python, dict(foo='x', bar='xxy'), None)
        assert isinstance(ew.widget_context.validation_error, fe.Invalid)
        text = self.w.display()
        assert '<label for="foo"' in text
        assert '<span class="fielderror">Enter' in text, text
        assert '<input name="foo" value="x"' in text
        assert '<input name="bar" value="xxy"' in text

class TestFieldsKajiki(_TestFields): ew=ew.kajiki_ew
class TestFieldSetKajiki(_TestFieldSet): ew=ew.kajiki_ew
class TestRowFieldKajiki(_TestRowField): ew=ew.kajiki_ew
class TestRepeatedFieldKajiki(_TestRepeatedField): ew=ew.kajiki_ew
class TestTableFieldKajiki(_TestTableField): ew=ew.kajiki_ew
class TestInputFieldsKajiki(_TestInputFields): ew=ew.kajiki_ew
class TestSelectFieldsKajiki(_TestSelectFields): ew=ew.kajiki_ew
class TestFormsKajiki(_TestForms): ew=ew.kajiki_ew

class TestFieldsJinja2(_TestFields): ew=ew.jinja2_ew
class TestFieldSetJinja2(_TestFieldSet): ew=ew.jinja2_ew
class TestRowFieldJinja2(_TestRowField): ew=ew.jinja2_ew
class TestRepeatedFieldJinja2(_TestRepeatedField): ew=ew.jinja2_ew
class TestTableFieldJinja2(_TestTableField): ew=ew.jinja2_ew
class TestInputFieldsJinja2(_TestInputFields): ew=ew.jinja2_ew
class TestSelectFieldsJinja2(_TestSelectFields): ew=ew.jinja2_ew
class TestFormsJinja2(_TestForms): ew=ew.jinja2_ew
