from elements import Elements
from datetime import datetime

ATOM_NAMESPACE = 'http://www.w3.org/2005/Atom'
APP_NAMESPACE = 'http://purl.org/atom/app#'

class Link(Elements):
    """The atom:link element"""

    _tag = 'link'
    _namespace = ATOM_NAMESPACE
    _children = Elements._children.copy()
    _attributes = Elements._attributes.copy()
    
    _attributes['rel'] = 'rel'
    _attributes['href'] = 'href'
    _attributes['type'] = 'type'
    _attributes['title'] = 'title'
    _attributes['length'] = 'length'
    _attributes['hreflang'] = 'hreflang'

    def __init__(self, href=None, rel=None, link_type=None, hreflang=None, 
        title=None, length=None, text=None):
        """Constructor for Link

        Args:
        href: string The href attribute of the link
        rel: string
        type: string
        hreflang: string The language for the href
        title: string
        length: string The length of the href's destination
        text: str The text data in the this element
        """

        self.href = href
        self.rel = rel
        self.type = link_type
        self.hreflang = hreflang
        self.title = title
        self.length = length
        
        self.text = text

class Person(Elements):
    """A foundation class from which atom:author and atom:contributor extend.

    A person contains information like name, email address, and web page URI for
    an author or contributor to an Atom feed. 
    """

    _children = Elements._children.copy()
    _attributes = Elements._attributes.copy()
    _namespace = ATOM_NAMESPACE
    
    _children['name'] = ('name', unicode)
    _children['email'] = ('email', unicode)
    _children['uri'] = ('uri', unicode)

    def __init__(self, name=None, email=None, uri=None, text=None):
        """Foundation from which author and contributor are derived.

        The constructor is provided for illustrative purposes, you should not
        need to instantiate a Person.

        Args:
        name: Name The person's name
        email: Email The person's email address
        uri: Uri The URI of the person's webpage
        for additional XML attributes of this element.
        text: String The text contents of the element. This is the contents
        of the Entry's XML text node. (Example: <foo>This is the text</foo>)
        """

        self.name = name
        self.email = email
        self.uri = uri
        self.text = text      
                      
class Contributor(Person):
    """The atom:contributor element"""

    _tag = 'contributor'
    _namespace = ATOM_NAMESPACE
    _children = Person._children.copy()
    _attributes = Person._attributes.copy()

    def __init__(self, name=None, email=None, uri=None, text=None):
        """Constructor for Contributor

        Args:
          name: Name
          email: Email
          uri: Uri
          text: str The text data in the this element
        """

        self.name = name
        self.email = email
        self.uri = uri
        self.text = text
                    
class Author(Person):             
    """The atom:author element      
                              
    An author is a required element in Feed. 
    """                             
                              
    _tag = 'author'                 
    _namespace = ATOM_NAMESPACE     

    _children = Person._children.copy()
    _attributes = Person._attributes.copy()
                              
    def __init__(self, name=None, email=None, uri=None, text=None):
        """Constructor for Author
                              
        Args:                         
          name: Name                  
          email: Email                
          uri: Uri                    
          text: str The text data in the this element
        """                           
                              
        self.name = name              
        self.email = email            
        self.uri = uri                
        self.text = text              
                                  
class Text(Elements):
  """A foundation class from which atom:title, summary, etc. extend.
  
  This class should never be instantiated.
  """

  _children = Elements._children.copy()
  _attributes = Elements._attributes.copy()

  _attributes['type'] = 'type'

  def __init__(self, text_type=None, text=None, extension_elements=None,
      extension_attributes=None):
    """Constructor for Text
    
    Args:
      text_type: string
      text: str The text data in the this element
      extension_elements: list A  list of ExtensionElement instances
      extension_attributes: dict A dictionary of attribute value string pairs
    """
    
    self.type = text_type
    self.text = text


class Title(Text):
    """The atom:title element"""

    _tag = 'title'
    _namespace = ATOM_NAMESPACE
    _children = Text._children.copy()
    _attributes = Text._attributes.copy()

    def __init__(self, title_type=None, text=None):
        """Constructor for Title

        Args:
          title_type: string
          text: str The text data in the this element
          extension_elements: list A  list of ExtensionElement instances
          extension_attributes: dict A dictionary of attribute value string pairs
        """

        self.type = title_type
        self.text = text

class Subtitle(Text):
    """The atom:subtitle element"""

    _tag = 'subtitle'
    _namespace = ATOM_NAMESPACE
    _children = Text._children.copy()
    _attributes = Text._attributes.copy()

    def __init__(self, subtitle_type=None, text=None):
        """Constructor for Subtitle

        Args:
          subtitle_type: string
          text: str The text data in the this element
          extension_elements: list A  list of ExtensionElement instances
          extension_attributes: dict A dictionary of attribute value string pairs
        """

        self.type = subtitle_type
        self.text = text

class Rights(Text):
    """The atom:rights element"""

    _tag = 'rights'
    _namespace = ATOM_NAMESPACE
    _children = Text._children.copy()
    _attributes = Text._attributes.copy()

    def __init__(self, rights_type=None, text=None):
        """Constructor for Rights

        Args:
          rights_type: string
          text: str The text data in the this element
          extension_elements: list A  list of ExtensionElement instances
          extension_attributes: dict A dictionary of attribute value string pairs
        """

        self.type = rights_type
        self.text = text

class Summary(Text):
    """The atom:summary element"""

    _tag = 'summary'
    _namespace = ATOM_NAMESPACE
    _children = Text._children.copy()
    _attributes = Text._attributes.copy()

    def __init__(self, summary_type=None, text=None):
        """Constructor for Summary

        Args:
          summary_type: string
          text: str The text data in the this element
          extension_elements: list A  list of ExtensionElement instances
          extension_attributes: dict A dictionary of attribute value string pairs
        """

        self.type = summary_type
        self.text = text

class Content(Text):
    """The atom:content element"""

    _tag = 'content'
    _namespace = ATOM_NAMESPACE
    _children = Text._children.copy()
    _attributes = Text._attributes.copy()

    _attributes['src'] = 'src'

    def __init__(self, content_type=None, src=None, text=None):
        """Constructor for Content

        Args:
          content_type: string
          src: string
          text: str The text data in the this element
          extension_elements: list A  list of ExtensionElement instances
          extension_attributes: dict A dictionary of attribute value string pairs
        """

        self.type = content_type
        self.src = src
        self.text = text

class Generator(Elements):        
    """The atom:generator element"""

    _tag = 'generator'
    _namespace = ATOM_NAMESPACE
    _children = Elements._children.copy()
    _attributes = Elements._attributes.copy()

    _attributes['uri'] = 'uri'
    _attributes['version'] = 'version'

    def __init__(self, uri=None, version=None, text=None):
        """Constructor for Generator

        Args:
          uri: string
          version: string
          text: str The text data in the this element
        """

        self.uri = uri
        self.version = version
        self.text = text
                          
class Category(Elements): 
    """The atom:category element"""
                      
    _tag = 'category'       
    _namespace = ATOM_NAMESPACE
    _children = Elements._children.copy()
    _attributes = Elements._attributes.copy()
    
    _attributes['term'] = 'term'
    _attributes['scheme'] = 'scheme'
    _attributes['label'] = 'label'
                      
    def __init__(self, term=None, scheme=None, label=None, text=None):
        """Constructor for Category

        Args:
          term: str
          scheme: str
          label: str
          text: str The text data in the this element
        """

        self.term = term
        self.scheme = scheme
        self.label = label
        self.text = text
    
class Control(Elements):
    """The app:control element indicating restrictions on publication.

    The APP control element may contain a draft element indicating whether or
    not this entry should be publicly available.
    """

    _tag = 'control'
    _namespace = APP_NAMESPACE
    _children = Elements._children.copy()
    _attributes = Elements._attributes.copy()

    _children['draft'] = ('draft', unicode)

    def __init__(self, draft=None, text=None):
        """Constructor for app:control"""

        self.draft = draft
        self.text = text
    
class FeedEntryParent(Elements):
    """A super class for atom:feed and entry, contains shared attributes"""

    _children = Elements._children.copy()
    _attributes = Elements._attributes.copy()
    
    _children['author'] = ('author', [Author])
    _children['category'] = ('category', [Category])
    _children['contributor'] = ('contributor', [Contributor])
    _children['id'] = ('id', unicode)
    _children['link'] = ('link', [Link])
    _children['rights'] = ('rights', Rights)
    _children['title'] = ('title', Title)
    _children['updated'] = ('updated', datetime, '%Y-%m-%dT%H:%M:%S.000%z')

    def __init__(self, author=None, category=None, contributor=None, 
        atom_id=None, link=None, rights=None, title=None, updated=None, 
        text=None):
        
        self.author = author or []
        self.category = category or []
        self.contributor = contributor or []
        self.id = atom_id
        self.link = link or []
        self.rights = rights
        self.title = title
        self.updated = updated
        self.text = text

class Source(FeedEntryParent):
    """The atom:source element"""

    _tag = 'source'
    _namespace = ATOM_NAMESPACE
    _children = FeedEntryParent._children.copy()
    _attributes = FeedEntryParent._attributes.copy()

    _children['generator'] = ('generator', Generator)
    _children['icon'] = ('icon', unicode)
    _children['logo'] = ('logo', unicode)
    _children['subtitle'] = ('subtitle', Subtitle)

    def __init__(self, author=None, category=None, contributor=None,
        generator=None, icon=None, atom_id=None, link=None, logo=None, 
        rights=None, subtitle=None, title=None, updated=None, text=None):
        """Constructor for Source

        Args:
          author: list (optional) A list of Author instances which belong to this
              class.
          category: list (optional) A list of Category instances
          contributor: list (optional) A list on Contributor instances
          generator: Generator (optional)
          icon: Icon (optional)
          id: Id (optional) The entry's Id element
          link: list (optional) A list of Link instances
          logo: Logo (optional)
          rights: Rights (optional) The entry's Rights element
          subtitle: Subtitle (optional) The entry's subtitle element
          title: Title (optional) the entry's title element
          updated: Updated (optional) the entry's updated element
          text: String (optional) The text contents of the element. This is the
              contents of the Entry's XML text node.
              (Example: <foo>This is the text</foo>)
        """

        self.author = author or []
        self.category = category or []
        self.contributor = contributor or []
        self.generator = generator
        self.icon = icon
        self.id = atom_id
        self.link = link or []
        self.logo = logo
        self.rights = rights
        self.subtitle = subtitle
        self.title = title
        self.updated = updated
        self.text = text
    
class Entry(FeedEntryParent):
    """The atom:entry element"""

    _tag = 'entry'
    _namespace = ATOM_NAMESPACE
    _children = FeedEntryParent._children.copy()
    _attributes = FeedEntryParent._attributes.copy()
    
    _children['content'] = ('content', Content)
    _children['published'] = ('published', unicode)
    _children['source'] = ('source', Source)
    _children['summary'] = ('summary', Summary)
    _children['{%s}control' % APP_NAMESPACE] = ('control', Control)

    def __init__(self, author=None, category=None, content=None, 
        contributor=None, atom_id=None, link=None, published=None, rights=None,
        source=None, summary=None, control=None, title=None, updated=None, 
        text=None):
        """Constructor for atom:entry

        Args:
          author: list A list of Author instances which belong to this class.
          category: list A list of Category instances
          content: Content The entry's Content
          contributor: list A list on Contributor instances
          id: Id The entry's Id element
          link: list A list of Link instances
          published: Published The entry's Published element
          rights: Rights The entry's Rights element
          source: Source the entry's source element
          summary: Summary the entry's summary element
          title: Title the entry's title element
          updated: Updated the entry's updated element
          control: The entry's app:control element which can be used to mark an 
              entry as a draft which should not be publicly viewable.
          text: String The text contents of the element. This is the contents
              of the Entry's XML text node. (Example: <foo>This is the text</foo>)
        """

        self.author = author or []
        self.category = category or []
        self.content = content
        self.contributor = contributor or []
        self.id = atom_id
        self.link = link or []
        self.published = published
        self.rights = rights
        self.source = source
        self.summary = summary
        self.title = title
        self.updated = updated
        self.control = control
        self.text = text

class Feed(Source):
    """The atom:feed element"""

    _tag = 'feed'
    _namespace = ATOM_NAMESPACE
    _children = Source._children.copy()
    _attributes = Source._attributes.copy()
    
    _children['entry'] = ('entry', [Entry])

    def __init__(self, author=None, category=None, contributor=None,
        generator=None, icon=None, atom_id=None, link=None, logo=None, 
        rights=None, subtitle=None, title=None, updated=None, entry=None, 
        text=None):
        """Constructor for Source

        Args:
          author: list (optional) A list of Author instances which belong to this
              class.
          category: list (optional) A list of Category instances
          contributor: list (optional) A list on Contributor instances
          generator: Generator (optional) 
          icon: Icon (optional) 
          id: Id (optional) The entry's Id element
          link: list (optional) A list of Link instances
          logo: Logo (optional) 
          rights: Rights (optional) The entry's Rights element
          subtitle: Subtitle (optional) The entry's subtitle element
          title: Title (optional) the entry's title element
          updated: Updated (optional) the entry's updated element
          entry: list (optional) A list of the Entry instances contained in the 
              feed.
          text: String (optional) The text contents of the element. This is the 
              contents of the Entry's XML text node. 
              (Example: <foo>This is the text</foo>)
          extension_elements: list (optional) A list of ExtensionElement instances
              which are children of this element.
          extension_attributes: dict (optional) A dictionary of strings which are 
              the values for additional XML attributes of this element.
        """

        self.author = author or []
        self.category = category or []
        self.contributor = contributor or []
        self.generator = generator
        self.icon = icon
        self.id = atom_id
        self.link = link or []
        self.logo = logo
        self.rights = rights
        self.subtitle = subtitle
        self.title = title
        self.updated = updated
        self.entry = entry or []
        self.text = text
