"""Stellar lifetimes as a function of stellar mass, and related functions.
"""
import math

import numpy
import scipy.interpolate

def main_sequence_life(mass):
    """Rough approxiamtion of main sequence lifetime in years.

    tauMS ~ 10^10 years * (M/Msun)^-2.5
    
    mass in solar masses.
    """
    return 1.e10 * mass**-2.5

def mass_from_main_sequence_life(life):
    """Approximate mass given main sequence lifetime in years.

    tauMS ~ 10^10 years * (M/Msun)^-2.5
    
    mass in solar masses.
    """
    return (life/1.e10)**(-1./(2.5))

def invert_function(function, xmin, xmax, xstep, bounds_error=False):
    """Invert function y = F(x) using interpolation.

    Returns an interpolation function giving x as a function of y.
    """
    xi = numpy.arange(xmin, xmax + xstep, xstep)
    yi = function(xi)
    ifunc = scipy.interpolate.interp1d(yi, xi, bounds_error=bounds_error)
    return ifunc

def invert_function_log(function, logxmin, logxmax, logxstep,
                        bounds_error=False):
    """Invert function y = F(x) using interpolation.

    Internally uses logarithmic scale in the x values and in
    performing the interpolation.

    Returns
    -------

    an interpolation function giving x as a function of y. NOTE:
    provide y, don't provide log(y).

    """
    logxi = numpy.arange(logxmin, logxmax + logxstep, 
                         logxstep)
    logyi = numpy.log10(function(10.**logxi))
    logifunc = scipy.interpolate.interp1d(logyi, logxi, 
                                          bounds_error=bounds_error)
    ifunc = lambda y: 10.**logifunc(numpy.log10(y))
    return ifunc

def invert_function_semilogy(function, xmin, xmax, xstep,
                             bounds_error=False):
    """Invert function y = F(x) using interpolation.

    Internally uses logarithmic scale in the y values.

    Returns
    -------

    an interpolation function giving x as a function of y. NOTE:
    provide y, don't provide log(y).

    """
    xi = numpy.arange(xmin, xmax + xstep, 
                      xstep)
    logyi = numpy.log10(function(xi))
    ifunclog = scipy.interpolate.interp1d(logyi, xi, 
                                       bounds_error=bounds_error)
    ifunc = lambda y: ifunclog(numpy.log10(y))
    return ifunc

def mass_from_main_sequence_life_function(function, mmin=0., mmax=1000., 
                                          mstep=0.01):
    """Returns a function giving stellar mass from lifetime.

    Uses invert_function_semilogy to define an interpolation function
    that will return stellar mass when given stellar lifetime.
    
    """
    return invert_function_semilogy(function, mmax, mmin, -1. * mstep)

def main_sequence_life_MM89(mass):
    """Lifetime in years from Maeder & Meynet (1989) via Romano et al. (2005). 

    The minlife attribute gives the minimum lifetime for a star as M->inf.

    Parameters
    ----------
    mass (1D array):
        Stellar mass in solar masses.

    Notes
    -----

    From Maeder & Meynet (1989), via Romano, Chiappini, Matteucci,
    Tosi, 2005, A\&A, 430, 491 (2005A&A...430..491R), who say:

      t = 
        10^(-0.6545 logm + 1) for m <= 1.3 MSun
        10^(-3.7 logm + 1.35) for 1.3 < m/MSun <= 3
        10^(-2.51 logm + 0.77) for 3 < m/MSun <= 7
        10^(-1.78 logm + 0.17) for 7 < m/MSun <= 15
        10^(-0.86 logm - 0.94) for 15 < m/MSun <= 60
        1.2 m^-1.85  +  0.003 for m > 60 MSun
      with t in units of Gyr.
      
    """

    import numpy

    mass = numpy.atleast_1d(mass)

    if mass.ndim > 1:
         raise ValueError("mass must be a 1D array or scalar.")

    logm = numpy.log10(mass)

    logmfactor = numpy.array([[-0.6545, -3.7, -2.51, -1.78, -0.86]]).transpose()
    constant = numpy.array([[1., 1.35, 0.77, 0.17, -0.94]]).transpose()
    mlimit = numpy.array([[0., 1.3, 3., 7., 15., 60.]]).transpose()

    t_star = numpy.sum(10**(logmfactor * logm + constant) * 
                       ((mass > mlimit[0:-1]) * (mass <= mlimit[1:])),
                       axis=0)
    t_star[mass > 60.] = 1.2 * mass[mass > 60.]**-1.85 + 0.003
    return 1.e9 * t_star
main_sequence_life_MM89.minlife = 0.003 * 1.e9

def main_sequence_life_K97(mass):
    """Lifetime in years from Kodama (1997) via Romano et al. (2005). 

    The minlife attribute gives the minimum lifetime for a star as M->inf.

    Parameters
    ----------
    mass: array
        Stellar mass in solar masses.

    Notes
    -----
    
    50 for m <= 0.56 MSun
    10^((0.334 - sqrt(1.790 - 0.2232*(7.764 - logm)))/0.1116) for m <= 6.6 MSun
    1.2 m^-1.85 + 0.003 for m > 6.6 MSun
    """
    import numpy
    mass = numpy.atleast_1d(mass)
    t = numpy.empty(mass.shape)

    t[mass <= 0.56] = 50.
    imask = numpy.logical_and(mass > 0.56, mass <= 6.6)
    t[imask] = \
        10.**((0.334 - numpy.sqrt(1.790 - 
                                  0.2232*(7.764 - numpy.log10(mass[imask])))
               ) / 0.1116)
    t[mass > 6.6] = 1.2 * mass[mass > 6.6]**-1.85 + 0.003

    # Make sure the max lifetime is preserved.
    t[t>50.] = 50.

    return 1.e9 * t
main_sequence_life_K97.minlife = 0.003 * 1.e9
main_sequence_life_K97.maxlife = 50. * 1e9

def test_conversion(lifetime_function):
    """Tests conversion of mass into lifetime and back. Note that this
    test is not passed right now because of the numerical
    discontinuties in the lifetime functions where their different
    segments meet up."""
    print lifetime_function.__name__
    import numpy.testing.utils as ntest
    mass = numpy.arange(0., 200., 0.1)
    time_from_mass = lifetime_function(mass)
    mass_from_time_func = \
                       mass_from_main_sequence_life_function(lifetime_function)
    mass_from_time_from_mass = mass_from_time_func(time_from_mass)

    frac_diff = (mass_from_time_from_mass-mass)/mass

    pylab.subplot(221)
    pylab.plot(mass, mass_from_time_from_mass)

    pylab.subplot(222)
    pylab.plot(mass, time_from_mass, label=lifetime_function.__name__)
    pylab.yscale('log')
    pylab.legend(loc='best')

    pylab.subplot(223)
    pylab.plot(mass, frac_diff)

    goodmask = numpy.ones(len(time_from_mass), dtype=numpy.bool)
    if hasattr(lifetime_function, 'minlife'):
        goodmask[time_from_mass <= lifetime_function.minlife] = False
    if hasattr(lifetime_function, 'maxlife'):
        goodmask[time_from_mass >= lifetime_function.maxlife] = False

    goodmask[numpy.logical_not(numpy.isfinite(time_from_mass))] = False

    print "Max frac. diff: ",
    print numpy.nanmax(numpy.abs(frac_diff[goodmask]))
    print "SKIPPING TEST THAT FAILS!!!!"
    if (False):
        threshold = 1e-9
        ntest.assert_array_less(numpy.abs(frac_diff[goodmask]),
                                threshold * numpy.ones(len(frac_diff[goodmask])),
                                err_msg='Error in mass recovery exceeds %s' %
                                threshold)
        print 'Mass recovery discrepency is below %s' % threshold

if __name__ == '__main__':

    import os
    import sys
    
    ### Argument parsing. ###
    if len(sys.argv)==1:
        print "Run with a filename argument to produce image files, e.g.:"
        print " python lifetime.py lifetime.png"
    if len(sys.argv) > 1:
        filename = sys.argv[1]
        prefix, extension = os.path.splitext(filename)
    else:
        filename = None
        
    # Get a list of all the main_sequence_life_* functions.
    module = sys.modules[__name__]
    functionlist = [getattr(module, name) for name in dir(module) if name.startswith('main_sequence_life_')]

    import pylab
    #import scipy
    #import scipy.integrate
    # Plot the functions by mass.
    for function in functionlist:
        pylab.figure(figsize=(11,6))
        test_conversion(function)

    ### Plot output code. ###
    if filename is None:
        pylab.show()
    else:
        from matplotlib import _pylab_helpers
        for manager in _pylab_helpers.Gcf.get_all_fig_managers():
            fig = manager.canvas.figure
            if len(fig.get_label()) > 0:
                label = fig.get_label()
            else:
                label = '_Fig' + str(manager.num)
            newfilename = prefix + '_' + label + extension
            fig.savefig(newfilename, bbox_inches="tight")
