# -----------------------------------------------------------------------------
# _output.py - Output generators.
#
# 2012, Ben Eills
#
# Copyright 2012, 2014, Ensoft Ltd.
# -----------------------------------------------------------------------------

from __future__ import absolute_import, print_function

"""Output generators."""

__all__ = (
    "EntrailsOutput",
    "ConsoleOutput",
    "DebugOutput",
    "HTMLOutput",
    "LogFileOutput",
    "PlainTextOutput",
    "XMLOutput",
)


import datetime
import sys

from . import _utils


class EntrailsOutput(object):
    """
    Base output adapter class.

    Use as template to implement custom output adapters.
    """
    def __init__(self):
        pass
    def close(self):
        pass
    def enter(self, o):
        pass
    def exit(self, o):
        pass
    def exception(self, o):
        pass
    def logging(self, o, event_type_string):
        pass


class PlainTextOutput(EntrailsOutput):
    """
    Output adapter generating plaintext of variable verbosity, one
    event per line.

    TODO: printf formatting, verbosity
    """
    def __init__(self, verbosity, prefix_func=None, depth_spacer=" ",
                 enter_func=None, exit_func=None, exception_func=None):
        self.depth_spacer = depth_spacer
        self.safe_write("Starting trace in program\n")
        self.depth = 0
        self.prefix_func = prefix_func or self._default_prefix_func
        self.enter_func = enter_func or self._default_enter_func
        self.exit_func = exit_func or self._default_exit_func
        self.exception_func = exception_func or self._default_exception_func

    def close(self):
        if not self.f.closed:
            self.f.close()

    def safe_write(self, data):
        if not self.f.closed:
            self.f.write(data)

    def _default_enter_func(self, o):
        return "Call " + o.short_call_string()

    def _default_exit_func(self, o):
        return "Return from {0} with value {1}".format(o["co_name"],
                                                       o.short_return_value())

    def _default_exception_func(self, o):
        return "Encountered exception" #@@@ more info TODO

    def _default_prefix_func(self, o):
        return ("[%s][%03d:%s]" + "  "*self.depth) % (o["label"],
                                                      o["f_lineno"],
                                                      _utils.pretty_resize_string(o.nice_filename(), 15, exact=True))

    def enter(self, o):
        self.depth += 1
        self.safe_write(self.prefix_func(o) + self.enter_func(o) + "\n")
        return None

    def exit(self, o):
        self.safe_write(self.prefix_func(o) + self.exit_func(o) + "\n")
        self.depth -= 1
        return None

    def exception(self, o):
        self.safe_write(self.prefix_func(o) + self.exception_func(o) + "\n")
        return None

    def logging(self, o, event_type_string):
        self.safe_write("ERROR TRACING with event: %s on line %d in %s\n" % (
                event_type_string, o["f_lineno"], o["co_filename"]))
        return None


class LogFileOutput(PlainTextOutput):
    """
    Convienience function to send plaintext output to a file.
    """
    def __init__(self, filename, verbosity, prefix_func=None, depth_spacer=" ",
                 enter_func=None, exit_func=None, exception_func=None):
        self.f = open(filename, 'a', 1) #line-buffering
        super(LogFileOutput, self).__init__(verbosity, prefix_func=prefix_func,
                                            depth_spacer=depth_spacer,
                                            enter_func=enter_func,
                                            exit_func=exit_func,
                                            exception_func=exception_func)


class ConsoleOutput(PlainTextOutput):
    """
    Convienience function to send plaintext output to stdout.
    """
    def __init__(self, filename, verbosity, prefix_func=None, depth_spacer=" ",
                 enter_func=None, exit_func=None, exception_func=None):
        self.f = sys.stdout
        super(ConsoleOutput, self).__init__(verbosity, prefix_func=prefix_func,
                                            depth_spacer=depth_spacer,
                                            enter_func=enter_func,
                                            exit_func=exit_func,
                                            exception_func=exception_func)


class DebugOutput(EntrailsOutput):
    """
    Displays all output data with little formatting.
    """
    def __init__(self):
        print("Entrails debug output started on {0}".format(
            datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")))

    def close(self):
        print("Entrails debug output ended on {0}".format(
            datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")))

    def enter(self, o):
        print("FUNCTION CALL: " + o.call_string())
        for k, v in o.items():
            print("\t{0}:{1}".format(k, str(v)))
        return None

    def exit(self, o):
        print("FUNCTION EXIT")
        for k, v in o.items():
            print("\t{0}:{1}".format(k, str(v)))
        return None

    def exception(self, o):
        print("EXCEPTION on line {0} in {1}".format(o["f_lineno"],
                                                    o["co_filename"]))
        for k, v in o.items():
          print(k + ": " + str(v))
        return None

    def logging(self, o, event_type_string):
        print("ERROR TRACING with event: {0} on line {1} in {2}".format(event_type_string,
                                                                        o["f_lineno"],
                                                                        o["co_filename"]))
        return None


class HTMLOutput(EntrailsOutput):
    """
    Not fully implemented.  Basic HTML output.
    """
    def __init__(self, filename, verbosity):
        self.depth = 0
        self.f = open(filename, 'a', 1) #line-buffering
        self.f.write("<html><body><h1>Trace of program started on {0}</h1>\n".format(datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")))

    def __del__(self):
        self.f.write("\n</body></html>\n")
        self.f.close()

    def enter(self, codeframe):
        self.depth += 1
        self.f.write(" > "*self.depth + "<b>Now entering:</b> " + codeframe["name"] + "<br />\n")
        return None

    def exit(self, codeframe):
        self.depth -= 1
        self.f.write(" > "*(self.depth+1) + "<b>and exiting:</b> " + codeframe["name"] + " with return value " + str(codeframe["return_value"]) + "<br />\n")
        return None

    def exception(self, o):
        self.f.write("<h3>Encountered exception!</h3>")
        return None

    def logging(self, o, event_type_string):
        self.f.write("<h4>ERROR TRACING with event: %s on line %d in %s</h4>" % (event_type_string, o["f_lineno"], o["co_filename"]))
        return None


class XMLOutput(EntrailsOutput):
    def __init__(self, filename):
        self.filename = filename
        self.root = ET.Element("data")

    def close(self):
        tree = ET.ElementTree(self.root)
        tree.write(self.filename, xml_declaration=True, method="xml")

    def write_out(self, codeframe, event_type):
        event = ET.SubElement(self.root, "event")
        event.attrib = {"timestamp" : "{0:f}".format(codeframe["timestamp"])}
        for k, v in codeframe.items():
            if k != "timestamp":
                ET.SubElement(event, k).text = str(v)
        ET.SubElement(event, "type").text = event_type
        ET.SubElement(event, "call_string").text = codeframe.call_string()
        ET.SubElement(event, "short_call_string").text = codeframe.short_call_string()
        ET.SubElement(event, "short_return_value").text = codeframe.short_return_value()
        ET.SubElement(event, "nice_filename").text = codeframe.nice_filename()

    def enter(self, codeframe):
        self.write_out(codeframe, "enter")

    def exit(self, codeframe):
        self.write_out(codeframe, "exit")

    def exception(self, codeframe):
        self.write_out(codeframe, "exception")

