# -----------------------------------------------------------------------------
# _trace.py - Execution tracer
#
# 2012, Ben Eills
#
# Copyright 2012, 2014, Ensoft Ltd.
# -----------------------------------------------------------------------------

from __future__ import absolute_import, print_function

"""Execution tracer."""

__all__ = (
    "Entrails",
)


import collections
import os.path
import sys
import time

from . import _filter
from . import _utils


class _Frame(dict):
    """
    Holds information about a frame.

    Use as a dict to access the important bits of the frame and arg
    parameters, along with timestamp and label.  Also has convienience
    methods to generate function call string and class info.  This is
    supplied to an output adapter's call(), return() and exception()
    methods.
    """
    def parameters(self):
        """
        Returns dict of form {parameter_name: argument_value}.
        """
        # We rely on CPython putting arguments at start of f_locals (in reverse
        #   order for some reason...).  This is undocumented, but consistent.
        parameters = [(k, v) for k, v in self["f_locals"].items()
                    if k in self["co_varnames"][0:self["co_argcount"]]]
        parameters.reverse()
        return collections.OrderedDict(parameters)

    def arguments(self):
        """
        Returns argument list, possibly excluding the first 'self' arg
        if the function is a method.

        Only designed for 'call' event (i.e. function entry)
        """
        # Possibly chop off first self argument
        ignorer = 0 if not self.is_method() else 1
        return list(self.parameters().values())[ignorer:]

    def string_arguments(self):
        """
        Returns list of nice string argument representations.

        If the argument object encounters an error during repr() call
          (some standard library classes can do this under certain conditions)
          fail cleanly.
        """
        strings = []
        for arg in self.arguments():
            try:
                strings.append(repr(arg))
            except:
                strings.append('<repr error>')
        return strings

    def call_string(self):
        """
        Returns a pretty string representation of the function call built
          from function name and args, nicely formatted/abbrieviated.

        Only designed for 'call' event (i.e. function entry)
        """
        if self.is_method():
            return '{0}.{1}({2})'.format(self.instance_name(), self['co_name'],
                                         ', '.join(self.string_arguments()))
        else:
            return '{0}({1})'.format(self['co_name'],
                                     ', '.join(self.string_arguments()))

    def short_call_string(self, length=40):
        """
        Returns a pretty string of the function call, shortened to at
          most length characters.

        Only designed for 'call' event (i.e. function entry)
        """
        return _utils.pretty_resize_string(self.call_string(), length)

    def short_return_value(self, length=40):
        """
        Returns a pretty string representation of the return value,
          shortened to at most length characters.

        Only designed for 'call' event (i.e. function entry)
        """
        try:
            return _utils.pretty_resize_string(repr(self['return_value']), length)
        except:
            return _utils.pretty_resize_string('<repr error>', length)

        # Do we close quotation marks in string return values?
        # if len(repr(rv)) > 40:
        #     if rv.__class__.__name__ == 'str':
        #         return repr(rv)[0:length-4]+"...'"
        #     else:
        #         return repr(rv)[0:length-3]+"..."
        # else:
        #     return repr(rv)

    def nice_filename(self):
        """
        Nice filename, without directory segment.
        """
        return os.path.basename(self["co_filename"])

    def is_method(self):
        """
        True if function is an instance method  (and the programmer
          did not defy all convention by abandonning the use of 'self'
          as the instance pointer).
        Only designed for 'call' event (i.e. function entry)

        TODO: What happens for class methods/other decorated methods?
        """
        if self['co_varnames'] and self['co_varnames'][0] == 'self':
            return True
        return False

    def instance_name(self):
        """
        Class instance name of method, or empty string if not method.
        Only designed for 'call' event (i.e. function entry)
        We cache value after first proper lookup.
        """
        if 'instance_name' not in self:
            self['instance_name'] = self._retrieve_instance_name()
        return self['instance_name']

    def _retrieve_instance_name(self):
        """
        Get class instance name by performance-intensive lookup.
        Should only be called on methods.
        """
        try:
            # This is confusing.  We go into the previous (parent) frame and
            #   find the local variable name in that pointing to a our class.
            for k, v in list(self['f_back'].f_locals.items()):
                if v == self['f_locals']['self']:
                    return k
        except:
            return '<lookup error>'

    def class_name(self):
        """
        Class name of method, or empty string if not method.
        Only designed for 'call' event (i.e. function entry)
        Should only be called on methods.
        """
        try:
            return self["f_locals"]["self"].__class__.__name__
        except:
            return '<lookup error>'


class Entrails(object):
    """
    Implements entrails functionality to monitor program execution flow.

    Instantiate at in process to be traced, add outputs and call start_trace().
    Run end_trace() to safely close outputs and stop tracing.
    """
    event_names = { 'call' : 'enter',
                    'return' : 'exit',
                    'exception' : 'exception' }

    def __init__(self,
                 label='<untitled>',
                 filters=(_filter.LibraryFilter(), _filter.SelfFilter())):
        self._outputs = []
        self._init_time = time.time()
        self._label = label
        self._filters = filters

    def filter(self, obj):
        """
        Returns True if object remains unfiltered, False otherise.
        """
        if all(f.filter(obj) for f in self._filters):
            return True
        return False

    def _trace_function(self, trace, event, arg):
        # Build Frame
        obj = _Frame()
        obj["label"] = self._label
        obj["timestamp"] = time.time() - self._init_time
        obj["f_lineno"] = trace.f_lineno
        obj["f_back"] = trace.f_back
        obj["f_locals"] = trace.f_locals
        obj["co_name"] = trace.f_code.co_name
        obj["co_argcount"] = trace.f_code.co_argcount
        obj["co_filename"] = trace.f_code.co_filename
        obj["co_names"] = trace.f_code.co_names
        obj["co_varnames"] = trace.f_code.co_varnames

        # Filter
        # If this object is to be discarded, return trace_function to
        #   indicate that we wish to continue tracing.
        if not self.filter(obj):
            return self._trace_function

        # Possibly add return_value.  If event is non-return, arg isn't really
        #   that useful.
        if event is 'return':
            obj["return_value"] = arg

        if event in self.event_names:
            for output in self._outputs:
                #try:
                getattr(output, self.event_names[event])(obj)
        return self._trace_function

    def add_output(self, output_object):
        """
        Adds output adapter, accepting argument of type EntrailsOutput.
        """
        self._outputs.append(output_object)

    def remove_output(self, output_object):
        """
        Cease sending tracing data to a particular output.
        Removes from internal list and runs output.__del__()@@@
        """
        output_object.close()
        if output_object in self._outputs:
            self._outputs.remove(output_object)

    def start_trace(self):
        """
        Start tracing.
        """
        sys.settrace(self._trace_function)

    def end_trace(self):
        """
        Stop tracing.  Safely removes all outputs.
        """
        for o in self._outputs:
            self.remove_output(o)
        sys.settrace(None)

