#-----------------------------------------------------------------------------
# Copyright (c) 2008  Raymond L. Buvel
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#-----------------------------------------------------------------------------

#-----------------------------------------------------------------------------
# Concatenate and transform the ASCII header and data files for the selected
# ephemeris into binary format.  Use this program for cases where the Unix data
# cannot be translated.  The resulting binary file is in native format.
#-----------------------------------------------------------------------------

# Edit the following values to select the ephemeris and point to the directory
# containing the ASCII format data files.

ephemNumber = '405'
asciiDir = '/home/rlb/ephemeris/ascii'

# This automatically selects an output file name based on the ephemeris.  If
# you don't like this convention, change it here.
outFileName = 'DE%s.bin' % ephemNumber

#-----------------------------------------------------------------------------
import os, fileinput, glob, struct

#-----------------------------------------------------------------------------
# Prepare the list of ASCII format files for processing.  The files must be in
# time sequence.

files = glob.glob(os.path.join(asciiDir,'asc*.%s' % ephemNumber))
lst1 = []
lst2 = []
for name in files:
    if name.find('ascm') >= 0:
        lst1.append(name)
    elif name.find('ascp') >= 0:
        lst2.append(name)
    else:
        raise ValueError('Unknown file')
lst1.sort()
lst1.reverse()
lst2.sort()
files = [os.path.join(asciiDir,'header.%s' % ephemNumber)] + lst1 + lst2
print 'Number of files =', len(files)-1

#-----------------------------------------------------------------------------
# Process lines in the header and build up the header dictionary which is used
# to write the binary header records.  When the data group is encountered, the
# header is written and processing switches to processData.

curGroup = None
header = {}

def processHeader(line):
    global curGroup, arraySize

    if line.startswith('GROUP'):
        curGroup = line.split()[-1]
        if curGroup == '1070':
            # Start of the data group.  Write out the header records.
            writeHeader()
            return
        header[curGroup] = []
    elif line.startswith('KSIZE='):
        lst = line.split()
        arraySize = int(lst[-1])
    else:
        header[curGroup].append(line)

#-----------------------------------------------------------------------------
# Process lines in the data section.  When a new record is detected, the
# previous record is written.

dataValues = []

def processData(line):
    line = line.replace('D','e')
    lst = line.split()
    if len(lst) == 2:
        # Start of a new record
        if arraySize != int(lst[1]):
            raise ValueError('Invalid array size')
        if dataValues:
            writeData()
    elif len(lst) == 3:
        dataValues.extend([float(x) for x in lst])
    else:
        raise ValueError('Invalid data line')

#-----------------------------------------------------------------------------
# Verify the time sequence and write out the record.  Sets the global variables
# startTime and endTime as the records are written.  These times are used to
# adjust the start and end times in the header record.

recCount = 0

def writeData():
    global recCount, lastTime, startTime, endTime

    if len(dataValues) < arraySize:
        raise ValueError('Invalid data record')

    if (dataValues[1] - dataValues[0]) != timeStep:
        raise ValueError('Invalid data record')

    if recCount:
        # If not the first record, check the time sequence
        dt = dataValues[0] - lastTime
        if dt == 0:
            # Skip duplicate records
            dataValues[:] = []
            return
        elif dt != timeStep:
            raise ValueError('Time sequence error')
    else:
        # Record the actual start time.
        startTime = dataValues[0]

    outFile.write(struct.pack('=%dd' % arraySize, *dataValues[:arraySize]))

    lastTime = dataValues[0]
    endTime = dataValues[1]
    dataValues[:] = []
    recCount += 1

    if (recCount % 500) == 0:
        print 'Record', recCount

#-----------------------------------------------------------------------------
# Write out the header records using the data collected in the header
# dictionary when the header file was read.  Sets the global variable timeStep
# which is used to verify the time sequence of the data records.

def writeHeader():
    global timeStep

    arrayBytes = arraySize * struct.calcsize('d')
    blanks = ' '*84
    outList = []

    # Process the title group
    group = '1010'
    if len(header[group]) != 3:
        raise invalidGroup(group)
    title = header[group]

    # Process the constant names
    group = '1040'
    numConst = int(header[group][0])
    names = []
    for line in header[group][1:]:
        names.extend(line.split())

    if len(names) < numConst:
        raise invalidGroup(group)
    names = names[:numConst]

    # Collect the constant values
    group = '1041'
    if numConst != int(header[group][0]):
        raise invalidGroup(group)
    values = []
    for line in header[group][1:]:
        line = line.replace('D','e')
        values.extend([float(x) for x in line.split()])

    if len(values) < numConst:
        raise invalidGroup(group)
    values = values[:numConst]

    # Create a constant dictionary since a few values need to be extracted.
    const = dict(zip(names, values))

    # Process the times group
    group = '1030'
    if len(header[group]) != 1:
        raise invalidGroup(group)
    times = [float(x) for x in header[group][0].split()]
    timeStep = times[2]

    # Process the structure group
    group = '1050'
    if len(header[group]) != 3:
        raise invalidGroup(group)
    lst = []
    for line in header[group]:
        lst.append([int(x) for x in line.split()])

    dataStruct = zip(*lst)
    if len(dataStruct) != 13:
        raise invalidGroup(group)

    # Construct the header records
    for x in title:
        outList.append(x + blanks[:84-len(x)])

    for x in names:
        outList.append(x + blanks[:6-len(x)])

    # Fill out the rest of the names block
    outList.append(' '*(6*(400-numConst)))

    outList.append(struct.pack('=3d', *times))
    outList.append(struct.pack('=i', numConst))
    outList.append(struct.pack('=d', const['AU']))
    outList.append(struct.pack('=d', const['EMRAT']))

    for x in dataStruct[:12]:
        outList.append(struct.pack('=3i', *x))

    outList.append(struct.pack('=i', int(const['DENUM'])))

    outList.append(struct.pack('=3i', *dataStruct[12]))

    # Compress the list into a single string so the length is available.
    outStr = ''.join(outList)
    outList[:] = [outStr]

    # Pad the first record with zeros
    outList.append('\0'*(arrayBytes - len(outStr)))

    # The constant values are placed at the start of the second record.
    constStr = struct.pack('=%dd' % numConst, *values)
    outList.append(constStr)

    # Pad the second record with zeros
    outList.append('\0'*(arrayBytes - len(constStr)))

    # Perform a final error check and write out the header records
    outStr = ''.join(outList)
    if len(outStr) != (2*arrayBytes):
        raise ValueError('Invalid header')

    outFile.write(outStr)


def invalidGroup(name):
    return ValueError('GROUP %s invalid' % name)

#-----------------------------------------------------------------------------

outFile = file(outFileName, 'wb')

for line in fileinput.input(files):
    line = line.strip()
    if not line:
        # Skip blank lines
        continue
    if curGroup == '1070':
        processData(line)
    else:
        processHeader(line)

# If the last record has not been written, do so now.
if dataValues:
    writeData()

# Update the time limits in the header
outFile.seek(3*84 + 6*400)
outFile.write(struct.pack('=2d', startTime, endTime))
outFile.close()

