#-----------------------------------------------------------------------------
# Copyright (c) 2008  Raymond L. Buvel
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#-----------------------------------------------------------------------------

#-----------------------------------------------------------------------------
# Concatenate Unix format binary files and translate Unix format data (big
# endian) into native format.  This has been tested on a PC running Linux and
# Windows.  However, it should work for any 32-bit machine using IEEE-754
# floating point numbers.  It may also work on 64-bit architectures using
# IEEE-754 as long as the C compiler used to build Python uses 32 bits for
# an int.
#-----------------------------------------------------------------------------

# Edit the following values to select the ephemeris and point to the directory
# containing the Unix format data files.
#
# NOTE: DE200, DE405, and DE406 all work corectly.  However, I was unable to
# get DE403 to work.  I didn't try the others.  If you want some of these other
# data sets, you will need to convert the ASCII files.

ephemNumber = '405'
unixDir = '/home/rlb/ephemeris/unix'

# This automatically selects an output file name based on the ephemeris.  If
# you don't like this convention, change it here.
outFileName = 'DE%s.bin' % ephemNumber

# The value for arraySize is the value of NCOEFF.  This value can be found in
# the ASCII header file for the selected ephemeris.  For each ephemeris, there
# is an entry in the following dictionary so that the array size selection can
# be handled by simply selecting the desired ephemeris.
arraySize = {
'200' : 826,
'405' : 1018,
'406' : 728,
}[ephemNumber]

arrayBytes = arraySize*8

#-----------------------------------------------------------------------------
import os,glob,struct

#-----------------------------------------------------------------------------
# Prepare the list of Unix format files for processing.  The files must be in
# time sequence.

files = glob.glob(os.path.join(unixDir,'unx*.%s' % ephemNumber))
lst1 = []
lst2 = []
for name in files:
    if name.find('unxm') >= 0:
        lst1.append(name)
    elif name.find('unxp') >= 0:
        lst2.append(name)
    else:
        raise ValueError('Unknown file')
lst1.sort()
lst1.reverse()
lst2.sort()
files = lst1 + lst2
print 'Number of files =', len(files)

#-----------------------------------------------------------------------------
# Define the structure of the header records.  These format parameters for the
# struct module are used to unpack the Unix format and pack the native format.
# If you are on a machine where the native format gets padded, you will need to
# rework these definitions to correct for that.

H1strSize = 84*3 + 400*6 # Title and constant names
H1dataFmt = '3d i 2d 36i i 3i'
H1dataSize = struct.calcsize('>'+H1dataFmt)

partsFmt = ''.join([
  # Header record 1
  '%ds' % H1strSize,
  '%ds' % H1dataSize,
  '%ds' % (arrayBytes - (H1strSize + H1dataSize)), # Pad bytes
  # Header record 2
  '%ds' % (400*8),  # Constant values
  '%ds' % (arrayBytes - 400*8),  # Pad bytes
])

#-----------------------------------------------------------------------------
# Collect header information from the first file
efile = file(files[0], 'rb')

# Save Unix header for error check
uHeader = struct.unpack(partsFmt, efile.read(2*arrayBytes))
_parts = list(uHeader)
H1 = struct.unpack('>'+H1dataFmt, _parts[1])
H2 = struct.unpack('>400d', _parts[3])
_parts[1] = struct.pack('='+H1dataFmt, *H1)
_parts[3] = struct.pack('=400d', *H2)
nHeader = ''.join(_parts)
efile.close()
assert len(nHeader) == 2*arrayBytes
del _parts

timeStep = H1[2]

#-----------------------------------------------------------------------------
# Open the output file and write the native format header.

efile = file(outFileName, 'wb')
efile.write(nHeader)
del nHeader

#-----------------------------------------------------------------------------
# Translate the Unix data to native format and perform some integrity checks so
# that the process fails early if it is not likely to work correctly.

nRec = 0
titleLst = []
def xlatRecords(name):
    global nRec, lastRecord
    print name
    inFile = file(name, 'rb')
    uFmt = '>%dd' % arraySize
    nFmt = '=%dd' % arraySize

    # Validate the header record.  Note that not every ephemeris contains
    # nutations and librations.  The structure information is not always
    # encoded the same if these sections are missing (e.g. the coefficient
    # index is sometimes set and sometimes not).  These sections of the data
    # structure are not checked as a result.
    rec = struct.unpack(partsFmt, inFile.read(2*arrayBytes))
    assert rec[0][84*3:] == uHeader[0][84*3:]
    assert rec[1][16:-28] == uHeader[1][16:-28]
    assert rec[3] == uHeader[3]
    hdrTimes = struct.unpack('>2d', rec[1][:16])
    titleLst.append(rec[0][:84*3])

    while True:
        rec = inFile.read(arrayBytes)
        if not rec:
            break

        if len(rec) != arrayBytes:
            filePos = inFile.tell() - len(rec)
            print 'File name: %s\nPosition: %s' % (name, filePos)
            raise ValueError('File size mis-match: read %s, expected %s' %
                    (len(rec), arrayBytes))

        data = struct.unpack(uFmt, rec)
        if nRec > 0:
            # Note: the time values are such that the floating point
            # calculations are exact.  Consequently, equality comparison is OK.
            dt = data[0] - lastRecord[0]
            if dt == 0:
                # Skip a duplicate record
                continue
            elif dt != timeStep:
                filePos = inFile.tell() - len(rec)
                print 'File name: %s\nPosition: %s' % (name, filePos)
                raise ValueError('Time sequence error: dt = %s, expected %s' %
                        (dt, timeStep))
        nRec += 1
        lastRecord = data
        efile.write(struct.pack(nFmt, *data))
    inFile.close()
    print (hdrTimes[1]-hdrTimes[0])/timeStep, nRec
    return hdrTimes + data[:2]

times = []
for name in files:
    times.append(xlatRecords(name))

#-----------------------------------------------------------------------------
# Validate the time information
for x in times:
    assert x[1] == x[-1]

#-----------------------------------------------------------------------------
# Verify the time span against the number of records.

startTime = times[0][0]
endTime = times[-1][-1]
n = (endTime-startTime)/timeStep
if abs(n-nRec) > 0.01:
    raise ValueError('Time span error: n = %s, nRec = %s' % (n, nRec))

#-----------------------------------------------------------------------------
# Update the title and time information

title = titleLst[0][:2*84] + titleLst[-1][2*84:]
efile.seek(0)
efile.write(title)
efile.seek(H1strSize)
efile.write(struct.pack('=2d', startTime, endTime))

efile.close()
