#     Copyright 2011 Newcastle University
#     Jacek Szpot, Maciej Machulak, Lukasz Moren
#
#     http://smartjisc.wordpress.com
#
#     Licensed under the Apache License, Version 2.0 (the "License");
#     you may not use this file except in compliance with the License.
#     You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
#     Unless required by applicable law or agreed to in writing, software
#     distributed under the License is distributed on an "AS IS" BASIS,
#     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#     See the License for the specific language governing permissions and
#     limitations under the License.

"""EsmeOAuth2. A box of magic, and a Token."""

__author__ = "Jacek Szpot"
__credits__ = ["Maciej Machulak", "Lukasz Moren"]
__copyright__ = "2011 Newcastle University"
__license__ = "Apache License 2.0"

import urlparse
import cgi
import urllib
import logging
import httplib
import time
# let's see what json parser we have available
try:
    import json
except ImportError:
    # as in Google's AppEngine
    from django.utils import simplejson as json

class NeedAuth(Exception):
    """Exception raised when the need for a (re-)auth is diagnosed or suspected."""
    def __init__(self, authz_uri):
        Exception.__init__(self)
        self.authz_uri = authz_uri

    def __str__(self):
        return repr(self.authz_uri)

class MissingData(Exception):
    """Raised when an action is attempted, but insufficient data was provided."""
    def __init__(self, problem):
        Exception.__init__(self)
        self.problem = problem

    def __str__(self):
        return repr(self.problem)

# still not sure.
class Config(object):
    """This is supposed to be a convenience object, wrapping all the setting into one.""" 
    def __init__(self):
        self.token_endpoint = None
        self.authz_endpoint = None
        self.client_secret = None
        self.client_id = None
        self.response_type = 'code' # default, here to ease future extensions
        self.scope = None

class Token(object):
    """The essence of this module - the omnipotent Token.

    This is an approach somewhat different to those seen elsewhere. 
    """
    def __init__(self, config=None, storage_method='appengine_datastore', retrieval_method='appengine_datastore'):
        """Provides basic initialization of all token-related parameters. Magic is absent."""
        # developer-provided
        # this isn't pretty.
        if config is not None:
            self.token_endpoint = config.token_endpoint
            self.authz_endpoint = config.authz_endpoint
            self.client_secret = config.client_secret
            self.redirect_url = config.redirect_url
            self.client_id = config.client_id
            self.response_type = 'code'
            self.scope = config.scope
        #xxx: scheduled for removal & restructuring
        elif config is None:
            self.token_endpoint = None
            self.authz_endpoint = None
            self.redirect_url = None
            self.client_secret = None
            self.client_id = None
            self.response_type = 'code'
            # todo: decide whether these belong here, or should be dropped
            # dynamically provided, on request?
            self.scope = None
        #xxx: end

        self.storage_method = storage_method
        self.retrieval_method = retrieval_method

        # automatically fetched
        self.access_token = None
        self.token_type = None
        self.expires_in = None
        self.refresh_token = None

        self.datastore_key = None

    # todo: (someday, maybe) provide a twin method, but based on callbacks -- compare usability
    # the essence of the whole class.
    def use_with(self, method, url, data=None):
        """Perform an HTTP(S) request, [hopefully] quietly taking care of access token affairs."""
        # todo: handle 403 messages? -- i.e. when a user has removed a scope from and application
        #       or is this too much of a border case to worry about?
        #       also, amusingly, the draft (rev18) doesn't even contain "403" in it.
        #       since a re-authorization will be needed anyway, I'll just add an == 403 case at the end.

        # mutable objects shouldn't be default arguments, so here we go:
        # also, we'll be using a list of tuples. The other option is a dict [todo].
        if data is None:
            data = []

        # check if there is an access token provided, raise MissingData
        if self.access_token is None:
            raise MissingData(".use_with() has been called, but .access_token is None. Initialize first.")

        # method check
        # todo: TRACE, HEAD, OPTIONS?
        if method.upper() not in ['GET', 'POST', 'PUT', 'DELETE']:
            raise ValueError('Unsupported method - has to be GET, POST, PUT or DELETE.')
 
        h = httplib.HTTPSConnection(urlparse.urlparse(url).netloc)
        # this has to be different for GET and different for POST - access_token should be in URI or body
        if method.upper() == 'GET':
            # todo: this ignores any parameters - fix this
            h.request(method, urlparse.urlparse(url).path+'?access_token='+self.access_token, urllib.urlencode(data))
        else:
            data.append(('access_token', self.access_token))
            h.request(method, urlparse.urlparse(url).path, urllib.urlencode(data))
        r = h.getresponse()
        logging.info('Token.use_with(): HTTPSConnection.getresponse(): '+str(r.status))
        if r.status == 401 or r.status == 400 and self.refresh_token is not None: # token expired or invalid, refresh token available
            logging.info('(1st .use_with attempt) Response from server was: '+r.read())
            logging.info('Attempting a token refresh -- hold on.')
            self.attempt_refresh()
            # not sure about this - could I just reuse `h`?
            h = httplib.HTTPSConnection(urlparse.urlparse(url).netloc)
            # again, we need to either include the access token in the query or the body of the request
            # data should already have access_token appended to it (a few lines earlier), if this is a POST
            if method.upper() == 'GET':
                h.request(method, urlparse.urlparse(url).path+'?access_token='+self.access_token, urllib.urlencode(data))
            elif method.upper() == 'POST':
                h.request(method, urlparse.urlparse(url).path, urllib.urlencode(data)) # access_token already in data
            r = h.getresponse()
            logging.info('Response to our refresh attempt: '+str(r.status))
            # a sad case we've been denied - again .. authorization will be required
            if r.status == 401 or r.status == 400: # temporary == 400
                # to make things clear: this code tries to automatically craft an authorization url
                # and return it via an exception - if, however, that turns out to be impossible (data missing)
                # the exception will simply contain a short error message instructing the user to provide missing
                # data, and calling create_authz_url manually.
                try:
                    authz_url = self.create_authz_url()
                except MissingData: # maybe this __should__ be a custom exception too? it probably should.
                    # todo: create a custom exception for use in create_authz_url(), e.g. MissingData()
                    # so that it is clear that the problem was in the module and not somewhere below
                    message = 'Looks like authorization is needed. I couldn\'t create an authz_url ' \
                              'for you, as some crucial data seems to be missing (i.e. is None). ' \
                              'Fix this and run create_authz_url() yourself, please.'
                    raise NeedAuth(message)
                # if we got here, means we got here with a shiny new authz_url
                raise NeedAuth(authz_url)

            # so let's say we got okay'd (200)
            if r.status == 200:
                return r # httplib.HTTPResponse

        # adding a 403 case in here -- for now
        if r.status == 401 or r.status == 400 or r.status == 403 and self.refresh_token is None:
            # *knock, knock* have you heard the bad news?
            logging.info(str(r.status)+', no refresh token. Server said: '+str(r.read()))
            # as before - DRY, DRY?
            try:
                authz_url = self.create_authz_url()
            except ValueError:
                # todo: put some effort in writing a proper error message
                message = 'I don\'t feel like coming up with a witty error message right now.'
                raise NeedAuth(message)
            raise NeedAuth(authz_url)
            # to be finished and thought about

        # if it went well
        if r.status == 200:
            return r # httplib.HTTPResponse

        # todo: structure this nicely, rethink
        # dead end
        logging.info('Neither 401 nor 200, was a '+str(r.status)+':\n'+r.read())
        raise Exception("Neither a 401 nor a 200 received.")

    def attempt_refresh(self):
        """Make a bold attempt of refreshing the access token via a refresh token."""
        h = httplib.HTTPSConnection(urlparse.urlparse(self.token_endpoint).netloc)
        headers = { 'Content-Type': 'application/x-www-form-urlencoded' }
        params = {
            'client_id': self.client_id,
            'client_secret': self.client_secret,
            'refresh_token': self.refresh_token,
            'grant_type': 'refresh_token',
        }
        logging.info('urlencoded params for this request:\n'+urllib.urlencode(params))
        h.request('POST', urlparse.urlparse(self.token_endpoint).path, urllib.urlencode(params), headers)
        response = h.getresponse()
        response_read = response.read()
        logging.info('Server responded with HTTP '+str(response.status)+' to our refresh request.')
        logging.info('Response was: '+response_read )

        self.parse_token_endpoint_response(response_read)
        
    def parse_token_endpoint_response(self, response_body):
        """Conduct a parsing effort and, on success, update the appropriate parameters.

        While this lacks explicitness, this method also saves the Token object state
        to a less ephemeral type of storage. Currently, the only supported - and therefore
        default - way is storing [parts of] the token in Google's AppEngine datastore.
        """
        try:
            parsed_response = json.loads(response_body)
            logging.info('Response containing token(s) was actually a JSON one.')
        except ValueError:
            logging.info('Response containing token(s) was ... not JSON, trying urlencoding..')
            # looks like we're dealing with something else than json
            # let's try urlencoding then
            parsed_response = dict(cgi.parse_qsl(response_body))
            # if we get an empty dictionary, something was wrong. not urlencoding, probably.
            # to be extended, if the need arises
                
        # we're through, time to update the object's data
        self.access_token = parsed_response.get('access_token', None)
        logging.info('parsed: access_token = '+str(self.access_token))
        # --- `expires` vs `expires_in`
        if 'expires' in parsed_response:
            self.expires_in = int(parsed_response['expires'])
        elif 'expires_in' in parsed_response:
            self.expires_in = int(parsed_response['expires_in'])
        else:
            self.expires_in = None
        logging.info('parsed: expires_in = '+str(self.expires_in))
        
        # --- should be done
        self.token_type = parsed_response.get('token_type', None)
        logging.info('parsed: token_type = '+str(self.token_type))

        # __IF__ a refresh token was provided, overwrite the current one
        new_refresh_token = parsed_response.get('refresh_token', None)
        if new_refresh_token is not None:
            self.refresh_token = new_refresh_token
        
        logging.info('parsed: refresh_token = '+str(self.refresh_token))

        logging.info('Looks like we\'re done parsing. Successfully.')

        # token data in datastore needs to be updated as well
        # todo: AppEngine only! and pretty concealed, this will need to change
        #       if we're looking at some sort of platform-independence
        logging.info('Invoking save_to_datastore().')

        #
        # custom storage method dispatching starts here
        #
        if self.storage_method == 'appengine_datastore':
            logging.info("storage method is appengine_datastore")
            self.save_to_datastore()
        else:
            logging.info("storage method is custom")
            # create a dict with the values to store
            token_values = {
                'access_token': self.access_token,
                'expires_in': self.expires_in,
                'refresh_token': self.refresh_token,
                'token_type': self.token_type, # todo: why do I store this?
            }
            key = self.storage_method(self.datastore_key, token_values)
            self.datastore_key = key

    def save_to_datastore(self):
        """Attempts to put the important parts of the Token in the datastore."""

        logging.info('[save_to_datastore] attempting to save token to datastore.')
        # any point in making this modifiable in the datastore?
        # or can I just serialize and store as a single property?
        # doing the latter for now, for simplicity
        # serialize -> store in db -> return key -> [[ put key in cookie ]]
        try:
            from google.appengine.api import memcache
            from google.appengine.ext import db
            logging.info('[save_to_datastore] looks like we\'re on GAE (imports ok\'d), good.')
        except ImportError, import_error:
            logging.info('[save_to_datastore] not GAE.')
            raise import_error

        # todo: possibly relocate this, for now: proving a concept
        class SavedTokenData(db.Model):
            access_token = db.StringProperty()
            expires_in = db.IntegerProperty()
            token_type = db.StringProperty()
            refresh_token = db.StringProperty()
            issued = db.IntegerProperty()

        # check if self.key is present - this means there was a cookie set, that this token has existed
        if self.datastore_key is not None:
            logging.info('token already in datastore, updating...')
            key = db.Key(self.datastore_key)
            # load from datastore
            saved = SavedTokenData.get(key)
        else:
            logging.info('creating new entry in datastore...')
            # new datastore entry
            saved = SavedTokenData()

        saved.access_token = self.access_token
        saved.expires_in = self.expires_in
        saved.token_type = self.token_type
        saved.refresh_token = self.refresh_token
        saved.issued = int(time.time()) # true issue time should be within a few seconds before
        saved.put()
        # even though the datastore key is part of the object,
        # it need not be stored in the datastore -- this is merely
        # a way of knowing, at runtime, whether or not an entity for
        # this token has been created (see in restore_state())
        self.datastore_key = str(saved.key())
        logging.info('done saving. key: '+self.datastore_key)

        # now that we have it in the datastore and we have the key
        # let's put it into memcache
        logging.info("one more thing -- memcache")
        data = {
            "access_token": saved.access_token,
            "expires_in": saved.expires_in,
            "token_type": saved.token_type,
            "refresh_token": saved.refresh_token,
            "issued": saved.issued,
        }
        logging.info("will put: %s" % (str(data)))
        memcache.set(str(saved.key()), data)
        logging.info("memcache part done")

    def restore_state(self, key):
        """Restore Token() state.

        This is a dispatcher -- it calls the proper restore function."""

        logging.info("restore_state dispatcher has been awoken.")
        if self.retrieval_method == 'appengine_datastore':
            self.restore_from_appengine(key)
        else:
            # YET UNTESTED and visionary in nature
            # dig up data from datastore
            data = self.retrieval_method(key)
            logging.info("restore_state, custom function returned: ")
            # set the values
            self.access_token = data.get('access_token', None)
            self.expires_in = data.get('access_token', None)
            self.refresh_token = data.get('refresh_token', None)
            self.token_type = data.get('token_type', None)
            self.datastore_key = key
            logging.info("done")

        logging.info("restore_state finished")

    def restore_from_appengine(self, key):
        """Restores object state from the AppEngine datastore contraption.

        This is never called directly."""

        logging.info("restore_from_appengine called")

        from google.appengine.api import memcache

        # but first let's see if we have that memcached
        cached_token_data = memcache.get(key)
        logging.info("reaching into memcache. got this: %s" % str(cached_token_data))
        if cached_token_data is not None:
            for name, value in cached_token_data.items():
                # security: arbitrary values in memcache will overwrite
                # those needed for normal functioning
                if name in ['access_token', 'expires_in', 'refresh_token', 'token_type']:
                    logging.info("setting attr %s to %s" % (name, value))
                    setattr(self, name, value)

            # this exit point may be a bit concealed
            logging.info("got it from memcache, no need to trouble the datastore. finishing.")
            self.datastore_key = key
            return
                
        # okay, we didn't have it in the memcache
        # restoring from datastore

        # todo: clean up these imports
        try:
            from google.appengine.ext import db
            logging.info('[save_to_datastore] looks like we\'re on GAE, good.')
        except ImportError, import_error:
            logging.info('[save_to_datastore] not GAE.')
            raise import_error
        
        class SavedTokenData(db.Model):
            access_token = db.StringProperty()
            expires_in = db.IntegerProperty()
            token_type = db.StringProperty()
            refresh_token = db.StringProperty()

        std = SavedTokenData.get(db.Key(key))
        logging.info('retrieved token: '+str(std.access_token)+','+str(std.refresh_token))
        self.access_token = std.access_token
        self.expires_in = std.expires_in
        self.token_type = std.token_type
        self.refresh_token = std.refresh_token
        self.datastore_key = key # used by save_to_datastore(), itself NOT saved in datastore

    def create_authz_url(self):
        """A convenience function: returns the authorization URI."""
        # first, check that we have all the necessary data
        if None in (self.authz_endpoint, self.client_id, self.redirect_url, self.scope, self.response_type):
            raise MissingData("One of the required values is missing (i.e. still 'None'). Fix that and try again.")

        # scope meddling - list of scopes should be a .. list, concatenate that in a nice way
        # check if scope is actually a list
        if not isinstance(self.scope, list):
            raise ValueError("Requested scope should be a list (even if it has one value) - it ain't. Fix that.")

        # 37% nicer than the previous idiotic way!
        scope_formatted = ','.join(self.scope)
        
        # carefully craft the authz url
        # order: client_id, redirect_uri, scope, response type
        # () should be microscopically faster than a +=. Hey, details, I know.
        #### ACTUALLY, tests have shown that ... no, this slower. microscopically.
        url = (self.authz_endpoint + '?client_id=' + str(self.client_id) + '&redirect_uri=' +
               self.redirect_url + '&scope=' + scope_formatted + '&response_type=' + self.response_type)

        return url

    def change_code_for_token(self, code):
        """The trader. Trades an authorization code for a marvelous access token."""
        h = httplib.HTTPSConnection(urlparse.urlparse(self.token_endpoint).netloc)
        params = {
            'client_id': self.client_id,
            'client_secret': self.client_secret,
            'code': code,
            'grant_type': 'authorization_code', # todo: perhaps this __shouldn't__ be static
            'redirect_uri': self.redirect_url,
        }
        params_encoded = urllib.urlencode(params)
        headers = {"Content-Type": "application/x-www-form-urlencoded"}
        logging.info('netloc was: '+urlparse.urlparse(self.token_endpoint).netloc)
        logging.info('path is:'+urlparse.urlparse(self.token_endpoint).path)
        h.request('POST', urlparse.urlparse(self.token_endpoint).path, params_encoded, headers)
        r = h.getresponse()
        response_body = r.read()

        logging.info('Tried to change authorization code for token and got a '+str(r.status)+' response.')
        logging.info('Feeding this into our sturdy parser:\n'+response_body)

        self.parse_token_endpoint_response(response_body)

        # in theory, the function could accept an argument with a default value set
        # which would control if the function ends in a call (the automagic path)
        # or if it returns the relevant data (in this case `response_body`)
        # for the developer to fool around with it. Any point in doing so?

    # todo: STUB. Still a good idea, but won't include any showstopper functionality.
    def validate_data(self):
        """Checks if all the limbs are part of the body."""
        if urlparse.urlparse(self.token_endpoint).scheme != 'https':
            raise ValueError('Token endpoint scheme must be https.')
