"""
Provides the PVS interactive tool, creating a PVS.pvs instance
"""

from etb.wrapper import Tool, BatchTool
from etb_subprocess import ETBSubproc
from etb_structs import Literal, Claim, ClaimEntry, Rule, Proof,\
    Tool, InteractiveTool, parse_claim, parse_rule

class PVSError(Exception):
    """General PVS Exception"""
    def __init__(self, errmsg):
        self.err = 'Error: '
        self.errmsg = errmsg.replace("\\n","\n")
    def __str__(self):
        return repr(self.errmsg)

class ParseError(PVSError):
    """PVS Parse Exception"""
    def __init__(self, errmsg):
        PVSError.__init__(self, errmsg)
        self.err = 'Parse Error: '

class TypeError(PVSError):
    """PVS Type Exception"""
    def __init__(self, errmsg):
        PVSError.__init__(self, errmsg)
        self.err = 'Type Error: '

class PVS (InteractiveTool, ETBSubproc):
    """
    PVS Class for running PVS interactively

    Creating an instance of PVS causes PVS to be run, and the
    instance includes methods for interacting with PVS.
    """
    def __init__ (self):
        self.prompt = 'pvs\(\d+\): '
        ETBSubproc.__init__(self, 'pvs -raw')
        ETBSubproc.run_command(self, '(load "etb-pvs")')
        self.context = None
        self.directory = os.getcwd()

    def interrupt(self):
        ETBSubproc.interrupt(self)
        self.run_command(':reset')

    def run_command(self, cmd):
        out = ETBSubproc.run_command(self, cmd)
        err = re.search('<pvserror msg="(.+)">\n"(.*)"\n</pvserror>', out)
        if err:
            if err.group(1) == 'type error':
                raise TypeError(err.group(2))
            elif err.group(1) == 'parse error':
                raise ParseError(err.group(2))
            else: raise PVSError(err.group(2))
        else:
            errm = re.search('^Error: (.+)', out)
            if errm:
                recm = re.search('^ (\d+): Return to Top Level', out)
                if recm:
                    etb_subprocess.etb_subproc.run_command(self, ':reset')
                    if errm.group(1) == 'Typecheck error':
                        raise TypeError(out)
                    elif errm.group(1) == 'Parse error':
                        raise ParseError(out)
                    else: raise PVSError(out)
            else:
                return out

    def change_context(self, directory):
        if isinstance(directory, str):
            if self.context != directory:
                out = self.run_command('(change-context "{}")'\
                                           .format(directory))
                m = re.search('Context changed to', out)
                if m:
                    print 'Context changed to {}'.format(directory)
                    self.context = directory
        else:
            raise PVSError('Directory must be a string')

    def typecheck_theory(self, rule_label, bindings):
        """
        typecheck_file expects bindings to contain Theory and Context
        Theory must be set, Context is optional, and defaults to the
        current context.
        """
        print 'bindings = {}'.format(bindings)
        if 'Theory' not in bindings:
            print 'Must instantiate the Theory'
            return None
        try:
            theory = format(bindings['Theory'])
            if 'Context' in bindings:
                self.change_context(bindings['Context'])
            out = self.run_command('(etb-typecheck-theory "{}")'.format(theory))
            tciter = re.finditer('^(.*) typechecked, (\d)/(\d) proved TCCs, depends on (\S+ )*', out, re.M)
            thclaims = {}
            for tc in tciter:
                subclaims = []
                th = tc.group(1)
                if tc.group(3) != 0:
                    prtccs = tc.group(2) == tc.group(3)
                    subclaims.append(
                        toolbus.add_claim(Claim(['pvs_theory_tccs_proved',
                                                 th, self.context,
                                                 tc.group(2),
                                                 tc.group(3)]),
                                          'closed' if prtccs else 'open',
                                          Proof(rule_label, []),
                                          [])
                        )
                if tc.group(4) is not None:
                    for ith in tc.group(4).split():
                        assert(ith in thclaims)
                        subclaims.append(thclaims[ith])
                thclaims[tc.group(1)] = toolbus.add_claim(Claim(['pvs_theory_typechecked',
                                                                 th, self.context]),
                                                          'closed' if prtccs else 'open',
                                                          Proof(rule_label, []),
                                                          subclaims)
        except PVSError as e:
            print '{}\n{}'.format(e.err, e.errmsg)

    def prove_formula(self, fmla, theory, strat):
        try:
            out = self.run_command('(prove-formula-decl "{}" "{}" "{}")'
                                   .format(fmla, theory, strat))
        except PVSError as e:
            print '{}\n{}'.format(e.err, e.errmsg)
        
    def typecheck_theory_claim(self, theory, numtccs):
        print 'typecheck_pvs_theory_claim {} {}'.format(theory, numtccs)
        claim = Claim('pvs_typechecked_theory', theory)
        proof = Proof(Literal('pvs_typecheck_theory', theory), [])
        if numtccs == 0:
            return toolbus.add_claim(claim, True, proof, [])
        else:
            cindx = toolbus.add_claim(Claim('pvs_theory_tccs_proved', theory), None, None, [])
            return toolbus.add_claim(claim, 'Pending', proof, [cindx])

    def add_rules(self):
        claim = Claim(['pvs_theory_typechecked', 'Theory', 'Context'])
        assert(isinstance(claim, Claim))
        toolbus.add_rule('pvs_typecheck_theory', # label
                         ['Theory', 'Context'],  # args
                         claim, # claim
                         self.typecheck_theory)    # preconds (or script)

# Create a PVS instance        
pvs = PVS()
# Register the primitive rules
pvs.add_rules()
