"""
Implementation of terms with embedded JSON, and hashconsing.  Also,
implementation of substitutions.

The best way to create terms is using mk_term(), mk_fresh_var(),
mk_const(), mk_var(), mk_array(), mk_map(), or mk_apply().
"""

import sys
import re
import weakref
import collections
import itertools
import json

# ----------------------------------------------------------------------
# terms
# ----------------------------------------------------------------------

def mk_term(obj):
    """Converts an object into a term, if compatible. Else,
    raise a ValueError.

    >>> mk_term("foo")
    foo
    >>> mk_term("foo") is mk_term("foo")
    True
    >>> mk_term(True)
    true
    >>> term = mk_term([42, "foo", mk_var(1), {"bar": True}])
    >>> term
    [42, foo, X1, {bar: true}]
    >>> class Test(object): pass
    >>> t = Test()
    >>> try:
    ...     mk_term(t)
    ... except ValueError as e:
    ...     print 'error'
    ...
    error
    >>> term.free_vars() == frozenset([mk_var(1)])
    True
    """
    if isinstance(obj, Term):
        return obj
    elif isinstance(obj, basestring) or isinstance(obj, float) \
        or isinstance(obj, int) or isinstance(obj, bool) or obj is None:
        return mk_const(obj)
    elif isinstance(obj, list) or isinstance(obj, tuple):
        obj = tuple(mk_term(x) for x in obj)
        return mk_array(obj)
    elif isinstance(obj, dict):
        obj = dict( (mk_term(k), mk_term(v)) for k, v in obj.iteritems() )
        return mk_map(obj)
    else:
        raise ValueError('cannot convert {0} of type {1} to term'
                         .format(obj, type(obj)))

def mk_const(val):
    """Make a constant term out of the given value"""
    t = Term(kind=Term.CONST)
    t.val = val
    return t.hashcons()

def mk_var(num):
    """Make a variable with given De Buijn index."""
    t = Term(kind=Term.VAR)
    t.val = num
    return t.hashcons()

def mk_array(elems):
    """Make an array from the given elements. Elements
    are converted to terms if they are not already terms.
    """
    elems = tuple(mk_term(t) for t in elems)
    t = Term(kind=Term.ARRAY)
    t.args = elems
    return t.hashcons()

def mk_map(entries):
    """Make a map from the given pairs. Pairs are converted to
    pairs of terms if needed.
    """
    entries = [(mk_term(k), mk_term(v)) for k, v in entries.iteritems()]
    t = Term(kind=Term.MAP)
    t.args = collections.OrderedDict(entries)
    return t.hashcons()

def mk_apply(pred, args):
    """makes a term out of pred, and args, and applies the
    former to the latter.

    >>> mk_apply("foo", ["bar", 42])
    foo(bar, 42)
    """
    t = Term(kind=Term.APPLY)
    t.val = mk_term(pred)
    t.args = tuple(mk_term(t) for t in args)
    return t.hashcons()

_count = 0

def mk_fresh_var():
    """Create a new variable.

    >>> mk_fresh_var() != mk_fresh_var()
    True
    """
    global _count
    v = mk_var(_count)
    _count += 1
    return v

def _occur_check(var, t):
    """Checks whether var occurs in t."""
    return var in t.free_vars()

class Term(object):
    """A datalog+JSON term, with hashconsing"""
    
    # hashconsing of terms
    __terms = weakref.WeakValueDictionary()

    # chars to escape
    escapestr = re.compile("""[\\\n\t:[]{} ]""")

    # kinds of terms
    VAR = 0
    CONST = 1
    APPLY = 2
    ARRAY = 3
    MAP = 4

    __slots__ = ['kind', 'val', 'args', '_hash', '_fvars',
                 '_volatile', '_normal_form', '__weakref__']

    def __eq__(self, other):
        """Equality of terms."""
        return self is other or self._equal(other)

    def _equal(self, other):
        """Structural equality check"""
        if not isinstance(other, Term) or self.kind != other.kind or hash(self) != hash(other):
            return False
        elif self.kind == Term.CONST or self.kind == Term.VAR:
            # be careful that True==1, and False==0, so we check for the type
            if type(self.val) == bool or type(other.val) == bool:
                return self.val is other.val
            return self.val == other.val
        elif self.kind == Term.APPLY:
            return self.val == other.val and self.args == other.args
        else:
            return self.args == other.args

    def __lt__(self, other):
        """Arbitrary order (with hash...)"""
        # TODO a better (total?) ordering
        return self != other and hash(self) < hash(other)

    def __call__(self, *args):
        """Apply the term to given terms.

        >>> p = mk_term('p')
        >>> p(mk_term('a'), mk_term('b'))
        p(a, b)
        """
        return mk_apply(self, args)

    def __init__(self, kind):
        """Initialize the term with the given kind."""
        self.kind = kind
        self.val = None
        self.args = None
        self._hash = None
        self._fvars = None
        self._normal_form = None
        self._volatile = False

    def __hash__(self):
        """Hash the term."""
        #if self._hash is None:
        if self.is_var() or self.is_const():
            self._hash = hash(self.val)
        elif self.is_apply():
            self._hash = hash( (self.val, self.args) )
        elif self.is_map():  # XXX slow...
            self._hash = hash(tuple(self.args.iteritems()))
        else:
            self._hash = hash(self.args)
        return self._hash

    def __repr__(self):
        """Representation of the term."""
        if self.is_var():
            return "X%d" % self.val if isinstance(self.val, int) else unicode(self.val)
        elif self.is_const():
            if isinstance(self.val, basestring) and \
               Term.escapestr.search(self.val):
               # must escape some chars
               return '"%s"' % str(self.val)
            elif isinstance(self.val, bool):
                return "true" if self.val else "false"
            elif self.val is None:
                return "null"
            else:
                return str(self.val)
        elif self.is_apply():
            return "%s(%s)" % (self.val,
                ', '.join(repr(a) for a in self.args))
        elif self.is_array():
            return "[%s]" % ', '.join(repr(a) for a in self.args)
        elif self.is_map():
            return "{%s}" % ', '.join(
                '%s: %s' % (k, v) for k, v in self.args.iteritems())
        assert False, 'unhandled case; __repr__ unhappy'

    def to_dot(self):
        if self.is_var():
            return "X%d" % self.val if isinstance(self.val, int) else unicode(self.val)
        elif self.is_const():
            if isinstance(self.val, basestring) and \
               Term.escapestr.search(self.val):
               return '"%s"' % str(self.val)
            elif isinstance(self.val, bool):
                return "true" if self.val else "false"
            elif self.val is None:
                return "null"
            else:
                return str(self.val)
        elif self.is_apply():
            return "%s(%s)" % (self.val,
                ', '.join(a.to_dot() for a in self.args))
        elif self.is_array():
            return "[%s]" % ', '.join(a.to_dot() for a in self.args)
        elif self.is_map():
            if mk_term('file') in self.get_args():
                return str(self.get_args()[mk_term('file')])
            return "[%s]" % ', '.join(
                '%s: %s' % (k, v) for k, v in self.args.iteritems())
        assert False, 'unhandled case; to_dot unhappy'
        
    def __nonzero__(self):
        if self.is_const():
            return bool(self.val)
        elif self.is_array():
            return bool(self.args)
        return False

    def hashcons(self):
        """Returns the term that is representative for the equivalence
        class of terms that are equal to self.

        >>> t = mk_term('foo')
        >>> t.hashcons() == t
        True
        >>> t.hashcons() is mk_term('foo').hashcons()
        True
        """
        return Term.__terms.setdefault(self, self)

    def is_var(self):
        """Check whether the term is a variable."""
        return self.kind == Term.VAR

    def is_const(self):
        """Check whether the term is a constant."""
        return self.kind == Term.CONST

    def is_apply(self):
        """Check whether the term is the application of a term
        to other terms."""
        return self.kind == Term.APPLY

    def is_map(self):
        """Check whether the term is a map."""
        return self.kind == Term.MAP

    def is_array(self):
        """Check whether the term is an array."""
        return self.kind == Term.ARRAY

    @staticmethod
    def all_terms():
        """Iterate through all current terms."""
        for t in Term.__terms.itervalues():
            yield t

    def apply(self, args):
        """Apply the term to a list of other terms.

        >>> p = mk_const("p")
        >>> p.apply( [mk_term("a"), mk_term("b")] )
        p(a, b)
        """
        return mk_apply(self, args)

    def unify(self, other):
        """Unify this term against the other. In case of success,
        returns a substitution (even empty), in case of failure, returns None.

        >>> mk_var(1).unify(mk_term('p'))
        subst(X1 = p)
        >>> mk_term('p').unify(mk_term('q'))
        >>> mk_term('p')('a', 'b', mk_var(1)).unify(
        ...   mk_term('p')('a', 'b', 'c'))
        subst(X1 = c)
        >>> mk_term('p')({ 'a': 'b', 'c': mk_var(1) }).unify(
        ...       mk_term('p')({ 'a': 'b', 'c': 'd'}))
        subst(X1 = d)
        """
        assert isinstance(other, Term), 'Unify only works for terms'
        assert not self.free_vars().intersection(other.free_vars()), 'Unify unhappy with the free vars'
        # create a substitution
        bindings = Subst()
        # create a stack of pairs of terms to unify 
        stack = [ (self, other) ]
        while stack:
            left, right = stack.pop()

            # apply the substitution to terms
            left = bindings(left)
            right = bindings(right)

            if left == right:
                continue
            elif left.is_var():
                if _occur_check(left, right):
                    return None
                bindings.bind(left, right)
            elif right.is_var():
                if _occur_check(right, left):
                    return None
                bindings.bind(right, left)
            elif left.is_apply() and right.is_apply():
                if len(left.args) != len(right.args):
                    return None  # failure
                # otherwise, just unify preds and arguments pairwise
                stack.append( (left.val, right.val) )
                for l, r in itertools.izip(left.args, right.args):
                    stack.append( (l, r) )
            elif left.is_array() and right.is_array() and \
                    len(left.args) == len(right.args):
                for l, r in itertools.izip(left.args, right.args):
                    stack.append( (l, r) )
            elif left.is_map() and right.is_map():
                # most interesting case: unify keys pairwise
                # only ground keys are authorized.
                if not left.args.viewkeys() == right.args.viewkeys():
                    return None
                for k, v in left.args.iteritems():
                    assert k.is_ground(), 'k is not ground; unify unhappy'
                    stack.append( (v, right.args[k]) )
            else:
                return None  # failure
        return bindings

    def is_volatile(self):
        """Check whether the term is volatile"""
        return self._volatile

    def set_volatile(self):
        """Mark the symbol as volatile."""
        self._volatile = True

    def rename(self, offset=None):
        """Performs an alpha-renaming of this term, obtained by replacing
        all variables in it by fresh variables.

        Returns (renaming, renamed_term)

        >>> t = mk_term("p").apply( [ mk_var(1),
        ...     mk_term( { "foo": mk_var(2) } ) ] )
        >>> t
        p(X1, {foo: X2})
        >>> t.free_vars() == frozenset((mk_var(1), mk_var(2)))
        True
        >>> renaming, t2 = t.rename()
        >>> t == t2
        False
        >>> t.unify(t2).is_renaming()
        True
        """
        free_vars = self.free_vars()
        if offset is None:
            offset = max(v.val for v in free_vars if isinstance(v.val, int)) + 1
        renaming = Subst()
        for i, v in enumerate(free_vars):
            renaming.bind(v, mk_var(i + offset))
        assert renaming.is_renaming(), 'renaming not a renaming; rename unhappy'
        return (renaming, renaming(self))

    def negative_rename(self):
        """Performs an alpha-renaming of the term, using
        only negative variables.

        >>> t = mk_term("p")( mk_var(1), { "foo": mk_var(2) } )
        >>> t
        p(X1, {foo: X2})
        >>> t.negative_rename()[1]
        p(X-3, {foo: X-2})
        """
        free_vars = self.free_vars()
        offset = max(v.val for v in free_vars if isinstance(v.val, int)) + 1
        return self.rename(offset=-offset)

    def is_ground(self):
        """Checks whether the term is ground.

        >>> t = mk_term('p').apply( [mk_var(1), mk_term('q')] )
        >>> t.is_ground()
        False
        >>> mk_term(["p", "q", { "foo": 42 }]).is_ground()
        True
        """
        return not self.free_vars()

    def free_vars(self):
        """Returns the set of free variables of this term.
        """
        if self._fvars is None:
            vars = set()
            self._compute_free_vars(vars)
            self._fvars = frozenset(vars)
        return self._fvars

    def _compute_free_vars(self, vars):
        """Adds the free vars of the term to vars"""
        if self.is_var():
            vars.add(self)
            return
        elif self.is_const():
            return
        if self.is_apply():
            self.val._compute_free_vars(vars)
        if self.is_apply() or self.is_array():
            for t in self.args:
                t._compute_free_vars(vars)
        elif self.is_map():
            for k, v in self.args.iteritems():
                k._compute_free_vars(vars)
                v._compute_free_vars(vars)

    def normalize(self):
        """Returns a normalized version of the term. Variables in it
        are replaced by X0...Xn-1) where n is the number of free
        variables in the term.

        Returns (renaming, term) where renaming is used to normalize.

        >>> t = mk_term("p").apply([mk_var(3),
        ...                         mk_term({ "foo": mk_var(2) })] )
        >>> t
        p(X3, {foo: X2})
        >>> t.normalize()[1]
        p(X0, {foo: X1})
        >>> t = mk_term("p").apply([mk_var(2),
        ...                         mk_term( { "foo": mk_var(1)} )] )
        >>> t
        p(X2, {foo: X1})
        >>> t.normalize()
        (subst(X2 = X0), p(X0, {foo: X1}))
        """
        if self.is_ground():
            return (Subst(), self)
        fvars = self.ordered_free_vars()
        renaming = Subst(dict( (v, mk_var(i)) for \
            i, v in enumerate(fvars) ))
        return (renaming, renaming(self))

    def is_normalized(self):
        """Checks whether the term is normalized"""
        if self._normal_form is None:
            self._normal_form = self.normalize()[1]
        return self._normal_form == self

    def ordered_free_vars(self, l=None):
        """Returns the list of variables in the term, by order of prefix
        traversal. Free vars may occur several times in the list
        """
        if l is None:
            l = []
        if self.is_var():
            if self not in l:  # avoid duplicates
                l.append(self)
        elif self.is_apply():
            for t in self.args:
                t.ordered_free_vars(l)
        elif self.is_array():
            for t in self.args:
                t.ordered_free_vars(l)
        elif self.is_map():
            for k, v in self.args.iteritems():
                k.ordered_free_vars(l)
                v.ordered_free_vars(l)
        return l

    def first_symbol(self):
        """finds the first symbol in the object

        >>> mk_term('a').first_symbol()
        a
        >>> mk_term('b')(mk_var(2)).first_symbol()
        b
        >>> mk_array([ mk_term('a'), mk_term('b')]).first_symbol()
        a
        """
        if self.is_const() or self.is_var():
            return self
        elif self.is_apply():
            return self.val.first_symbol()
        elif self.is_array():
            return self.args[0].first_symbol()
        elif self.is_map():
            raise AssertionError('map term has no first symbol')
        else:
            raise ValueError('unhandled case for first_symbol: ' + \
                             repr(self))

    def get_val(self):
        """The value of the term (for variables/const)"""
        assert self.kind == Term.CONST or self.kind == Term.VAR, 'value of non var/const; get_val unhappy'
        return self.val

    def get_pred(self):
        """Predicate (for apply)"""
        assert self.kind == Term.APPLY, 'term not an APPLY; get_pred unhappy'
        return self.val

    def get_args(self):
        """List of arguments (for apply/map/array)"""
        #iam says: give the puzzled user a clue
        assert self.kind in [Term.APPLY, Term.MAP, Term.ARRAY], 'Term.get_args: %s of kind %s not APPLY, MAP, or ARRAY' % (repr(self), self.kind)
        return self.args

# ----------------------------------------------------------------------
# clauses
# ----------------------------------------------------------------------

class Clause(object):
    """A Horn clause, with a head and a (possibly empty) body.

    >>> Clause(mk_term('p'), [])
    p.
    >>> Clause(mk_term('p'), [mk_var(1), mk_term({'foo': 42})])
    p :- X1, {foo: 42}.
    """
    def __init__(self, head, body, temp=False):
        self.head = head
        self.body = tuple(body)
        # compute the free variables (used to be local to the non temp case)
        self._free_vars = frozenset([]).union(* (x.free_vars() for x in body))
        # check that the clause is well-formed: all variables in the head are
        # bound in the body
        if not temp:
            if not (head.free_vars() <= self._free_vars):
                print >>sys.stderr, head
                for b in body:
                    print >>sys.stderr, b
                print >>sys.stderr, head.free_vars()
                print >>sys.stderr, self._free_vars
                assert False, 'datalog restriction fails! Clause __init__ unhappy'
        self._is_ground = head.is_ground() and all(x.is_ground() for x in body)
        self._done = True  # freeze

    def __setattr__(self, attr, val):
        if getattr(self, '_done', False):
            raise ValueError('immutable term')
        super(Clause, self).__setattr__(attr, val)

    def __hash__(self):
        h = hash(self.head)
        for x in self.body:
            h = hash( (h, x) )
        return h

    def __eq__(self, other):
        return isinstance(other, Clause) and self.head == other.head and \
            self.body == other.body

    def __repr__(self):
        if self.body:
            return '{0} :- {1}.'.format(self.head,
                ', '.join(repr(x) for x in self.body))
        else:
            return '{0}.'.format(self.head)

    def is_ground(self):
        """Checks whether the clause contains no variables"""
        return self._is_ground

    def free_vars(self):
        """Free variables of the clause (equivalent to the free variables
        of the body of the clause).
        """
        return self._free_vars

    def rename(self, offset=None):
        """Perform a renaming of the variables in the clause, using
        variables that do not occur in the clause.

        If an offset is provided, it will be used instead of finding one
        that ensures the absence of variables collisions.

        Returns (renaming, renamed_clause)

        >>> c = Clause(mk_var(1), [mk_var(1), mk_var('X')])
        >>> _, c2 = c.rename()
        >>> c2.free_vars().intersection(c.free_vars())
        frozenset([])
        >>> c2.head.free_vars() <= c2.free_vars()
        True
        >>> c.rename(offset=4)
        (subst(X1 = X4, X = X5), X4 :- X4, X5.)
        """
        fvars = self.free_vars()
        if not fvars:
            return [], self
        elif offset is None:
            # compute an offset: by adding this number to the variables,
            # we are sure never to collide with fvars
            offset = max(v.get_val() for v in fvars if isinstance(v.get_val(), int)) + 1
        renaming = Subst()
        for i, v in enumerate(fvars):
            renaming.bind(v, mk_var(i + offset))
        assert renaming.is_renaming(), 'renaming no a renaming; rename unhappy'
        return (renaming, renaming(self))

class DerivationRule(Clause):
    def __init__(self, head, body, temp=False):
        Clause.__init__(self, head, body, temp)
    
class InferenceRule(Clause):
    def __init__(self, head, body, temp=False):
        Clause.__init__(self, head, body, temp)
        
# ----------------------------------------------------------------------
# claims
# ----------------------------------------------------------------------

class Claim(object):
    """
    A claim is a pair of a ground term and an explanation.

    The explanation can be a derivation rule, an inference rule, or an
    application of an interpreted predicate.
    """
    def __init__(self, literal, reason):
        """Create the claim. reason can be a string or a Clause."""
        assert literal.is_ground(), 'Non-ground claim'
        self.literal = literal
        if isinstance(reason, Clause):
            self.subst = reason.head.unify(literal)
        self.reason = reason
        self._hash = hash((self.literal, self.reason))

    def __hash__(self):
        return self._hash

    def __eq__(self, other):
        return isinstance(other, self.__class__) and other.literal == self.literal and \
            self.reason == other.reason

    def __repr__(self):
        return 'claim({0}, reason={1})'.format(self.literal, self.reason)

class DerivedClaim(Claim):
    '''
    Claim establised by derivation.
    '''
    def __init__(self, literal, reason):
        assert isinstance(reason, DerivationRule), 'reason not a DerivationRule, DerivedClaim __init__ unhappy.'
        Claim.__init__(self, literal, reason)        
        
    def __repr__(self):
        return 'derivedClaim({0}, reason={1})'.format(self.literal, self.reason)
        
class ProvedClaim(Claim):
    '''
    Claim establised by inference.
    '''
    def __init__(self, literal, reason):
        assert isinstance(reason, InferenceRule), 'reason not a InferenceRule, ProvedClaim __init__ unhappy.'
        Claim.__init__(self, literal, reason)        

    def __repr__(self):
        return 'provedClaim({0}, reason={1})'.format(self.literal, self.reason)
        
class InterpretedClaim(Claim):
    '''
    Claim established by an interpretation.
    '''
    def __init__(self, literal, reason):
        Claim.__init__(self, literal, reason)

    def __repr__(self):
        return 'interpretedClaim({0}, reason={1})'.format(self.literal, self.reason)
        
# ----------------------------------------------------------------------
# substitutions and renamings
# ----------------------------------------------------------------------

def mk_subst(**bindings):
    """Named arguments constructor for substitutions. It builds
    terms from its named arguments.

    >>> mk_subst(X=42, Y=[1,2,3])
    subst(X = 42, Y = [1, 2, 3])
    """
    term_bindings = dict( (mk_var(k), mk_term(v)) for k, v \
                          in bindings.iteritems())
    return Subst(term_bindings)

class Subst(object):
    """A substitution.

    >>> s = Subst( { mk_var(1): mk_term('p'),
    ...         mk_var(2): mk_term('u') } )
    >>> s
    subst(X1 = p, X2 = u)
    >>> sorted(s.domain())
    [X1, X2]
    >>> sorted(s.range())
    [p, u]
    >>> s.is_empty()
    False
    >>> len(s)
    2
    >>> s.restrict( [mk_var(1)] )
    subst(X1 = p)
    """

    __slots__ = ['_bindings', '_hash', '_introduced',
                 '_timestamp', '_introduced_timestamp', '__weakref__']

    def __init__(self, bindings=None):
        """Initialize the substitution"""
        self._bindings = []
        self._hash = None
        self._timestamp = 0  # modification timestamp
        self._introduced_timestamp = 0  # last update of introduced
        self._introduced = set()  # set of introduced variables
        if bindings is not None:
            if isinstance(bindings, dict):
                for k, v in bindings.iteritems():
                    self.bind(k, v)
            elif isinstance(bindings, list) or isinstance(bindings, tuple):
                for k, v in bindings:
                    self.bind(k, v)
            else:
                assert False, 'unknown kind of substitution'

    def __eq__(self, other):
        """Equality between substitutions."""
        return isinstance(other, Subst) and self._bindings == other._bindings

    def __hash__(self):
        return hash(frozenset(self._bindings))
    
    def __repr__(self):
        """Representation of the subst"""
        return "subst(%s)" % ', '.join(
            '%s = %s' % (k, v) for k, v in self._bindings)

    def __len__(self):
        """Number of bindings in the subst."""
        return len(self._bindings)

    def __call__(self, t):
        """Apply the substitution to a term.

        >>> s = Subst( {mk_var(1): mk_term('foo')} )
        >>> s
        subst(X1 = foo)
        >>> s(mk_var(1))
        foo
        >>> s(mk_var(2))
        X2
        >>> t = mk_term( ["foo", 42, mk_term('foo')( {"bar": mk_var(1)} ) ] )
        >>> t.is_ground()
        False
        >>> t
        [foo, 42, foo({bar: X1})]
        >>> s(t)
        [foo, 42, foo({bar: foo})]
        >>> s(t) != t
        True
        """
        if isinstance(t, Term):
            if t.is_ground():
                return t
            elif t.is_var():
                for i in xrange(len(self._bindings)):
                    if self._bindings[i][0] == t:
                        return self._bindings[i][1]
                return t
            elif t.is_const():
                return t
            elif t.is_apply():
                return mk_apply(self(t.val), map(self, t.args))
            elif t.is_array():
                return mk_array(map(self, t.args))
            elif t.is_map():
                return mk_map(dict((self(k), self(v)) for k, v in \
                              t.args.iteritems()))
            else:
                assert False, 'unknown kind of term in __call__'
        elif isinstance(t, Clause):
            return t.__class__(self(t.head), map(self, t.body))
        else:
            print t.__class__
            print t
            assert False, 'bad arg %s of class %s; __call__ unhappy' % (t, t.__class__)
        
    def __nonzero__(self):
        """A substitution, even empty, is to be considered as a true value"""
        return True

    def __contains__(self, var):
        """Checks whether var is bound by the substitution

        >>> s = Subst({ mk_var(1): mk_const(42)})
        >>> mk_var(1) in s
        True
        >>> mk_var(2) in s
        False
        """
        assert var.is_var(), 'var ain\'t a var; __contains__ unhappy'
        for k, _ in self._bindings:
            if k == var:
                return True
        return False

    def get(self, str):
        t = self(mk_var(str))
        return dumps(t)
    
    def get_bindings(self):
        return self._bindings
    
    def clone(self):
        """Return a copy of the substitution."""
        s = Subst()
        for k, v in self._bindings:
            s.bind(k, v)
        return s

    def bind(self, var, t):
        """Bind var to t in the substitution. Var must not
        be already bound.
        """
        assert var.is_var(), 'var ain\'t a var; bind unhappy'
        assert isinstance(t, Term)
        if var == t:
            return  # no-op
        assert self(var) == var, 'var not bound; bind unhappy'
        self._bindings.append( (var, t) )
        self._bindings.sort()
        self._timestamp += 1  # update timestamp

    def is_empty(self):
        """Checks whether the substitution is empty."""
        return not self._bindings

    def range(self):
        """Values of the substitution"""
        for _, v in self._bindings:
            yield v

    def domain(self):
        """Variables bound by the substitution"""
        for k, _ in self._bindings:
            yield k

    def introduced(self):
        """Variables introduced by the substitution (iterator)"""
        if self._timestamp > self._introduced_timestamp:
            # must update the set of introduced variables
            self._introduced.clear()
            for t in self.range():
                self._introduced.update(t.free_vars())
            self._introduced_timestamp = self._timestamp
        for var in self._introduced:
            yield var  # yield content of the set

    def compose(self, other):
        """Composes the two substitutions,  self o other.
        The resulting substitution
        is { x -> other(self(x)) } for x in
        domain(self) union domain(other)

        be careful that this is backward w.r.t. function composition,
        since t \sigma \theta = t (\sigma o \theta)

        >>> s = Subst({mk_var(1): mk_var(3)})
        >>> t = Subst({mk_var(2): mk_term('p')(mk_var(1)),
        ...     mk_var(3): mk_term('b')})
        >>> s.compose(t) == Subst({mk_var(1): mk_term('b'),
        ...     mk_var(2): mk_term('p')(mk_var(1)),
        ...     mk_var(3): mk_term('b')})
        True
        >>> s.compose(s) == s
        True
        """
        assert isinstance(other, Subst)
        s = Subst()
        for var in self.domain():
            s.bind(var, other(self(var)))
        for var in other.domain():
            if var not in self:
                s.bind(var, other(var))
        return s

    def join(self, other):
        """Take the join of the two substitutions, self . other,
        the resulting substitution is::

        { x -> other(self(x)) for x in domain(self) } union
        { x -> other(x) } for x in domain(other) vars(range(self)).

        >>> s = Subst({mk_var(1): mk_var(3)})
        >>> t = Subst({mk_var(2): mk_term('p')(mk_var(1)),
        ...            mk_var(3): mk_term('b')})
        >>> s.join(t) == Subst({mk_var(1): mk_term('b'),
        ...     mk_var(2): mk_term('p')(mk_var(1)) })
        True
        """
        assert isinstance(other, Subst)
        s = Subst()
        for var, t in self._bindings:
            s.bind(var, other(t))
        for var in other.domain():
            if var not in self.introduced():
                s.bind(var, other(var))
        return s

    def is_renaming(self):
        """Checks whether the substitution is a renaming.

        >>> Subst( { mk_var(1): mk_term("a") } ).is_renaming()
        False
        >>> Subst( { mk_var(1): mk_var(2) } ).is_renaming()
        True
        """
        return all(x.is_var() for x in self.range()) and \
            len(list(self.domain())) == len(list(self.range()))

    def restrict(self, domain):
        """Returns a new substitution, which is the same but
        restricted to the given domain.

        >>> s = Subst({mk_var(2): mk_term('p')(mk_var(1)),
        ...            mk_var(3): mk_term('b')})
        >>> s.restrict([mk_var(2)])
        subst(X2 = p(X1))
        """
        s = Subst()
        for var, t in self._bindings:
            if var in domain:
                s.bind(var, t)
        return s

# ----------------------------------------------------------------------
# json parsing and printing
# ----------------------------------------------------------------------

class TermJSONEncoder(json.JSONEncoder):
    """Custom encoder in JSON. It deals with terms, clauses,
    substitutions and claims.
    """

    def default(self, o):
        "try to encode terms"
        if isinstance(o, Term):
            if o.is_var():
                return { '__Var': o.get_val() }
            elif o.is_const():
                return o.get_val()
            elif o.is_apply():
                return {'__Apply': [o.get_pred()] + list(o.get_args())}
            elif o.is_array():
                return list(o.get_args())
            elif o.is_map():
                return dict((repr(k), v) for k, v in o.get_args().iteritems())
        elif isinstance(o, Clause):
            return {'__Clause': [o.head] + list(o.body)}
        elif isinstance(o, Subst):
            return {'__Subst': list(o.get_bindings())}
        elif isinstance(o, Claim):
            return {'__Claim': o.literal,
                    '__Reason': o.reason }
        return json.JSONEncoder.default(self, o)  # defer to default

class TermReadableJSONEncoder(json.JSONEncoder):
    """Custom encoder in JSON. It deals with terms, clauses,
    substitutions and claims, but prints them more readably for clients.
    """

    def default(self, o):
        "try to encode terms"
        if isinstance(o, Term):
            if o.is_var():
                return {'name': o.get_val()}
            elif o.is_const():
                return o.get_val()
            elif o.is_apply():
                return {'pred': o.get_pred(), 'args': list(o.get_args())}
            elif o.is_array():
                return list(o.get_args())
            elif o.is_map():
                return dict((repr(k), v) for k, v in o.get_args().iteritems())
        elif isinstance(o, Clause):
            return {'head': o.head, 'body': list(o.body)}
        elif isinstance(o, Subst):
            return {'__Subst': list(o.get_bindings())}
        elif isinstance(o, Claim):
            return {'__Claim': o.literal,
                    '__Reason': o.reason }
        return json.JSONEncoder.default(self, o)  # defer to default

def term_object_hook(o):
    """Given the JSON object o (a dict), tries to parse terms,
    claims, clauses and substs from it.
    """
    # detect special kinds of maps
    if '__Var' in o:
        return mk_var(o['__Var'])
    elif '__Apply' in o:
        l = o['__Apply']
        assert len(l) >= 1
        return mk_term(l[0])(*(mk_term(x) for x in l[1:]))
    elif '__Clause' in o:
        l = o['__Clause']
        assert len(l) >= 1
        return Clause(mk_term(l[0]), [mk_term(x) for x in l[1:]])
    elif '__Subst' in o:
        l = o['__Subst']
        return Subst(bindings=dict([ [mk_term(k), mk_term(v)] for [ k, v ] in l ]))
    elif '__Claim' in o and '__Reason' in o:
        lit = mk_term(o['__Claim'])
        reason = o['__Reason']
        return Claim(lit, reason=reason)
    elif '__Subst' in o:
        bindings = o['__Subst']
        return Subst( [(mk_term(k), mk_term(v)) for k, v in bindings] )
    # default choice: just return the object
    return o

def dump(obj, filedesc, *args, **kwargs):
    """Print the object in JSON on the given file descriptor."""
    json.dump(obj, filedesc, cls=TermJSONEncoder, *args, **kwargs)

def dumps(obj, *args, **kwargs):
    """Print the object in JSON into a string"""
    return json.dumps(obj, cls=TermJSONEncoder, *args, **kwargs)

def dumps_readably(obj, *args, **kwargs):
    """Print the object in JSON into a string, old form of term"""
    return json.dumps(obj, cls=TermReadableJSONEncoder, *args, **kwargs)

def load(filedesc, *args, **kwargs):
    """Print the object in JSON on the given file descriptor."""
    return json.load(filedesc, object_hook=term_object_hook, *args, **kwargs)

def loads(s, *args, **kwargs):
    return json.loads(s, object_hook=term_object_hook, *args, **kwargs)
