"""
ETB interface for HiLiTE

HiLiTE rules

hilite_model_analysis(Model, External_input_ranges, Model_summary) :-
    hilite_command_file_from_template("VVFCS_ModelAnalysisTemplate.htpl", Model,
                                      External_input_ranges, Command_file),
    hilite_output_directory_from_model(Model, Output_directory),
    hilite_static_analysis(Command_file, Output_directory, Model_summary).

hilite_range_bounds_check(Model, External_input_ranges, Model_summary) :-
    hilite_command_file_from_template("VVFCS_RangeBoundsCheckTemplate.htpl", Model,
                                      External_input_ranges, Command_file),
    hilite_output_directory_from_model(Model, Output_directory),
    hilite_static_analysis(Command_file, Output_directory, Model_summary).
"""

import os
import re
import subprocess
import datetime
import logging
from xml.etree.ElementTree import ElementTree
from etb.structs import isvariable, Literal
from etb.wrapper import Tool

class HiLiTE (Tool):
    def __init__(self, *args, **kwargs):
        self.template_dir = os.environ.get("HILITE_ETB")
        Tool.__init__(self, *args, **kwargs)

    @Tool.predicate("+cmd_template: file, +output_directory: value, "
                    "-model_summary: file")
    def hilite_static_analysis(self, cmd_template, output_directory, model_summary):
        """
        Run HiLiTE command on the cmd_template, which specifies the model,
        External_input_ranges, and Intermediate_ranges.  Generates the
        range_bound_errors and the model_summary in the specified directory.
        """
        # Invoke HiLiTE
        try:
            subprocess.call(["hilite", cmd_template, output_directory])
            summary = self.get_summary_report_in_dir(cmd_template, output_directory)
            #print("Coverage: " + repr(self.get_coverage_from_summary(summary)))
            if isvariable(model_summary):
                return [{model_summary: summary}]
            elif model_summary == summary:
                # If name the same, but file is different, ETB should catch it
                return [{}]
            else:
                return [] #failure
        except Exception as e:
            print("HiLiTE invocation problem: {0}".format(e))
            return []

    @Tool.predicate("+model: file, -directory: value")
    def hilite_output_directory_from_model(self, model, directory):
        """
        Get the model path, which may be empty if it's cwd, and create the new
        output directory.
        """
        if not os.path.exists(model):
            print("No such path " + model)
            return []
        
        model_dirname = os.path.dirname(model)
        model_basename = os.path.splitext(os.path.basename(model))[0]
        output_dirname = os.path.join(model_dirname, model_basename+"Output")
        if os.path.exists(output_dirname):
            output_dirname += datetime.datetime.utcnow().strftime("%Y%m%d_%H_%M_%S")
        try:
            os.mkdir(output_dirname)
        except Exception as e:
            print("Error creating {0}: {1}".format(output_dirname, e))
            return []
        print("Output directory is {0}".format(output_dirname))
        return [{directory: output_dirname}]
        
    @Tool.predicate("+template_file: file, +model: file, +external_input_range: file, "
                    "-command_file: file")
    def hilite_command_file_from_template(self, template_file, model, external_input_range, command_file):
        """
        Generate the command file from the template in the given directory.
        Replace all instances of "[ModelName]" in the template with modelname.
        Replace ExternalInputRangeFileName and RangeBoundsFileName with information from
        like-named bindings.
        """
        logging.getLogger('etb').info(
            "command_file_from_template called: model {1}, external_input_range {2} "
            "templateFile {0}, commandFile {3}".format(model, external_input_range,
                                                       template_file, command_file))
        model_dir = os.path.dirname(model)
        model_base = os.path.splitext(os.path.basename(model))[0]
        command_filename = os.path.splitext(model)[0]+".hilite"
        modelname_regex = re.compile(r"\[ModelName\]")
        input_range_regex = re.compile(r"\[ExternalInputRangeFileName\]")
        with open(template_file, "r") as tf:
            template = tf.readlines()
        with open(command_filename, "w") as cf:
            for line in template:
                sub_line = modelname_regex.sub(os.path.join(model_dir, model_base), line)
                sub_line = input_range_regex.sub(external_input_range, sub_line)
                cf.write(sub_line)
        if isvariable(command_file):
            return [{command_file: command_filename}] # Bind the variable
        elif command_file == command_filename:
            return [{}] # shouldn't happen - empty substitution
        else:
            return [] # failure

    @Tool.predicate("+summary_file: file, -coverage: value")
    def hilite_get_coverage_from_summary(self, summary_file):
        """
        Extract the Simulink coverage metric from the given HiLiTE summary file.
        Return as a float. Return -1 if there are errors parsing the file.
        """
        tree = ElementTree()
        tree.parse(summary_file)
        simulinkCoverage = tree.find("SimulinkTestGeneration")
        if simulinkCoverage is None:
            print("No SimulinkTestGeneration element in " + summary_file)
            return []
        status = simulinkCoverage.attrib["HiLiTEStatus"]
        if status != "Normal":
            print("HiLiTE did not terminate normally. Check for Errors in the log file.")
            return []
        coverage_val = simulinkCoverage.attrib["HiLiTEMetric"].strip('%')
        if isvariable(coverage):
            return [{coverage: coverage}]
        elif coverage == coverage_val:
            return [{}]
        else:
            return [] # Failure

    @Tool.predicate("+summary_file: file, -range_check_result: value")
    def hilite_get_range_check_result_from_summary(self, summary_file):
        """
        Check the given summary file for a TYPE_OR_RANGE defect. If none present all range
        bounds conform. If at least one, the check failed.
        """
        tree = ElementTree()
        tree.parse(summary_file)
        for defect in tree.iterfind("Defects/Defect"):
            if defect.attrib["Type"] == "TYPE_OR_RANGE":
                return [] #One defect fails the test
        # Return False
        #unless we find a defect, all range bounds passed
        if isvariable(range_check_result):
            return [{range_check_result: "True"}]
        elif range_check_result == "True":
            return [{}]
        else:
            return [] # Failure

    def get_summary_report_in_dir(self, cmd_filename, output_dirname):
        """
        Return the summary file or None if the run terminated abnormally
        """
        with open(cmd_filename, "r") as cf:
            cmd = cf.readlines()
        modelname_regex = re.compile(r'<Model name="(.+)"/>')
        modelname = None
        for line in cmd:
            m = modelname_regex.search(line)
            if m is not None:
                modelname = m.group(1)
                break
        model_basename = os.path.splitext(os.path.basename(modelname))[0]
        summaryname = model_basename+"_Summary.xml"
        summary = os.path.join(output_dirname, "Reports", summaryname)
        if os.path.exists(summary):
            return summary
        else:
            print("Summary file {0} not found. Check the HiLiTE log for errors.".format(summary))
            return None

def register(toolbus):
    "Register the tool"
    toolbus.add_tool(HiLiTE())
