# Batch wrapper for SAL

import os.path

from etb.wrapper import Tool, BatchTool

class SalBatch(BatchTool):
    """Batch wrapper for SAL"""

    # Utilities

    def context_from_filename(self, filename) :
        (ctx, _) = os.path.splitext(filename)
        return ctx

    # Error handling

    def parseResult(self, (stdout, stderr)):
        if 'Error' in stderr:
            self.fail('Error detected (%s)' % stderr)
        return stdout

    # Simple predicates

    @Tool.predicate("+salFile:file")
    def sal_file(self, salFile):
        if salFile['file'].endswith('.sal'):
            return { 'substs' : [{}], 'claims' : [] }
        else:
            return { 'substs' : [],
                     'claims' : [ 'error("Not a SAL file: %s")' % salFile['file'] ] }

    # Tools wrappers

    @Tool.predicate("+model: file, +prop: value")
    def sal_smc(self, model, prop) :
        """Calls sal-smc on the given model, for the given property."""
        context = self.context_from_filename(model['file'])
        return self.run('proved.\n', 'sal-smc', context, str(prop))

    @Tool.predicate("+model: file, +module: value")
    def sal_deadlock_check(self, model, module) :
        """Calls sal-deadlock-checker on the given module,
           in the given model. """
        context = self.context_from_filename(model['file'])
        return self.run('ok (module does NOT contain deadlock states).\n',
                        'sal-deadlock-checker', context, str(module))

    @Tool.predicate("+model:file,+assertion:value,+depth:value")
    def sal_bmc(self, model, assertion, depth):
        '''Calls sal-bmc'''
        return self.run('no counterexample between depths: [0, %s].\n' % depth,
                        'sal-bmc',
                        '--to=%s' % depth,
                        model, assertion)

    @Tool.predicate("+model:file,+assertion:value,+depth:value")
    def sal_inf_bmc(self, model, assertion, depth):
        '''Calls sal-inf-bmc'''
        return self.run('no counterexample between depths: [0, %s].\n' % depth,
                        'sal-inf-bmc',
                        '--to=%s' % depth,
                        model, assertion)

    @Tool.predicate("+in: file, -out: file")
    def sal_yices(self, inF, outF):
        '''Call yices from SAL'''
        (r, out, err) = self.callTool('yices', '--evidence', '--dimacs', '-v', '3', inF['file'])
        with open('sal.out', 'w') as oc:
            print >>oc, out
        outref = self.fs.put_file('sal.out')
        return [self.bindResult(outF, outref)]
    
def register(etb):
    "Register the tool"
    etb.add_tool(SalBatch(etb))




    # rules = [ 'bmc(F,P,D) :- sal_file(F), sal_bmc(F,P,D).',
    #           'bmc(F,P,D) :- sal_file(F), sal_inf_bmc(F,P,D).',
    #           'prove(F,M,P) :- sal_file(F), sal_smc(F,P), sal_deadlock_check(F,M).' ]
    # etb.add_rules(rules)
