"""
A parser for datalog+Json terms.

The grammar, largely based on Datalog, but with JSON extension:

Lexical:

 Upper :- A-Z

 Lower :- a-z

 Nums  :- 0-9

 Alnum :- Upper+Lower+Nums

 Print :- !#$&'*+-/;<>@[\\]^_`{|}

 Variable :- [ Upper ][ Alnum+_ ]*

 Identifier :- [ Lower+Nums+Print ][ Alnum+Print ]

 String :- [ " Any, with \ for escape " ] or with single quotes

Grammar:

 Const := Identifier | String

 Array := '[' Term*',' ']'  # JSON

 Object := '{' (String ':' Term)*',' '}' # JSON

 Term := Variable | Const | Array | Object |
                 (Const|Variable) [ '(' InfixTerm+',' ')' ]

 InfixTerm := Term = Term | Term != Term | Term

 Clause := InfixTerm [ ':-' InfixTerm+',' ]

 Assertion := Clause '.'

 Retraction := Clause '~'

 Query := Literal '?'

 Statement := Assertion | Retraction

 Program := Statement* [ Query ]

 Rule := Const '(' Variable*',' ')' : literal [ 'if' Literal+',' ]

"""

import terms
import string
import pyparsing
from pyparsing import Literal as Lit, Word, ZeroOrMore, Optional,\
    nums, alphas, alphanums, Forward, delimitedList, QuotedString, \
    Group, oneOf, restOfLine, ParseException

def parseToConst(s, loc, toks):
    tok = toks[0]
    if isinstance(tok, basestring):
        tok = tok.strip()
        if tok[0] in '{[':
            return term.parseString(tok)
        else :
            return terms.mk_const(tok)
    else:
        return terms.mk_const(tok)

def parseToMap(s, loc, toks):
    d = {}
    for tok in toks.asList():
        d[tok[0]] = tok[1]
    return terms.mk_map(d)

def parseToVariable(s, loc, toks):
    name = toks.asList()[0]
    return terms.mk_var(name)

def parseToApply(s, loc, toks):
    pargs = toks.asList()
    return pargs[0].apply(pargs[1:])

def parseToEq(s, loc, toks):
    [l, r] = toks.asList()
    return terms.mk_term('equal')(l, r)

def parseToNeq(s, loc, toks):
    [l, r] = toks.asList()
    return terms.mk_term('different')(l, r)

def parseToClause(s, loc, toks):
    tail = toks.asList()
    head = tail.pop(0)
    return terms.DerivationRule(head, tail)

def parseToInferenceRule(s, loc, toks):
    tail = toks.asList()
    head = tail.pop(0)
    return terms.InferenceRule(head, tail)

def parseToNum(s, loc, toks):
    l = toks.asList()
    if len(l) == 1:
        return int(l[0])
    elif len(l) == 2:
        return float(l[0]+'.'+l[1])

def stripString(s, loc, toks):
    return [t.strip("'\"") for t in toks]

null = Lit("null")

true = Lit("true")
false = Lit("false")

lp = Lit("(").suppress()
rp = Lit(")").suppress()
lk = Lit("[").suppress()
rk = Lit("]").suppress()
lb = Lit("{").suppress()
rb = Lit("}").suppress()

eq = Lit("=").suppress()
neq = Lit("!=").suppress()

ts = Lit(":-").suppress()
infer = Lit("<=").suppress()

pd = Lit(".").suppress()
tl = Lit("~").suppress()
qm = Lit("?").suppress()
qt = Lit('"').suppress()
sq = Lit("'").suppress()
if_ = Lit('if').suppress()
cl = Lit(':').suppress()

variable = Word(string.uppercase, alphanums+"_")
variable.setParseAction(parseToVariable)
identifier = Word(string.lowercase+nums+"!#$&*+-/;<>@\\^_`|",
                  alphanums+"!#$&*+-/;<>@\\^_`|")
name = Word(alphas, alphanums+"!#$&*+-/;<>@\\^_`|")

sstring = QuotedString('\'', multiline=True)
dstring = QuotedString('"', multiline=True)
qstring = sstring | dstring

true.setParseAction(lambda s, loc, toks: True)
false.setParseAction(lambda s, loc, toks: False)
null.setParseAction(lambda s, loc, toks: None)
singleLineComment = Group("%" + restOfLine)

# Terms (with json?)
term = Forward()
infixTerm = Forward()
num = Word(nums, nums) + Optional(pd + Word(nums, nums))
const = num | null | true | false | identifier | qstring
array = lk + Optional(delimitedList(infixTerm | term)) + rk
objpair = const + cl + (infixTerm | term)
obj = lb + Optional(delimitedList(objpair)) + rb
apply = (const | variable) + lp + delimitedList(infixTerm | term) + rp
eq_term = term + eq + term
neq_term = term + neq + term
term << (apply | const | variable | array | obj)
infixTerm << (eq_term | neq_term)

terms_ = delimitedList(infixTerm | term)

num.setParseAction(parseToNum)
const.setParseAction(parseToConst)
array.setParseAction(lambda s,loc,toks: terms.mk_array(toks.asList()))
objpair.setParseAction(lambda s,loc,toks: [toks.asList()])
obj.setParseAction(parseToMap)
apply.setParseAction(parseToApply)
eq_term.setParseAction(parseToEq)  # FIXME: crashes?!
neq_term.setParseAction(parseToNeq)
terms_.setParseAction(lambda s,loc,toks: toks.asList())

# Clauses
clause = (infixTerm | term) + ts + delimitedList(infixTerm | term) + pd
clause.setParseAction(parseToClause)

# Inference rule
inference_rule = (infixTerm | term) + infer + delimitedList(infixTerm | term) + pd
inference_rule.setParseAction(parseToInferenceRule)

# Fact
fact = term + pd

# Statement
statement = fact | clause | inference_rule
statement.setWhitespaceChars(" \t\n")

statements = ZeroOrMore(statement)
statements.ignore(singleLineComment)
statements.setWhitespaceChars(" \t\n")
statements.setParseAction(lambda s,loc,toks: toks.asList())

# helpers
def parse_term(s):
    """Parse a term from a string."""
    return term.parseString(s, parseAll=True)
def parse_terms(s):
    """Parse a list of terms from a string."""
    return terms_.parseString(s, parseAll=True)
