# ezdialog.py - Mike Callahan - Python 2.7 - 06/4/14
# Makes it easy to create dialog and message boxes
#
# History:
# 1.00 - initial version
# 1.01 - make master, title an optional parameter
# 1.10 - major rewrite
# 1.20 - add spin blank assignment
###################################################################

try:
    import Tkinter as tk                               # support Python 2  
except ImportError:
    import tkinter as tk                               # support Python 3
import time, ttk, sys, warnings, tkFont
from tkFileDialog import askopenfilename, asksaveasfilename

class EzDialog(tk.Frame):
    """ easy dialog window creator """

    def __init__(self, master=None):
        """ set-up window and create result dictionary
        master:tk.Toplevel - toplevel window """
        tk.Frame.__init__(self, master)                # create frame 
        self.result = {}                               # frame must manually grid
        top = self.winfo_toplevel()
        if 'destroy' in top.protocol(name='WM_DELETE_WINDOW'): # set close window to _cancel
            top.protocol(name='WM_DELETE_WINDOW', func=self._cancel)
                
    def __getitem__(self, row):
        """ support indexed retreival
        row:int - widget row """
        return self.getParameter(row)                  # support value = dialog[row]

    def __setitem__(self, row, value):
        """ support indexed assignment
        row:int - widget row
        value:object - assignment object """
        self.setParameter(row, value)                  # support dialog[row] = value
        
    def setTitle(self, prompt):
        """ set the title for a window
        prompt:str - the title """
        self.master.title(prompt)

    def makeLabel(self, row, init='', style='', gap=3):
        """ create a label
        row:int - row number
        init:str - label text
        style:string - 'bold' and/or 'italic'
        gap:int - vertical space between widgets frames
        -> ttk.Label """
        self.result[row] = ['label', tk.StringVar()]   # init result to tk var
        label = ttk.Label(self, textvariable=self.result[row][1]) # create label
        self.result[row][1].set(init)                  # set the init value
        font = label['font']                           # get the current font
        font = tkFont.Font(family=font)                # break it down
        weight = 'bold' if 'bold' in style else 'normal'     # set weight 
        slant = 'italic' if 'italic' in style else 'roman'   # set slant
        font.configure(weight=weight, slant=slant)     # change weight, slant
        label['font'] = font                           # use it
        label.grid(row=row, pady=gap)                  # grid it
        return label                                   # label

    def makeLine(self, row, gap=3):
        """ create a horizontal line
        row:int - row number
        gap:int - vertical space between widgets frames
        -> ttk.Separator """
        line = ttk.Separator(self, orient='horizontal') # create line
        line.grid(row=row, sticky='we', pady=gap)      # stretch across entire frame
        return line                                    # separator 

    def makeEntry(self, row, width, prompt, gap=3):
        """ create an Entry Box
        row:int - row number
        width:int - width of entry widget
        prompt:str - text of frame label
        gap:int - vertical space between widget frames
        -> ttk.Entry """
        self.result[row] = ['entry', tk.StringVar()]   # init result to tk var
        frame = ttk.LabelFrame(self, text=prompt)      # create titled frame
        entry = ttk.Entry(frame, width=width, textvariable=self.result[row][1]) #create entry
        entry.grid(sticky='w')                         # grid entry
        frame.grid(row=row, pady=gap)                  # grid titled frame
        return entry                                   # entry
    
    def makeCombo(self, row, width, prompt, alist=[], gap=3):
        """ create a combobox which can be dynamic
        row:int - row number
        width:int - width of entry widget
        prompt:str - text of frame label
        alist:list - items in pulldown
        gap:int - vertcal space between widget frames
        -> ttk.Comobox """
        self.result[row] = ['combo', tk.StringVar()]
        frame = ttk.LabelFrame(self, text=prompt)
        combobox = ttk.Combobox(frame, width=width, textvariable=self.result[row][1],
            values=alist)                              # create combobox
        combobox.grid(sticky='w')                      # grid combobox
        frame.grid(row=row, padx=10, pady=gap)         # grid frame
        return combobox                                # combobox
    
    def makeChecks(self, row, width, prompt, alist, orient='horizontal', gap=3):
        """ create a series of checkbuttons
        row:int - row number
        width:int - width of entry widget
        prompt:str - text of frame label
        alist:list - items in pulldown
        orient:str - 'horizontal' or 'vertical'
        gap:int - vertcal space between widget frames
        -> [ttk.Checkbuttons] """
        self.result[row] = ['checks', {}]              # init result to dict 
        frame = ttk.LabelFrame(self, text=prompt)
        cbuttons = []                                  # list for checkbuttons
        for n, item in enumerate(alist):               # for every item in the given list
            temp = tk.BooleanVar()                     # create the booleanvar
            self.result[row][1][item] = temp           # set boolean into result
            checkbutton = ttk.Checkbutton(frame, width=width, variable=temp,
                text=item)                             # create checkbutton
            if orient == 'vertical':
                checkbutton.grid(row=n, column=0)      # grid it
            else:
                checkbutton.grid(row=0, column=n)
            cbuttons.append(checkbutton)               # add checkbutton to list
        frame.grid(row=row, padx=10, pady=gap)
        return cbuttons                                # list of checkbuttons

    def makeRadios(self, row, width, prompt, alist, orient='horizontal', gap=3):
        """ create a series of radiobuttons
        row:int - row number
        width:int - width of entry widget
        prompt:str - text of frame label
        alist:list - items in pulldown
        orient:str - 'horizontal' or 'vertical'
        gap:int - vertcal space between widget frames
        -> [ttk.Radiobuttons] """
        self.result[row] = ['radio', tk.StringVar()]   # init var to tk var
        frame = ttk.LabelFrame(self, text=prompt)
        rbuttons = []                                  # list for radiobuttons 
        for n, item in enumerate(alist):               # for every item in the given list
            radiobutton = ttk.Radiobutton(frame, width=width, text=item,
                variable=self.result[row][1], value=item) # create the radiobutton
            if orient == 'vertical':
                radiobutton.grid(row=n, column=0)      # grid it
            else:
                radiobutton.grid(row=0, column=n)
            rbuttons.append(radiobutton)
        frame.grid(row=row, padx=10, pady=gap)         # grid frame
        return rbuttons                                # list of radiobuttons
        
    def makeOpen(self, row, width, prompt, gap=3, **parms):
        """ create a open file entry with a browse button
        row:int - row number
        width:int - width of entry widget
        prompt:str - text of frame label
        gap:int - vertcal space between widget frames
        parms:dictionary - labeled arguments to askopenfilename
        -> ttk.Entry """
        self.result[row] = ['open', tk.StringVar()]    # init var to tk var
        frame = ttk.LabelFrame(self, text=prompt)
        entry = ttk.Entry(frame, textvariable=self.result[row][1], width=width) # create entry
        entry.grid(padx=3, pady=3)                     # grid entry in frame
        command = (lambda: self._openDialog(row, **parms))
        button = ttk.Button(frame, width=7, text='Browse', command=command) # create button
        button.grid(row=0, column=1, padx=3, pady=3)   # grid button in frame
        frame.grid(row=row, padx=10, pady=gap)         # grid frame in window
        return [entry, button]                         # list of entry and button
    
    def _openDialog(self, row, **parms):
        """ create the file browsing window
        row:int - row number
        parms:dictionary - labeled arguments to askopenfilename """
        fn = askopenfilename(**parms)                  # create the file dialog
        if fn:                                         # user selected file?
            self.setParameter(row, fn)                 # store it in result
    
    def makeSaveAs(self, row, width, prompt, gap=3, **parms):
        """ create a open file entry with a browse button
        row:int - row number
        width:int - width of entry widget
        prompt:str - text of frame label
        gap:int - vertcal space between widget frames
        parms:dictionary - labeled arguments to asksaveasfilename
        -> [ttk.Entry, ttk.Button] """
        self.result[row] = ['saveas', tk.StringVar()]  # very similar to makeInput
        frame = ttk.LabelFrame(self, text=prompt) 
        entry = ttk.Entry(frame, textvariable=self.result[row][1], width=width)
        entry.grid(padx=3, pady=3)
        button = ttk.Button(frame, width=7, text='Browse',
            command=(lambda: self._saveDialog(row, **parms)))
        button.grid(row=0, column=1, padx=3, pady=3)
        frame.grid(row=row, column=0, padx=10, pady=gap)
        return [entry, button]                         # list of entry and button

    def _saveDialog(self, row, **parms):
        """ create the file browsing window
        row:int - row number
        parms:dictionary - labeled arguments to askopenfilename """
        fn = asksaveasfilename(**parms)                # similar to _openDialog
        if fn:
            self.setParameter(row, fn)
               
    def makeTreeList(self, row, height, prompt, columns, rows, gap=3):
        """ create a treeview that displays only lists
        row:int - row number
        height:int - height of widget 
        prompt:str - text of frame label
        columns:list - the column headers and width
        rows:list - list of widget row results to add to list
        gap:int - vertcal space between widget frames
        -> [ttk.Treeview, ttk.Button, ttk.Button] """
        # for some reason both tree.get_children and tree.get were causing
        # a TCL error, so this implementation works around those methods
        frame = ttk.LabelFrame(self, text=prompt)
        titles = [item[0] for item in columns]         # create the column titles
        tree = ttk.Treeview(frame, columns=titles, show='headings',
            selectmode='browse', height=height)        # create treeview
        self.result[row] = ['tree', [[], tree]]
        xscroll = ttk.Scrollbar(frame, orient='horizontal', command=tree.xview)
        yscroll = ttk.Scrollbar(frame, orient='vertical', command=tree.yview)
        tree.configure(xscroll=xscroll.set, yscroll=yscroll.set)
        for title, width in columns:                   # init column headers
            tree.heading(title, text=title, anchor='w') # set the title
            tree.column(title, width=width)            # set the width
        tree.grid(row=0, column=0, pady=3)             # grid the tree
        xscroll.grid(row=1, column=0, sticky='we')
        yscroll.grid(row=0, column=1, sticky='ns')
        buttonFrame = ttk.Frame(frame)                 # create the add/delete frame
        buttonFrame.grid(row=0, column=2, padx=3)      # grid the frame
        addbutton = ttk.Button(buttonFrame, width=6, text='Add',
            command=(lambda: self._addrow(row, rows))) # create add button
        subbutton = ttk.Button(buttonFrame, width=6, text='Delete',
            command=(lambda: self._delrow(row)))       # create delete button
        addbutton.grid(row=0, padx=1, pady=3)          # grid buttons
        subbutton.grid(row=1, padx=1, pady=3)
        frame.grid(row=row, padx=10, pady=gap)
        return [tree, addbutton, subbutton]            # list of treeview and buttons

    def _addrow(self, row, rows):
        """ add items from other widgets to the treeview
        row:int - the treeview row
        rows:list - the row numbers of the other widgets """
        tree = self.result[row][1][1]                  # get the treeview widget
        items = []                                     # create the items list
        for i in rows:                                 # check each widgets
            items.append(self.getParameter(i))         # get the contents of the other widget
            self.setParameter(i, '')
        tree.insert('', 'end', values=items)           # add list to treeview
        self.result[row][1][0].append(items)           # KLUGE add to row list
        
    def _delrow(self, row):
        """ delete selected row from treeview
        row:int - the treeview row """
        tree = self.result[row][1][1]
        select = tree.selection()                      # get the selection
        if select:
            index = tree.index(select)
            del self.result[row][1][0][index]          # KLUGE remove from row list
            tree.delete(select)                        # remove from treeview
  
    def makeScale(self, row, length, width, prompt, parms, gap=3):
        """ create a integer scale with entry box
        row:int - row number
        length:int - length of scale
        width:int - width of entry widget
        prompt:str - text of frame label
        parms:list - parms of scale [from, to, initial]
        gap:int - vertcal space between widget frames
        -> ttk.Scale """
        self.result[row] = ['scale', tk.IntVar()]
        from_, to, init = parms
        frame = ttk.LabelFrame(self, text=prompt)      # create frame 
        scale = ttk.Scale(frame, length=length, from_=from_, to=to, value=init, 
            variable=self.result[row][1], orient='horizontal',
            command=lambda x: self.setParameter(row, int(float(x)))) # create scale
        # the lambda causes the values to always be integers
        current = ttk.Entry(frame, width=width, textvariable=self.result[row][1]) # create entry
        self.setParameter(row, init)                   # initialize scale
        scale.grid(row=0, column=0)
        current.grid(row=0, column=1, padx=3)
        frame.grid(row=row, pady=gap)
        return [scale, current]                        # list of scale and entry

    def makeSpin(self, row, width, prompt, parms, between='', gap=3):
        """ create a group of spinboxes
        row:int - row number
        width:int - width of spinboxes
        prompt:str - text of frame label
        parms:list - the parmeters for each spinbox [from, to, initial]
        between: str - the label between each box
        gap:int - vertcal space between widget frames
        -> [ttk.Spinboxes] """
        self.result[row] = ['spin', [[], between]]     # data is list  
        frame = ttk.LabelFrame(self, text=prompt)
        col = 0                                        # set col
        spins = []
        for parm in parms:
            from_, to, init = parm                     # extract spinbox parms 
            self.result[row][1][0].append(tk.IntVar()) # add tkvar to list
            spin = tk.Spinbox(frame, width=width, from_=from_, to=to, 
                textvariable=self.result[row][1][0][col/2])
            self.result[row][1][0][col/2].set(init)
            spin.grid(row=0, column=col)
            spins.append(spin)
            col += 1
            label = ttk.Label(frame, text=between)
            label.grid(row=0, column=col)              # grid the separator
            col += 1
        label.destroy()                                # remove last separator
        frame.grid(row=row, pady=gap)
        return spins                                   # list of spinboxes
                
    def makeButtons(self, row, cmd=[], space=3, gap=3):
        """ create a button bar, defaults to Ok - Cancel
        row:int - row number
        cmd:list - [label:str, callback:function] for each button
        space:int - horizontal space between buttons
        gap:int - vertical space between widgets
        -> [ttk.Buttons] """
        if not cmd:                                    # use default buttons
            cmd = [['Ok',self._collect],['Cancel',self._cancel]] 
        frame = ttk.Frame(self)
        buttons = []                                   # list for created buttons 
        col = 0
        for label, callback in cmd:
            button = ttk.Button(frame, width=12, text=label,
              command=callback)                        # create button
            button.grid(row=0, column=col, padx=space) # grid it
            col += 1
            buttons.append(button)                     # add to list
        frame.grid(row=row, padx=gap, pady=gap)
        return buttons                                 # list of created buttons

    def _collect(self):
        """ close the dialog box, data is stored in self.result """
        self.master.destroy()                          # close the dialog window...
                                                       # but keep result
        
    def _cancel(self):
        """ close the dialog box, self.result is cleared """
        self.result = None                             # clear result
        self.master.destroy()                          # close the dialog window

    # support functions
        
    def getParameter(self, row):
        """ get the contents of the widget, same as value = dialog[row]
        row:int - the row number
        -> object """
        if self.result[row]:
            widget, temp = self.result[row]
            if widget in ('label','entry','combo','radio','open','saveas','scale'):
                value = temp.get()
            elif widget == 'checks':              # data type is dict
                value = []                        # create list
                for key in temp:                  # for every key...
                    if temp[key].get():           # check boolean value
                        value.append(key)         # add key to list
            elif widget == 'tree':
                rows, tree = temp                 # split temp
                value = rows                      # get the list
            elif widget == 'spin':
                ints, sep = temp
                value = ''
                for item in ints:
                    value += str(item.get())      # get the tk var
                    value += sep
                value = value[:-1]    
            return value    
    
    def setParameter(self, row, value):
        """ set the contents of the widget, same as dialog[row] = value
        row:init - row number
        value:object - value to set """
        widget, data = self.result[row]           # get the var
        if widget in ('label','entry','combo','radio','open','saveas','scale'):
            data.set(value)
        elif widget == 'checks':
            for key in data:                      # for every key in list...
                data[key].set(key in value)       # set tk boolean
        elif widget == 'tree':
            rows, tree = data                     # split temp
            for item in value:
                tree.insert('', 'end', values=item)
                self.result[row][1][0].append(item)
        elif widget == 'spin':
            for item in data[0]:
                if value == '':
                    item.set('')
                else:
                    item.set(value.pop(0))

    def waitforUser(self):                         
        """ alias for mainloop, better label for beginners """
        self.mainloop()
               

    # the following widgets are containers to other widgets

    def makeNotebook(self, tabs):
        """ create a tabbed notebook
        tab:[str] - titles of each tab page
        -> [notebook, [pages]] """
        pages = []                                # pages will be other frames
        notebook = ttk.Notebook(self)             # create notebook   
        for page in tabs:                         # each tab is a page
            frame = ttk.Frame(self)               # create frame
            notebook.add(frame, text=page, sticky='wens') # fill up the entire page
            pages.append(frame)                   # remember created frames
        return [notebook, pages]                  # must manually grid

    def makeFrame(self, prompt):
        """ create a labeled lrame
        prompt:str - frame label """
        frame = ttk.LabelFrame(self, text=prompt)
        return frame                              # must manually grid
        

class EzMessage(tk.Frame):
    """ easy message window creator """

    def __init__(self, master=None, wait=True, width=60, height=20):
        """ create an output window
        master:tk.toplevel - container widget
        wait:bool - create a button which lets user close window
        width:int - width of window, chars
        height:int - height of window, chars """
        tk.Frame.__init__(self, master)           # create frame
        self._buildWidgets(wait, width, height)   # build the widgets
        
    def _buildWidgets(self, wait, width, height):    
        """ build the widget
        wait:bool - create a button which lets user close window
        width:int - width of window, chars
        height:int - height of window, chars """ 
        self.text = tk.Text(self, width=width, height=height) # make text widget
        self.text.grid(row=0, sticky='wens')      # fill entire frame
        vbar = ttk.Scrollbar(self)                # make scrollbars
        hbar = ttk.Scrollbar(self, orient='horizontal')
        self.text['yscrollcommand'] = vbar.set    # connect scrollbars to text 
        self.text['xscrollcommand'] = hbar.set
        vbar['command'] = self.text.yview
        hbar['command'] = self.text.xview
        vbar.grid(row=0, column=1, sticky='ns')
        hbar.grid(row=1, column=0, sticky='we')
        if wait:                                  # need acknowledge button?
            self.button = ttk.Button(self, text='Ok', # create acknowledge button
                width=8, command=self._exit)
            self.button.grid(row=2, column=0)
        self.text.rowconfigure(0, weight=1)       # make the text widget resizable
        self.text.columnconfigure(0, weight=1)
        self.rowconfigure(0, weight=1)            # make the frame resizable 
        self.columnconfigure(0, weight=1)
        top = self.winfo_toplevel()               # make the window resizable
        top.rowconfigure(0, weight=1)
        top.columnconfigure(0, weight=1)
        self.grid(row=0, column=0, sticky='wens') # grid message window    

    def setTitle(self, prompt):
        """ set the title for a window
        prompt:str - the title """
        self.master.title(prompt)
        
    def addMessage(self, message, delay=0):
        """ print a message in the output window
        message:str - message
        delay:int - secs to delay """
        self.text.insert('end', message)          # add message
        self.text.see('end')                      # scroll text so it is visible
        self.text.update()                        # update display
        time.sleep(delay)                         # pause delay seconds

    def clear(self):
        """ clear the output window """
        self.text.delete('1.0', 'end')            # clear everything
        self.text.update()                        # update display

    def _exit(self):
        self.master.destroy()
        
    def catchExcept(self, wait=True):
        """ send all error messages to message window """
        import traceback
        msg = traceback.format_exc()
        self.addMessage('\n'+msg+'\n')            # add error message
        if wait:
            self.mainloop()

    def waitforUser(self):                         
        """ alias for mainloop, better label for beginners """
        self.mainloop()

