var ipv6parser = $.tablesorter.getParserById('ipv6Address').format,
ipv6regex = $.tablesorter.regex.ipv6Validate,
ipv6test = function(result, str, expect){
	if (result) {
		// ok( $.tablesorter.regex.ipv6Validate.test(str), "valid: " + str );
		// second parameter = true, so we get the hex format (false or undefined returns decimal)
		var t = ipv6parser(str, true);
		equal( t, expect, 'valid: ' + t + ' \u2190 "' + str + '"' );
	} else {
		ok( !ipv6regex.test(str), 'invalid: "' + str + '"' );
	}
},
ipv6tests = function(){

	test( "ipv6 parser", function() {
		expect(483);

		// IPV6 tests by Rich Brown copied from http:// download.dartware.com/thirdparty/test-ipv6-regex.pl
		// modified to compare to canonical ipv6 output using http://www.v6decode.com/ to produce the result
		ipv6test(!1,""); // empty string
		ipv6test(1,"::1", "0000:0000:0000:0000:0000:0000:0000:0001"); // loopback, compressed, non-routable
		ipv6test(1,"::", "0000:0000:0000:0000:0000:0000:0000:0000"); // unspecified, compressed, non-routable
		ipv6test(1,"0:0:0:0:0:0:0:1", "0000:0000:0000:0000:0000:0000:0000:0001"); // loopback, full
		ipv6test(1,"0:0:0:0:0:0:0:0", "0000:0000:0000:0000:0000:0000:0000:0000"); // unspecified, full
		ipv6test(1,"2001:db8:0:0:8:800:200c:417a", "2001:0db8:0000:0000:0008:0800:200c:417a"); // unicast, full
		ipv6test(1,"ff01:0:0:0:0:0:0:101", "ff01:0000:0000:0000:0000:0000:0000:0101"); // multicast, full
		ipv6test(1,"2001:db8::8:800:200c:417a", "2001:0db8:0000:0000:0008:0800:200c:417a"); // unicast, compressed
		ipv6test(1,"ff01::101", "ff01:0000:0000:0000:0000:0000:0000:0101"); // multicast, compressed
		ipv6test(!1,"2001:db8:0:0:8:800:200c:417a:221"); // unicast, full
		ipv6test(!1,"ff01::101::2"); // multicast, compressed
		ipv6test(1,"fe80::217:f2ff:fe07:ed62", "fe80:0000:0000:0000:0217:f2ff:fe07:ed62");
	
		ipv6test(1,"2001:0000:1234:0000:0000:c1c0:abcd:0876", "2001:0000:1234:0000:0000:c1c0:abcd:0876");
		ipv6test(1,"3ffe:0b00:0000:0000:0001:0000:0000:000a", "3ffe:0b00:0000:0000:0001:0000:0000:000a");
		ipv6test(1,"ff02:0000:0000:0000:0000:0000:0000:0001", "ff02:0000:0000:0000:0000:0000:0000:0001");
		ipv6test(1,"0000:0000:0000:0000:0000:0000:0000:0001", "0000:0000:0000:0000:0000:0000:0000:0001");
		ipv6test(1,"0000:0000:0000:0000:0000:0000:0000:0000", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(!1,"02001:0000:1234:0000:0000:c1c0:abcd:0876"); // extra 0 not allowed!
		ipv6test(!1,"2001:0000:1234:0000:00001:c1c0:abcd:0876"); // extra 0 not allowed!
		// ipv6test(1," 2001:0000:1234:0000:0000:c1c0:abcd:0876"); // leading space
		// ipv6test(1,"2001:0000:1234:0000:0000:c1c0:abcd:0876"); // trailing space
		// ipv6test(1," 2001:0000:1234:0000:0000:c1c0:abcd:0876 "); // leading and trailing space
		ipv6test(!1,"2001:0000:1234:0000:0000:c1c0:abcd:0876 0"); // junk after valid address
		ipv6test(!1,"2001:0000:1234: 0000:0000:c1c0:abcd:0876"); // internal space
	
		ipv6test(!1,"3ffe:0b00:0000:0001:0000:0000:000a"); // seven segments
		ipv6test(!1,"ff02:0000:0000:0000:0000:0000:0000:0000:0001"); // nine segments
		ipv6test(!1,"3ffe:b00::1::a"); // double "::"
		ipv6test(!1,"::1111:2222:3333:4444:5555:6666::"); // double "::"
		ipv6test(1,"2::10", "0002:0000:0000:0000:0000:0000:0000:0010");
		ipv6test(1,"ff02::1", "ff02:0000:0000:0000:0000:0000:0000:0001");
		ipv6test(1,"fe80::", "fe80:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"2002::", "2002:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"2001:db8::", "2001:0db8:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"2001:0db8:1234::", "2001:0db8:1234:0000:0000:0000:0000:0000");
		ipv6test(1,"::ffff:0:0", "0000:0000:0000:0000:0000:ffff:0000:0000");
		ipv6test(1,"1:2:3:4:5:6:7:8", "0001:0002:0003:0004:0005:0006:0007:0008");
		ipv6test(1,"1:2:3:4:5:6::8", "0001:0002:0003:0004:0005:0006:0000:0008");
		ipv6test(1,"1:2:3:4:5::8", "0001:0002:0003:0004:0005:0000:0000:0008");
		ipv6test(1,"1:2:3:4::8", "0001:0002:0003:0004:0000:0000:0000:0008");
		ipv6test(1,"1:2:3::8", "0001:0002:0003:0000:0000:0000:0000:0008");
		ipv6test(1,"1:2::8", "0001:0002:0000:0000:0000:0000:0000:0008");
		ipv6test(1,"1::8", "0001:0000:0000:0000:0000:0000:0000:0008");
		ipv6test(1,"1::2:3:4:5:6:7", "0001:0000:0002:0003:0004:0005:0006:0007");
		ipv6test(1,"1::2:3:4:5:6", "0001:0000:0000:0002:0003:0004:0005:0006");
		ipv6test(1,"1::2:3:4:5", "0001:0000:0000:0000:0002:0003:0004:0005");
		ipv6test(1,"1::2:3:4", "0001:0000:0000:0000:0000:0002:0003:0004");
		ipv6test(1,"1::2:3", "0001:0000:0000:0000:0000:0000:0002:0003");
		ipv6test(1,"1::8", "0001:0000:0000:0000:0000:0000:0000:0008");
		ipv6test(1,"::2:3:4:5:6:7:8", "0000:0002:0003:0004:0005:0006:0007:0008");
		ipv6test(1,"::2:3:4:5:6:7", "0000:0000:0002:0003:0004:0005:0006:0007");
		ipv6test(1,"::2:3:4:5:6", "0000:0000:0000:0002:0003:0004:0005:0006");
		ipv6test(1,"::2:3:4:5", "0000:0000:0000:0000:0002:0003:0004:0005");
		ipv6test(1,"::2:3:4", "0000:0000:0000:0000:0000:0002:0003:0004");
		ipv6test(1,"::2:3", "0000:0000:0000:0000:0000:0000:0002:0003");
		ipv6test(1,"::8", "0000:0000:0000:0000:0000:0000:0000:0008");
		ipv6test(1,"1:2:3:4:5:6::", "0001:0002:0003:0004:0005:0006:0000:0000");
		ipv6test(1,"1:2:3:4:5::", "0001:0002:0003:0004:0005:0000:0000:0000");
		ipv6test(1,"1:2:3:4::", "0001:0002:0003:0004:0000:0000:0000:0000");
		ipv6test(1,"1:2:3::", "0001:0002:0003:0000:0000:0000:0000:0000");
		ipv6test(1,"1:2::", "0001:0002:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"1::", "0001:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"1:2:3:4:5::7:8", "0001:0002:0003:0004:0005:0000:0007:0008");
		ipv6test(!1,"1:2:3::4:5::7:8"); // double "::"
		ipv6test(!1,"12345::6:7:8");
		ipv6test(1,"1:2:3:4::7:8", "0001:0002:0003:0004:0000:0000:0007:0008");
		ipv6test(1,"1:2:3::7:8", "0001:0002:0003:0000:0000:0000:0007:0008");
		ipv6test(1,"1:2::7:8", "0001:0002:0000:0000:0000:0000:0007:0008");
		ipv6test(1,"1::7:8", "0001:0000:0000:0000:0000:0000:0007:0008");
	
		// ipv4 addresses as dotted-quads
		ipv6test(1,"1:2:3:4:5:6:1.2.3.4", "0001:0002:0003:0004:0005:0006:0102:0304");
		ipv6test(1,"1:2:3:4:5::1.2.3.4", "0001:0002:0003:0004:0005:0000:0102:0304");
		ipv6test(1,"1:2:3:4::1.2.3.4", "0001:0002:0003:0004:0000:0000:0102:0304");
		ipv6test(1,"1:2:3::1.2.3.4", "0001:0002:0003:0000:0000:0000:0102:0304");
		ipv6test(1,"1:2::1.2.3.4", "0001:0002:0000:0000:0000:0000:0102:0304");
		ipv6test(1,"1::1.2.3.4", "0001:0000:0000:0000:0000:0000:0102:0304");
		ipv6test(1,"1:2:3:4::5:1.2.3.4", "0001:0002:0003:0004:0000:0005:0102:0304");
		ipv6test(1,"1:2:3::5:1.2.3.4", "0001:0002:0003:0000:0000:0005:0102:0304");
		ipv6test(1,"1:2::5:1.2.3.4", "0001:0002:0000:0000:0000:0005:0102:0304");
		ipv6test(1,"1::5:1.2.3.4", "0001:0000:0000:0000:0000:0005:0102:0304");
		ipv6test(1,"1::5:11.22.33.44", "0001:0000:0000:0000:0000:0005:0b16:212c");
		ipv6test(!1,"1::5:400.2.3.4");
		ipv6test(!1,"1::5:260.2.3.4");
		ipv6test(!1,"1::5:256.2.3.4");
		ipv6test(!1,"1::5:1.256.3.4");
		ipv6test(!1,"1::5:1.2.256.4");
		ipv6test(!1,"1::5:1.2.3.256");
		ipv6test(!1,"1::5:300.2.3.4");
		ipv6test(!1,"1::5:1.300.3.4");
		ipv6test(!1,"1::5:1.2.300.4");
		ipv6test(!1,"1::5:1.2.3.300");
		ipv6test(!1,"1::5:900.2.3.4");
		ipv6test(!1,"1::5:1.900.3.4");
		ipv6test(!1,"1::5:1.2.900.4");
		ipv6test(!1,"1::5:1.2.3.900");
		ipv6test(!1,"1::5:300.300.300.300");
		ipv6test(!1,"1::5:3000.30.30.30");
		ipv6test(!1,"1::400.2.3.4");
		ipv6test(!1,"1::260.2.3.4");
		ipv6test(!1,"1::256.2.3.4");
		ipv6test(!1,"1::1.256.3.4");
		ipv6test(!1,"1::1.2.256.4");
		ipv6test(!1,"1::1.2.3.256");
		ipv6test(!1,"1::300.2.3.4");
		ipv6test(!1,"1::1.300.3.4");
		ipv6test(!1,"1::1.2.300.4");
		ipv6test(!1,"1::1.2.3.300");
		ipv6test(!1,"1::900.2.3.4");
		ipv6test(!1,"1::1.900.3.4");
		ipv6test(!1,"1::1.2.900.4");
		ipv6test(!1,"1::1.2.3.900");
		ipv6test(!1,"1::300.300.300.300");
		ipv6test(!1,"1::3000.30.30.30");
		ipv6test(!1,"::400.2.3.4");
		ipv6test(!1,"::260.2.3.4");
		ipv6test(!1,"::256.2.3.4");
		ipv6test(!1,"::1.256.3.4");
		ipv6test(!1,"::1.2.256.4");
		ipv6test(!1,"::1.2.3.256");
		ipv6test(!1,"::300.2.3.4");
		ipv6test(!1,"::1.300.3.4");
		ipv6test(!1,"::1.2.300.4");
		ipv6test(!1,"::1.2.3.300");
		ipv6test(!1,"::900.2.3.4");
		ipv6test(!1,"::1.900.3.4");
		ipv6test(!1,"::1.2.900.4");
		ipv6test(!1,"::1.2.3.900");
		ipv6test(!1,"::300.300.300.300");
		ipv6test(!1,"::3000.30.30.30");
		ipv6test(1,"fe80::217:f2ff:254.7.237.98", "fe80:0000:0000:0000:0217:f2ff:fe07:ed62");
		ipv6test(1,"::ffff:192.168.1.26", "0000:0000:0000:0000:0000:ffff:c0a8:011a");
		ipv6test(!1,"2001:1:1:1:1:1:255z255x255y255"); // garbage instead of "." in ipv4
		ipv6test(!1,"::ffff:192x168.1.26"); // ditto
		ipv6test(1,"::ffff:192.168.1.1", "0000:0000:0000:0000:0000:ffff:c0a8:0101");
		ipv6test(1,"0:0:0:0:0:0:13.1.68.3", "0000:0000:0000:0000:0000:0000:0d01:4403"); // ipv4-compatible ipv6 address, full, deprecated
		ipv6test(1,"0:0:0:0:0:ffff:129.144.52.38", "0000:0000:0000:0000:0000:ffff:8190:3426"); // ipv4-mapped ipv6 address, full
		ipv6test(1,"::13.1.68.3", "0000:0000:0000:0000:0000:0000:0d01:4403"); // ipv4-compatible ipv6 address, compressed, deprecated
		ipv6test(1,"::ffff:129.144.52.38", "0000:0000:0000:0000:0000:ffff:8190:3426"); // ipv4-mapped ipv6 address, compressed
		ipv6test(1,"fe80:0:0:0:204:61ff:254.157.241.86", "fe80:0000:0000:0000:0204:61ff:fe9d:f156");
		ipv6test(1,"fe80::204:61ff:254.157.241.86", "fe80:0000:0000:0000:0204:61ff:fe9d:f156");
		ipv6test(1,"::ffff:12.34.56.78", "0000:0000:0000:0000:0000:ffff:0c22:384e");
		ipv6test(!1,"::ffff:2.3.4");
		ipv6test(!1,"::ffff:257.1.2.3");
		ipv6test(!1,"1.2.3.4");
	
		ipv6test(!1,"1.2.3.4:1111:2222:3333:4444::5555"); // aeron
		ipv6test(!1,"1.2.3.4:1111:2222:3333::5555");
		ipv6test(!1,"1.2.3.4:1111:2222::5555");
		ipv6test(!1,"1.2.3.4:1111::5555");
		ipv6test(!1,"1.2.3.4::5555");
		ipv6test(!1,"1.2.3.4::");
	
		// testing ipv4 addresses represented as dotted-quads
		// leading zero's in ipv4 addresses not allowed: some systems treat the leading "0" in ".086" as the start of an octal number
		// update: the bnf in rfc-3986 explicitly defines the dec-octet (for ipv4 addresses) not to have a leading zero
		ipv6test(!1,"fe80:0000:0000:0000:0204:61ff:254.157.241.086");
		ipv6test(1,"::ffff:192.0.2.128", "0000:0000:0000:0000:0000:ffff:c000:0280"); // but this is ok, since there's a single digit
		ipv6test(!1,"xxxx:xxxx:xxxx:xxxx:xxxx:xxxx:1.2.3.4");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:00.00.00.00");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:000.000.000.000");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:256.256.256.256");
	
		// not testing address with subnet mask
		// ipv6test(1,"2001:0db8:0000:cd30:0000:0000:0000:0000/60"); // full, with prefix
		// ipv6test(1,"2001:0db8::cd30:0:0:0:0/60"); // compressed, with prefix
		// ipv6test(1,"2001:0db8:0:cd30::/60"); // compressed, with prefix // 2
		// ipv6test(1,"::/128"); // compressed, unspecified address type, non-routable
		// ipv6test(1,"::1/128"); // compressed, loopback address type, non-routable
		// ipv6test(1,"ff00::/8"); // compressed, multicast address type
		// ipv6test(1,"fe80::/10"); // compressed, link-local unicast, non-routable
		// ipv6test(1,"fec0::/10"); // compressed, site-local unicast, deprecated
		// ipv6test(!1,"124.15.6.89/60"); // standard ipv4, prefix not allowed
	
		ipv6test(1,"fe80:0000:0000:0000:0204:61ff:fe9d:f156", "fe80:0000:0000:0000:0204:61ff:fe9d:f156");
		ipv6test(1,"fe80:0:0:0:204:61ff:fe9d:f156", "fe80:0000:0000:0000:0204:61ff:fe9d:f156");
		ipv6test(1,"fe80::204:61ff:fe9d:f156", "fe80:0000:0000:0000:0204:61ff:fe9d:f156");
		ipv6test(1,"fe80::", "fe80:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"fe80::1", "fe80:0000:0000:0000:0000:0000:0000:0001");
		ipv6test(!1,":");
		ipv6test(1,"::ffff:c000:280", "0000:0000:0000:0000:0000:ffff:c000:0280");
	
		// aeron supplied these test cases
		ipv6test(!1,"1111:2222:3333:4444::5555:");
		ipv6test(!1,"1111:2222:3333::5555:");
		ipv6test(!1,"1111:2222::5555:");
		ipv6test(!1,"1111::5555:");
		ipv6test(!1,"::5555:");
		ipv6test(!1,":::");
		ipv6test(!1,"1111:");
		ipv6test(!1,":");
	
		ipv6test(!1,":1111:2222:3333:4444::5555");
		ipv6test(!1,":1111:2222:3333::5555");
		ipv6test(!1,":1111:2222::5555");
		ipv6test(!1,":1111::5555");
		ipv6test(!1,":::5555");
		ipv6test(!1,":::");
	
	
		// additional test cases
		// from http:// rt.cpan.org/public/bug/display.html?id=50693
	
		ipv6test(1,"2001:0db8:85a3:0000:0000:8a2e:0370:7334", "2001:0db8:85a3:0000:0000:8a2e:0370:7334");
		ipv6test(1,"2001:db8:85a3:0:0:8a2e:370:7334", "2001:0db8:85a3:0000:0000:8a2e:0370:7334");
		ipv6test(1,"2001:db8:85a3::8a2e:370:7334", "2001:0db8:85a3:0000:0000:8a2e:0370:7334");
		ipv6test(1,"2001:0db8:0000:0000:0000:0000:1428:57ab", "2001:0db8:0000:0000:0000:0000:1428:57ab");
		ipv6test(1,"2001:0db8:0000:0000:0000::1428:57ab", "2001:0db8:0000:0000:0000:0000:1428:57ab");
		ipv6test(1,"2001:0db8:0:0:0:0:1428:57ab", "2001:0db8:0000:0000:0000:0000:1428:57ab");
		ipv6test(1,"2001:0db8:0:0::1428:57ab", "2001:0db8:0000:0000:0000:0000:1428:57ab");
		ipv6test(1,"2001:0db8::1428:57ab", "2001:0db8:0000:0000:0000:0000:1428:57ab");
		ipv6test(1,"2001:db8::1428:57ab", "2001:0db8:0000:0000:0000:0000:1428:57ab");
		ipv6test(1,"0000:0000:0000:0000:0000:0000:0000:0001", "0000:0000:0000:0000:0000:0000:0000:0001");
		ipv6test(1,"::ffff:0c22:384e", "0000:0000:0000:0000:0000:ffff:0c22:384e");
		ipv6test(1,"2001:0db8:1234:0000:0000:0000:0000:0000", "2001:0db8:1234:0000:0000:0000:0000:0000");
		ipv6test(1,"2001:0db8:1234:ffff:ffff:ffff:ffff:ffff", "2001:0db8:1234:ffff:ffff:ffff:ffff:ffff");
		ipv6test(1,"2001:db8:a::123", "2001:0db8:000a:0000:0000:0000:0000:0123");
		ipv6test(1,"fe80::", "fe80:0000:0000:0000:0000:0000:0000:0000");
	
		ipv6test(!1,"123");
		ipv6test(!1,"ldkfj");
		ipv6test(!1,"2001::ffd3::57ab");
		ipv6test(!1,"2001:db8:85a3::8a2e:37023:7334");
		ipv6test(!1,"2001:db8:85a3::8a2e:370k:7334");
		ipv6test(!1,"1:2:3:4:5:6:7:8:9");
		ipv6test(!1,"1::2::3");
		ipv6test(!1,"1:::3:4:5");
		ipv6test(!1,"1:2:3::4:5:6:7:8:9");
	
		// new from aeron
		ipv6test(1,"1111:2222:3333:4444:5555:6666:7777:8888", "1111:2222:3333:4444:5555:6666:7777:8888");
		ipv6test(1,"1111:2222:3333:4444:5555:6666:7777::", "1111:2222:3333:4444:5555:6666:7777:0000");
		ipv6test(1,"1111:2222:3333:4444:5555:6666::", "1111:2222:3333:4444:5555:6666:0000:0000");
		ipv6test(1,"1111:2222:3333:4444:5555::", "1111:2222:3333:4444:5555:0000:0000:0000");
		ipv6test(1,"1111:2222:3333:4444::", "1111:2222:3333:4444:0000:0000:0000:0000");
		ipv6test(1,"1111:2222:3333::", "1111:2222:3333:0000:0000:0000:0000:0000");
		ipv6test(1,"1111:2222::", "1111:2222:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"1111::", "1111:0000:0000:0000:0000:0000:0000:0000");
		// ipv6test(1,"::", ""); // duplicate
		ipv6test(1,"1111:2222:3333:4444:5555:6666::8888", "1111:2222:3333:4444:5555:6666:0000:8888");
		ipv6test(1,"1111:2222:3333:4444:5555::8888", "1111:2222:3333:4444:5555:0000:0000:8888");
		ipv6test(1,"1111:2222:3333:4444::8888", "1111:2222:3333:4444:0000:0000:0000:8888");
		ipv6test(1,"1111:2222:3333::8888", "1111:2222:3333:0000:0000:0000:0000:8888");
		ipv6test(1,"1111:2222::8888", "1111:2222:0000:0000:0000:0000:0000:8888");
		ipv6test(1,"1111::8888", "1111:0000:0000:0000:0000:0000:0000:8888");
		ipv6test(1,"::8888", "0000:0000:0000:0000:0000:0000:0000:8888");
		ipv6test(1,"1111:2222:3333:4444:5555::7777:8888", "1111:2222:3333:4444:5555:0000:7777:8888");
		ipv6test(1,"1111:2222:3333:4444::7777:8888", "1111:2222:3333:4444:0000:0000:7777:8888");
		ipv6test(1,"1111:2222:3333::7777:8888", "1111:2222:3333:0000:0000:0000:7777:8888");
		ipv6test(1,"1111:2222::7777:8888", "1111:2222:0000:0000:0000:0000:7777:8888");
		ipv6test(1,"1111::7777:8888", "1111:0000:0000:0000:0000:0000:7777:8888");
		ipv6test(1,"::7777:8888", "0000:0000:0000:0000:0000:0000:7777:8888");
		ipv6test(1,"1111:2222:3333:4444::6666:7777:8888", "1111:2222:3333:4444:0000:6666:7777:8888");
		ipv6test(1,"1111:2222:3333::6666:7777:8888", "1111:2222:3333:0000:0000:6666:7777:8888");
		ipv6test(1,"1111:2222::6666:7777:8888", "1111:2222:0000:0000:0000:6666:7777:8888");
		ipv6test(1,"1111::6666:7777:8888", "1111:0000:0000:0000:0000:6666:7777:8888");
		ipv6test(1,"::6666:7777:8888", "0000:0000:0000:0000:0000:6666:7777:8888");
		ipv6test(1,"1111:2222:3333::5555:6666:7777:8888", "1111:2222:3333:0000:5555:6666:7777:8888");
		ipv6test(1,"1111:2222::5555:6666:7777:8888", "1111:2222:0000:0000:5555:6666:7777:8888");
		ipv6test(1,"1111::5555:6666:7777:8888", "1111:0000:0000:0000:5555:6666:7777:8888");
		ipv6test(1,"::5555:6666:7777:8888", "0000:0000:0000:0000:5555:6666:7777:8888");
		ipv6test(1,"1111:2222::4444:5555:6666:7777:8888", "1111:2222:0000:4444:5555:6666:7777:8888");
		ipv6test(1,"1111::4444:5555:6666:7777:8888", "1111:0000:0000:4444:5555:6666:7777:8888");
		ipv6test(1,"::4444:5555:6666:7777:8888", "0000:0000:0000:4444:5555:6666:7777:8888");
		ipv6test(1,"1111::3333:4444:5555:6666:7777:8888", "1111:0000:3333:4444:5555:6666:7777:8888");
		ipv6test(1,"::3333:4444:5555:6666:7777:8888", "0000:0000:3333:4444:5555:6666:7777:8888");
		ipv6test(1,"::2222:3333:4444:5555:6666:7777:8888", "0000:2222:3333:4444:5555:6666:7777:8888");
		ipv6test(1,"1111:2222:3333:4444:5555:6666:123.123.123.123", "1111:2222:3333:4444:5555:6666:7b7b:7b7b");
		ipv6test(1,"1111:2222:3333:4444:5555::123.123.123.123", "1111:2222:3333:4444:5555:0000:7b7b:7b7b");
		ipv6test(1,"1111:2222:3333:4444::123.123.123.123", "1111:2222:3333:4444:0000:0000:7b7b:7b7b");
		ipv6test(1,"1111:2222:3333::123.123.123.123", "1111:2222:3333:0000:0000:0000:7b7b:7b7b");
		ipv6test(1,"1111:2222::123.123.123.123", "1111:2222:0000:0000:0000:0000:7b7b:7b7b");
		ipv6test(1,"1111::123.123.123.123", "1111:0000:0000:0000:0000:0000:7b7b:7b7b");
		ipv6test(1,"::123.123.123.123", "0000:0000:0000:0000:0000:0000:7b7b:7b7b");
		ipv6test(1,"1111:2222:3333:4444::6666:123.123.123.123", "1111:2222:3333:4444:0000:6666:7b7b:7b7b");
		ipv6test(1,"1111:2222:3333::6666:123.123.123.123", "1111:2222:3333:0000:0000:6666:7b7b:7b7b");
		ipv6test(1,"1111:2222::6666:123.123.123.123", "1111:2222:0000:0000:0000:6666:7b7b:7b7b");
		ipv6test(1,"1111::6666:123.123.123.123", "1111:0000:0000:0000:0000:6666:7b7b:7b7b");
		ipv6test(1,"::6666:123.123.123.123", "0000:0000:0000:0000:0000:6666:7b7b:7b7b");
		ipv6test(1,"1111:2222:3333::5555:6666:123.123.123.123", "1111:2222:3333:0000:5555:6666:7b7b:7b7b");
		ipv6test(1,"1111:2222::5555:6666:123.123.123.123", "1111:2222:0000:0000:5555:6666:7b7b:7b7b");
		ipv6test(1,"1111::5555:6666:123.123.123.123", "1111:0000:0000:0000:5555:6666:7b7b:7b7b");
		ipv6test(1,"::5555:6666:123.123.123.123", "0000:0000:0000:0000:5555:6666:7b7b:7b7b");
		ipv6test(1,"1111:2222::4444:5555:6666:123.123.123.123", "1111:2222:0000:4444:5555:6666:7b7b:7b7b");
		ipv6test(1,"1111::4444:5555:6666:123.123.123.123", "1111:0000:0000:4444:5555:6666:7b7b:7b7b");
		ipv6test(1,"::4444:5555:6666:123.123.123.123", "0000:0000:0000:4444:5555:6666:7b7b:7b7b");
		ipv6test(1,"1111::3333:4444:5555:6666:123.123.123.123", "1111:0000:3333:4444:5555:6666:7b7b:7b7b");
		ipv6test(1,"::2222:3333:4444:5555:6666:123.123.123.123", "0000:2222:3333:4444:5555:6666:7b7b:7b7b");
	
		// playing with combinations of "0" and "::"
		// nb: these are all sytactically correct, but are bad form
		// because "0" adjacent to "::" should be combined into "::"
		ipv6test(1,"::0:0:0:0:0:0:0", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"::0:0:0:0:0:0", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"::0:0:0:0:0", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"::0:0:0:0", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"::0:0:0", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"::0:0", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"::0", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"0:0:0:0:0:0:0::", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"0:0:0:0:0:0::", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"0:0:0:0:0::", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"0:0:0:0::", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"0:0:0::", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"0:0::", "0000:0000:0000:0000:0000:0000:0000:0000");
		ipv6test(1,"0::", "0000:0000:0000:0000:0000:0000:0000:0000");
	
		// new invalid from aeron
		// invalid data
		ipv6test(!1,"xxxx:xxxx:xxxx:xxxx:xxxx:xxxx:xxxx:xxxx");
	
		// too many components
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:7777:8888:9999");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:7777:8888::");
		ipv6test(!1,"::2222:3333:4444:5555:6666:7777:8888:9999");
	
		// too few components
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:7777");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666");
		ipv6test(!1,"1111:2222:3333:4444:5555");
		ipv6test(!1,"1111:2222:3333:4444");
		ipv6test(!1,"1111:2222:3333");
		ipv6test(!1,"1111:2222");
		ipv6test(!1,"1111");
	
		// missing :
		ipv6test(!1,"11112222:3333:4444:5555:6666:7777:8888");
		ipv6test(!1,"1111:22223333:4444:5555:6666:7777:8888");
		ipv6test(!1,"1111:2222:33334444:5555:6666:7777:8888");
		ipv6test(!1,"1111:2222:3333:44445555:6666:7777:8888");
		ipv6test(!1,"1111:2222:3333:4444:55556666:7777:8888");
		ipv6test(!1,"1111:2222:3333:4444:5555:66667777:8888");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:77778888");
	
		// missing : intended for ::
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:7777:8888:");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:7777:");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:");
		ipv6test(!1,"1111:2222:3333:4444:5555:");
		ipv6test(!1,"1111:2222:3333:4444:");
		ipv6test(!1,"1111:2222:3333:");
		ipv6test(!1,"1111:2222:");
		ipv6test(!1,"1111:");
		ipv6test(!1,":");
		ipv6test(!1,":8888");
		ipv6test(!1,":7777:8888");
		ipv6test(!1,":6666:7777:8888");
		ipv6test(!1,":5555:6666:7777:8888");
		ipv6test(!1,":4444:5555:6666:7777:8888");
		ipv6test(!1,":3333:4444:5555:6666:7777:8888");
		ipv6test(!1,":2222:3333:4444:5555:6666:7777:8888");
		ipv6test(!1,":1111:2222:3333:4444:5555:6666:7777:8888");
	
		// :::
		ipv6test(!1,":::2222:3333:4444:5555:6666:7777:8888");
		ipv6test(!1,"1111:::3333:4444:5555:6666:7777:8888");
		ipv6test(!1,"1111:2222:::4444:5555:6666:7777:8888");
		ipv6test(!1,"1111:2222:3333:::5555:6666:7777:8888");
		ipv6test(!1,"1111:2222:3333:4444:::6666:7777:8888");
		ipv6test(!1,"1111:2222:3333:4444:5555:::7777:8888");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:::8888");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:7777:::");
	
		// double ::");
		ipv6test(!1,"::2222::4444:5555:6666:7777:8888");
		ipv6test(!1,"::2222:3333::5555:6666:7777:8888");
		ipv6test(!1,"::2222:3333:4444::6666:7777:8888");
		ipv6test(!1,"::2222:3333:4444:5555::7777:8888");
		ipv6test(!1,"::2222:3333:4444:5555:7777::8888");
		ipv6test(!1,"::2222:3333:4444:5555:7777:8888::");
	
		ipv6test(!1,"1111::3333::5555:6666:7777:8888");
		ipv6test(!1,"1111::3333:4444::6666:7777:8888");
		ipv6test(!1,"1111::3333:4444:5555::7777:8888");
		ipv6test(!1,"1111::3333:4444:5555:6666::8888");
		ipv6test(!1,"1111::3333:4444:5555:6666:7777::");
	
		ipv6test(!1,"1111:2222::4444::6666:7777:8888");
		ipv6test(!1,"1111:2222::4444:5555::7777:8888");
		ipv6test(!1,"1111:2222::4444:5555:6666::8888");
		ipv6test(!1,"1111:2222::4444:5555:6666:7777::");
	
		ipv6test(!1,"1111:2222:3333::5555::7777:8888");
		ipv6test(!1,"1111:2222:3333::5555:6666::8888");
		ipv6test(!1,"1111:2222:3333::5555:6666:7777::");
	
		ipv6test(!1,"1111:2222:3333:4444::6666::8888");
		ipv6test(!1,"1111:2222:3333:4444::6666:7777::");
	
		ipv6test(!1,"1111:2222:3333:4444:5555::7777::");
	
	
		// too many components"
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:7777:8888:1.2.3.4");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:7777:1.2.3.4");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666::1.2.3.4");
		ipv6test(!1,"::2222:3333:4444:5555:6666:7777:1.2.3.4");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:1.2.3.4.5");
	
		// too few components
		ipv6test(!1,"1111:2222:3333:4444:5555:1.2.3.4");
		ipv6test(!1,"1111:2222:3333:4444:1.2.3.4");
		ipv6test(!1,"1111:2222:3333:1.2.3.4");
		ipv6test(!1,"1111:2222:1.2.3.4");
		ipv6test(!1,"1111:1.2.3.4");
		ipv6test(!1,"1.2.3.4");
	
		// missing :
		ipv6test(!1,"11112222:3333:4444:5555:6666:1.2.3.4");
		ipv6test(!1,"1111:22223333:4444:5555:6666:1.2.3.4");
		ipv6test(!1,"1111:2222:33334444:5555:6666:1.2.3.4");
		ipv6test(!1,"1111:2222:3333:44445555:6666:1.2.3.4");
		ipv6test(!1,"1111:2222:3333:4444:55556666:1.2.3.4");
		ipv6test(!1,"1111:2222:3333:4444:5555:66661.2.3.4");
	
		// missing .
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:255255.255.255");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:255.255255.255");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:255.255.255255");
	
		// missing : intended for ::
		ipv6test(!1,":1.2.3.4");
		ipv6test(!1,":6666:1.2.3.4");
		ipv6test(!1,":5555:6666:1.2.3.4");
		ipv6test(!1,":4444:5555:6666:1.2.3.4");
		ipv6test(!1,":3333:4444:5555:6666:1.2.3.4");
		ipv6test(!1,":2222:3333:4444:5555:6666:1.2.3.4");
		ipv6test(!1,":1111:2222:3333:4444:5555:6666:1.2.3.4");
	
		// :::
		ipv6test(!1,":::2222:3333:4444:5555:6666:1.2.3.4");
		ipv6test(!1,"1111:::3333:4444:5555:6666:1.2.3.4");
		ipv6test(!1,"1111:2222:::4444:5555:6666:1.2.3.4");
		ipv6test(!1,"1111:2222:3333:::5555:6666:1.2.3.4");
		ipv6test(!1,"1111:2222:3333:4444:::6666:1.2.3.4");
		ipv6test(!1,"1111:2222:3333:4444:5555:::1.2.3.4");
	
		// double ::
		ipv6test(!1,"::2222::4444:5555:6666:1.2.3.4");
		ipv6test(!1,"::2222:3333::5555:6666:1.2.3.4");
		ipv6test(!1,"::2222:3333:4444::6666:1.2.3.4");
		ipv6test(!1,"::2222:3333:4444:5555::1.2.3.4");
	
		ipv6test(!1,"1111::3333::5555:6666:1.2.3.4");
		ipv6test(!1,"1111::3333:4444::6666:1.2.3.4");
		ipv6test(!1,"1111::3333:4444:5555::1.2.3.4");
	
		ipv6test(!1,"1111:2222::4444::6666:1.2.3.4");
		ipv6test(!1,"1111:2222::4444:5555::1.2.3.4");
	
		ipv6test(!1,"1111:2222:3333::5555::1.2.3.4");
	
		// missing parts
		ipv6test(!1,"::.");
		ipv6test(!1,"::..");
		ipv6test(!1,"::...");
		ipv6test(!1,"::1...");
		ipv6test(!1,"::1.2..");
		ipv6test(!1,"::1.2.3.");
		ipv6test(!1,"::.2..");
		ipv6test(!1,"::.2.3.");
		ipv6test(!1,"::.2.3.4");
		ipv6test(!1,"::..3.");
		ipv6test(!1,"::..3.4");
		ipv6test(!1,"::...4");
	
		// extra : in front
		ipv6test(!1,":1111:2222:3333:4444:5555:6666:7777::");
		ipv6test(!1,":1111:2222:3333:4444:5555:6666::");
		ipv6test(!1,":1111:2222:3333:4444:5555::");
		ipv6test(!1,":1111:2222:3333:4444::");
		ipv6test(!1,":1111:2222:3333::");
		ipv6test(!1,":1111:2222::");
		ipv6test(!1,":1111::");
		ipv6test(!1,":::");
		ipv6test(!1,":1111:2222:3333:4444:5555:6666::8888");
		ipv6test(!1,":1111:2222:3333:4444:5555::8888");
		ipv6test(!1,":1111:2222:3333:4444::8888");
		ipv6test(!1,":1111:2222:3333::8888");
		ipv6test(!1,":1111:2222::8888");
		ipv6test(!1,":1111::8888");
		ipv6test(!1,":::8888");
		ipv6test(!1,":1111:2222:3333:4444:5555::7777:8888");
		ipv6test(!1,":1111:2222:3333:4444::7777:8888");
		ipv6test(!1,":1111:2222:3333::7777:8888");
		ipv6test(!1,":1111:2222::7777:8888");
		ipv6test(!1,":1111::7777:8888");
		ipv6test(!1,":::7777:8888");
		ipv6test(!1,":1111:2222:3333:4444::6666:7777:8888");
		ipv6test(!1,":1111:2222:3333::6666:7777:8888");
		ipv6test(!1,":1111:2222::6666:7777:8888");
		ipv6test(!1,":1111::6666:7777:8888");
		ipv6test(!1,":::6666:7777:8888");
		ipv6test(!1,":1111:2222:3333::5555:6666:7777:8888");
		ipv6test(!1,":1111:2222::5555:6666:7777:8888");
		ipv6test(!1,":1111::5555:6666:7777:8888");
		ipv6test(!1,":::5555:6666:7777:8888");
		ipv6test(!1,":1111:2222::4444:5555:6666:7777:8888");
		ipv6test(!1,":1111::4444:5555:6666:7777:8888");
		ipv6test(!1,":::4444:5555:6666:7777:8888");
		ipv6test(!1,":1111::3333:4444:5555:6666:7777:8888");
		ipv6test(!1,":::3333:4444:5555:6666:7777:8888");
		ipv6test(!1,":::2222:3333:4444:5555:6666:7777:8888");
		ipv6test(!1,":1111:2222:3333:4444:5555:6666:1.2.3.4");
		ipv6test(!1,":1111:2222:3333:4444:5555::1.2.3.4");
		ipv6test(!1,":1111:2222:3333:4444::1.2.3.4");
		ipv6test(!1,":1111:2222:3333::1.2.3.4");
		ipv6test(!1,":1111:2222::1.2.3.4");
		ipv6test(!1,":1111::1.2.3.4");
		ipv6test(!1,":::1.2.3.4");
		ipv6test(!1,":1111:2222:3333:4444::6666:1.2.3.4");
		ipv6test(!1,":1111:2222:3333::6666:1.2.3.4");
		ipv6test(!1,":1111:2222::6666:1.2.3.4");
		ipv6test(!1,":1111::6666:1.2.3.4");
		ipv6test(!1,":::6666:1.2.3.4");
		ipv6test(!1,":1111:2222:3333::5555:6666:1.2.3.4");
		ipv6test(!1,":1111:2222::5555:6666:1.2.3.4");
		ipv6test(!1,":1111::5555:6666:1.2.3.4");
		ipv6test(!1,":::5555:6666:1.2.3.4");
		ipv6test(!1,":1111:2222::4444:5555:6666:1.2.3.4");
		ipv6test(!1,":1111::4444:5555:6666:1.2.3.4");
		ipv6test(!1,":::4444:5555:6666:1.2.3.4");
		ipv6test(!1,":1111::3333:4444:5555:6666:1.2.3.4");
		ipv6test(!1,":::2222:3333:4444:5555:6666:1.2.3.4");
	
		// extra : at end
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:7777:::");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666:::");
		ipv6test(!1,"1111:2222:3333:4444:5555:::");
		ipv6test(!1,"1111:2222:3333:4444:::");
		ipv6test(!1,"1111:2222:3333:::");
		ipv6test(!1,"1111:2222:::");
		ipv6test(!1,"1111:::");
		ipv6test(!1,":::");
		ipv6test(!1,"1111:2222:3333:4444:5555:6666::8888:");
		ipv6test(!1,"1111:2222:3333:4444:5555::8888:");
		ipv6test(!1,"1111:2222:3333:4444::8888:");
		ipv6test(!1,"1111:2222:3333::8888:");
		ipv6test(!1,"1111:2222::8888:");
		ipv6test(!1,"1111::8888:");
		ipv6test(!1,"::8888:");
		ipv6test(!1,"1111:2222:3333:4444:5555::7777:8888:");
		ipv6test(!1,"1111:2222:3333:4444::7777:8888:");
		ipv6test(!1,"1111:2222:3333::7777:8888:");
		ipv6test(!1,"1111:2222::7777:8888:");
		ipv6test(!1,"1111::7777:8888:");
		ipv6test(!1,"::7777:8888:");
		ipv6test(!1,"1111:2222:3333:4444::6666:7777:8888:");
		ipv6test(!1,"1111:2222:3333::6666:7777:8888:");
		ipv6test(!1,"1111:2222::6666:7777:8888:");
		ipv6test(!1,"1111::6666:7777:8888:");
		ipv6test(!1,"::6666:7777:8888:");
		ipv6test(!1,"1111:2222:3333::5555:6666:7777:8888:");
		ipv6test(!1,"1111:2222::5555:6666:7777:8888:");
		ipv6test(!1,"1111::5555:6666:7777:8888:");
		ipv6test(!1,"::5555:6666:7777:8888:");
		ipv6test(!1,"1111:2222::4444:5555:6666:7777:8888:");
		ipv6test(!1,"1111::4444:5555:6666:7777:8888:");
		ipv6test(!1,"::4444:5555:6666:7777:8888:");
		ipv6test(!1,"1111::3333:4444:5555:6666:7777:8888:");
		ipv6test(!1,"::3333:4444:5555:6666:7777:8888:");
		ipv6test(!1,"::2222:3333:4444:5555:6666:7777:8888:");
	
		// additional cases: http:// crisp.tweakblogs.net/blog/2031/ipv6-validation-%28and-caveats%29.html
		ipv6test(1,"0:a:b:c:d:e:f::", "0000:000a:000b:000c:000d:000e:000f:0000");
		ipv6test(1,"::0:a:b:c:d:e:f", "0000:0000:000a:000b:000c:000d:000e:000f"); // syntactically correct, but bad form (::0:... could be combined)
		ipv6test(1,"a:b:c:d:e:f:0::", "000a:000b:000c:000d:000e:000f:0000:0000");
		ipv6test(!1,"':10.0.0.1");
	});
};