# -*- coding: utf-8 -*-
# data_filter_test.py

# Licence
#
# FilterPype is a process-flow pipes-and-filters Python framework.
# Copyright (c) 2009-2012 Flight Data Services Ltd
# http://www.filterpype.org
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.


import unittest
import hashlib
from configobj import ConfigObj
import sys
import math
import timeit
import os
import shutil
import mock
import bz2
import random

import filterpype.data_fltr_base as dfb
import filterpype.data_filter as df
import filterpype.filter_utils as fut
import filterpype.filter_factory as ff
import filterpype.pipeline as ppln
import filterpype.ppln_demo as ppln_demo

k_run_long_tests = True

data_dir5 = os.path.join(fut.abs_dir_of_file(__file__), 
                         'test_data', 'tst_data5')

# Sample data 1 is in a media type with values:
# file header length = 11
# block size = 104 bytes
# including a 24-byte header

sample_data1 = '''\
file_header\
=====block_header01=====\
$13_567_9AB_DEF_\
ABC_EFG_IJK_MNP_\
XXX_XXX_XXX_XXX_\
0000000011111111\
$23_567_9AB_DEF_\
=====block_header02=====\
ABC_EFG_IJK_MNP_\
XXX_XXX_XXX_XXX_\
0000000022222222\
$33_567_9AB_DEF_\
ABC_EFG_IJK_MNP_\
=====block_header03=====\
XXX_XXX_XXX_XXX_\
0000000033333333\
$43_567_9AB_DEF_\
ABC_EFG_IJK_MNP_\
XXX_XXX_XXX_XXX_\
'''

class TestAttributeChangeDetection(unittest.TestCase):
    """ Test that the AttributeChangeDetection filter properly
assigns an attribute to a packet.
    """
    def setUp(self):
        pass
    
    def tearDown(self):
        pass
    
    def test_raises_attribute_error(self):
        """
        Change if filter starts handling exceptions on missing packet attributes.
        """
        fltr = df.AttributeChangeDetection(attributes=("test_attr",), 
                                           packet_change_flag="change_flag")
        pkt = dfb.DataPacket()
        # Originally AttributeError, due to the nature of exception handling
        # within FilterPypeFDS, this will now raise a FilterProcessingException.
        self.assertRaises(AttributeError, fltr.send, pkt)
        ##self.assertRaises(dfb.FilterProcessingException, fltr.send, pkt)
    
    def test_single_attribute(self):
        fltr = df.AttributeChangeDetection(attributes=("test_attr",), 
                                           packet_change_flag="change_flag")
        initial = 1
        changed_1 = 2
        changed_2 = 3
        initial_pkt = dfb.DataPacket(test_attr=initial)
        unchanged_pkt = dfb.DataPacket(test_attr=initial)
        changed_pkt = dfb.DataPacket(test_attr=changed_1)
        revert_pkt = dfb.DataPacket(test_attr=initial)
        last_pkt = dfb.DataPacket(test_attr=changed_2)
        fltr.send(initial_pkt)
        self.assertEqual(initial_pkt.change_flag, False)
        fltr.send(unchanged_pkt)
        self.assertEqual(unchanged_pkt.change_flag, False)
        fltr.send(changed_pkt)
        self.assertEqual(changed_pkt.change_flag, True)
        fltr.send(revert_pkt)
        self.assertEqual(revert_pkt.change_flag, False)
        fltr.send(last_pkt)
        self.assertEqual(last_pkt.change_flag, True)
        
    def test_multiple_attributes(self):
        fltr = df.AttributeChangeDetection(attributes=("test_attr_1",
                                                       "test_attr_2"), 
                                           packet_change_flag="change_flag")
        test_attr_1_initial = "a"
        test_attr_1_changed = "b"
        test_attr_2_initial = 2.5
        test_attr_2_changed = 3.5
        
        initial_pkt = dfb.DataPacket(test_attr_1=test_attr_1_initial,
                                     test_attr_2=test_attr_2_initial)
        unchanged_pkt = dfb.DataPacket(test_attr_1=test_attr_1_initial,
                                       test_attr_2=test_attr_2_initial)
        changed_pkt_1 = dfb.DataPacket(test_attr_1=test_attr_1_initial,
                                       test_attr_2=test_attr_2_changed)
        revert_pkt_1 = dfb.DataPacket(test_attr_1=test_attr_1_initial,
                                      test_attr_2=test_attr_2_initial)
        changed_pkt_2 = dfb.DataPacket(test_attr_1=test_attr_1_changed,
                                       test_attr_2=test_attr_2_initial)
        revert_pkt_2 = dfb.DataPacket(test_attr_1=test_attr_1_initial,
                                      test_attr_2=test_attr_2_initial)
        changed_pkt_3 = dfb.DataPacket(test_attr_1=test_attr_1_changed,
                                       test_attr_2=test_attr_2_changed)
        fltr.send(initial_pkt)
        self.assertEqual(initial_pkt.change_flag, False)
        fltr.send(unchanged_pkt)
        self.assertEqual(unchanged_pkt.change_flag, False)
        fltr.send(changed_pkt_1)
        self.assertEqual(changed_pkt_1.change_flag, True)
        fltr.send(revert_pkt_1)
        self.assertEqual(revert_pkt_1.change_flag, False)
        fltr.send(changed_pkt_2)
        self.assertEqual(changed_pkt_2.change_flag, True)
        fltr.send(revert_pkt_2)
        self.assertEqual(revert_pkt_2.change_flag, False)
        fltr.send(changed_pkt_3)
        self.assertEqual(changed_pkt_3.change_flag, True)
        
class TestInitialValueChangeDetection(unittest.TestCase):
    """.."""
    def setUp(self):
        pass
    def tearDown(self):
        pass
    
    def test_single_attr(self):
        fltr = df.InitialValueChangeDetection(attribute_map={"test_attr_1":5}, 
                                              packet_change_flag="change_flag")
        pkt_1 = dfb.DataPacket(test_attr_1=5)
        fltr.send(pkt_1)
        self.assertFalse(pkt_1.change_flag)
        pkt_2 = dfb.DataPacket(test_attr_1=1)
        fltr.send(pkt_2)
        self.assertTrue(pkt_2.change_flag)
        pkt_3 = dfb.DataPacket(test_attr_1=5)
        fltr.send(pkt_3)
        #self.assertTrue(pkt_3.change_flag)
        pkt_4 = dfb.DataPacket(test_attr_1=1)
        fltr.send(pkt_4)
        self.assertTrue(pkt_4.change_flag)
        pkt_5 = dfb.DataPacket(test_attr_1=1)
        fltr.send(pkt_5)
        self.assertTrue(pkt_5.change_flag)
        
        
        

class TestAttributeExtractor(unittest.TestCase):
    """ Test the extraction of attributes from text strings based on a
    delmiter
    """
    def setUp(self):
        self.att_equ = df.AttributeExtractor(delimiter='equals')
        self.att_col = df.AttributeExtractor(delimiter='colon')
        self.att_tab = df.AttributeExtractor(delimiter='tabular')
        # whitespace
        self.att_whi = df.AttributeExtractor()
        
        self.sink = df.Sink()
        
        self.att_equ.next_filter = self.sink
        self.att_col.next_filter = self.sink
        self.att_tab.next_filter = self.sink
        self.att_whi.next_filter = self.sink


    def tearDown(self):
        pass
    
    def test_extract_attributes(self):
        
        # where are we applying the attributes? to the packet or the filter?
        
        data1 = dfb.DataPacket("pat's name=pat")
        data2 = dfb.DataPacket("  His Gender : mail")
        data3 = dfb.DataPacket(" job\tpostMan ")
        
        self.att_equ.send(data1)
        self.att_col.send(data2)
        self.att_tab.send(data3)
        self.att_whi.send(data3)

        # no longer storing the attributes on the filter, only the packet
        ##self.assertEqual(self.att_equ.pats_name, 'pat')
        ##self.assertEqual(self.att_col.his_gender, 'mail')
        ##self.assertEqual(self.att_tab.job, 'postMan')
        ##self.assertEqual(self.att_whi.job, 'postMan')
        
        # check the packet for the results
        pkts = self.sink.results
        self.assertEqual(pkts[0].pats_name, 'pat')
        self.assertEqual(pkts[1].his_gender, 'mail')
        self.assertEqual(pkts[2].job, 'postMan')
        self.assertEqual(pkts[3].job, 'postMan')
        
        

class TestBatch(unittest.TestCase):

    def setUp(self):
        self.sink = df.Sink()

    def tearDown(self):
        pass
    
    def test_flush_buffer_no_packets(self):
        """Should not raise exception."""
        batch_filter = df.Batch(size=8192)
        batch_filter.flush_buffer()
    
    def test_flush_buffer_after_sending_empty_packet(self):
        """Should not raise exception."""
        empty_packet = dfb.DataPacket()
        batch_filter = df.Batch(size=8192)
        batch_filter.next_filter = self.sink
        batch_filter.send(empty_packet)
        batch_filter.flush_buffer()
        
    def test_flush_buffer_after_sending_packet_1(self):
        """Should not raise exception."""
        packet = dfb.DataPacket(240 * '\x00')
        batch_filter = df.Batch(size=8192)
        batch_filter.next_filter = self.sink
        batch_filter.send(packet)
        batch_filter.flush_buffer()
        self.assertEqual(len(self.sink.all_data), 1)
        
    def test_flush_buffer_after_sending_packet_2(self):
        """Should not raise exception."""
        packet = dfb.DataPacket(20240 * '\x00')
        batch_filter = df.Batch(size=8192)
        batch_filter.next_filter = self.sink
        batch_filter.send(packet)
        batch_filter.flush_buffer()
        self.assertEqual(len(self.sink.all_data), 3)
        
    def test_flush_buffer_after_sending_packets(self):
        pass
        
        

    def test_batch_continuous1(self):
        self.assertRaises(dfb.FilterAttributeError, df.Batch)
        block_split1 = df.Batch(size=90)
        block_split1.next_filter = self.sink
        packet1 = dfb.DataPacket('abcdef')
        block_split1.send(packet1)
        block_split1.shut_down()
        self.assertEquals(self.sink.results[0].data, 'abcdef')

        self.sink2 = df.Sink()
        block_split2 = df.Batch(size=4)
        block_split2.next_filter = self.sink2
        packet2 = dfb.DataPacket('abcdef')
        block_split2.send(packet2)
        block_split2.shut_down()
        self.assertEquals(self.sink2.results[-2].data, 'abcd')
        self.assertEquals(self.sink2.results[-1].data, 'ef')

    def test_batch_continuous2(self):
        block_split4 = df.Batch(size=3)
        block_split4.next_filter = self.sink
        packet1 = dfb.DataPacket('abcdefg')
        packet2 = dfb.DataPacket('1234')
        block_split4.send(packet1)
        block_split4.send(packet2)
        block_split4.shut_down()
        self.assertEquals(self.sink.results[-4].data, 'abc')
        self.assertEquals(self.sink.results[-3].data, 'def')
        self.assertEquals(self.sink.results[-2].data, 'g12')
        self.assertEquals(self.sink.results[-1].data, '34')

    def test_batch_initial_branch1(self):
        return  # Replace this initial_branch test <<<<<<< TO-DO
        config = '''\
        
        '''
        block_split2 = df.Batch(size=4, initial_branch_size=3)
        hidden_branch_route = dfb.HiddenBranchRoute()
        block_split2.next_filter = hidden_branch_route
        main_sink = df.Sink()
        branch_sink = df.Sink()
        hidden_branch_route.next_filter = main_sink
        hidden_branch_route.branch_filter = branch_sink
        packet2 = dfb.DataPacket('xxx135713571357')
        block_split2.send(packet2)
        block_split2.shut_down()
        self.assertEquals(branch_sink.results[0].data, 'xxx')
        self.assertEquals('|'.join(main_sink.all_data), '1357|1357|1357')

    def test_batch_initial_branch2(self):
        return  # Replace this initial_branch test <<<<<<< TO-DO
        block_split2 = df.Batch(size=2, initial_branch_size=5)
        hidden_branch_route = dfb.HiddenBranchRoute()
        block_split2.next_filter = hidden_branch_route
        main_sink = df.Sink()
        branch_sink = df.Sink()
        hidden_branch_route.next_filter = main_sink
        hidden_branch_route.branch_filter = branch_sink
        packet2 = dfb.DataPacket('xxxxx135713571357')
        block_split2.send(packet2)
        block_split2.shut_down()
        self.assertEquals(branch_sink.results[0].data, 'xxxxx')
        self.assertEquals('|'.join(main_sink.all_data), '13|57|13|57|13|57')
        self.assertEquals(len(main_sink.results), 6)

    def test_batch_initial_branch3(self):
        return  # Replace this initial_branch test <<<<<<< TO-DO
        block_split3 = df.Batch(size=2, initial_branch_size=6)
        hidden_branch_route = dfb.HiddenBranchRoute()
        block_split3.next_filter = hidden_branch_route
        main_sink = df.Sink()
        branch_sink = df.Sink()
        hidden_branch_route.next_filter = main_sink
        hidden_branch_route.branch_filter = branch_sink
        packet3 = dfb.DataPacket('branch12345')
        block_split3.send(packet3)
        block_split3.shut_down()
        self.assertEquals(branch_sink.results[0].data, 'branch')
        self.assertEquals('|'.join(main_sink.all_data), '12|34|5')
        self.assertEquals(len(main_sink.results), 3)
        self.assertEquals(main_sink.results[0].data, '12')
        self.assertEquals(main_sink.results[1].data, '34')
        self.assertEquals(main_sink.results[2].data, '5')
        
    def test_zero_batch_size(self):
        def make_batch(size):
            return df.Batch(size=size)
        # Bad batch size
        self.assertRaises(dfb.FilterAttributeError, make_batch, 0)

    def test_batch_batch1(self):
        return # <<<< No use case for this found
        block_splitter = df.Batch(size=3)
        block_splitter.next_filter = self.sink
        packet1 = dfb.DataPacket('123456789')
        block_splitter.send(packet1)
        #print '**2060** [0] =', self.sink.results[0].data
        #print '**2060** [1] =', self.sink.results[1].data
        #print '**2060** [2] =', self.sink.results[2].data
        self.assertEquals(self.sink.results[0].data, '123')
        self.assertEquals(self.sink.results[1].data, '456')
        self.assertEquals(self.sink.results[2].data, '789')

        packet2 = dfb.DataPacket('abcde')
        block_splitter.send(packet2)
        block_splitter.shut_down()
        self.assertEquals(self.sink.results[-3].data, '789')
        self.assertEquals(self.sink.results[-2].data, 'abc')
        self.assertEquals(self.sink.results[-1].data, 'de')

    def test_batch_batch2(self):
        return # <<<< No use case for this found
        block_splitter = df.Batch(size=3, stream_mode='batch')
        block_splitter.next_filter = self.sink
        packet1 = dfb.DataPacket('1234567890')
        block_splitter.send(packet1)
        #print '**2060** [0] =', self.sink.results[0].data
        #print '**2060** [1] =', self.sink.results[1].data
        #print '**2060** [2] =', self.sink.results[2].data
        self.assertEquals(self.sink.results[0].data, '123')
        self.assertEquals(self.sink.results[1].data, '456')
        self.assertEquals(self.sink.results[2].data, '789')
        self.assertEquals(self.sink.results[3].data, '0')

        packet2 = dfb.DataPacket('abcde')
        block_splitter.send(packet2)
        block_splitter.shut_down()
        self.assertEquals(self.sink.results[-3].data, '0')
        self.assertEquals(self.sink.results[-2].data, 'abc')
        self.assertEquals(self.sink.results[-1].data, 'de')


class TestBranchClone(unittest.TestCase):

    def setUp(self):
        self.sink_main = df.Sink()
        self.sink_branch = df.Sink()
        self.branch_clone = df.BranchClone()
        self.hidden_branch_route = dfb.HiddenBranchRoute()
        self.branch_clone.next_filter = self.hidden_branch_route
        self.hidden_branch_route.next_filter = self.sink_main
        self.hidden_branch_route.branch_filter = self.sink_branch    
        
    def tearDown(self):
        pass

    def test_branch_clone(self):
        packet1 = dfb.DataPacket(data='abcdef')
        self.branch_clone.send(packet1)
        self.assertEquals(self.sink_main.results[-1].data, 'abcdef')
        self.assertEquals(self.sink_branch.results[-1].data, 'abcdef')

        
class TestBranchFirstPart(unittest.TestCase):

    def setUp(self):
        self.sink_main = df.Sink()
        self.sink_branch = df.Sink()
        self.branch_first_part = df.BranchFirstPart()
        self.hidden_branch_route = dfb.HiddenBranchRoute()
        self.branch_first_part.next_filter = self.hidden_branch_route
        self.hidden_branch_route.next_filter = self.sink_main
        self.hidden_branch_route.branch_filter = self.sink_branch    
        
    def tearDown(self):
        pass

    def test_branch_first_part(self):
        packet1 = dfb.DataPacket(data='abcdef', branch_up_to=2)
        self.branch_first_part.send(packet1)
        self.assertEquals(self.sink_branch.results[-1].data, 'ab')
        self.assertEquals(self.sink_main.results[-1].data, 'cdef')

    def test_branch_nothing(self):
        packet1 = dfb.DataPacket(data='abcdef', branch_up_to=0)
        self.branch_first_part.send(packet1)
        self.assertEquals(len(self.sink_branch.results), 0)
        self.assertEquals(self.sink_main.results[-1].data, 'abcdef')

    def test_branch_no_data(self):
        packet1 = dfb.DataPacket(data='', branch_up_to=3)
        self.branch_first_part.send(packet1)
        self.assertEquals(len(self.sink_branch.results), 0)
        self.assertEquals(self.sink_main.results[-1].data, '')

    def test_branch_all(self):
        packet1 = dfb.DataPacket(data='abcdef', branch_up_to=6)
        self.branch_first_part.send(packet1)
        self.assertEquals(self.sink_branch.results[-1].data, 'abcdef')
        self.assertEquals(self.sink_main.results[-1].data, '')

    def test_branch_too_many(self):
        packet1 = dfb.DataPacket(data='abcdef', branch_up_to=22)
        self.branch_first_part.send(packet1)
        self.assertEquals(self.sink_branch.results[-1].data, 'abcdef')
        self.assertEquals(self.sink_main.results[-1].data, '')

        
class TestBranchIf(unittest.TestCase):

    def setUp(self):
        self.hidden_branch_route = dfb.HiddenBranchRoute()
        self.sink_main = df.Sink()
        self.sink_branch = df.Sink()
        self.hidden_branch_route.next_filter = self.sink_main
        self.hidden_branch_route.branch_filter = self.sink_branch    
        
    def tearDown(self):
        pass

    def test_branch_if_no_param(self):
        packet1 = dfb.DataPacket(data='1234')
        branch_if = df.BranchIf(branch_key='strike_rate_high',
                                comparison='equals',
                                compare_value=True,
                                branch_on_packet=False)
        self.assertRaises(AttributeError, branch_if.send, packet1)
        
    def test_branch_if_on_filter(self):
        branch_if = df.BranchIf(branch_key='strike_rate_high',
                                comparison='equals',
                                compare_value=True,
                                branch_on_packet=False)
        branch_if.next_filter = self.hidden_branch_route
        packet1 = dfb.DataPacket(data='5678')
        branch_if.strike_rate_high = True
        branch_if.send(packet1)
        self.assertEquals(len(self.sink_main.results), 0)
        self.assertEquals(len(self.sink_branch.results), 1)
        branch_if.strike_rate_high = False
        packet2 = dfb.DataPacket(data='998877')
        packet3 = dfb.DataPacket(data='665544')
        branch_if.send(packet2)
        branch_if.send(packet3)
        self.assertEquals(len(self.sink_main.results), 2)
        self.assertEquals(len(self.sink_branch.results), 1)

    def test_branch_if_on_data(self):
        branch_if = df.BranchIf(branch_key='over_flange',
                                comparison='equals',
                                compare_value=True,
                                branch_on_packet=True)
        branch_if.next_filter = self.hidden_branch_route
        packet1 = dfb.DataPacket(data='5678')
        packet1.over_flange = True
        branch_if.send(packet1)
        self.assertEquals(len(self.sink_main.results), 0)
        self.assertEquals(len(self.sink_branch.results), 1)
        packet2 = dfb.DataPacket(data='998877')
        packet2.over_flange = False
        packet3 = dfb.DataPacket(data='665544')
        packet3.over_flange = True
        branch_if.send(packet2)
        branch_if.send(packet3)
        self.assertEquals(len(self.sink_main.results), 1)
        self.assertEquals(len(self.sink_branch.results), 2)
        
    def test_branch_if_on_data_assume_equals_true1(self):
        branch_if = df.BranchIf(branch_key='over_flange')
        branch_if.next_filter = self.hidden_branch_route
        packet1 = dfb.DataPacket(data='5566')
        packet1.over_flange = True
        branch_if.send(packet1)
        self.assertEquals(len(self.sink_main.results), 0)
        self.assertEquals(len(self.sink_branch.results), 1)
        
    def test_branch_if_on_data_assume_equals_true2(self):
        branch_if = df.BranchIf(branch_key='over_flange')
        branch_if.next_filter = self.hidden_branch_route
        packet1 = dfb.DataPacket(data='5566')
        packet1.over_flange = True
        branch_if.send(packet1)
        packet2 = dfb.DataPacket(data='531531')
        packet2.over_flange = False
        packet3 = dfb.DataPacket(data='642642')
        packet3.over_flange = True
        branch_if.send(packet2)
        branch_if.send(packet3)
        self.assertEquals(len(self.sink_main.results), 1)
        self.assertEquals(len(self.sink_branch.results), 2)
        
    def test_branch_if_branch_doesnt_exist(self):  # TO-DO <<< 
        # Martin's route needed a (waste) filter to work
        pass

    
class TestBranchOnceTriggered(unittest.TestCase):
    
    def setUp(self):
        self.fltr = df.BranchOnceTriggered(watch_attribute="data",
                                           watch_value="b")
        self.hidden_branch_route = dfb.HiddenBranchRoute()
        self.sink_main = df.Sink()
        self.sink_branch = df.Sink()
        self.fltr.next_filter = self.hidden_branch_route
        self.hidden_branch_route.next_filter = self.sink_main
        self.hidden_branch_route.branch_filter = self.sink_branch   
    
    def tearDown(self):
        pass
    
    def test_branch_once_triggered(self):
        # Expecting 2 down the main and all other packets down the branch.
        for packet_data in ["a", "a", "b", "c", "d", "a"]:
            curr_packet = dfb.DataPacket(data=packet_data)
            self.fltr.send(curr_packet)
        self.assertEqual(len(self.sink_branch.results), 4)
        self.assertEqual(len(self.sink_main.results), 2)


class TestBranchRef(unittest.TestCase):

    def setUp(self):
        # Construct branching pipeline by hand
        self.sink_main = df.Sink()
        self.sink_branch = df.Sink()
        self.branch_ref = df.BranchRef()
        self.hidden_branch_route = dfb.HiddenBranchRoute()
        self.branch_ref.next_filter = self.hidden_branch_route
        self.hidden_branch_route.next_filter = self.sink_main
        self.hidden_branch_route.branch_filter = self.sink_branch    
        
    def tearDown(self):
        pass

    def test_branch_ref(self):
        packet1 = dfb.DataPacket(data='abcdef')
        self.branch_ref.send(packet1)
        self.assertEquals(self.sink_main.results[-1].data, 'abcdef')
        self.assertEquals(self.sink_branch.results[-1].data, 'abcdef')
        self.assertEquals(self.sink_main.results[-1], 
                          self.sink_branch.results[-1])

        
class TestBranchRoute(unittest.TestCase):

    def setUp(self):
        self.hidden_branch_route = dfb.HiddenBranchRoute()
        self.sink = df.Sink()
        self.hidden_branch_route.next_filter = self.sink

    def tearDown(self):
        pass

    def test_branch_route(self):
        packet = dfb.DataPacket(data='abcdef')
        self.assertRaises(dfb.FilterRoutingError, 
                          self.hidden_branch_route.send_on,
                          packet, 'xxx')
        
        
class TestBZip2Filter(unittest.TestCase):

    def setUp(self):
        self.bzip_compressor = df.BZipCompress()
        self.bzip_decompressor = df.BZipDecompress()
        self.sink = df.Sink()

    def tearDown(self):
        pass

    def test_bzip_compress_decompress(self):
        self.bzip_compressor.next_filter = self.bzip_decompressor
        self.bzip_decompressor.next_filter = self.sink
        input1 = 'abcdef'
        # Data will be compressed and decompressed
        self.bzip_compressor.send(dfb.DataPacket(input1))
        # Flush data from compression buffer
        self.bzip_compressor.shut_down()
        self.assertEquals('|'.join(self.sink.all_data), '|' + input1)

    def test_double_bzip_compress_decompress(self):
        bzip_compressor2 = df.BZipCompress()
        bzip_decompressor2 = df.BZipDecompress()

        self.bzip_compressor.next_filter = bzip_compressor2
        bzip_compressor2.next_filter = self.bzip_decompressor
        self.bzip_decompressor.next_filter = bzip_decompressor2
        bzip_decompressor2.next_filter = self.sink

        input2 = '123456789'
        # Data will be compressed and decompressed
        self.bzip_compressor.send(dfb.DataPacket(input2))
        # Flush data from compression buffer
        self.bzip_compressor.shut_down()
        self.assertEquals('|'.join(self.sink.all_data), '||' + input2)  # TO-DO why ||??
    
    def test_bzip_compress_to_file(self):
        dest_file_name = os.path.join(data_dir5, 'compressed.tmp')
        write_file = df.WriteFile(dest_file_name=dest_file_name)
        self.bzip_compressor.next_filter = write_file
        input3 = 'abcdef'
        self.bzip_compressor.send(dfb.DataPacket(input3))
        input4 = 'efghc'
        self.bzip_compressor.send(dfb.DataPacket(input4))
        self.bzip_compressor.shut_down()
        out_file_handle = bz2.BZ2File(dest_file_name, 'r')
        self.assertEqual(out_file_handle.read(), input3 + input4)
        out_file_handle.close()
        os.remove(dest_file_name)

        
class TestCalcSlope(unittest.TestCase):
    
    def setUp(self):
        packets = [dfb.DataPacket(height=1),
                   dfb.DataPacket(height=2),
                   dfb.DataPacket(height=3),
                   dfb.DataPacket(height=4),
                   dfb.DataPacket(height=5),
                   dfb.DataPacket(height=6)]
        self.group_pkt1 = dfb.DataPacket(packets[:5])
        self.group_pkt2 = dfb.DataPacket(packets[1:])
        self.group_pkt_bad = dfb.DataPacket(packets[3:])
        self.group_pkt_same = dfb.DataPacket([dfb.DataPacket(height=2)
                                              for j in xrange(5)])
        self.slope_calc = df.CalcSlope(calc_source_name='height')
        
    def tearDown(self):
        pass
    
    def test_calc_slope(self):
        self.slope_calc.send(self.group_pkt1)
        self.assertEquals(self.group_pkt1.data[2].height_slope, 1.0)
        self.slope_calc.send(self.group_pkt2)
        self.assertEquals(self.group_pkt2.data[2].height_slope, 1.0)
        
    def test_calc_slope_not_five_values(self):
        def get_slope():
            return self.group_pkt_bad.data[2].height_slope
        self.slope_calc.send(self.group_pkt_bad)
        self.assertRaises(AttributeError, get_slope)
        
    def test_calc_no_slope(self):
        self.slope_calc.send(self.group_pkt_same)
        self.assertEquals(self.group_pkt_same.data[2].height_slope, 0.0)

    
class TestCalculate(unittest.TestCase):
    
    def setUp(self):
        self.lhs_value = 3
        self.rhs_value = 7
        self.sink = df.Sink()
        self.packet = dfb.DataPacket(data='')
        
    def tearDown(self):
        pass
    
    def test_add(self):
        calc = df.Calculate(lhs_value=self.lhs_value, operator='add',
                            rhs_value=self.rhs_value, param_result='ADDED')
        calc.next_filter = self.sink
        calc.send(self.packet)
        self.assertEquals(self.sink.results[-1].ADDED, 10)
    
    def test_subtract(self):
        calc = df.Calculate(lhs_value=self.lhs_value, operator='subtract',
                            rhs_value=self.rhs_value, param_result='SUBTRACTED')
        calc.next_filter = self.sink
        calc.send(self.packet)
        self.assertEquals(self.sink.results[-1].SUBTRACTED, -4)
    
    def test_multiply(self):
        calc = df.Calculate(lhs_value=self.lhs_value, operator='multiply',
                            rhs_value=self.rhs_value, param_result='MULTIPLIED')
        calc.next_filter = self.sink
        calc.send(self.packet)
        self.assertEquals(self.sink.results[-1].MULTIPLIED, 21)
    
    def test_divide(self):
        calc = df.Calculate(lhs_value=self.lhs_value, operator='divide',
                            rhs_value=self.rhs_value, param_result='DIVIDED')
        calc.next_filter = self.sink
        calc.send(self.packet)
        self.assertEquals(self.sink.results[-1].DIVIDED, 0.42857142857142855)

    def test_divide_by_zero_raises(self):
        calc = df.Calculate(lhs_value=0, operator='divide',
                            rhs_value=0, param_result='DIVIDED')
        calc.next_filter = self.sink
        self.assertRaises(dfb.FilterLogicError, calc.send, self.packet)

    def test_raises_when_given_non_numerics(self):
        calc = df.Calculate(lhs_value='a', operator='divide',
                            rhs_value=1, param_result='DIVIDED')
        calc.next_filter = self.sink
        self.assertRaises(dfb.FilterAttributeError, calc.send, self.packet)
        
        
class TestCallbackOnAttribute(unittest.TestCase):
    
    def setUp(self):
        self.mocked_method = mock.Mock()
        self.mocked_method_env = mock.Mock()
        self.mocked_method_twice = mock.Mock()
        self.callbacker = df.CallbackOnAttribute(
            watch_attr='holy',
            callback=self.mocked_method,
        )

        self.caller_env = df.CallbackOnAttribute(
            watch_attr='gone',
            callback=self.mocked_method_env,
            environ=dict(home='sweet home'),
        )
        
        self.caller_twice = df.CallbackOnAttribute(
            watch_attr='monty',
            callback=self.mocked_method_twice,
            num_watch_pkts=2,
        )
        
        self.caller_count_to_conf = df.CallbackOnAttribute(
            watch_attr='watch',
            callback=self.mocked_method,
            count_to_confirm=3,
            num_watch_pkts=3,
        )
        
    def tearDown(self):
        pass
    
    def test_callback_not_found_on_close(self):
        packet = dfb.DataPacket()
        callback = mock.Mock()
        callbacker = df.CallbackOnAttribute(watch_attr='not here',
                                            callback=callback,
                                            close_when_found=False)
        for counter in xrange(100):
            callbacker.send(packet)
                
        callbacker.shut_down()
        self.assertEquals(len(callback.call_args_list), 1)
        self.assertEquals(callback.call_args, (('not_found:not here',), {}))
    
    def test_callback_not_called_when_closing_after_found(self):
        """ test_callback_not_called_when_closing_after_found
        
            Test that after an attribute has been found, the not_found does not
            get called when closing down the pipeline.
        """
        packet = dfb.DataPacket()
        packet2 = dfb.DataPacket()
        packet2.not_here = 'meep'
        callback = mock.Mock()
        callbacker = df.CallbackOnAttribute(watch_attr='not_here',
                                            callback=callback,
                                            close_when_found=False)
        for counter in xrange(100):
            if counter == 50:
                callbacker.send(packet2)
            else:
                callbacker.send(packet)
                
        callbacker.shut_down()
        self.assertEquals(len(callback.call_args_list), 1)
        self.assertEquals(callback.call_args, (('found:not_here',),
                                               {'not_here': 'meep'}))       
    
    def test_close_when_found(self):
        """ test_close_when_found
            
            Test that the pipeline closes down when attribute is found
        """
        # This requires a pipeline in order to work
        config = '''
        [--main--]
        keys=callback
        ftype=aaa
        description=bbb
        
        [callback_on_attribute]
        watch_attr = not_here
        callback = ${callback}
        close_when_found = true
        [--route--]
        callback_on_attribute >>>
        pass_through
        '''
        
        factory = ff.DemoFilterFactory()
        sink = df.Sink(max_results=100)
        packet = dfb.DataPacket()
        packet2 = dfb.DataPacket()
        packet2.not_here = 'meep'
        callback = mock.Mock()
        pipeline = ppln.Pipeline(factory=factory,
                                 config=config, callback=callback)
        pipeline.next_filter = sink
        #callbacker = df.CallbackOnAttribute(watch_attr='not_here',
                                            #callback=callback,
                                            #close_when_found=True)
        #callbacker.next_filter = sink
        for counter in xrange(100):
            if counter == 50 and not pipeline.shutting_down:  
                pipeline.send(packet2)
            elif not pipeline.shutting_down: 
                pipeline.send(packet)
                
        self.assertEquals(len(sink.results), 51)

    def test_close_when_found_does_not_close(self):
        """ test_close_when_found
            
            Test that the pipeline does not close down when attr is found
        """
        # This requires a pipeline in order to work
        config = '''
        [--main--]
        keys=callback
        ftype=aaa
        description=bbb
        
        [callback_on_attribute]
        watch_attr = not_here
        callback = ${callback}
        close_when_found = false
        [--route--]
        callback_on_attribute >>>
        pass_through
        '''
        
        factory = ff.DemoFilterFactory()
        sink = df.Sink(max_results=100)
        packet = dfb.DataPacket()
        packet2 = dfb.DataPacket()
        packet2.not_here = 'meep'
        callback = mock.Mock()
        pipeline = ppln.Pipeline(factory=factory,
                                 config=config, callback=callback)
        pipeline.next_filter = sink
        #callbacker = df.CallbackOnAttribute(watch_attr='not_here',
                                            #callback=callback,
                                            #close_when_found=True)
        #callbacker.next_filter = sink
        for counter in xrange(100):
            if counter == 50 and not pipeline.shutting_down:
                pipeline.send(packet2)
            elif not pipeline.shutting_down:  # TO-DO pipeline.refinery.shutting_down
                pipeline.send(packet)
                
        self.assertEquals(len(sink.results), 100)
    
    def test_callback_using_msg_bottle(self):
        msg_bottle = dfb.MessageBottle('callback_on_attribute', 'no_message',
                                       some_param='something')
        self.callbk_on_msg = df.CallbackOnAttribute(
            watch_attr='some_param',
            callback=self.mocked_method,
            )
        self.callbk_on_msg.send(msg_bottle)
        self.assertEqual(self.mocked_method.call_count, 1)        
    
    def test_callback(self):
        packet = dfb.DataPacket()
        packet.holy = "grail"
        self.callbacker.send(packet)
        self.assertEqual(self.mocked_method.call_args, 
                         ( ("found:holy",), dict(holy='grail') ) )
        empty_packet = dfb.DataPacket()
        self.callbacker.send(empty_packet)
        self.assertEqual(self.mocked_method.call_count, 1)
        
        gone_packet = dfb.DataPacket()
        gone_packet.gone = 'fishing'
        self.caller_env.send(gone_packet)
        self.assertEqual(
            self.mocked_method_env.call_args,
            ( ("found:gone",), dict(gone='fishing',
                              home='sweet home') ) 
        )
    def test_callback_not_found(self):
        # can't find the attribute monty
        packet = dfb.DataPacket()
        self.caller_twice.send(packet)
        self.caller_twice.send(packet)
        self.assertEqual(self.mocked_method_twice.call_args, 
                         ( ("not_found:monty",), {} ) )
        self.caller_twice.send(packet)
        # ensure that it still only gets called the once
        self.assertEqual(self.mocked_method_twice.call_count, 1)
       
    def test_callback_count_to_confirm_unsatisfied(self):
        packet1 = dfb.DataPacket()
        packet1.watch = 'out'
        packet2 = packet1.clone()
        packet3 = packet1.clone()
        self.caller_count_to_conf.send(packet1)
        self.caller_count_to_conf.send(packet2)
        self.caller_count_to_conf.send(packet3)
        self.assertEqual(self.mocked_method.call_count, 1)
        self.assertEqual(self.mocked_method.call_args, 
                         ( ("found:watch",), dict(watch='out') ) 
                         )
    
    #def test_
    
    def test_callback_count_to_confirm_satisfied(self):
        ##packet1 = dfb.DataPacket()
        ##packet1.watch = 'out'
        # Create packet attributes using kwargs directly
        packet1 = dfb.DataPacket(watch='out')
        packet2 = packet1.clone()
        packet3 = dfb.DataPacket(watch='')
        packet4 = dfb.DataPacket(watch='out')
        packet5 = dfb.DataPacket(watch='out')
        self.caller_count_to_conf.send(packet1)
        self.caller_count_to_conf.send(packet2)
        self.caller_count_to_conf.send(packet3)
        self.caller_count_to_conf.send(packet4)
        self.caller_count_to_conf.send(packet5)
        # All of the packets have a watch and the required number for a callback
        # is 3. This means that the packet count with the watch attribute will
        # be 3 or more TWICE. Is it supposed to reset to zero when the required
        # number are found?? This assertion would suggest it should.
        self.assertEqual(self.mocked_method.call_count, 2)
        self.assertEqual(self.mocked_method.call_args, 
                         ( ("not_found:watch",), {'values_found': ['', 'out'],
                                                  'watch': ''} ) 
                         )
    def test_callback_on_attribute_change(self):
        # watch for the attribute in the packet to change value
        # this includes when the packet is set initially
        caller_watch_for_change = df.CallbackOnAttribute(
            watch_attr='percent_read',
            callback=self.mocked_method,
            ##count_to_confirm = 3,
            ##num_watch_pkts = 3,
            watch_for_change = True,
            include_in_environ = ['another_param', 'some_param']
        )
        
        # define packets
        pkt1 = dfb.DataPacket(percent_read=0, another_param='another')
        pkt2 = dfb.DataPacket(percent_read=1)
        pkt3 = dfb.DataPacket(percent_read=99, some_param='some')
        
        caller_watch_for_change.send(pkt1)
        self.assertEqual(self.mocked_method.call_count, 1)
        self.assertEqual(self.mocked_method.call_args,
                         ( ("found:percent_read",), {'percent_read':0, 
                                                     'another_param':'another'} ) )
        # send the same attr value and there should be no change
        caller_watch_for_change.send(pkt1)
        caller_watch_for_change.send(pkt1)
        self.assertEqual(self.mocked_method.call_count, 1)
        # send a new value and it should respond
        caller_watch_for_change.send(pkt2)
        self.assertEqual(self.mocked_method.call_count, 2)
        self.assertEqual(self.mocked_method.call_args,
                         ( ("found:percent_read",), {'percent_read':1,
                                                     'another_param':'another'} ) )
        # another new value straight away
        caller_watch_for_change.send(pkt3)
        self.assertEqual(self.mocked_method.call_args,
                         ( ("found:percent_read",), {'percent_read':99,
                                                     'another_param':'another',
                                                     'some_param':'some'} ) )
        
    # this turned out not to have the desired effect
    ##def test_callback_on_attribute_change_with_count_to_confirm(self):
        ### watch for the attribute in the packet to change value
        ### this includes when the packet is set initially
        ##caller_watch_for_change = df.CallbackOnAttribute(
            ##watch_attr='percent_read',
            ##callback=self.mocked_method,
            ##count_to_confirm = 2,
            ##num_watch_pkts = 5,
            ##watch_for_change = True,
        ##)
        
        ### define packets
        ##pkt1 = dfb.DataPacket(percent_read=0)
        ##pkt1 = dfb.DataPacket(percent_read=1)
        ##pkt1 = dfb.DataPacket(percent_read=2)
        
        ##caller_watch_for_change.send(pkt1)
        ##self.assertEqual(self.mocked_method.call_count, 0)
        ##caller_watch_for_change.send(pkt1)
        ##self.assertEqual(self.mocked_method.call_count, 1)
        ##self.assertEqual(self.mocked_method.call_args,
                         ##( ("found:percent_read",), {'percent_read':0} ) )
        ### send the same attr value and there should be no change
        ##caller_watch_for_change.send(pkt1)
        ##caller_watch_for_change.send(pkt1)
        ##self.assertEqual(self.mocked_method.call_count, 1)
        ### send a new value and it should respond after two the same have arrived
        ##caller_watch_for_change.send(pkt2)
        ##caller_watch_for_change.send(pkt1)
        ##caller_watch_for_change.send(pkt2)
        ##self.assertEqual(self.mocked_method.call_count, 1)
        ##caller_watch_for_change.send(pkt2)
        ##self.assertEqual(self.mocked_method.call_count, 2)
        ##self.assertEqual(self.mocked_method.call_args,
                         ##( ("found:percent_read",), {'percent_read':1} ) )

                
        
class TestCombine(unittest.TestCase):
    
    def setUp(self):
        self.combiner = df.Combine(source_field_names=['aa', 'bb', 'cc'],
                                   target_field_name='p.temp')
        self.sink = df.Sink()
        self.combiner.next_filter = self.sink
    
    def tearDown(self):
        pass

    def test_combine_constants(self):
        packet = dfb.DataPacket()
        self.combiner.send(packet)
        self.assertEquals(self.sink.results[-1].temp, 'aabbcc')
        
    def test_combine_attributes(self):
        self.combiner.source_field_names=['f.foo', ', ', 'f.bar', ' ', 'p.baz']
        self.combiner.foo = 'Hello'
        self.combiner.bar = 'Python'
        packet = dfb.DataPacket()
        packet.baz = 'World'
        self.combiner.send(packet)
        self.assertEquals(self.sink.results[-1].temp, 'Hello, Python World')
        
class TestConvertToInt(unittest.TestCase):
    
    def setUp(self):
        self.sink = df.Sink()
    
    def tearDown(self):
        pass
    
    def test_convert_hex_chars_to_int(self):
        data_input = '\x00\x81'
        
        packet = dfb.DataPacket()
        packet.THIS_VALUE = data_input
        # The filter
        converter = df.ConvertBytesToInt(param_names='THIS_VALUE')
        converter.next_filter = self.sink
        converter.send(packet)
        converter.shut_down()
        
        self.assertEquals(self.sink.results[-1].THIS_VALUE, 129)

    def test_raises_FilterAttributeError(self):
        packet = dfb.DataPacket()
        converter = df.ConvertBytesToInt(param_names='NON_EXISTANT')
        converter.next_filter = self.sink
        # FilterProcessingException is now raised for all filter exceptions
        # due to the StopIteration issue explained within
        # ProblemsWeHaveEncountered (trac).
        ##self.assertRaises(dfb.FilterProcessingException, converter.send, packet)
        converter.shut_down()
    
    def test_convert_muliple_values(self):
        data_input_one = '\x00\x81'
        data_input_two = '\x00\xc4'
        
        packet = dfb.DataPacket()
        packet.THIS_VALUE = data_input_one
        packet.THAT_VALUE = data_input_two
        # The filter
        converter = df.ConvertBytesToInt(param_names=['THIS_VALUE',
                                                       'THAT_VALUE'])
        converter.next_filter = self.sink
        converter.send(packet)
        converter.shut_down()
        
        self.assertEquals(self.sink.results[-1].THIS_VALUE, 129)
        self.assertEquals(self.sink.results[-1].THAT_VALUE, 196)

        
class TestDataLength(unittest.TestCase):
    def setUp(self):
        self.datalen = df.DataLength()
        self.sink = df.Sink(capture_msgs=True)
        self.datalen.next_filter = self.sink
    
    def tearDown(self):
        pass
    
    def test_find_data_length(self):
        # find the data, excluding \x00
        # it should clock up as it goes along
        zeros = 0x00
        ten_zeros = chr(zeros) * 10
        self.datalen.send(dfb.DataPacket(ten_zeros))
        self.assertEqual(self.datalen.data_last_seen, 0)
        self.assertEqual(self.datalen.bytes_seen, 10)
        # check that it remembers length of packets seen
        self.datalen.send(dfb.DataPacket('a' + chr(zeros) * 9))
        self.assertEqual(self.datalen.data_last_seen, 11)
        self.assertEqual(self.datalen.bytes_seen, 20)
        # check the last item in the sink contains the packet with 
        # attribute data_last_seen
        self.datalen.shut_down()
        import pprint 
        pprint.pprint([pkt.__dict__ for pkt in self.sink.results])
        # This error is because for some reason the message bottle packet
        # is not getting to the sink
        # TO-DO: Error here -- test_find_data_length
        self.assertEqual(self.sink.results[-1].total_data_length, 11)
        

class TestDedupeData(unittest.TestCase):
    
    def setUp(self):
        self.deduper = df.DedupeData()
        self.sink = df.Sink()
        self.deduper.next_filter = self.sink
    
    def tearDown(self):
        pass
    
    def test_deduping_ints(self):
        packet = dfb.DataPacket([1, 4, 3, 3, 3, 2])
        self.deduper.send(packet)
        self.assertEquals(self.sink.results[-1].data, [1, 2, 3, 4])

    def test_deduping_strings(self):
        packet = dfb.DataPacket(['c', 'DEF', 'fred', 'a'])
        self.deduper.send(packet)
        self.assertEquals(self.sink.results[-1].data, ['a', 'c', 'DEF', 'fred'])
        
        
class TestConvertFilenameToPath(unittest.TestCase):
    
    def setUp(self):
        self.sink = df.Sink()
    
    def tearDown(self):
        pass
    
    def test_converts_from_filename_to_path(self):
        expected_filename = os.path.join(os.path.dirname(__file__), 'test_file')
        packet = dfb.DataPacket(data='')
        packet.FILE_NAME = 'test_file'
        fil = df.ConvertFilenameToPath(input_file_name=expected_filename,
                                       in_attr='FILE_NAME',
                                       out_attr='FILE_PATH')
        fil.next_filter = self.sink
        fil.send(packet)
        
        self.assertEquals(self.sink.results[-1].FILE_PATH, expected_filename)


class TestCountBytes(unittest.TestCase):
    
    def setUp(self):
        self.byte_counter = df.CountBytes()
        self.sink = df.Sink()
        self.byte_counter.next_filter = self.sink
    
    def tearDown(self):
        pass

    def test_counting_bytes(self):
        for j in xrange(10):
            packet = dfb.DataPacket('X' * j)
            self.byte_counter.send(packet)
        self.assertEquals(self.byte_counter.counted_bytes, 45)

    def test_double_counting_bytes(self):
        for j in xrange(10):
            packet = dfb.DataPacket('xy' * j)
            self.byte_counter.send(packet)
            self.byte_counter.send(packet)
        self.assertEquals(self.byte_counter.counted_bytes, 180)

    
class TestCountLoops(unittest.TestCase):
    
    def setUp(self):
        self.counter = df.CountLoops()
        self.sink = df.Sink()
        self.counter.next_filter = self.sink
        self.counter2 = df.CountLoops()
        self.sink2 = df.Sink()
        self.counter2.next_filter = self.sink2
    
    def tearDown(self):
        pass

    def test_counting_loops(self):
        packet = dfb.DataPacket()
        self.counter.send(packet)
        self.assertEquals(self.sink.results[-1].loop_num, 1)
        self.counter.send(packet)
        self.assertEquals(self.sink.results[-1].loop_num, 2)
        self.counter.send(packet)
        self.counter.send(packet)
        self.counter.send(packet)
        self.assertEquals(self.sink.results[-1].loop_num, 5)
        self.counter2.send(packet)
        self.counter2.send(packet)
        self.assertEquals(self.sink2.results[-1].loop_num, 7)
                        
        
class TestCountPackets(unittest.TestCase):
    
    def setUp(self):
        self.packet_counter = df.CountPackets()
        self.sink = df.Sink()
        self.packet_counter.next_filter = self.sink
    
    def tearDown(self):
        pass

    def test_counting_packets(self):
        for j in xrange(10):
            packet = dfb.DataPacket()
            self.packet_counter.send(packet)
        self.assertEquals(self.packet_counter.counted_packets, 10)

    def test_double_counting_packets(self):
        for j in xrange(10):
            packet = dfb.DataPacket()
            self.packet_counter.send(packet)
            self.packet_counter.send(packet)
        self.assertEquals(self.packet_counter.counted_packets, 20)

        
class TestDistillHeader(unittest.TestCase):

    def setUp(self):
        self.hidden_branch_route = dfb.HiddenBranchRoute()
        self.main_sink = df.Sink()
        self.branch_sink = df.Sink()
        self.hidden_branch_route.next_filter = self.main_sink
        self.hidden_branch_route.branch_filter = self.branch_sink

    def tearDown(self):
        pass

    def test_distill_header_batch1(self):
        head_strip2 = df.DistillHeader(header_size=11, distill_mode='repeated')
        head_strip2.next_filter = self.hidden_branch_route
        packet1 = dfb.DataPacket(sample_data1)
        head_strip2.send(packet1)
        head_strip2.shut_down()
        self.assertEquals(self.branch_sink.results[0].data, 'file_header')
        self.assertEquals(len(self.main_sink.results[0].data), 312)

    def test_distill_header_batch2(self):
        head_strip3 = df.DistillHeader(header_size=3, distill_mode='repeated')
        head_strip3.next_filter = self.hidden_branch_route
        packet1 = dfb.DataPacket('hhh12345')
        packet2 = dfb.DataPacket('hhh67890')
        head_strip3.send(packet1)
        head_strip3.send(packet2)
        head_strip3.shut_down()
        print '**2110** %d main_rcvr inputs' % len(self.main_sink.results)
        print '**2110** %d branch sink inputs' % len(
                                          self.branch_sink.results)
        print '**2110** main inputs =', [x.__dict__ 
                                         for x in self.main_sink.results]
        print '**2110** branch inputs =', [x.__dict__ 
                                           for x in self.branch_sink.results]
        self.assertEquals(self.branch_sink.results[0].data, 'hhh')
        self.assertEquals(self.main_sink.results[0].data, '12345')
        self.assertEquals(self.branch_sink.results[1].data, 'hhh')
        self.assertEquals(self.main_sink.results[1].data, '67890')
        self.assertEquals(len(self.branch_sink.results), 2)
        self.assertEquals(len(self.main_sink.results), 2)

    def test_distill_header_batch3(self):
        head_strip3 = df.DistillHeader(header_size=14, distill_mode='repeated')
        head_strip3.next_filter = self.hidden_branch_route
        packet1 = dfb.DataPacket('hhh12345')
        packet2 = dfb.DataPacket('hhh67890')
        head_strip3.send(packet1)
        head_strip3.send(packet2)
        head_strip3.shut_down()
        print '**2130** %d main_rcvr inputs' % len(self.main_sink.results)
        print '**2130** %d branch sink inputs' % len(
                                          self.branch_sink.results)
        print '**2130** main inputs =', [x.__dict__ 
                                         for x in self.main_sink.results]
        print '**2130** branch inputs =', [x.__dict__ 
                                           for x in self.branch_sink.results]
        self.assertEquals(self.branch_sink.results[0].data, 'hhh12345')
        self.assertEquals(self.branch_sink.results[1].data, 'hhh67890')
        self.assertEquals(len(self.branch_sink.results), 2)
        self.assertEquals(len(self.main_sink.results), 0)

    def test_distill_header_continuous1(self):
        self.assertRaises(dfb.FilterAttributeError, df.DistillHeader)
        head_strip1 = df.DistillHeader(header_size=11, distill_mode='once')
        head_strip1.next_filter = self.hidden_branch_route
        packet1 = dfb.DataPacket(sample_data1)
        head_strip1.send(packet1)
        head_strip1.shut_down()
        self.assertEquals(self.branch_sink.results[0].data, 'file_header')
        self.assertEquals(len(self.main_sink.results[0].data), 312)

    def test_distill_header_continuous2(self):
        head_strip3 = df.DistillHeader(header_size=3, distill_mode='once')
        head_strip3.next_filter = self.hidden_branch_route
        packet1 = dfb.DataPacket('hhh12345')
        packet2 = dfb.DataPacket('hhh67890')
        head_strip3.send(packet1)
        head_strip3.send(packet2)
        head_strip3.shut_down()
        print '**2090** %d main_rcvr inputs' % len(self.main_sink.results)
        print '**2090** %d branch sink inputs' % len(
                                          self.branch_sink.results)
        print '**2090** main inputs =', [x.__dict__ 
                                         for x in self.main_sink.results]
        print '**2090** branch inputs =', [x.__dict__ 
                                           for x in self.branch_sink.results]
        self.assertEquals(self.branch_sink.results[0].data, 'hhh')
        self.assertEquals(self.main_sink.results[0].data, '12345')
        self.assertEquals(self.main_sink.results[1].data, 'hhh67890')
        self.assertEquals(len(self.branch_sink.results), 1)
        self.assertEquals(len(self.main_sink.results), 2)

    def test_distill_header_continuous3(self):
        head_strip3 = df.DistillHeader(header_size=14, distill_mode='once')
        head_strip3.next_filter = self.hidden_branch_route
        packet1 = dfb.DataPacket('hhh12345')
        packet2 = dfb.DataPacket('hhh67890')
        head_strip3.send(packet1)
        head_strip3.send(packet2)
        head_strip3.shut_down()
        print '**2120** %d main_rcvr inputs' % len(self.main_sink.results)
        print '**2120** %d branch sink inputs' % len(
                                          self.branch_sink.results)
        print '**2120** main inputs =', [x.__dict__ 
                                         for x in self.main_sink.results]
        print '**2120** branch inputs =', [x.__dict__ 
                                           for x in self.branch_sink.results]
        self.assertEquals(self.branch_sink.results[0].data, 'hhh12345hhh678')
        self.assertEquals(self.main_sink.results[0].data, '90')
        self.assertEquals(len(self.branch_sink.results), 1)
        self.assertEquals(len(self.main_sink.results), 1)
        

class TestReadBatch(unittest.TestCase):

    def setUp(self):
        self.sink = df.Sink()
        self.file_name1 = os.path.join(data_dir5, 'short.dat')
        ##self.file_name_out = os.path.join(data_dir5, 'short.out')
        f1 = open(self.file_name1, 'wb')
        try:
            f1.write('one two three four five six')
        finally:
            f1.close()

    def tearDown(self):
        pass

    def test_read_batch(self):
        read_batch1 = df.ReadBatch(batch_size=10)
        read_batch1.next_filter = self.sink
        read_batch1.send(dfb.DataPacket(self.file_name1))
        read_batch1.shut_down()
        self.assertEquals(self.sink.results[0].data, 'one two th')
        self.assertEquals(self.sink.results[1].data, 'ree four f')
        self.assertEquals(self.sink.results[2].data, 'ive six')
        
    def test_read_batch_with_file_obj(self):
        read_batch1 = df.ReadBatch(batch_size=10)
        read_batch1.next_filter = self.sink
        file_obj = open(self.file_name1, 'rb')
        read_batch1.send(dfb.DataPacket(file_obj))
        read_batch1.shut_down()
        self.assertEquals(self.sink.results[0].data, 'one two th')
        self.assertEquals(self.sink.results[1].data, 'ree four f')
        self.assertEquals(self.sink.results[2].data, 'ive six')

    def test_read_batch_with_two_file_objs(self):
        read_batch1 = df.ReadBatch(batch_size=10)
        read_batch1.next_filter = self.sink
        file_obj1 = open(self.file_name1, 'rb')
        read_batch1.send(dfb.DataPacket(file_obj1))
        file_obj2 = open(self.file_name1, 'rb')
        read_batch1.batch_size = 7
        read_batch1.send(dfb.DataPacket(file_obj2))
        read_batch1.shut_down()
        self.assertEquals(self.sink.results[0].data, 'one two th')
        self.assertEquals(self.sink.results[1].data, 'ree four f')
        self.assertEquals(self.sink.results[2].data, 'ive six')
        self.assertEquals(self.sink.results[3].data, 'one two')
        self.assertEquals(self.sink.results[4].data, ' three ')
        self.assertEquals(self.sink.results[5].data, 'four fi')
        self.assertEquals(self.sink.results[6].data, 've six')

    def test_read_batch_sending_file_name(self):
        read_batch2 = df.ReadBatch(batch_size=1000)
        sink2 = df.Sink()
        read_batch2.next_filter = sink2
        read_batch2.send(dfb.DataPacket(self.file_name1))
        read_batch2.shut_down()
        self.assertEquals(sink2.results[0].data, 'one two three four five six')
        self.assertEquals(sink2.results[0].source_file_name, self.file_name1)

    def test_read_batch_using_default_batch_size(self):
        read_batch2 = df.ReadBatch()
        sink2 = df.Sink()
        read_batch2.next_filter = sink2
        read_batch2.send(dfb.DataPacket(self.file_name1))
        read_batch2.shut_down()
        self.assertEquals(sink2.results[0].data, 'one two three four five six')
        self.assertEquals(sink2.results[0].source_file_name, self.file_name1)

    def test_read_batch_with_file_name_attribute(self):
        read_batch2 = df.ReadBatch(source_file_name=self.file_name1,
                                 batch_size=1000)
        sink2 = df.Sink()
        read_batch2.next_filter = sink2
        read_batch2.send(dfb.DataPacket(None))
        read_batch2.shut_down()
        self.assertEquals(sink2.results[0].data, 'one two three four five six')
        self.assertEquals(sink2.results[0].source_file_name, self.file_name1)
        
    def test_read_batch_with_initial_skip(self):
        read_batch3 = df.ReadBatch(batch_size=10, initial_skip=1)
        sink3 = df.Sink()
        read_batch3.next_filter = sink3
        read_batch3.send(dfb.DataPacket(self.file_name1))
        read_batch3.shut_down()
        self.assertEquals(sink3.results[0].data, 'ree four f')
        self.assertEquals(sink3.results[1].data, 'ive six')
        read_batch4 = df.ReadBatch(batch_size=3, initial_skip=4)
        sink4 = df.Sink()
        read_batch4.next_filter = sink4
        read_batch4.send(dfb.DataPacket(self.file_name1))
        read_batch4.shut_down()
        self.assertEquals(sink4.results[-5].data, 'e f')
        self.assertEquals(sink4.results[-4].data, 'our')
        self.assertEquals(sink4.results[-3].data, ' fi')
        self.assertEquals(sink4.results[-2].data, 've ')
        self.assertEquals(sink4.results[-1].data, 'six')

    def test_read_batch_with_max_reads(self):
        max_reads = 2
        read_batch1 = df.ReadBatch(batch_size=10, max_reads=max_reads)
        read_batch1.next_filter = self.sink
        read_batch1.send(dfb.DataPacket(self.file_name1))
        read_batch1.shut_down()
        self.assertEquals(len(self.sink.results), max_reads)
        self.assertEquals(self.sink.results[0].data, 'one two th')
        self.assertEquals(self.sink.results[1].data, 'ree four f')

        max_reads = 0  # No limit
        sink5 = df.Sink()
        read_batch2 = df.ReadBatch(batch_size=10, max_reads=max_reads)
        read_batch2.next_filter = sink5
        read_batch2.send(dfb.DataPacket(self.file_name1))
        read_batch2.shut_down()
        self.assertEquals(len(sink5.results), 3)
        self.assertEquals(sink5.results[0].data, 'one two th')
        self.assertEquals(sink5.results[1].data, 'ree four f')
        self.assertEquals(sink5.results[2].data, 'ive six')


class TestReadBytes(unittest.TestCase):
    def setUp(self):
        self.sink = df.Sink()
        self.file_name1 = os.path.join(data_dir5, 'short2.dat')
        f1 = open(self.file_name1, 'wb')
        try:
            f1.write('one two three four five six seven eight nine ten eleven')
        finally:
            f1.close()

    def tearDown(self):
        pass

    def test_read_start(self):
        read_filter = df.ReadBytes(source_file_name=self.file_name1,
                                       start_byte=0,
                                       size=20, block_size=20)
        read_filter.next_filter = self.sink
        packet = dfb.DataPacket(data='')
        read_filter.send(packet)
        self.assertEquals(len(self.sink.results), 1)
        self.assertEquals(self.sink.results[-1].data, 'one two three four f')
    
    def test_read_end(self):
        read_filter = df.ReadBytes(source_file_name=self.file_name1,
                                       start_byte=-20, size=-1, whence=1)
        read_filter.next_filter = self.sink
        packet = dfb.DataPacket(data='')
        read_filter.send(packet)
        self.assertEquals(len(self.sink.results), 1)
        self.assertEquals(self.sink.results[-1].data, 'ight nine ten eleven')
    
    def test_read_middle(self):
        read_filter = df.ReadBytes(source_file_name=self.file_name1,
                                       start_byte=10, size=38, block_size=19)
        read_filter.next_filter = self.sink
        packet = dfb.DataPacket(data='')
        read_filter.send(packet)
        self.assertEquals(len(self.sink.results), 2)
        self.assertEquals(self.sink.results[0].data, 'ree four five six s')
        self.assertEquals(self.sink.results[1].data, 'even eight nine ten')
    
    def test_read_all(self):
        read_filter = df.ReadBytes(source_file_name=self.file_name1,
                                       start_byte=0, size=-1)
        read_filter.next_filter = self.sink
        packet = dfb.DataPacket(data='')
        read_filter.send(packet)
        self.assertEquals(len(self.sink.results), 1)
        self.assertEquals(
            self.sink.results[-1].data,
            'one two three four five six seven eight nine ten eleven')
    
    def test_read_chunk_from_end(self):
        read_filter = df.ReadBytes(source_file_name=self.file_name1,
                                       start_byte=-20, size=10, whence=1)
        read_filter.next_filter = self.sink
        packet = dfb.DataPacket(data='')
        read_filter.send(packet)
        self.assertEquals(len(self.sink.results), 1)
        self.assertEquals(self.sink.results[-1].data, 'ight nine ')

    def test_read_multiple_chunks_from_end(self):
        read_filter = df.ReadBytes(source_file_name=self.file_name1,
                                       start_byte=-40, size=-1, whence=1,
                                       block_size=5)
        read_filter.next_filter = self.sink
        packet = dfb.DataPacket(data='')
        read_filter.send(packet)
        self.assertEquals(len(self.sink.results), 8)
        self.assertEquals(self.sink.results[0].data, 'our f')
        self.assertEquals(self.sink.results[-1].data, 'leven')
    
    def test_read_too_much(self):
        read_filter = df.ReadBytes(source_file_name=self.file_name1,
                                       start_byte=0, size=30000)
        read_filter.next_filter = self.sink
        packet = dfb.DataPacket(data='')
        read_filter.send(packet)
        self.assertEquals(len(self.sink.results), 1)
        self.assertEquals(
            self.sink.results[-1].data,
            'one two three four five six seven eight nine ten eleven')

    def test_stops_reading_at_expected_point(self):
        read_filter = df.ReadBytes(source_file_name=self.file_name1,
                                       start_byte=0, size=15, block_size=9)
        read_filter.next_filter = self.sink
        packet = dfb.DataPacket(data='')
        read_filter.send(packet)
        self.assertEquals(len(self.sink.results), 2)
        self.assertEquals(self.sink.results[0].data, 'one two t')
        self.assertEquals(self.sink.results[1].data, 'hree f')
    
    def test_reads_end_of_file_with_large_batch_size(self):
        read_filter = df.ReadBytes(source_file_name=self.file_name1,
                                       start_byte=0, size=55, block_size=15)
        read_filter.next_filter = self.sink
        packet = dfb.DataPacket(data='')
        read_filter.send(packet)
        self.assertEquals(len(self.sink.results), 4)
        self.assertEquals(self.sink.results[-1].data, 'ten eleven')
    
    def test_read_small_data_bigger_total_block(self):
        read_filter = df.ReadBytes(source_file_name=self.file_name1,
                                       start_byte=0,
                                       size=20, block_size=7)
        read_filter.next_filter = self.sink
        packet = dfb.DataPacket(data='')
        read_filter.send(packet)
        self.assertEquals(len(self.sink.results), 3)
        self.assertEquals(self.sink.results[-1].data, 'four f')

class SomePythonFunctions(ppln.Pipeline):

    config = '''
    [--main--]
    ftype = some_python_functions
    description = TO-DO: docstring
    
    [py_fancy_mean:x:y:z]
    return (x + y + z) / 3.0
    
    [py_scale_airspeed]
    x = sum([2, 4, 6])        # = 12
    y = x * 2                 # = 24
    z = x + (2 * y / 6)       # = 20
    ## **10610** This is a comment
    print 'Hello World'
    name_that_could_be_a_filter = 5.5
    abc = [name_that_could_be_a_filter]
    
    [py_mean_g]
    def mean_g(g3, g4, g5, RADALT):
        g10 = (g3 + g4 + g5) / 3.0
        ft56 = g10 + RADALT
        return ft56
    # This mean_g only available within this function
    
    [py_no_code]
        
    [py_use_fancy_mean]
    bar = 2.7
    foo = fancy_mean(bar, 5, 7.8)
    print 'Using fancy_mean, foo = %s' % foo
    
    [py_capitalise:text]  
    return text.upper()
    
    [py_exceptions]
    try:
        assert 5 == 6
    except AssertionError:
        print 'AssertionError found'
        
    [py_try_out_capitalise_packet_data]
    print '**12420** packet.data before capitalise() =', packet.data
    packet.data = capitalise(packet.data)  
    print '**12430** packet.data after capitalise() =', packet.data
    
    [py_calc_wps] 
    WPS = packet.half_wps * 2

    [py_calc_subframe]
    SUBF = WPS / 4
    
    [py_print_subframe_length]
    print '**12300** Subframe length =', SUBF
    
    [py_all_sorts_of_attributes]
    print '**12310** Params = 123ABC, ABC345, A_B_C'
    print 'MixedCaseIgnored, A_B_C, $@XYZ%%, _X_Y_Z'
    
    [py_set_batch_size]
    import random
    BATCH_SIZE = random.randrange(2, 25)
    
    [batch]
    dynamic = true
    
    [--route--]
    ## py_use_fancy_mean >>>
    ## py_calc_subframe >>>
    ## py_calc_wps >>>
    ## py_calc_subframe >>>
    ## py_print_subframe_length >>>
    ## py_try_out_capitalise_packet_data >>>
    py_set_batch_size >>>
    batch:%BATCH_SIZE
    '''

class Dummy(object): pass
    
class TestEmbedPython(unittest.TestCase):
    
    def setUp(self):
        self.factory = ff.DemoFilterFactory()
        
    def tearDown(self):
        pass
    
    def test_subst_keys(self):
        """"""
        embed_python = df.EmbedPython()
        embed_python.pipeline = Dummy()
        embed_python.pipeline.path = "C:\\temp"
        self.assertEqual(embed_python.subst_keys('${path}'),
                         "'%s'" % embed_python.pipeline.path)
    
    def test_embed_python(self):
        filter_with_python = SomePythonFunctions(factory=self.factory)
        module1 = filter_with_python.emb_module
        print '**10640** made embed_python filter "%s" with module "%s"' % (
            filter_with_python.name, module1.__name__)
        for key, value in module1.__dict__.iteritems():
            if not key.startswith('__'):
                print '**10635** %s: %s' % (key, value)
        print '**10630** mean = %s' % (module1.fancy_mean(1, 2, 3))
        print '\n**10660**', 60 * '-'
        print '\n'.join(filter_with_python.module_loc._python_code_lines)
        print '**10660**', 60 * '-'
        batcher = filter_with_python.getf('batch')
        #self.assertEquals(
            #filter_with_python.module_loc._python_code_lines[-3],
            #'fred = fancy_double(bill)')
        packet1 = dfb.DataPacket('hello world', half_wps=128)
        filter_with_python.send(packet1)
        return 
    
        self.assertEqual(batcher.size, 1)  # <<<<<< TO-DO 18 != 1
        packet2 = dfb.DataPacket('alas alas', half_wps=256)
        filter_with_python.send(packet2)
        self.assertEqual(batcher.size, 2)
        packet3 = dfb.DataPacket('freight train', half_wps=512)
        filter_with_python.send(packet3)
        self.assertEqual(batcher.size, 3)
        
    def test_embedded_batch_sizing(self):
        data_in = '15xxxxxyyyyyzzzzz12aaabbbcccddd07eeeeeee04fghj'
        # Pipeline should batch alternately into 2 bytes and the read amount
        filter_with_python2 = ppln_demo.VariableBatchPipeline(
                                                       factory=self.factory)
        print '**10662** %s Embedded Python start %s' % (30 * '-', 30 * '-')
        print '\n'.join(filter_with_python2._python_code_lines)
        print '**10662** %s Embedded Python end %s' % (30 * '-', 30 * '-')
        filter_with_python2.send(dfb.DataPacket(data_in))

    def test_update_live(self):
        config = '''\
        [--main--]
        ftype = testing_update_live
        description = Reading live data from Python module
        keys = foo, bar, bill:55, baz:20
            
        [py_simple]
        import random
        x = ${bar}
        status = ${foo}
        BAR_3 = x
        try:
            FOO += BAR_3
        except TypeError: # FOO has not been initialised...
            FOO = BAR_3
        print 'FOO = %d' % FOO
        
        [py_call_simple_twice]
        #print 'BAR_3 = %s' % BAR_3
        simple()
        simple()
        BAZ = 3
        
        [batch_one]
        dynamic = true
        
        [batch_two]
        dynamic = true
        size = %FOO
        
        [--route--]
        py_simple >>>
        py_call_simple_twice >>>
        batch_one:%FOO >>>
        batch_two >>>
        sink
        '''
        live_updater = ppln.Pipeline(factory=self.factory, 
                                     config=config,
                                     foo='margin', bar=5)
        #batcher = live_updater.getf('batch')
        module3 = live_updater.emb_module
        print '**10670** %s Embedded Python start %s' % (30 * '-', 30 * '-')
        code = live_updater.module_loc._python_code_lines
        if code:
            print '\n'.join(code)
        print '**10670** %s Embedded Python end %s' % (30 * '-', 32 * '-')
        batch1 = live_updater.getf('batch_one')
        for j, expected_size in zip(xrange(6), range(15,91,15)):
            packet1 = dfb.DataPacket('hello')
            live_updater.send(packet1)
            # batch_size should start at 15 and increment by 15 each time
            self.assertEqual(batch1.size, expected_size)
            print '\n**10680** batch_size =', batch1.size
    
    def test_python_embedding_handles_percent_literals(self):
        # Test to make sure that the python embedding handles when a filter key 
        # literal value begins with a % character. EG, read_batch:%OUTFILE.out 
        # when we want to read from a file called "%OUTFILE.out"
        
        config = '''
        [--main--]
        ftype = testing_handles_percent_literals
        description = Reading live data from Python module
        keys = read_file_name
        '''
        
        
    def test_embed_has_access_to_refinery_required_keywords(self):
        # embedded python can access pipeline keywords and the defaults
        # if none are provided
        return # TO-DO Need to discuss: test_embed_has_access...
        config = '''
        [--main--]
        ftype = keys_access
        description = Test keys access
        keys = required
        
        [py_access_required_pipeline_keys]
        assert REQUIRED is not None
        assert REQUIRED.value == [32]
        
        [--route--]
        py_access_required_pipeline_keys 
        
        '''
        class Req(object): value = [32]
        req = Req()        
        
        keys_pipe = ppln.Pipeline(factory=self.factory, 
                                  config=config,
                                  required=req)
        # send data; raises AssertionError if fails
        keys_pipe.send(dfb.DataPacket(data='hi'))

    def test_embed_has_access_to_refinery_keywords_and_defaults(self):
        # embedded python can access pipeline keywords and the defaults
        # if none are provided
        return # TO-DO Need to discuss: test_embed_has_access...
        config = '''
        [--main--]
        ftype = keys_access
        description = Test keys access
        keys1 = required, optional_not_provided:not_much
        keys2 = optional_provided:not_here
        
        [py_access_optional_pipeline_keys]
        assert OPTIONAL_NOT_PROVIDED is not None
        assert OPTIONAL_NOT_PROVIDED == 'not_much'
        
        assert OPTIONAL_PROVIDED is not None
        assert OPTIONAL_PROVIDED != 'not_here'
        assert OPTIONAL_PROVIDED.parameter == 'here it is'
        
        [py_access_required_pipeline_keys]
        assert REQUIRED is not None
        assert REQUIRED.value == [32]
        
        [--route--]
        py_access_required_pipeline_keys >>>
        py_access_optional_pipeline_keys
        
        '''
        class Something(object): value = [32]
        req = Something()        
        opt = Something()
        opt.parameter = 'here it is'
        
        keys_pipe = ppln.Pipeline(factory=self.factory, 
                                  config=config,
                                  required=req,
                                  optional_provided=opt)
        ##try:
        keys_pipe.send(dfb.DataPacket(data='hi'))

        
class TestGetBytes(unittest.TestCase):
    
    def setUp(self):
        self.sink = df.Sink()
    
    def tearDown(self):
        pass
    
    def test_got_correct_bytes(self):
        test_string = 'ABCDEF'
        
        packet = dfb.DataPacket(data=test_string)
        bytes1 = df.GetBytes(start_byte=4, bytes_to_get=1, param_name='VALUE')
        bytes1.next_filter = self.sink
        bytes1.send(packet)
        bytes1.shut_down()
        
        data_out = self.sink.all_data[0]
        
        print '**2367**', fut.data_to_hex_string(data_out)
        
        self.assertEquals(self.sink.results[-1].VALUE, 'E')
        
        
class TestHashSHA256(unittest.TestCase):

    def setUp(self):
        self.sink = df.Sink()

    def tearDown(self):
        pass

    def test_calc_hash(self):
        hash_sha256 = df.HashSHA256()
        hash_sha256.next_filter = self.sink
        input1 = 'abcdef'
        hash_sha256.send(dfb.DataPacket(input1))
        hash_obj = hashlib.sha256()
        hash_obj.update(input1)
        self.assertEquals(self.sink.results[-1].data, input1)
        self.assertEquals(hash_sha256.hasher.hexdigest(), hash_obj.hexdigest())
        input2 = '123456'
        hash_sha256.send(dfb.DataPacket(input2))
        hash_obj2 = hashlib.sha256()
        hash_obj2.update(input1 + input2)
        self.assertEquals('|'.join(self.sink.all_data), input1 + '|' + input2)
        self.assertEquals(hash_sha256.hasher.hexdigest(), hash_obj2.hexdigest())

class TestHeaderAsAttribute(unittest.TestCase):
    
    def setUp(self):
        self.test_header_size = 32
        self.test_header_attr = "hdr"
        self.fltr = df.HeaderAsAttribute(header_size=self.test_header_size,
                                         header_attribute=self.test_header_attr)
        self.sink = df.Sink()
        self.fltr.next_filter = self.sink
        self.header_byte = "\x00"
        self.data_byte = "\xFF"
    
    def tearDown(self):
        pass
    
    def test_header_as_attribute_1(self):
        for send_on in (True, False):
            
            for x in range(100):
                header_size = random.randint(0,256)
                data_size = random.randint(0,1536)
                header_bytes = header_size * self.header_byte
                data_bytes = data_size * self.data_byte
                fltr = df.HeaderAsAttribute(header_size=header_size,
                                            header_attribute=self.test_header_attr,
                                            send_on_if_only_header=send_on)
                sink = df.Sink()
                fltr.next_filter = sink
                
                packet = dfb.DataPacket(data=header_bytes + data_bytes)
                
                fltr.send(packet)
                self.assertEqual(getattr(sink.results[0],
                                 self.test_header_attr),
                                 header_bytes)
                self.assertEqual(sink.results[0].data,
                                 data_bytes)
    
    def test_header_as_attribute_send_on_packet(self):
        for x in range(10):
            # Data too small for header.
            header_size = random.randint(0,255)
            header_bytes = header_size * self.header_byte
            fltr = df.HeaderAsAttribute(header_size=header_size+1,
                                        header_attribute=self.test_header_attr,
                                        send_on_if_only_header=False)
            sink = df.Sink()
            fltr.next_filter = sink
            packet = dfb.DataPacket(data=header_bytes)
            fltr.send(packet)
            self.assertEqual(len(sink.results), 0)

# nice idea, but not the solution to the current problem
##class TestIndex(unittest.TestCase):
    ##def setUp(self):
        ##self.indexer = df.Index(substring='456', param_name='ind')
        ##self.sink = df.Sink()
        ##self.indexer.next_filter = self.sink
        
    ##def test_index(self):
        ##pkt1 = dfb.DataPacket('0123456789')
        
        ##self.indexer.send(pkt1)
        ##self.assertEqual(self.sink.results[0].ind, '4')
        
class TestJoin(unittest.TestCase):
    
    def setUp(self):
        self.joiner = df.Join()
        self.sink = df.Sink()
        self.joiner.next_filter = self.sink

    def tearDown(self):
        pass

    def test_join(self):
        packet1 = dfb.DataPacket('one')
        packet2 = dfb.DataPacket('two')
        packet3 = dfb.DataPacket('three')
        packet4 = dfb.DataPacket(None)
        self.joiner.send(packet1)
        self.joiner.send(packet2)
        self.joiner.send(packet3)
        self.assertEquals(len(self.sink.results), 0)
        self.joiner.send(packet4)
        self.assertEquals(self.sink.results[-1].data, 'one two three')
        
    def test_join_with_close(self):
        packet1 = dfb.DataPacket('four')
        packet2 = dfb.DataPacket('five')
        packet3 = dfb.DataPacket('six')
        self.joiner.send(packet1)
        self.joiner.send(packet2)
        self.joiner.send(packet3)
        self.joiner.join_str = '~'
        self.assertEquals(len(self.sink.results), 0)
        self.joiner.shut_down()
        self.assertEquals(self.sink.results[-1].data, 'four~five~six')
        

class TestPassNonZero(unittest.TestCase):
    
    def setUp(self):
        self.pass_non_zero = df.PassNonZero(check_byte_count=4)
        self.sink = df.Sink()
        self.pass_non_zero.next_filter = self.sink
    
    def tearDown(self):
        pass
    
    def test_pass_non_zero_data(self):
        packet1 = dfb.DataPacket('abcdef')
        packet2 = dfb.DataPacket('123456')
        packet3 = dfb.DataPacket('ghijkl')
        self.pass_non_zero.send(packet1, packet2, packet3)
        self.assertEquals('|'.join(self.sink.all_data), 'abcdef|123456|ghijkl')
    
    def test_dont_pass_zero_data(self):
        packet1 = dfb.DataPacket(fut.hex_string_to_data('00 00 00 00'))
        packet2 = dfb.DataPacket('hello')
        packet3 = dfb.DataPacket(fut.hex_string_to_data('FF FF FF FF FF FF'))
        self.pass_non_zero.send(packet1, packet2, packet3)
        self.assertEquals(''.join(self.sink.all_data), 'hello')
    
    def test_zero_data_too_short_to_check(self):
        packet1 = dfb.DataPacket(fut.hex_string_to_data('00 00 00'))
        packet2 = dfb.DataPacket('hello')
        packet3 = dfb.DataPacket(fut.hex_string_to_data('FF FF FF'))
        self.pass_non_zero.send(packet1, packet2, packet3)
        self.assertEquals('|'.join(self.sink.all_data), 
                          fut.hex_string_to_data('00 00 00') + '|hello|' + \
                          fut.hex_string_to_data('FF FF FF'))
        
        
class TestPassThrough(unittest.TestCase):

    def setUp(self):
        self.sink = df.Sink()

    def tearDown(self):
        pass
    
    def test_pass_through(self):
        pass_through = df.PassThrough()
        pass_through.next_filter = self.sink
        packet1 = dfb.DataPacket(data='abcdef')
        pass_through.send(packet1)
        self.assertEquals(self.sink.results[-1].data, 'abcdef')
        self.assertEquals('|'.join(self.sink.all_data), 'abcdef')
        packet2 = dfb.DataPacket(data='12345')
        pass_through.send(packet2)
        self.assertEquals(self.sink.results[-1].data, '12345')
        self.assertEquals('|'.join(self.sink.all_data), 'abcdef|12345')

    def test_pass_through_multiple_send(self):
        pass_through = df.PassThrough()
        pass_through.next_filter = self.sink
        packet1 = dfb.DataPacket(data='abcdef')
        packet2 = dfb.DataPacket(data='12345')
        pass_through.send(packet1, packet2)
        self.assertEquals(self.sink.results[-1].data, '12345')
        self.assertEquals('|'.join(self.sink.all_data), 'abcdef|12345')


class TestPeek(unittest.TestCase):
    
    def setUp(self):
        self.peek = df.Peek(peek_ahead=3)
        self.sink = df.Sink()
        self.peek.next_filter = self.sink
    
    def tearDown(self):
        pass
    
    def test_peek(self):
        self.assertRaises(dfb.FilterAttributeError, df.Peek)
        packet1 = dfb.DataPacket('abcdef')
        self.peek.send(packet1)
        self.assertEquals(len(self.sink.results), 0)
        packet2 = dfb.DataPacket('ghijkl')
        self.peek.send(packet2)
        self.assertEquals(len(self.sink.results), 1)
        self.assertEquals(self.sink.results[-1], packet1)
        self.assertEquals(packet1.peek, 'ghi')
        self.peek.shut_down()
        self.assertEquals(len(self.sink.results), 2)
        self.assertEquals(self.sink.results[-1], packet2)
        self.assertEquals(packet2.peek, '')
        
        
class TestRenameFile(unittest.TestCase):

    def setUp(self):
        file_name_in = os.path.join(data_dir5, 'some_file.dat')
        self.assertTrue(os.path.exists(file_name_in))
        # Copy data file to source, before renaming
        self.file_name_copy = os.path.join(data_dir5, 'some_file-tmp.dat')
        shutil.copy(file_name_in, self.file_name_copy)
        self.assertTrue(os.path.exists(self.file_name_copy))
        # Setup pipeline to exercise RenameFile filter
        self.rename_file_filter = df.RenameFile()

    def tearDown(self):
        # Delete renamed temporary file
        for file_name in fut.all_files('*short-tmp.tsc', data_dir5):
            os.remove(file_name)
    
    def test_rename_file(self):
        # Do the rename
        from_name = self.file_name_copy
        to_name = os.path.join(data_dir5, 'TAILNO-41deba-short-tmp.tsc')
        packet1 = dfb.DataPacket(from_filename=from_name, to_filename=to_name)
        self.rename_file_filter.send(packet1)
        # Check renamed file exists
        self.assertFalse(os.path.exists(from_name))
        self.assertTrue(os.path.exists(to_name))
    
    def test_rename_nonexistent_file(self):
        file_name_copy = os.path.join(data_dir5, 'this_file_does_not_exist.tsc')
        self.assertFalse(os.path.exists(file_name_copy))
        # Do the rename
        from_name = file_name_copy
        to_name = os.path.join(data_dir5, 'this_file_is_also_nonexistant.tsc')
        packet1 = dfb.DataPacket(from_filename=from_name, to_filename=to_name)
        self.rename_file_filter.send(packet1)
        # Check renamed file doesn't exist
        self.assertFalse(os.path.exists(from_name))

    def test_rename_to_same_file(self):
        # Do the rename
        from_name = self.file_name_copy
        to_name = self.file_name_copy
        packet1 = dfb.DataPacket(from_filename=from_name, to_filename=to_name)
        # FilterProcessingException now raised instead of FilterError due to
        # alleviating the StopIteration issue as described within
        # ProblemsWeHaveEncountered (trac).
        self.assertRaises(dfb.DataError, self.rename_file_filter.send, packet1)
        ##self.assertRaises(dfb.FilterProcessingException,
        ##                  self.rename_file_filter.send,
        ##                  packet1)
        
    def test_rename_to_already_exists(self):
        # Do the rename
        from_name = self.file_name_copy
        to_name = os.path.join(data_dir5, 'hello.txt')
        open(to_name, 'w').close()
        packet1 = dfb.DataPacket(from_filename=from_name, to_filename=to_name)
        self.rename_file_filter.send(packet1)
        # Check renamed file exists
        self.assertFalse(os.path.exists(from_name))
        self.assertTrue(os.path.exists(to_name))


class TestReset(unittest.TestCase):

    def setUp(self):
        # Although Reset is a data filter, it needs a pipeline to test it
        # because it uses the pipeline to find its target filter.
        self.factory = ff.DemoFilterFactory()
        config = '''
        [--main--]
        ftype = testing_reset 
        description = TO-DO: docstring
                        
        [--route--]
        r111eset_branch:bar >>>
            (sink)
        sink
        '''
        self.pipeline1 = ppln.Pipeline(factory=ff.DemoFilterFactory(),
                                       config=config)
    
    def tearDown(self):
        pass
    
    def test_reset(self):
        sink = self.pipeline1.getf('sink')
        sink.bar = 'hello'
        packet1 = dfb.DataPacket('one')
        self.pipeline1.send(packet1)
        self.assertEquals(sink.bar, 0)
    
    # What was this test for?
    #def test_reset_from_packet_attrib(self):
        #sink = self.pipeline1.getf('sink')
        ##sink = df.Sink()
        #sink.WIBBLE = 'FOO'
        #packet1 = dfb.DataPacket(data='')
        #packet1.WIBBLE = 'ABC'
        #self.pipeline1.send(packet1)
        ##filter1 = df.Reset(target_filter_name='sink', param_name='FOO',
                           ##value='WIBBLE')
        ##filter1.next_filter = sink
        ##filter1.send(packet1)
        ## TO-DO test_reset_from_packet_attrib() fails
        #self.assertEquals(sink.results[-1].data, 2626)
        
    ##def test_reset_from_packet(self):
        ##resetter = self.pipeline1.getf('reset')
        ##sink = self.pipeline1.getf('sink')
        ##resetter.source_packet_field_name = 'foo'
        ##sink.bar = 'hello'
        ##packet2 = dfb.DataPacket('two')
        ##packet2.foo = 99
        ##resetter.send(packet2)
        ##self.assertEquals(sink.bar, 99)        
        
        
class TestSwapTwoBytes(unittest.TestCase):
    
    def setUp(self):
        self.sink = df.Sink()
        
    def tearDown(self):
        pass
    
    def test_swap_bytes_in_a_2char_string(self):
        """ test_reverse_a_string
            
        """
        test_string = '\x00\xFF'
        
        packet = dfb.DataPacket(data=test_string)
        reverse = df.SwapTwoBytes()
        reverse.next_filter = self.sink
        reverse.send(packet)
        reverse.shut_down() 
        
        self.assertNotEqual(self.sink.results[-1].data , test_string)
        self.assertEquals(self.sink.results[-1].data, '\xFF\x00')
        
    def test_swap_bytes_in_a_4char_string(self):
        """ test_reverse_a_string
            
        """
        test_string = '\x12\x34\x56\x78'
        
        packet = dfb.DataPacket(data=test_string)
        reverse = df.SwapTwoBytes()
        reverse.next_filter = self.sink
        reverse.send(packet)
        reverse.shut_down() 
        
        self.assertNotEqual(self.sink.results[-1].data , test_string)
        self.assertEquals(self.sink.results[-1].data, '\x34\x12\x78\x56');
    
    def test_swap_non_string(self):
        """ test_reverse_a_non_string
            
        """
        test_string = 12345
        
        packet = dfb.DataPacket(data=test_string)
        reverse = df.SwapTwoBytes()
        reverse.next_filter = self.sink        
        # FilterProcessingException now raised instead of FilterError due to
        # alleviating the StopIteration issue as described within
        # ProblemsWeHaveEncountered (trac).
        self.assertRaises(TypeError, reverse.send, packet)
        ##self.assertRaises(dfb.FilterProcessingException, reverse.send, packet)


class TestReverseString(unittest.TestCase):
    
    def setUp(self):
        self.sink = df.Sink()
        
    def tearDown(self):
        pass
    
    def test_reverse_a_string(self):
        """ test_reverse_a_string
            
        """
        test_string = '\x00\x00\xFF\xFF'
        
        packet = dfb.DataPacket(data=test_string)
        reverse = df.ReverseString()
        reverse.next_filter = self.sink
        reverse.send(packet)
        reverse.shut_down() 
        
        self.assertNotEqual(self.sink.results[-1].data , test_string)
        self.assertEquals(self.sink.results[-1].data, '\xFF\xFF\x00\x00')

    
class TestSeqPacket(unittest.TestCase):
    
    def setUp(self):
        self.numberer = df.SeqPacket()
        self.sink = df.Sink()
        self.numberer.next_filter = self.sink
        self.numberer2 = df.SeqPacket()
        self.sink2 = df.Sink()
        self.numberer2.next_filter = self.sink2
    
    def tearDown(self):
        pass

    def test_numbering(self):
        for j in xrange(5):
            packet = dfb.DataPacket()
            self.numberer.send(packet)
        self.assertEquals(self.sink.results[0].seq_num, 0)
        self.assertEquals(self.sink.results[-1].seq_num, 4)
            
    def test_not_renumbering(self):
        for j in xrange(5):
            packet = dfb.DataPacket()
            self.numberer.send(packet)
        # Send the last packet round again
        self.numberer2.send(self.sink.results[-1])
        # seq_num should be left as it was
        self.assertEquals(self.sink2.results[0].seq_num, 4)
        
    def test_with_different_field_name(self):  # <<<<<< TO-DO:TEST <<<<<<
        pass 

    def test_with_counter_reset(self):  # <<<<<< TO-DO:TEST <<<<<<
        pass 
    
    def test_with_message1(self):
        for j in xrange(5): 
            packet = dfb.DataPacket()
            self.numberer.send(packet)  
        self.assertEquals(self.sink.results[-1].seq_num, 4)
        bottle = dfb.MessageBottle('none', 'reset')
        self.numberer.send(bottle)
        self.numberer.send(dfb.DataPacket())
        self.assertEquals(self.sink.results[-1].seq_num, 5)
    
    # No such thing as a 'plug_size' on sink
    ##def test_with_message2(self):
        ##for j in xrange(5):
            ##packet = dfb.DataPacket()
            ##self.numberer.send(packet)  
        ##self.assertEquals(self.sink.results[-1].seq_num, 4)
        ### Send message to self.sink
        ##bottle = dfb.MessageBottle('sink', 'reset', 
                                   ##param_name='plug_size', new_value=3.5)
        ##self.numberer.send(bottle)
        ##self.numberer.send(dfb.DataPacket())
        ##self.assertEquals(self.sink.results[-1].seq_num, 5)
        ##self.assertEquals(self.sink.plug_size, 3.5)

class TestSetAttributesToData(unittest.TestCase):
    def setUp(self):
        self.sink = df.Sink()

    def tearDown(self):
        pass
    
    def test_set_attributes_gets_attributes(self):
        packet = dfb.DataPacket(data='')
        packet.ATTR1 = 'foo'
        packet.ATTR2 = 'bar'
        packet.ATTR3 = 'wibble'
        packet.ATTR4 = 3
        packet.ATTR5 = 'dont deal with me'
        
        attributes_to_set = ['ATTR1', 'ATTR2', 'ATTR3', 'ATTR4']

        filter1 = df.SetAttributesToData(attribute_list=attributes_to_set,
                                         separator='', 
                                         eol='', write_field_headers=False)
        filter1.next_filter = self.sink
        filter1.send(packet)
        
        self.assertEquals(self.sink.results[-1].data, 'foobarwibble3')

    def test_set_attributes_empty(self):
        packet = dfb.DataPacket(data='wibblefish')
        
        attributes_to_set = []

        filter1 = df.SetAttributesToData(attribute_list=attributes_to_set,
                                         write_field_headers=False)
        filter1.next_filter = self.sink
        filter1.send(packet)
        
        self.assertEquals(self.sink.results[-1].data, 'wibblefish')

    def test_set_attributes_no_attributes_given(self):
        packet = dfb.DataPacket(data='thecakeisalie')

        filter1 = df.SetAttributesToData(write_field_headers=False)
        filter1.next_filter = self.sink
        filter1.send(packet)
        
        self.assertEquals(self.sink.results[-1].data, 'thecakeisalie')
        
    def test_set_attributes_list_params(self):
        packet = dfb.DataPacket(data = '',nums=[1,2,3], 
                                strs=['hey','hi','woo'],)
        
        filter1 = df.SetAttributesToData(attribute_list=['nums','strs'],
                                         write_field_headers=False)
        filter1.next_filter = self.sink
        filter1.send(packet)
        # Changed by Rob -- is this right?
        ##self.assertEquals(self.sink.results[-1].data, '1,hey\n2,hi\n3,woo')
        self.assertEquals(self.sink.results[-1].data, '\n1,hey\n2,hi\n3,woo')
    
    def test_set_attributes_list_params_using_diff_output_formats(self):
        packet = dfb.DataPacket(data = '',nums=[1,2,15], 
                                strs=['hey','hi','woo'],)
        # Test Hex
        filter1 = df.SetAttributesToData(attribute_list=['nums','strs'],
                                         write_field_headers=False,
                                         output_format='hex')
        filter1.next_filter = self.sink
        filter1.send(packet)
        # Changed by Rob -- is this right?
##        self.assertEquals(self.sink.results[-1].data, '0x1,hey\n0x2,hi\n0xf,woo')
        self.assertEquals(self.sink.results[-1].data, 
                          '\n0x1,hey\n0x2,hi\n0xf,woo')
        # Test Binary
        filter2 = df.SetAttributesToData(attribute_list=['nums','strs'],
                                         write_field_headers=False,
                                         output_format='binary')
        filter2.next_filter = self.sink
        filter2.send(packet)
        # Changed by Rob -- is this right?
##        self.assertEquals(self.sink.results[-1].data, '0b1,hey\n0b10,hi\n0b1111,woo')
        self.assertEquals(self.sink.results[-1].data, 
                          '\n0b1,hey\n0b10,hi\n0b1111,woo')
        # Test Octal
        filter3 = df.SetAttributesToData(attribute_list=['nums','strs'],
                                         write_field_headers=False,
                                         output_format='octal')
        filter3.next_filter = self.sink
        filter3.send(packet)
        # Changed by Rob -- is this right?
##        self.assertEquals(self.sink.results[-1].data, '01,hey\n02,hi\n017,woo')
        self.assertEquals(self.sink.results[-1].data, 
                          '\n01,hey\n02,hi\n017,woo')
    
        
    def test_set_headers(self):
        packet = dfb.DataPacket(data = '',nums=[1,2,3], 
                                strs=['hey','hi','woo'],)
        
        filter1 = df.SetAttributesToData(write_field_headers=True,
                                         attribute_list=['nums','strs'])
        filter1.next_filter = self.sink
        filter1.send(packet)
        
        self.assertEquals(self.sink.results[-1].data, 
                          'nums,strs\n1,hey\n2,hi\n3,woo')

    def test_with_line_nos(self):
        packet = dfb.DataPacket(data = '',nums=[1,2,3], strs=['hey','hi','woo'])
        
        filter1 = df.SetAttributesToData(write_field_headers=True,
                                         attribute_list=['nums','strs'],
                                         line_numbers=True)
        filter1.next_filter = self.sink
        filter1.send(packet)
        
        self.assertEquals(self.sink.results[-1].data,
                          'line,nums,strs\n1,1,hey\n2,2,hi\n3,3,woo')
        

class TestSink(unittest.TestCase):
    
    def setUp(self):
        self.sink = df.Sink()
    
    def tearDown(self):
        pass
    
    def test_sink_for_packet(self):
        packet = dfb.DataPacket(data='abcdef')
        self.sink.send(packet)
        self.assertEquals(self.sink.results[-1].data, 'abcdef')
        self.assertEquals('|'.join(self.sink.all_data), 'abcdef')

    def test_sink_for_strings(self):
        # We can pass anything around, but packets should be the standard ?? <<<<<<<<<<<<<<<<<
        self.sink.send(dfb.DataPacket('hello'))
        self.assertEquals(self.sink.results[-1].data, 'hello')
        self.sink.send(dfb.DataPacket(''))
        self.assertEquals(self.sink.results[-1].data, '')
        self.sink.send(dfb.DataPacket([1, 2, 3]))
        self.assertEquals(self.sink.results[-1].data, [1, 2, 3])
        self.sink.send(dfb.DataPacket(fut.hex_string_to_data('00 00 00')))
        self.assertEquals(self.sink.results[-1].data,
                          fut.hex_string_to_data('00 00 00'))
        self.sink.shut_down()
        
    def test_send_on(self):
        sink2 = df.Sink()
        sink2.send_on(dfb.MessageBottle('blah', 'my_message'))
        self.assertEqual(len(sink2.results), 0)
    
    def test_send_on_with_message_catching(self):
        sink_message_catcher = df.Sink(capture_msgs=True)
        sink_message_catcher.send_on(
            dfb.MessageBottle('blah2', 'my_second_message'))
        self.assertEqual(len(sink_message_catcher.results), 1)
        self.assertEqual(sink_message_catcher.results[0].message, 
                         "my_second_message")
            
        
class TestSplitWords(unittest.TestCase):
    
    def setUp(self):
        self.splitter = df.SplitWords()
        self.sink = df.Sink()
        self.splitter.next_filter = self.sink

    def tearDown(self):
        pass

    def test_split_words(self):
        packet1 = dfb.DataPacket('one two three four five  ')
        self.splitter.send(packet1)
        self.assertEquals(self.sink.results[0].data, 'one')
        self.assertEquals(self.sink.results[-1].data, 'five')
        self.assertEquals('|'.join(self.sink.all_data),
                          'one|two|three|four|five')
        
class TestSplitWords(unittest.TestCase):
    
    def setUp(self):
        self.splitter = df.SplitWords()
        self.sink = df.Sink()
        self.splitter.next_filter = self.sink

    def tearDown(self):
        pass

    def test_split_words(self):
        custom_splitter = df.SplitWords(split_on_str=". ")
        custom_splitter.next_filter = self.sink
        packet1 = dfb.DataPacket('THE TIME CHIP SERIAL IS: 23. SOME_OTHER: 5. A')
        custom_splitter.send(packet1)
        self.assertEquals(self.sink.results[0].data, 'THE TIME CHIP SERIAL IS: 23')
        self.assertEquals(self.sink.results[1].data, 'SOME_OTHER: 5')
        self.assertEquals(self.sink.results[2].data, 'A')

class TestSplitLines(unittest.TestCase):
    
    def setUp(self):
        self.splitter = df.SplitLines()
        self.sink = df.Sink()
        self.splitter.next_filter = self.sink

    def tearDown(self):
        pass

    def test_split_lines(self):
        data1 = 'one\r\ntwo\nthree\rfour \r\n five  \n six '
        packet1 = dfb.DataPacket(data1)
        
        # split lines shall ignore empty lines
        packet2 = dfb.DataPacket(
        """line1
        line2
        
        
        \rline3
        \n\n\r\r\n\n\r\r\n
        line4""")
        self.splitter.send(packet1)
        self.assertEquals(self.sink.results[0].data, 'one')
        self.assertEquals(self.sink.results[-1].data, ' six ')
        self.assertEqual(len(self.sink.results), 6)
        vals = [res.data for res in self.sink.results]
        self.assertEqual(vals, data1.splitlines())
        self.assertEqual(vals, 
                         ['one', 'two', 'three', 'four ', ' five  ', ' six '])
        ##self.assertEquals(self.sink.all_data, 'onetwothreefourfive')
        self.splitter.send(packet2)
        self.assertEquals(self.sink.results[6].data, 'line1')
        self.assertEquals(self.sink.results[-1].data, '        line4')
        self.assertEqual(len(self.sink.results), 10)
        
        
class TestTankBranch(unittest.TestCase):

    def setUp(self):
        self.sink = df.Sink()
        self.factory = ff.DemoFilterFactory()

    def tearDown(self):
        pass
    
    def test_tank_all_data(self):
        param_dict = dict(ftype='tank_branch', tank_size=3)  # Hold 3 packets
        tank_branch = self.factory.create_filter(param_dict)   
        self.assertEquals(tank_branch._priority_queue.queue_size(), 3)
        self.assertEquals(tank_branch.packets_held, 0)
        packet1 = dfb.DataPacket('123')
        packet2 = dfb.DataPacket('456')
        packet3 = dfb.DataPacket('789')
        packet4 = dfb.DataPacket('0AB')
        packet5 = dfb.DataPacket('CDE')
        self.assertEquals(tank_branch.all_data, [])
        tank_branch.send(packet1)
        self.assertEquals(tank_branch.packets_held, 1)
        self.assertEquals(tank_branch.all_data, ['123'])
        tank_branch.send(packet2)
        self.assertEquals(tank_branch.packets_held, 2)
        self.assertEquals(tank_branch.all_data, ['123', '456'])
        tank_branch.send(packet3)
        self.assertEquals(tank_branch.packets_held, 3)
        self.assertEquals(tank_branch.all_data, ['123', '456', '789'])
        tank_branch.send(packet4)
        self.assertEquals(tank_branch.packets_held, 3)
        self.assertEquals(tank_branch.all_data, ['456', '789', '0AB'])
        tank_branch.send(packet5)
        self.assertEquals(tank_branch.packets_held, 3)
        self.assertEquals(tank_branch.all_data, ['789', '0AB', 'CDE'])
        tank_branch.tank_size = 2
        self.assertEquals(tank_branch.packets_held, 2)
        self.assertEquals(tank_branch.all_data, ['0AB', 'CDE'])
        
    def test_tank_branch_keys(self):
        param_dict = dict(ftype='tank_branch', tank_size=3)
        tank_branch = self.factory.create_filter(param_dict)   
        self.assertEquals(tank_branch._keys, ['tank_size',
                                              'priority_field_name',                                              
                                              ])

        
class TestTankQueue(unittest.TestCase):

    def setUp(self):
        self.sink = df.Sink()

    def tearDown(self):
        pass
    
    def test_load_unlimited_tank(self):
        # Test that TankQueue takes all packets if tank_size == -1
##        self.tank_queue = df.TankQueue(tap_open=False)
        self.tank_queue = df.TankQueue(tank_size=-1)  # Hold all packets
        self.tank_queue.next_filter = self.sink
        packet1 = dfb.DataPacket('123')
        packet2 = dfb.DataPacket('456')
        packet3 = dfb.DataPacket('789')
        self.tank_queue.send(packet1)
        self.tank_queue.send(packet2)
        self.tank_queue.send(packet3)
        self.assertEquals(self.tank_queue.packets_held, 3)
        self.assertEquals(self.sink.results, [])
##        self.tank_queue.open_tap()
        self.tank_queue.tank_size = 0  # Release all packets
        self.assertEquals(self.tank_queue.packets_held, 0)
        self.assertEquals(len(self.sink.results), 3)
        
    def test_load_sized_tank(self):
        # tests the contents and behaviour of the tank when we start with a tank
        # size of 2 and make various changes to the tank size, while adding 
        # packets
        self.tank_queue = df.TankQueue(tank_size=2)  # Hold 2 packets
        self.assertEquals(self.tank_queue._priority_queue.queue_size(), 2)
        self.assertEquals(self.tank_queue.packets_held, 0)
        self.tank_queue.next_filter = self.sink
        packet1 = dfb.DataPacket('123')
        packet2 = dfb.DataPacket('456')
        packet3 = dfb.DataPacket('789')
        packet4 = dfb.DataPacket('ABC')
        packet5 = dfb.DataPacket('DEF')
        self.tank_queue.send(packet1)
        self.assertEquals(self.tank_queue.packets_held, 1)
        self.tank_queue.send(packet2)
        self.assertEquals(self.tank_queue.packets_held, 2)
        self.tank_queue.send(packet3)
        self.assertEquals(self.tank_queue.packets_held, 2)
        self.assertEquals(len(self.sink.results), 1)
        self.tank_queue.tank_size = -1  # Hold all packets; nothing changes
        self.assertEquals(self.tank_queue.packets_held, 2)
        self.assertEquals(self.tank_queue._priority_queue.queue_size(), 2)
        self.assertEquals(len(self.sink.results), 1)
        self.tank_queue.send(packet4)
        self.assertEquals(self.tank_queue.packets_held, 3)        
        self.tank_queue.tank_size = 5  
        self.assertEquals(self.tank_queue._priority_queue.queue_size(), 5)
        self.assertEquals(self.tank_queue.packets_held, 3)        
        self.tank_queue.send(packet5)
        self.assertEquals(self.tank_queue.packets_held, 4)
        self.assertEquals(len(self.sink.results), 1)
        self.tank_queue.tank_size = 0  
        self.assertEquals(len(self.sink.results), 5)
        
    def test_tank_all_data(self):
        # test the .all_data() functionality of tank queue, all_data() should 
        # concatenate the data attributes of packets held in the tank, in order 
        # of priority
        self.tank_queue = df.TankQueue(tank_size=3)  # Hold 3 packets
        self.assertEquals(self.tank_queue._priority_queue.queue_size(), 3)
        self.assertEquals(self.tank_queue.packets_held, 0)
        packet1 = dfb.DataPacket('123')
        packet2 = dfb.DataPacket('456')
        packet3 = dfb.DataPacket('789')
        self.assertEquals(self.tank_queue.all_data, [])
        self.tank_queue.send(packet1)
        self.assertEquals(self.tank_queue.packets_held, 1)
        self.assertEquals(self.tank_queue.all_data, ['123'])
        self.tank_queue.send(packet2)
        self.assertEquals(self.tank_queue.packets_held, 2)
        self.assertEquals(self.tank_queue.all_data, ['123', '456'])
        self.tank_queue.send(packet3)
        self.assertEquals(self.tank_queue.packets_held, 3)
        self.assertEquals(self.tank_queue.all_data, ['123', '456', '789'])

    def test_tank_sorted_packets(self):
        # 
        self.tank_queue = df.TankQueue(tank_size=3)  # Hold 3 packets
        self.assertEquals(self.tank_queue._priority_queue.queue_size(), 3)
        self.assertEquals(self.tank_queue.packets_held, 0)
        packet1 = dfb.DataPacket('789')
        packet2 = dfb.DataPacket('456')
        packet3 = dfb.DataPacket('123')
        self.assertEquals(self.tank_queue.sorted_packets, [])
        self.tank_queue.send(packet1)
        self.assertEquals(self.tank_queue.sorted_packets, [packet1])
        self.tank_queue.send(packet2)
        self.assertEquals(self.tank_queue.sorted_packets, [packet1, packet2])
        self.tank_queue.send(packet3)
        self.assertEquals(self.tank_queue.sorted_packets, 
                          [packet1, packet2, packet3])
        
        
class TestWaste(unittest.TestCase):
    def setUp(self):
        self.sink = df.Sink()
    
    def tearDown(self):
        pass
    
    def test_waste(self):
        waste = df.Waste()
        waste.next_filter = self.sink
        packet1 = dfb.DataPacket('abcdef')
        waste.send(packet1)
        self.assertEquals(len(self.sink.results), 0)
        self.assertEquals(self.sink.all_data, [])
        
class TestWrap(unittest.TestCase):

    def setUp(self):
        self.sink = df.Sink()

    def tearDown(self):
        pass
    
    def test_wrap_nothing(self):
        wrap = df.Wrap()
        wrap.next_filter = self.sink
        packet = dfb.DataPacket(data=u'abcdef')
        wrap.send(packet)
        self.assertEquals(self.sink.results[-1].data, u'abcdef')

    ##def test_wrap_prefix(self):
        ##wrap = df.Wrap(data_prefix=u'===')
        ##wrap.next_filter = self.sink
        ##packet = dfb.DataPacket(data=u'abcdef')
        ##wrap.send(packet)
        ##self.assertEquals(self.sink.results[-1].data, u'===abcdef')

    def test_wrap_prefix(self):
        # Not using unicode
        wrap = df.Wrap(data_prefix='===')
        wrap.next_filter = self.sink
        packet = dfb.DataPacket(data='abcdef\xfa')  # Includes non-ascii char
        wrap.send(packet)
        self.assertEquals(self.sink.results[-1].data, '===abcdef\xfa')

        
    def test_wrap_unicode_prefix(self):
        wrap = df.Wrap(data_prefix=u'\xff\xff')
        wrap.next_filter = self.sink
        packet = dfb.DataPacket(data=u'abcdef')
        wrap.send(packet)
        self.assertEquals(self.sink.results[-1].data, u'\xff\xffabcdef')

    def test_wrap_suffix(self):
        wrap = df.Wrap(data_suffix='+++')
        wrap.next_filter = self.sink
        packet = dfb.DataPacket(data='abcdef')
        wrap.send(packet)
        self.assertEquals(self.sink.results[-1].data, 'abcdef+++')
    
    def test_wrap_both_ends(self):
        wrap = df.Wrap(data_prefix=u'===', data_suffix=u'+++')
        wrap.next_filter = self.sink
        packet = dfb.DataPacket(data=u'abcdef\xfa')
        wrap.send(packet)
        self.assertEquals(self.sink.results[-1].data, u'===abcdef\xfa+++')
        
class TestWriteConfigObjFile(unittest.TestCase):
    def setUp(self):
        self.dest_file_name = "write_configobj_file_test"
        self.dest_file_suffix = "cfg"
        self.dest_file_path = self.dest_file_name + '.' + self.dest_file_suffix
        
        self.fltr = df.WriteConfigObjFile(dest_file_name=self.dest_file_name,
                                          dest_file_suffix=self.dest_file_suffix)
        self.sink = df.Sink()
        self.fltr.next_filter = self.sink
    
    def tearDown(self):
        if os.path.exists(self.dest_file_path):
            os.remove(self.dest_file_path)
    
    def test_write_config_obj_file_single_section(self):
        test_section = {"test_section": {"value1":'1',
                                         "value2":'2'}}
        section_packet = dfb.DataPacket(data=test_section)
        self.fltr.send(section_packet)
        # Assert that the filter sends on the incoming packet unchanged.
        self.assertEqual(section_packet,
                         self.sink.results[0])
        
        # Shut down the filter to cause the writing of the file.
        self.fltr.shut_down()
        
        # Assert that the section has been written with configobj.
        with open(self.dest_file_path, 'r') as out_file_obj:
            config_obj = ConfigObj(out_file_obj)
            self.assertEqual(config_obj.dict(),
                             test_section)
            
    def test_write_config_obj_file_multiple_sections(self):
        all_test_sections = {}
        for x in range(10):
            test_section_key = "test_section_%d" % x
            test_section_value = {"key_%d" % x: "%d" % x,
                                  "key_%d": "%d" % x}
            test_section = {"test_section_%d" % x: test_section_value}
            all_test_sections[test_section_key] = test_section_value
            section_packet = dfb.DataPacket(data=test_section)
            self.fltr.send(section_packet)
            # Assert that the latest packet has arrived in the sink.
            self.assertEqual(section_packet,
                             self.sink.results[-1])
        self.fltr.shut_down()
        with open(self.dest_file_path, 'r') as out_file_obj:
            config_obj = ConfigObj(out_file_obj)
            self.assertEqual(config_obj.dict(),
                             all_test_sections)


class TestWriteFile(unittest.TestCase):
    def setUp(self):
        self.file_name_out = os.path.join(data_dir5, 'short.tmp')

    def tearDown(self):
        if os.path.exists(self.file_name_out):
            os.remove(self.file_name_out)

    def test_write_file(self):
        write_file = df.WriteFile(dest_file_name=self.file_name_out)
        packet = dfb.DataPacket(data='abcdef')
        write_file.send(packet)
        write_file.shut_down()
        read_batch2 = df.ReadBatch(batch_size=0x2000)
        sink2 = df.Sink()
        read_batch2.next_filter = sink2
        read_batch2.send(dfb.DataPacket(self.file_name_out))
        read_batch2.shut_down()
        self.assertEquals(sink2.results[0].data, 'abcdef')

    def test_write_file_append(self):
        write_file1 = df.WriteFile(dest_file_name=self.file_name_out)
        packet1 = dfb.DataPacket(data='abcdef')
        write_file1.send(packet1)
        write_file1.shut_down()
        write_file2 = df.WriteFile(dest_file_name=self.file_name_out,
                                   append=True)
        packet2 = dfb.DataPacket(data='ghijkl')
        write_file2.send(packet2)
        write_file2.shut_down()
        read_batch2 = df.ReadBatch(batch_size=0x2000)
        sink2 = df.Sink()
        read_batch2.next_filter = sink2
        read_batch2.send(dfb.DataPacket(self.file_name_out))
        read_batch2.shut_down()
        self.assertEquals(sink2.results[0].data, 'abcdefghijkl')
        
    def test_write_file_with_suffix(self):
        write_file = df.WriteFile(dest_file_name=self.file_name_out)
        write_file.name = 'writer'
        suffix_messenger = dfb.MessageBottle('writer',
                                             'change_write_suffix',
                                             file_name_suffix='new')
        write_file.send(suffix_messenger)
        data_to_write = dfb.DataPacket(data='wooo')
        write_file.send(data_to_write)
        write_file.shut_down()
        dir_content = os.listdir(os.path.dirname(self.file_name_out))
        expected_file_name = 'short.out.new'
        self.assertEqual(dir_content.count(expected_file_name), 1)
        
    def test_write_file_bzip2(self):
        write_file = df.WriteFile(dest_file_name = self.file_name_out, \
                                  compress='bzip')
        test_data = 'abcdef'
        write_file.send(dfb.DataPacket(data=test_data))
        write_file.shut_down()
        open_bz2 = bz2.BZ2File(self.file_name_out + '.bz2', mode='r')
        self.assertEqual(open_bz2.read(), test_data)
        open_bz2.close()
        
    def test_write_file_with_suffix_bzip2(self):
        write_file = df.WriteFile(dest_file_name = self.file_name_out, \
                                  compress='bzip')
        write_file.name = 'writer'
        suffix_messenger = dfb.MessageBottle('writer',
                                             'change_write_suffix',
                                             file_name_suffix='tmp')
        string_1 = 'first file'
        string_2 = 'second file'
        data_to_write_1 = dfb.DataPacket(data=string_1)
        data_to_write_2 = dfb.DataPacket(data=string_2)
        write_file.send(data_to_write_1)
        write_file.send(suffix_messenger)
        write_file.send(data_to_write_2)
        write_file.shut_down()
        open_bz2_1 = bz2.BZ2File(self.file_name_out + '.bz2', mode='r')
        open_bz2_2 = bz2.BZ2File(self.file_name_out + '.tmp' + '.bz2', mode='r')
        self.assertEqual(open_bz2_1.read(), string_1)
        self.assertEqual(open_bz2_2.read(), string_2)
        for fhandle in (open_bz2_1, open_bz2_2):
            fhandle.close()
        os.remove(self.file_name_out + '.tmp' + '.bz2')
        
    def test_write_file_none(self):
        # Ensure None does not raise an exception.
        write_file = df.WriteFile(dest_file_name = None)
        string_1 = 'first file'
        data_to_write_1 = dfb.DataPacket(data=string_1)
        write_file.send(data_to_write_1)
        write_file.shut_down()



if __name__ == '__main__':  #pragma: nocover
##    TestEmbedPython('test_embed_has_access_to_refinery_keywords_and_defaults').run()
    TestEmbedPython('test_update_live').run()
    ##TestDataLength('test_find_data_length').run()
##    TestCallbackOnAttribute('test_callback_using_msg_bottle').run()
        ##TestPipelines('test_filter11').run()
##    runner = unittest.TextTestRunner()
##    runner.run(TestPipelines('test_filter13'))
##    TestTank('test_load_unlimited_tank').run()
##    TestPassThrough('test_pass_through_multiple_send').run()
##    TestBranchClone('test_branch_clone').run()
##    TestCountLoops('test_counting').run()
   ##TestSetAttributesToData('test_set_attributes_list_params').run()
    ##TestSetAttributesToData('test_set_headers').run()
    ##TestSetAttributesToData('test_with_line_nos').run()
    ##TestWriteFile('test_write_file_with_suffix').run()
    ##TestPassThrough('test_pass_through').run()
    #TestBranchFirstPart('test_branch_first_part').run()
    print '\n**1910** Finished'
