#!/usr/bin/env python

"""
This module contains some of the most central FireWorks classes:


- A Workflow is a sequence of FireWorks as a DAG (directed acyclic graph)
- A FireWork defines a workflow step and contains one or more FireTasks along
 with its Launches.
- A Launch describes the run of a FireWork on a computing resource.
- A FireTaskBase defines the contract for tasks that run within a FireWork (
FireTasks)
- A FWAction encapsulates the output of a FireTask and tells FireWorks what
to do next after a job completes
"""

from collections import defaultdict, OrderedDict
import abc
from datetime import datetime
import os
import pprint

from monty.io import reverse_readline, zopen
from monty.os.path import zpath
from six import add_metaclass

from fireworks.fw_config import TRACKER_LINES, NEGATIVE_FWID_CTR
from fireworks.core.fworker import FWorker
from fireworks.utilities.dict_mods import apply_mod
from fireworks.utilities.fw_serializers import FWSerializable, \
    recursive_serialize, recursive_deserialize, serialize_fw
from fireworks.utilities.fw_utilities import get_my_host, get_my_ip, \
    NestedClassGetter


__author__ = "Anubhav Jain"
__credits__ = "Shyue Ping Ong"
__copyright__ = "Copyright 2013, The Materials Project"
__version__ = "0.1"
__maintainer__ = "Anubhav Jain"
__email__ = "ajain@lbl.gov"
__date__ = "Feb 5, 2013"


@add_metaclass(abc.ABCMeta)
class FireTaskMeta(type):
    def __call__(cls, *args, **kwargs):
        o = type.__call__(cls, *args, **kwargs)
        for k in cls.required_params:
            if k not in o:
                raise ValueError("Required parameter {} not specified!"
                                 .format(k))
        return o


@add_metaclass(FireTaskMeta)
class FireTaskBase(defaultdict, FWSerializable):
    """
    FireTaskBase is used like an abstract class that defines a computing task
    (FireTask). All FireTasks should inherit from FireTaskBase.

    You can set parameters of a FireTask like you'd use a dict.
    """

    # Specify required parameters with class variable. Consistency will be
    # checked upon init.
    required_params = []

    def __init__(self, *args, **kwargs):
        dict.__init__(self, *args, **kwargs)

    @abc.abstractmethod
    def run_task(self, fw_spec):
        """
        This method gets called when the FireTask is run. It can take in a
        FireWork spec, perform some task using that data, and then return an
        output in the form of a FWAction.

        :param fw_spec: (dict) a FireWork spec
        :return: (FWAction)
        """
        pass

    @serialize_fw
    @recursive_serialize
    def to_dict(self):
        return dict(self)

    @classmethod
    @recursive_deserialize
    def from_dict(cls, m_dict):
        return cls(m_dict)


class FWAction(FWSerializable):
    """
    A FWAction encapsulates the output of a FireTask (it is returned by a
    FireTask after the FireTask completes). The
     FWAction allows a user to store rudimentary output data as well as
     return commands that alter the workflow.
    """

    def __init__(self, stored_data=None, exit=False, update_spec=None,
                 mod_spec=None, additions=None, detours=None,
                 defuse_children=False):
        """
        :param stored_data: (dict) data to store from the run. Does not
        affect the operation of FireWorks.
        :param exit: (bool) if set to True, any remaining FireTasks within
        the same FireWork are skipped.
        :param update_spec: (dict) specifies how to update the child FW's spec
        :param mod_spec: ([dict]) update the child FW's spec using the
        DictMod language (more flexible than update_spec)
        :param additions: ([Workflow]) a list of WFs/FWs to add as children
        :param detours: ([Workflow]) a list of WFs/FWs to add as children (
        they will inherit the current FW's children)
        :param defuse_children: (bool) defuse all the original children of
        this FireWork
        """
        mod_spec = mod_spec if mod_spec is not None else []
        additions = additions if additions is not None else []
        detours = detours if detours is not None else []

        self.stored_data = stored_data if stored_data else {}
        self.exit = exit
        self.update_spec = update_spec if update_spec else {}
        self.mod_spec = mod_spec if isinstance(mod_spec, list) else [mod_spec]
        self.additions = additions if isinstance(additions, list) else [
            additions]
        self.detours = detours if isinstance(detours, list) else [detours]
        self.defuse_children = defuse_children

    @recursive_serialize
    def to_dict(self):
        return {'stored_data': self.stored_data, 'exit': self.exit,
                'update_spec': self.update_spec,
                'mod_spec': self.mod_spec, 'additions': self.additions,
                'detours': self.detours,
                'defuse_children': self.defuse_children}

    @classmethod
    @recursive_deserialize
    def from_dict(cls, m_dict):
        d = m_dict
        additions = [Workflow.from_dict(f) for f in d['additions']]
        detours = [Workflow.from_dict(f) for f in d['detours']]
        return FWAction(d['stored_data'], d['exit'], d['update_spec'],
                        d['mod_spec'], additions, detours,
                        d['defuse_children'])

    @property
    def skip_remaining_tasks(self):
        """
        If the FWAction gives any dynamic action, we skip the subsequent
        FireTasks

        :return: (bool)
        """
        return self.exit or self.detours or self.additions or self.defuse_children

    def __str__(self):
        return "FWAction\n" + pprint.pformat(self.to_dict())


class FireWork(FWSerializable):
    """
    A FireWork is a workflow step and might be contain several FireTasks
    """

    STATE_RANKS = {'ARCHIVED': -1, 'DEFUSED': 0, 'WAITING': 1, 'READY': 2,
                   'RESERVED': 3, 'FIZZLED': 4, 'RUNNING': 5, 'COMPLETED': 7}

    def __init__(self, tasks, spec=None, name=None, launches=None,
                 archived_launches=None, state='WAITING', created_on=None,
                 fw_id=None):
        """
        :param tasks: ([FireTask]) a list of FireTasks to run in sequence
        :param spec: (dict) specification of the job to run. Used by the
        FireTask
        :param launches: ([Launch]) a list of Launch objects of this FireWork
        :param archived_launches: ([Launch]) a list of archived Launch
        objects of this FireWork
        :param state: (str) the state of the FW (e.g. WAITING, RUNNING,
        COMPLETED, ARCHIVED)
        :param fw_id: (int) an identification number for this FireWork
        """

        tasks = tasks if isinstance(tasks, list) else [tasks]

        self.tasks = tasks
        self.spec = spec if spec else {}
        self.spec['_tasks'] = [t.to_dict() for t in
                               tasks]  # put tasks in a special location of the spec

        self.name = name or 'Unnamed FW'  # do it this way to prevent None
        # names
        if fw_id is not None:
            self.fw_id = fw_id
        else:
            global NEGATIVE_FWID_CTR
            NEGATIVE_FWID_CTR -= 1
            self.fw_id = NEGATIVE_FWID_CTR

        self.launches = launches if launches else []
        self.archived_launches = archived_launches if archived_launches else []
        self.created_on = created_on or datetime.utcnow()

        self.state = state

    @recursive_serialize
    def to_dict(self):
        m_dict = {'spec': self.spec, 'fw_id': self.fw_id,
                  'created_on': self.created_on}

        # only serialize these fields if non-empty
        if len(self.launches) > 0:
            m_dict['launches'] = self.launches

        if len(self.archived_launches) > 0:
            m_dict['archived_launches'] = self.archived_launches

        if self.state != 'WAITING':
            m_dict['state'] = self.state

        m_dict['name'] = self.name

        return m_dict

    def _rerun(self):
        """
        Moves all Launches to archived Launches and resets the state to
        'WAITING'. The FireWork can thus be re-run \
        even if it was Launched in the past. This method should be called by
        a Workflow because a refresh is needed \
        after calling this method.

        """

        self.archived_launches.extend(self.launches)
        self.launches = []
        self.state = 'WAITING'

    def to_db_dict(self):
        m_dict = self.to_dict()
        m_dict['launches'] = [l.launch_id for l in
                              self.launches]  # the launches are stored
        # separately
        m_dict['archived_launches'] = [l.launch_id for l in
                                       self.archived_launches]  # the
        # archived launches are stored separately
        m_dict['state'] = self.state
        return m_dict

    @classmethod
    @recursive_deserialize
    def from_dict(cls, m_dict):
        tasks = m_dict['spec']['_tasks']
        launches = [Launch.from_dict(tmp) for tmp in m_dict.get('launches', [])]
        archived_launches = [Launch.from_dict(tmp) for tmp in
                             m_dict.get('archived_launches', [])]
        fw_id = m_dict.get('fw_id', -1)
        state = m_dict.get('state', 'WAITING')
        created_on = m_dict.get('created_on', None)
        name = m_dict.get('name', None)

        return FireWork(tasks, m_dict['spec'], name, launches, archived_launches,
                        state, created_on, fw_id)

    def __str__(self):
        return 'FireWork object: (id: %i , name: %s)' % (self.fw_id, self.fw_name)


class Tracker(FWSerializable, object):
    """
    A Tracker monitors a file and returns the last N lines for updating the Launch object
    """

    MAX_TRACKER_LINES = 1000

    def __init__(self, filename, nlines=TRACKER_LINES, content=''):
        if nlines > self.MAX_TRACKER_LINES:
            raise ValueError("Tracker only supports a maximum of {} lines; you put {}.".format(
                self.MAX_TRACKER_LINES, nlines))
        self.filename = filename
        self.nlines = nlines
        self.content = content

    def track_file(self, launch_dir=None):
        """
        Reads the monitored file and returns back the last N lines
        :param launch_dir: directory where job was launched in case of relative filename
        :return:
        """
        m_file = self.filename
        if launch_dir and not os.path.isabs(self.filename):
            m_file = os.path.join(launch_dir, m_file)

        lines = []
        if os.path.exists(m_file):
            with zopen(zpath(m_file)) as f:
                for l in reverse_readline(f):
                    lines.append(l)
                    if len(lines) == self.nlines:
                        break
            self.content = '\n'.join(reversed(lines))

        return self.content

    def to_dict(self):
        m_dict = {'filename': self.filename, 'nlines': self.nlines}
        if self.content:
            m_dict['content'] = self.content
        return m_dict

    @classmethod
    def from_dict(cls, m_dict):
        return Tracker(m_dict['filename'], m_dict['nlines'], m_dict.get('content', ''))

    def __str__(self):
        return '### Filename: {}\n{}'.format(self.filename, self.content)


class Launch(FWSerializable, object):
    """
    A Launch encapsulates data about a specific run of a FireWork on a
    computing resource
    """

    def __init__(self, state, launch_dir, fworker=None, host=None, ip=None,
                 trackers=None, action=None, state_history=None,
                 launch_id=None, fw_id=None):
        """
        :param state: (str) the state of the Launch (e.g. RUNNING, COMPLETED)
        :param launch_dir: (str) the directory where the Launch takes place
        :param fworker: (FWorker) The FireWorker running the Launch
        :param host: (str) the hostname where the launch took place (set
        automatically if None)
        :param ip: (str) the IP address where the launch took place (set
        automatically if None)
        :param trackers: ([Tracker]) File Trackers for this Launch
        :param action: (FWAction) the output of the Launch
        :param state_history: ([dict]) a history of all states of the Launch
        and when they occurred
        :param launch_id: (int) launch_id set by the LaunchPad
        :param fw_id: (int) id of the FireWork this Launch is running
        """

        if state not in FireWork.STATE_RANKS:
            raise ValueError("Invalid launch state: {}".format(state))

        self.launch_dir = launch_dir
        self.fworker = fworker or FWorker()
        self.host = host or get_my_host()
        self.ip = ip or get_my_ip()
        self.trackers = trackers if trackers else []
        self.action = action if action else None
        self.state_history = state_history if state_history else []
        self.state = state
        self.launch_id = launch_id
        self.fw_id = fw_id

    def touch_history(self, update_time=None):
        """
        Updates the update_at field of the state history of a Launch. Used to
         ping that a Launch is still alive.
        """
        update_time = update_time or datetime.utcnow()
        self.state_history[-1]['updated_on'] = update_time

    def set_reservation_id(self, reservation_id):
        """
        Adds the job_id to the reservation

        :param reservation_id: (str) the id of the reservation (e.g.,
        queue reservation)
        """
        for data in self.state_history:
            if data['state'] == 'RESERVED' and 'reservation_id' not in data:
                data['reservation_id'] = str(reservation_id)
                break

    @property
    def state(self):
        """
        :return: (str) The current state of the Launch.
        """
        return self._state

    @state.setter
    def state(self, state):
        """
        Setter for the the Launch's state. Automatically triggers an update
        to state_history.

        :param state: (str) the state to set for the Launch
        """
        self._state = state
        self._update_state_history(state)

    @property
    def time_start(self):
        """
        :return: (datetime) the time the Launch started RUNNING
        """
        return self._get_time('RUNNING')

    @property
    def time_end(self):
        """
        :return: (datetime) the time the Launch was COMPLETED or FIZZLED
        """
        return self._get_time(['COMPLETED', 'FIZZLED'])

    @property
    def time_reserved(self):
        """
        :return: (datetime) the time the Launch was RESERVED in the queue
        """
        return self._get_time('RESERVED')

    @property
    def last_pinged(self):
        """
        :return: (datetime) the time the Launch last pinged a heartbeat that
        it was still running
        """
        return self._get_time('RUNNING', True)

    @property
    def runtime_secs(self):
        """
        :return: (int) the number of seconds that the Launch ran for
        """
        start = self.time_start
        end = self.time_end
        if start and end:
            return (end - start).total_seconds()

    @property
    def reservedtime_secs(self):
        """
        :return: (int) number of seconds the Launch was stuck as RESERVED in
        a queue
        """
        start = self.time_reserved
        if start:
            end = self.time_start if self.time_start else datetime \
                .utcnow()
            return (end - start).total_seconds()

    @recursive_serialize
    def to_dict(self):
        return {'fworker': self.fworker, 'fw_id': self.fw_id,
                'launch_dir': self.launch_dir, 'host': self.host,
                'ip': self.ip, 'trackers': self.trackers,
                'action': self.action, 'state': self.state,
                'state_history': self.state_history,
                'launch_id': self.launch_id}

    @recursive_serialize
    def to_db_dict(self):
        m_d = self.to_dict()
        m_d['time_start'] = self.time_start
        m_d['time_end'] = self.time_end
        m_d['runtime_secs'] = self.runtime_secs
        if self.reservedtime_secs:
            m_d['reservedtime_secs'] = self.reservedtime_secs
        return m_d

    @classmethod
    @recursive_deserialize
    def from_dict(cls, m_dict):
        fworker = FWorker.from_dict(m_dict['fworker']) if m_dict['fworker'] else None
        action = FWAction.from_dict(m_dict['action']) if m_dict.get(
            'action') else None
        trackers = [Tracker.from_dict(f) for f in m_dict['trackers']] if m_dict.get(
            'trackers') else None
        return Launch(m_dict['state'], m_dict['launch_dir'], fworker,
                      m_dict['host'], m_dict['ip'], trackers, action,
                      m_dict['state_history'], m_dict['launch_id'],
                      m_dict['fw_id'])

    def _update_state_history(self, state):
        """
        Internal method to update the state history whenever the Launch state
         is modified

        :param state:
        """
        last_state = self.state_history[-1]['state'] if len(
            self.state_history) > 0 else None
        if state != last_state:
            now_time = datetime.utcnow()
            self.state_history.append({'state': state, 'created_on': now_time})
            if state in ['RUNNING', 'RESERVED']:
                self.touch_history()  # add updated_on key

    def _get_time(self, states, use_update_time=False):
        """
        Internal method to help get the time of various events in the Launch
        (e.g. RUNNING) from the state history

        :param states: match one of these states
        :param use_update_time: use the "updated_on" time rather than
        "created_on"
        :return: (datetime)
        """
        states = states if isinstance(states, list) else [states]
        for data in self.state_history:
            if data['state'] in states:
                if use_update_time:
                    return data['updated_on']
                return data['created_on']


class Workflow(FWSerializable):
    """
    A Workflow connects a group of FireWorks in an execution order
    """

    class Links(dict, FWSerializable):
        """
        An inner class for storing the DAG links between FireWorks
        """

        def __init__(self, *args, **kwargs):
            super(Workflow.Links, self).__init__(*args, **kwargs)

            for k, v in list(self.items()):
                if not isinstance(v, list):
                    self[k] = [v]  # v must be list
                if not isinstance(k, int):
                    try:
                        self[int(k)] = self[k]  # k must be int
                    except:
                        pass  # garbage input
                    del self[k]

        @property
        def nodes(self):
            allnodes = list(self.keys())
            for v in self.values():
                allnodes.extend(v)
            return list(set(allnodes))

        @property
        def parent_links(self):
            # note: if performance of parent_links becomes an issue,
            # override delitem/setitem to update parent_links
            child_parents = defaultdict(list)
            for (parent, children) in self.items():
                for child in children:
                    child_parents[child].append(parent)
            return dict(child_parents)

        def to_dict(self):
            # convert to str form for Mongo, which cannot have int keys
            return dict([(str(k), v) for (k, v) in self.items()])

        def to_db_dict(self):
            # convert to str form for Mongo, which cannot have int keys
            m_dict = {
                'links': dict([(str(k), v) for (k, v) in self.items()]),
                'parent_links': dict(
                    [(str(k), v) for (k, v) in self.parent_links.items()]),
                'nodes': self.nodes}
            return m_dict

        @classmethod
        def from_dict(cls, m_dict):
            return Workflow.Links(m_dict)

        def __setstate__(self, state):
            for k, v in state:
                self[k] = v

        def __reduce__(self):
            # to support Pickling of inner classes (for multi-job launcher's multiprocessing)
            # return a class which can return this class when called with the
            # appropriate tuple of arguments
            state = list(self.items())
            return (NestedClassGetter(),
                    (Workflow, self.__class__.__name__, ),
                    state)

    def __init__(self, fireworks, links_dict=None, name=None, metadata=None, created_on=None,
                 updated_on=None):
        """
        :param fireworks: ([FireWork]) - all FireWorks in this workflow
        :param links_dict: (dict) links between the FWs as (parent_id):[(
        child_id1, child_id2)]
        :param metadata: (dict) metadata for this Workflow
        """

        name = name or 'unnamed WF'# do it this way to prevent None names

        links_dict = links_dict if links_dict else {}

        self.id_fw = {}  # main dict containing mapping of an id to a
        # FireWork object
        for fw in fireworks:
            if fw.fw_id in self.id_fw:
                raise ValueError('FW ids must be unique!')
            self.id_fw[fw.fw_id] = fw

            if fw.fw_id not in links_dict:
                links_dict[fw.fw_id] = []

        self.links = Workflow.Links(links_dict)

        self.name = name

        # sanity: make sure the set of nodes from the links_dict is equal to
        # the set of nodes from id_fw
        if set(self.links.nodes) != set(map(int, self.id_fw.keys())):
            raise ValueError("Specified links don't match given FW")

        self.metadata = metadata if metadata else {}
        self.created_on = created_on or datetime.utcnow()
        self.updated_on = updated_on or datetime.utcnow()

    @property
    def fws(self):
        return list(self.id_fw.values())

    @property
    def state(self):

        # get state of workflow
        m_state = 'READY'
        states = [fw.state for fw in self.fws]
        if all([s == 'COMPLETED' for s in states]):
            m_state = 'COMPLETED'
        elif all([s == 'ARCHIVED' for s in states]):
            m_state = 'ARCHIVED'
        elif any([s == 'DEFUSED' for s in states]):
            m_state = 'DEFUSED'
        elif any([s == 'FIZZLED' for s in states]):
            m_state = 'FIZZLED'
        elif any([s == 'COMPLETED' for s in states]) or any([s == 'RUNNING' for s in states]):
            m_state = 'RUNNING'
        elif any([s == 'RESERVED' for s in states]):
            m_state = 'RESERVED'

        return m_state

    def apply_action(self, action, fw_id):
        """
        Apply a FWAction on a FireWork in the Workflow

        :param action: (FWAction) action to apply
        :param fw_id: (int) id of FireWork on which to apply the action
        :return: ([int]) list of FireWork ids that were updated or new
        """

        updated_ids = []

        # update the spec of the children FireWorks
        if action.update_spec:
            for cfid in self.links[fw_id]:
                self.id_fw[cfid].spec.update(action.update_spec)
                updated_ids.append(cfid)

        # update the spec of the children FireWorks using DictMod language
        if action.mod_spec:
            for cfid in self.links[fw_id]:
                for mod in action.mod_spec:
                    apply_mod(mod, self.id_fw[cfid].spec)
                    updated_ids.append(cfid)

        # defuse children
        if action.defuse_children:
            for cfid in self.links[fw_id]:
                self.id_fw[cfid].state = 'DEFUSED'
                updated_ids.append(cfid)

        # add detour FireWorks
        # this should be done *before* additions
        if action.detours:
            for wf in action.detours:
                new_updates = self._add_wf_to_fw(wf, fw_id, True)
                if len(set(updated_ids).intersection(new_updates)) > 0:
                    raise ValueError(
                        "Cannot use duplicated fw_ids when dynamically detouring workflows!")
                updated_ids.extend(new_updates)

        # add additional FireWorks
        if action.additions:
            for wf in action.additions:
                new_updates = self._add_wf_to_fw(wf, fw_id, False)
                if len(set(updated_ids).intersection(new_updates)) > 0:
                    raise ValueError(
                        "Cannot use duplicated fw_ids when dynamically adding workflows!")
                updated_ids.extend(new_updates)

        return list(set(updated_ids))

    def rerun_fw(self, fw_id, updated_ids=None):
        """
        Archives the launches of a FireWork so that it can be re-run.
        :param fw_id: (int)
        :return: ([int]) list of FireWork ids that were updated
        """

        updated_ids = updated_ids if updated_ids else set()
        m_fw = self.id_fw[fw_id]
        m_fw._rerun()
        updated_ids.add(fw_id)

        # re-run all the children
        for child_id in self.links[fw_id]:
            updated_ids = updated_ids.union(
                self.rerun_fw(child_id, updated_ids))

        # refresh the WF to get the states updated
        return self.refresh(fw_id, updated_ids)

    def _add_wf_to_fw(self, wf, fw_id, detour):
        """
        Internal method to add a workflow as a child to a FireWork

        :param wf: New Workflow to add
        :param fw_id: id of the FireWork on which to add the Workflow
        :param detour: whether to add the children of the current FireWork to
         the Workflow's leaves
        :return: ([int]) list of FireWork ids that were updated or new
        """
        updated_ids = []

        root_ids = wf.root_fw_ids
        leaf_ids = wf.leaf_fw_ids

        for new_fw in wf.fws:
            if new_fw.fw_id > 0:
                raise ValueError(
                    'FireWorks to add must use a negative fw_id! Got fw_id: '
                    '{}'.format(
                        new_fw.fw_id))

            self.id_fw[new_fw.fw_id] = new_fw  # add new_fw to id_fw

            if new_fw.fw_id in leaf_ids:
                if detour:
                    self.links[new_fw.fw_id] = list(
                        self.links[fw_id])  # add children of current FW to new FW
                else:
                    self.links[new_fw.fw_id] = []
            else:
                self.links[new_fw.fw_id] = wf.links[new_fw.fw_id]
            updated_ids.append(new_fw.fw_id)

        for root_id in root_ids:
            self.links[fw_id].append(root_id)  # add the root id as my child

        return updated_ids

    def refresh(self, fw_id, updated_ids=None):
        """
        Refreshes the state of a FireWork and any affected children.

        :param fw_id: (int) id of the FireWork on which to perform the refresh
        :param updated_ids: ([int])
        :return: ([int]) list of FireWork ids that were updated
        """

        updated_ids = updated_ids if updated_ids else set()  # these are the
        # fw_ids to re-enter into the database

        fw = self.id_fw[fw_id]
        prev_state = fw.state

        # if we're defused or archived, just skip altogether
        if fw.state == 'DEFUSED' or fw.state == 'ARCHIVED':
            return updated_ids

        # what are the parent states?
        parent_states = [self.id_fw[p].state for p in
                         self.links.parent_links.get(fw_id, [])]

        completed_parent_states = ['COMPLETED']
        if fw.spec.get('_allow_fizzled_parents'):
            completed_parent_states.append('FIZZLED')

        if len(parent_states) != 0 and not all(
                [s in completed_parent_states for s in parent_states]):
            m_state = 'WAITING'

        else:
            # my state depends on launch whose state has the highest 'score'
            # in STATE_RANKS
            max_score = 0
            m_state = 'READY'
            m_action = None

            # TODO: pick the first launch in terms of end date that matches
            # 'COMPLETED'; multiple might exist
            for l in fw.launches:
                if FireWork.STATE_RANKS[l.state] > max_score:
                    max_score = FireWork.STATE_RANKS[l.state]
                    m_state = l.state
                    if m_state == 'COMPLETED':
                        m_action = l.action

            # This part is confusing and rare - report any FIZZLED parents if allow_fizzed
            # allows us to handle FIZZLED jobs
            if fw.spec.get('_allow_fizzled_parents'):
                parent_fws = [self.id_fw[p].to_dict() for p in self.links.parent_links.get(fw_id, []) if self.id_fw[p].state == 'FIZZLED']
                if len(parent_fws) > 0:
                    fw.spec['_fizzled_parents'] = parent_fws
                    updated_ids.add(fw_id)

        fw.state = m_state

        if m_state != prev_state:
            updated_ids.add(fw_id)

            if m_state == 'COMPLETED':
                updated_ids = updated_ids.union(self.apply_action(m_action,
                                                                  fw.fw_id))

            # refresh all the children
            for child_id in self.links[fw_id]:
                updated_ids = updated_ids.union(
                    self.refresh(child_id, updated_ids))

        self.updated_on = datetime.utcnow()

        return updated_ids

    @property
    def root_fw_ids(self):
        """
        Gets root FireWorks of this workflow (those with no parents)

        :return: ([int]) FireWork ids of root FWs
        """

        all_ids = set(self.links.nodes)
        child_ids = set(self.links.parent_links.keys())
        root_ids = all_ids.difference(child_ids)
        return list(root_ids)

    @property
    def leaf_fw_ids(self):
        """
        Gets leaf FireWorks of this workflow (those with no children)

        :return: ([int]) FireWork ids of leaf FWs
        """

        leaf_ids = []
        for id, children in self.links.items():
            if len(children) == 0:
                leaf_ids.append(id)
        return leaf_ids

    def _reassign_ids(self, old_new):
        """
        Internal method to reassign FireWork ids, e.g. due to database insertion

        :param old_new: (dict)
        """

        # update id_fw
        new_id_fw = {}
        for (fwid, fws) in self.id_fw.items():
            new_id_fw[old_new.get(fwid, fwid)] = fws
        self.id_fw = new_id_fw

        # update the Links
        new_l = {}
        for (parent, children) in self.links.items():
            new_l[old_new.get(parent, parent)] = [old_new.get(child, child) for
                                                  child in children]
        self.links = Workflow.Links(new_l)

    def to_dict(self):
        return {'fws': [f.to_dict() for f in self.id_fw.values()],
                'links': self.links.to_dict(),
                'name': self.name,
                'metadata': self.metadata, 'updated_on': self.updated_on}

    def to_db_dict(self):
        m_dict = self.links.to_db_dict()
        m_dict['metadata'] = self.metadata
        m_dict['state'] = self.state
        m_dict['name'] = self.name
        m_dict['created_on'] = self.created_on
        m_dict['updated_on'] = self.updated_on
        return m_dict

    def to_display_dict(self):
        m_dict = self.to_db_dict()
        nodes = sorted(m_dict['nodes'])
        m_dict['name--id'] = self.name + '--' + str(nodes[0])
        m_dict['launch_dirs'] = OrderedDict(
            [(self._str_fw(x), [l.launch_dir for l in self.id_fw[x].launches])
             for x in nodes])
        m_dict['states'] = OrderedDict(
            [(self._str_fw(x), self.id_fw[x].state) for x in nodes])
        m_dict['nodes'] = [self._str_fw(x) for x in nodes]
        m_dict['links'] = OrderedDict(
            [(self._str_fw(k), [self._str_fw(v) for v in a]) for k, a in
             m_dict['links'].items()])
        m_dict['parent_links'] = OrderedDict(
            [(self._str_fw(k), [self._str_fw(v) for v in a]) for k, a in
             m_dict['parent_links'].items()])
        m_dict['states_list'] = '-'.join([a[0:4] for a in m_dict['states'].values()])
        return m_dict

    def _str_fw(self, fw_id):
        return self.id_fw[int(fw_id)].name + '--' + str(fw_id)

    @classmethod
    def from_dict(cls, m_dict):
        # accept either a Workflow dict or a FireWork dict
        if 'fws' in m_dict:
            created_on = m_dict.get('created_on')
            updated_on = m_dict.get('updated_on')
            return Workflow([FireWork.from_dict(f) for f in m_dict['fws']],
                            Workflow.Links.from_dict(m_dict['links']), m_dict.get('name'),
                            m_dict['metadata'], created_on, updated_on)
        else:
            return Workflow.from_FireWork(FireWork.from_dict(m_dict))

    @classmethod
    def from_FireWork(cls, fw, name=None, metadata=None):
        name = name if name else fw.name
        return Workflow([fw], None, name=name, metadata=metadata)

    def __str__(self):
        return 'Workflow object: (fw_ids: {} , name: {})'.format(self.id_fw.keys(), self.name)
