# coding: utf-8
import os
import glob2
import yaml
from flask import url_for
from jinja2 import Markup
from webassets import Bundle, Environment
from webassets.filter import Filter, register_filter

__all__ = ('register_assets')


class G(object):
    """Shared states in module."""
    env = None
    debug = False
    js_config = {}
    css_config = {}


def register_assets(app):
    """Register js and css assets."""
    static_path = app.config.get('STATIC_PATH')
    output_path = app.config.get('OUTPUT_PATH')
    output_url = app.config.get('OUTPUT_URL', '/static')

    # Load global state
    G.debug = app.debug
    G.env = Environment(static_path, url_mapping={
        output_path: output_url
    })
    G.js_config = yaml.load(open(os.path.join(static_path, 'js.yml'), 'r'))
    G.css_config = yaml.load(open(os.path.join(static_path, 'css.yml'), 'r'))

    register_filter(JsPageFilter)
    register_filter(CssPageFilter)

    # JS bundle
    js_page_paths = glob2.glob(
        os.path.join(static_path, G.js_config['page']) + '/**/*.js')
    js = Bundle(
        Bundle(*G.js_config['public']),
        Bundle(*js_page_paths, filters='jspage'),
        filters='jsmin',
        output=os.path.join(output_path, 'app.js'))
    G.env.register('app_js', js)

    # CSS bundle
    css_page_paths = glob2.glob(
        os.path.join(static_path, G.css_config['page']) + '/**/*.css')
    css = Bundle(
        Bundle(*G.css_config['public']),
        Bundle(css_page_paths, filters='csspage, less'),
        filters='cssmin',
        output=os.path.join(output_path, 'app.css'))
    G.env.register('app_css', css)

    # Attach webassets env to app
    app.webassets_env = G.env

    # Register jinja functions
    app.jinja_env.globals['app_link'] = app_link
    app.jinja_env.globals['app_script'] = app_script
    app.jinja_env.globals['get_page_name'] = get_page_name


def app_link(template_reference):
    """Generate css link tags for Flask app."""
    # if True:
    if G.debug:
        # 全局css引用
        links = [link(path) for path in G.css_config['public']]
        # 单页css引用
        page_link = link(get_page_css_path(template_reference))
        links.append(page_link)
        return Markup(''.join(links))
    else:
        return Markup("<link rel='stylesheet' href='%s'>" %
                      G.env['app_css'].urls()[0])


def app_script(template_reference):
    """Generate js script tags for Flask app."""
    # if True:
    if G.debug:
        # 全局js引用
        scripts = [script(path) for path in G.js_config['public']]
        # 单页js引用
        page_script = script(get_page_js_path(template_reference))
        scripts.append(page_script)
        return Markup(''.join(scripts))
    else:
        return Markup("<script type='text/javascript' src='%s'></script>" %
                      G.env['app_js'].urls()[0])


class JsPageFilter(Filter):
    """Filter for single page js codes."""
    name = 'jspage'

    def input(self, _in, out, **kwargs):
        js_page_path = G.js_config['page']
        relative_page_path = kwargs['source_path'].split(js_page_path)[1]
        page_name = get_page_name_from_template_name(relative_page_path)
        pre = "$(function(){ if($('body')[0].id == '%s') {" % page_name
        pro = "}});"
        out.write(pre + _in.read() + pro)

    def output(self, _in, out, **kwargs):
        out.write(_in.read())


class CssPageFilter(Filter):
    """Filter for single page css codes."""
    name = 'csspage'

    def input(self, _in, out, **kwargs):
        css_page_path = G.css_config['page']
        relative_page_path = kwargs['source_path'].split(css_page_path)[1]
        page_name = get_page_name_from_template_name(relative_page_path)
        pre = "#%s {" % page_name
        pro = "}"
        out.write(pre + _in.read() + pro)

    def output(self, _in, out, **kwargs):
        out.write(_in.read())


def static(filename):
    """Geenrate static resource url."""
    return url_for('static', filename=filename)


def script(path):
    """Generate script tag."""
    return Markup("<script type='text/javascript' src='%s'></script>" % static(path))


def link(path):
    """Generate css link tag."""
    return Markup("<link rel='stylesheet' href='%s'>" % static(path))


def get_page_name(template_reference):
    """Get current page name, with format as page-site-index."""
    template_name = get_template_name(template_reference)
    return get_page_name_from_template_name(template_name)


def get_page_name_from_template_name(template_name):
    """Get page name from template name."""
    return "page-%s" % template_name.strip('/') \
        .replace('/', '-').replace('_', '-') \
        .split('.')[0]


def get_page_js_path(template_reference):
    """Get js script path for current page."""
    template_name = get_template_name(template_reference)
    return os.path.join(G.js_config['page'], template_name.replace('html', 'js'))


def get_page_css_path(template_reference):
    """Get css script path for current page."""
    template_name = get_template_name(template_reference)
    return os.path.join(G.css_config['page'], template_name.replace('html', 'css'))


def get_template_name(template_reference):
    """Get current template name."""
    return template_reference._TemplateReference__context.name