# -*- encoding: utf-8 -*-
"""
    flask_bridgekeeper.bridgekeeper
    -------------------------------

    The heart of the extension.

    :copyright: (c) 2013 by Morgan Delahaye-Prat.
    :license: BSD, see LICENSE for more details.
"""


from __future__ import absolute_import
from __future__ import unicode_literals

from .identity import Anonymous

import flask, datetime


def context_processor():
    """
    Inject variables in Jinja.
    """
    return {
        'current_identity': getattr(flask.g, 'bridgekeeper'),
    }


class Bridgekeeper(object):
    """
    Central controller class that can be used to configure how 
    Flask-Bridgekeeper behaves. Each application that wants to use it has to
    create, or run :meth:`init_app` on, an instance of this class after the
    configuration was initialized.
    """

    def __init__(self, app=None, auth_provider=None, info_provider=None):

        if app is not None:
            self.init_app(app)
        self.auth_provider = auth_provider
        self.info_provider = info_provider

    def init_app(self, app):
        """
        Set up this instance for use with *app*, if no app was passed to the
        constructor.
        """
        app.bridgekeeper = self         # register the bridgekeeper
        app.before_request(self.identify)

        app.context_processor(context_processor)

    @staticmethod
    def deactivate():
        if hasattr(flask.g, 'bridgekeeper'):
            delattr(flask.g, 'bridgekeeper')
        flask.session.pop('bridgekeeper', None)

    def identify(self):

        anonymous = Anonymous()

        # sanity check
        if self.auth_provider is None or self.info_provider is None:
            anonymous.activate()
            return                              # early return

        full = None     # This will hold the full identity with its duties

        # get a previous Identity from the session if it exists and
        # is not outdated.
        former = flask.session.get('bridgekeeper', None)
        if former is not None and former.timeout > datetime.datetime.now():
            former.duties.intersection([])
            full = self.info_provider(former)

        # if no valid and complete identity was restored, retrieve it from
        # the normal authentication provider.
        if former is None or full is None:
            dummy = self.auth_provider()
            if dummy is None:
                anonymous.activate()
                return                          # early return
            full = self.info_provider(dummy)

        # activate the full identity or an Anonymous identity.
        if full is None:
            anonymous.activate()
        else:
            full.activate()

