# -*- encoding: utf-8 -*-
"""
    flask_bridgekeeper.identity
    ---------------------------

    :copyright: (c) 2013 by Morgan Delahaye-Prat.
    :license: BSD, see LICENSE for more details.
"""


from __future__ import absolute_import
from __future__ import unicode_literals

import flask, datetime

from .duty import DutySet


class Identity(object):
    """
    """

    def __init__(self, id, duties=None):

        self.id = id

        # the Identity is oudated by default
        self.timeout = datetime.datetime.now()

        self.duties = DutySet()
        if duties is not None:
            self.duties.update(duties)

    def __repr__(self):
        return '<Identity 0x{} id="{}", duties={}>'.format(id(self),
                                                           self.id,
                                                           list(self.duties))

    def activate(self):
        """
        Set as the active `Identity` for one request.
        """
        setattr(flask.g, 'bridgekeeper', self)

    def persist(self, timeout=0):
        """
        Save the `Identity` in the user session.

        If an `Identity` is found in the user session, the *authentication*
        step is skipped and the found identity is directly used by the
        *information* step.

        :param timeout: the time in seconds for which the `Identity` in the
                        session remains valid.
        """
        self.activate()
        self.timeout += datetime.timedelta(seconds=timeout)
        flask.session['bridgekeeper'] = self           # save it in the session

    def deactivate(self):
        """
        Remove the `Identity` from `flask.g` and the user session.
        """
        flask.current_app.bridgekeeper.deactivate()


class Anonymous(Identity):
    """
    An anonymous user with no roles by default.
    """
    def __init__(self):
        super(Anonymous, self).__init__(None)

    def persist(self, timeout=0):
        """
        The anonymous user cannot be made persistent. This action will clean
        the user's session and make the anonymous identity active.
        """
        self.activate()
        flask.session.pop('bridgekeeper', None)
