# -*- encoding: utf-8 -*-
"""
    flask_bridgekeeper.permissions
    ------------------------------

    :copyright: (c) 2013 by Morgan Delahaye-Prat.
    :license: BSD, see LICENSE for more details.
"""


from __future__ import absolute_import
from __future__ import unicode_literals

from .identity import Anonymous
from .duty import DutySet
from .exc import PermissionDenied

import flask, sys, inspect


def active_duties():
    """
    Retrieve from the application context the active duties of the current
    identity
    """
    identity = getattr(flask.g, 'bridgekeeper', None)
    if identity is not None:
        return identity.duties
    return []


class WithHackError(RuntimeError):
    pass


class Permissions(object):
    """
    Store a set of duties
    """

    def __init__(self, duties=None):
        self.duties = DutySet()
        if duties is not None:
            self.duties.update(duties)
        self._test = None

    # the duty checking operation can be time delayed. For this reason, the
    # operation is registered as a lambda for a later use.
    def any(self):
        # at least one duty active is in the permission
        self._test = lambda: True in [duty in self.duties for duty in active_duties()]
        return self

    def all(self):
        # all the duties in the permission are in the active duties
        self._test = lambda: not False in [duty in active_duties() for duty in self.duties]
        return self

    def none(self):
        # none of the duties in the permission are in the active duties
        self._test = lambda: not True in [duty in self.duties for duty in active_duties()]
        return self

    def __call__(self, func):

        # Make a closure on the right duty checking operation
        if self._test is None:
            self.any()
        registered_test = self._test
        self._test = None

        def wrapped(*args, **kwargs):

            # No access for the anonymous user.
            if type(getattr(flask.g, 'bridgekeeper', None)) is Anonymous:
                raise PermissionDenied
            if not len(self.duties) or registered_test():
                return func(*args, **kwargs)
            raise PermissionDenied

        return wrapped

    def __nonzero__(self):
        # python 2.x support
        return self.__bool__()

    def __bool__(self):
        # anonymous is always not allowed to access
        if type(getattr(flask.g, 'bridgekeeper', None)) is Anonymous:
            return False

        # is no duty has been set, the permission is always granted
        if not len(self.duties):
            return True

        # if no test is set, any() is call instead
        if self._test is None:
            self.any()
        test_todo = self._test
        self._test = None

        return test_todo()


    def _trace(self, frame, event, arg):
        raise WithHackError

    def __enter__(self):
        if not self:
            if flask.current_app.config.get('BRIDGEKEEPER_WITH_HACK', False):
                # inspired by https://github.com/rfk/withhacks
                sys.settrace(lambda *args, **keys: None)
                frame = inspect.currentframe(1)
                frame.f_trace = self._trace
            else:
                raise PermissionDenied()

    def __exit__(self, exc_type, *args):
        return exc_type is WithHackError