# -*- encoding: utf-8 -*-
"""
    tests
    -----

    :copyright: (c) 2013 by Morgan Delahaye-Prat.
    :license: BSD, see LICENSE for more details.
"""


from __future__ import absolute_import
from __future__ import unicode_literals

from contextlib import contextmanager
from flask import Flask, g, session, appcontext_pushed
from flask_bridgekeeper import Bridgekeeper, Identity, Duty


def auth_provider():
    """
    This is a test auth provider returning an `Identity` if `flask.g.test_user`
    is set to `valid_user`. Else it returns `None`.
    """
    test_user = getattr(g, 'test_user', None)
    if test_user in ['incomplete_user', 'valid_user',
                     'alternate_user', 'full_user']:
        return Identity(test_user)


def info_provider(identity):
    """
    This is a test info provider returning a full `Identity`
    """
    if identity.id in ['valid_user', 'alternate_user']:
        return Identity(identity.id, duties=[Duty('test', 'valid')])
    if identity.id == 'full_user':
        return Identity(identity.id, duties=[Duty('test', 'valid'),
                                             Duty('test', 'other')])

@contextmanager
def user_set(app, user, session_user, session_user_timeout):
    """
    Set a test_user in the application context for the test auth_provider
    """
    with app.test_request_context():
        g.test_user = user
        if session_user is not None:            # create a session user
            session_user.persist(session_user_timeout)
        app.preprocess_request()                # call request pre processing.
        yield


class BaseTestClass(object):
    """
    A base test class providing a valid flask application and a valid
    `Bridgekeeper` configuration with some users :

        - valid_user : a valid user
        - alternate_user : another valid user
        - incomplete_user : a misconfigured user
        - invalid_user : an invalid user
    """
    def setup(self):

        app = Flask(__name__)
        app.secret_key = 'test'

        bridgekeeper = Bridgekeeper(
            auth_provider=auth_provider,
            info_provider=info_provider
        )
        bridgekeeper.init_app(app)

        def ctx_factory(user=None, session_user=None, session_user_timeout=0):
            return user_set(app, user, session_user, session_user_timeout)

        self.ctx = ctx_factory
        self.app = app