"""Flask-Router --- Improved flask's router
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

Provides improved flask URL router.  
"""
from flask import request, abort

__all__ = ('Router', )

class Router(object):
    """ Improved flask's router (`app.route`)

    You can set multiple functions for one endpoint.  

    Here an example::   

        from flask.ext.router import Router

        # Create new router
        router = Router(app_or_blueprint, 'item_view')

        # Read view
        @router('/item/<int:item_id>')
        def item_view(item_id):
            return get_item(item_id)

        # Index view
        @router('/item/')
        def item_view():
            return get_all_items()

        # Create view
        @router('/item/', methods=['POST'])
        def item_view():
            label = request.form['label']
            item = create_item(label)
            return item

        # Delete view
        @router('/item/<int:item_id>', methods=['DELETE'])
        def item_view(item_id):
            item = get_item(item_id)
            delete_item(item)
            return 'Done'

        # Another router
        another = Router(app_or_blueprint, 'another')

        # Returns value
        @another('/answerme/<value>')
        @another('/answerme/', defaults={'value':'default'})
        def answerme(value):
            return value

    """

    default = 'default'
    """ Default key for functions whose method is not specified.  
    """

    def __init__(self, app, endpoint, *args, **kwargs):
        """ Initializer

        :param app: `~flask.app.Flask` app or `~flask.blueprints.Blueprint`
        :param endpoint: endpoint string that will be used

        """
        self.app = app
        # Function map, key is rule and method pair
        self.funcs = {}
        self.endpoint = endpoint

    def dispatch(self, *args, **kwargs):
        """ Dispatches view functions by URL rules and methods.  
        """
        # Method is case insensitive
        method = request.method.lower()

        rule = request.url_rule.rule

        # Find by rule and method
        if (rule, method) in self.funcs.keys():
            return self.funcs[(rule, method)](*args, **kwargs)

        # If not exists, use default function
        if (rule, self.default) in self.funcs.keys():
            return self.funcs[(rule, self.default)](*args, **kwargs)
        abort(405)

    def __call__(self, rule, **options):
        """ Decorator for adding URL rule, almost same as `~flask.app.Flask`'s
            `route` method but without endpoint.  
        """
        
        # Add dispatch rule
        self.app.add_url_rule(rule, endpoint=self.endpoint,
            view_func=self.dispatch, **options)

        def decorator(func):
            """ Decorator generated by router.  
            """
            self.app.add_url_rule(rule, endpoint=self.endpoint, **options)
            methods = options.get('methods', None)
            if methods:
                for method in methods:
                    # Method is case insensitive
                    method = method.lower()

                    # Register view Function
                    self.funcs[(rule, method)] = func
            else:
                # Set as default function
                self.funcs[(rule, self.default)] = func
            return func
        return decorator
