# coding: utf-8
import json

import requests

from .utils import is_valid_url

__all__ = ['Shortener',]

class Shortener(object):
    def __new__(cls, app):
        if not app.config.get('SHORTENER_ENGINE'):
            raise ValueError(u'Please enter a valid shortener engine')

        engine = app.config.get('SHORTENER_ENGINE')
        module = __import__('flaskext.shorteners')
        try:
            _class = getattr(module.shorteners, engine)
        except AttributeError:
            raise AttributeError(u'Please enter a valid shortener.')
        return _class(app)


class GoogleShortener(object):
    """
    Based on:
    https://github.com/avelino/django-googl/blob/master/googl/short.py
    Googl Shortener Implementation
    Doesn't need anything from the app

    """
    api_url = "https://www.googleapis.com/urlshortener/v1/url"

    def __init__(self, app):
        self.app = app

    def short(self, url):
        if isinstance(url, unicode):
            url = url.encode('utf-8')

        if not is_valid_url(url):
            raise ValueError(u'Please enter a valid url')

        params = json.dumps({'longUrl': url})
        headers = {'content-type': 'application/json'}
        response = requests.post(self.api_url, data=params,
                                 headers=headers)
        if response.ok:
            try:
                data = response.json()
            except:
                return u''
            if 'id' in data:
                self.shorten = data['id']
                return data['id']
        return u''

    def expand(self, url):
        params = {'shortUrl': url}
        response = requests.get(self.api_url, params=params)
        if response.ok:
            try:
                data = response.json()
            except:
                return u''
            if 'longUrl' in data:
                return data['longUrl']
        return u''


class BitlyShortener(object):
    """
    Bit.ly shortener Implementation
    needs on app.config:
    BITLY_LOGIN - Your bit.ly login user
    BITLY_API_KEY - Your bit.ly api key
    """
    shorten_url = 'http://api.bit.ly/shorten'
    expand_url = 'http://api.bit.ly/expand'

    def __init__(self, app):
        if not app.config.get('BITLY_LOGIN') and \
           not app.config.get('BITLY_API_KEY'):
            raise ValueError(u'BITLY_LOGIN AND BITLY_API_KEY missing from '
                             u'app config')

        self.login = app.config.get('BITLY_LOGIN')
        self.api_key = app.config.get('BITLY_API_KEY')


    def short(self, url):
        if isinstance(url, unicode):
            url = url.encode('utf-8')

        if not is_valid_url(url):
            raise ValueError(u'Please enter a valid url')

        params = dict(
            version="2.0.1",
            longUrl=url,
            login=self.login,
            apiKey=self.api_key,
        )
        response = requests.post(self.shorten_url, data=params)
        if response.ok:
            data = response.json()
            if 'statusCode' in data and data['statusCode'] == 'OK':
                key = self.url
                return data['results'][key]['shortUrl']
        return u''

    def expand(self, url):
        params = dict(
            version="2.0.1",
            shortUrl=url,
            login=self.login,
            apiKey=self.api_key,
        )
        response = requests.get(self.expand_url, params=params)
        if response.ok:
            data = response.json()
            if 'statusCode' in data and data['statusCode'] == 'OK':
                # get the hash key that contains the longUrl
                hash_key = data['results'].keys()[0]
                return data['results'][hash_key]['longUrl']
        return u''


class TinyurlShortener(object):
    """
    TinyURL.com shortener implementation
    No config params needed
    """
    api_url = "http://tinyurl.com/api-create.php"

    def __init__(self, app):
        self.app = app

    def short(self, url):
        if isinstance(url, unicode):
            url = url.encode('utf-8')

        if not is_valid_url(url):
            raise ValueError(u'Please enter a valid url')

        response = requests.get(self.api_url, params=dict(url=url))
        if response.ok:
            return response.text
        return u''

    def expand(self, url):
        response = requests.get(url)
        if response.ok:
            return response.url
        return u''
