from __future__ import unicode_literals, division, absolute_import
import logging
import re

from urllib import quote

from flexget import plugin
from flexget import validator
from flexget.entry import Entry
from flexget.event import event
from flexget.utils.soup import get_soup
from flexget.utils.search import torrent_availability, normalize_unicode, clean_title
from flexget.utils.requests import Session

log = logging.getLogger('search_torrentshack')

CATEGORIES = {
    'Apps/PC': 100,
    'Apps/misc': 150,
    'eBooks': 180,
    'Games/PC': 200,
    'Games/PS3': 240,
    'Games/Xbox360': 260 ,
    'HandHeld': 280,
    'Movies/x264': 300,
    'REMUX': 320,
    'Movies/DVD-R': 350,
    'Movies/XviD': 400,
    'Music/MP3': 450,
    'Music/FLAC': 480,
    'Music/Videos': 500,
    'TV/x264-HD': 600,
    'TV/x264-SD': 620,
    'TV/DVDrip': 700,
    'Misc': 800,
    'Anime': 850,
    'Foreign': 960,
    'Full Blu-ray': 970,
    'TV-SD Pack': 980,
    'TV-HD Pack': 981,
    'Movies-HD Pack': 982,
    'Movies-SD Pack': 983,
    'MP3 Pack': 984,
    'FLAC Pack': 985,
    'Games Pack': 986
}

URL = 'http://torrentshack.eu/'

class TorrentShackSearch(object):
    """ TorrentShack Search plugin

    == Basic usage:

    torrentshack:
        username: XXXX              (required)
        password: XXXX              (required)
        category: Movies/x264       (optional)
        gravity_multiplier: 200     (optional)

    == Categories
    +---------------+--------------+--------------+----------------+
    | Apps/PC       | Movies/x264  | TV/x264-HD   | TV-SD Pack     |
    | Apps/misc     | REMUX        | TV/x264-SD   | TV-HD Pack     |
    | eBooks        | Movies/DVD-R | TV/DVDrip    | Movies-HD Pack |
    | Games/PC      | Movies/XviD  | Misc         | Movies-SD Pack |
    | Games/PS3     | Music/MP3    | Anime        | MP3 Pack       |
    | Games/Xbox360 | Music/FLAC   | Foreign      | FLAC Pack      |
    | HandHeld      | Music/Videos | Full Blu-ray | Games Pack     |
    +---------------+--------------+--------------+----------------+

    You can specify either a single category or list of categories, example:

    category: Movies/x264

    or

    category:
        - Movies/XviD
        - Movies/x264

    Specifying specific category ID is also possible. You can extract ID from URL - for example
    if you hover or click on category on the site you'll see similar address:

    http://torrentshack.URL/torrents.php?filter_cat[300]=1

    In this particular example, category id is 300.

    == Priority

    gravity_multiplier is optional parameter that increases odds of downloading found matches from torrentshack
    instead of other search providers, that may have higer odds due to their higher number of peers.
    Although torrentshack  does not have many peers as some public trackers, the torrents are usually faster.
    By default, Flexget give higher priority to found matches according to following formula:

    gravity = number of seeds * 2 + number of leechers

    gravity_multiplier will multiply the above number by specified amount.
    If you use public trackers for searches, you may want to use this feature.
    """

    def validator(self):
        """Return config validator."""
        root = validator.factory('dict')
        root.accept('text', key='username', required=True)
        root.accept('text', key='password', required=True)
        root.accept('number', key='gravity_multiplier')

        root.accept('choice', key='category').accept_choices(CATEGORIES)
        root.accept('number', key='category')
        categories = root.accept('list', key='category')
        categories.accept('choice', key='category').accept_choices(CATEGORIES)
        categories.accept('number', key='category')

        return root

    @plugin.internet(log)
    def search(self, entry, config=None):

        try:
            multip = int(config['gravity_multiplier'])
        except KeyError:
            multip = 1

        if not isinstance(config['category'], list):
            config['category'] = [config['category']]

        categories_id = list()
        for category in config['category']:
            if not isinstance(category, int):
                categories_id.append(CATEGORIES.get(category))
            else:
                categories_id.append(category)
        category_url_fragment = ''.join(
            ['&' + quote('filter_cat[%s]' % id) + '=1' for id in categories_id])

        params = {
            'username': config['username'],
            'password': config['password'],
            'keeplogged': '1',
            'login': 'Login'
        }

        session = Session()
        log.debug('Logging in to %s...' % URL)
        session.post(URL + 'login.php', data=params)

        entries = set()
        for search_string in entry.get('search_strings', [entry['title']]):
            search_string_normalized = normalize_unicode(clean_title(search_string))
            search_string_url_fragment = 'searchstr=' + quote(search_string_normalized.encode('utf8'))

            url = URL + 'torrents.php?' + search_string_url_fragment + category_url_fragment
            log.debug('Fetching URL for `%s`: %s' % (search_string, url))

            page = session.get(url).content
            soup = get_soup(page)

            for result in soup.findAll('tr', attrs={'class': 'torrent'}):
                entry = Entry()
                entry['title'] = result.find('span', attrs={'class': 'torrent_name_link'}).string
                entry['url'] = URL + result.find('a',
                                                 href=re.compile(r'torrents.php\?action=download'),
                                                 attrs={'title': 'Download'})['href']
                entry['torrent_seeds'] = result.findAll('td')[-3].string
                entry['torrent_leeches'] = result.findAll('td')[-2].string
                entry['search_sort'] = torrent_availability(entry['torrent_seeds'], entry['torrent_leeches']) * multip

                size = result.findAll('td')[-5].string
                size = re.search('(\d+(?:[.,]\d+)*)\s?([KMG]B)', size)

                if size:
                        if size.group(2) == 'GB':
                            entry['content_size'] = int(float(size.group(1).replace(',', '')) * 1000 ** 3 / 1024 ** 2)
                        elif size.group(2) == 'MB':
                            entry['content_size'] = int(float(size.group(1).replace(',', '')) * 1000 ** 2 / 1024 ** 2)
                        elif size.group(2) == 'KB':
                            entry['content_size'] = int(float(size.group(1).replace(',', '')) * 1000 / 1024 ** 2)
                        else:
                            entry['content_size'] = int(float(size.group(1).replace(',', '')) / 1024 ** 2)

                entries.add(entry)
        return entries

@event('plugin.register')
def register_plugin():
    plugin.register(TorrentShackSearch, 'torrentshack', groups=['search'], api_ver=2)
