try:
    from PyRTF import *
    rtf_import = True
except ImportError:
    rtf_import = False

try:
    from xhtml2pdf import pisa
    pdf_import = True
except ImportError:
    pdf_import = False

from os import path, remove
from shutil import move
from datetime import datetime
import csv

# local imports
from ..utils.utils import ROOT_DIR
from ..model import ForemanOptions


def open_file(file_location):
    return file(file_location, 'w')

def create_rtf(notes):
    doc = Document()
    ss = doc.StyleSheet
    section = Section()
    doc.Sections.append(section)

    title = 'Case Notes for {} - {}'.format(notes[0].task.case.case_name, notes[0].task.task_name)

    logo = path.join(ROOT_DIR, 'static', 'images', 'logo', 'logo.png')
    image = Image(logo)
    section.Header.append(image)
    section.Header.append(title)
    today = ForemanOptions.get_date(datetime.now())
    section.Footer.append('Case notes generated by Foreman on {}'.format(today))

    p = Paragraph(ss.ParagraphStyles.Heading1)
    p.append(title)
    section.append(p)

    for note in notes:
        p = Paragraph(ss.ParagraphStyles.Heading2)
        p.append('{}\t{}'.format(note.date, note.author.fullname))
        section.append(p)

        body = rtf_encode(note.note)
        paragraphs = body.split("\n")
        for para in paragraphs:
            p = Paragraph(ss.ParagraphStyles.Normal)
            p.append(para)
            section.append(p)

        p = Paragraph(ss.ParagraphStyles.Normal)
        p.append('Hash of note: ' + rtf_encode(note.hash))
        section.append(p)

    return doc


def rtf_encode_char(unichar):
    code = ord(unichar)
    if code < 128:
        return str(unichar)
    return '\\u' + str(code if code <= 32767 else code-65536) + '?'


def rtf_encode(unistr):
    return ''.join(rtf_encode_char(c) for c in unistr)


def get_name(notes, ext):
    location = path.join(ROOT_DIR, 'notes')
    name = '{} {} Notes'.format(notes[0].task.case.case_name, notes[0].task.task_name)
    file_location = path.join(location, '{}.{}'.format(name, ext))
    return file_location, name


def render_rtf(notes):
    if rtf_import:
        dr = Renderer()
        doc = create_rtf(notes)
        file_location, name = get_name(notes, "rtf")
        dr.Write(doc, open_file(name))
        move(name, file_location)
        return file_location
    else:
        return None


def render_csv(notes):
    file_location, name = get_name(notes, "csv")

    with open(file_location, 'wb') as f:
        writer = csv.writer(f)
        writer.writerow(["Date", "Author", "Hash", "Note"])
        for note in notes:
            writer.writerow([note.date, note.author.fullname, note.hash, note.note.encode("utf-8")])

    return file_location


def convert_html_to_pdf(source_html, output_filename):
    if path.exists(output_filename):
        remove(output_filename)
    with open(output_filename, "wb") as result_file:
        pisa.CreatePDF(source_html, dest=result_file)


def convert_notes_into_html(notes, name):
    logo = path.abspath(path.join(ROOT_DIR, 'static', 'images', 'logo', 'logo.jpg'))
    html = u'''<html>
    <head>
    <style>
        @page {{
            size: a4 portrait;
            background-image: url('{}');
            font-size: 200%;
            @frame header_frame {{           /* Static Frame */
                -pdf-frame-content: header_content;
                left: 50pt; width: 512pt; top: 50pt; height: 40pt;
            }}
            @frame content_frame {{          /* Content Frame */
                left: 50pt; width: 512pt; top: 90pt; height: 632pt;

            }}
            @frame footer_frame {{           /* Another static Frame */
                -pdf-frame-content: footer_content;
                left: 50pt; width: 512pt; top: 772pt; height: 20pt;
            }}
        }}
    </style>
    </head>
    <body>
    <div id="header_content">{}</div>
    <div id="footer_content">
        Created by Foreman - page <pdf:pagenumber>
    </div>
    <h1>{}</h1>'''.format(logo, name, name)
    for note in notes:
        html += u"<h2>{} - {}</h2>".format(note.date, note.author.fullname,)
        paragraphs = note.note.split("\n")
        for para in paragraphs:
            html += u"<p>{}</p>".format(para)
        html += u"<p>Hash of note: {}</p>".format(note.hash)
    html += u"</body></html>"

    return html


def render_pdf(notes):
    if pdf_import:
        file_location, name = get_name(notes, "pdf")
        convert_html_to_pdf(convert_notes_into_html(notes, name), name)
        if path.exists(file_location):
            remove(file_location)
        move(name, file_location)
        return file_location
    else:
        return None