from nose.tools import assert_raises, assert_almost_equal
import re
import hashlib
import datetime
from decimal import Decimal
import pytz
from webob.multidict import MultiDict as MD
from formosa import fields, TranslationError, ErrorSet

class TestString(object):
    def setup(self):
        self.field = fields.String('name')
        self.required_field = fields.String('name', False)
        self.optional_field = fields.String('notes', True)
        self.max_length_field = fields.String('name', max_length=20)
        self.min_length_field = fields.String('name', min_length=5)

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate, MD())
        assert_raises(TranslationError, self.required_field.translate,
                      MD(name=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(name='   '))
        assert self.required_field.translate(MD(name='asdf')) == 'asdf'

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD(notes='')) == None
        assert self.optional_field.translate(MD(notes='     ')) == None
        assert self.optional_field.translate(MD()) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(name=''))
        assert not self.field.has_value(MD(name='    '))
        assert self.field.has_value(MD(name='asdf'))

    def test_max_length(self):
        assert_raises(TranslationError, self.max_length_field.translate,
                      MD(name='Aloysius Snuffleupagus'))
        assert self.max_length_field.translate(MD(name='   Fred Flintstone      ')) \
            == 'Fred Flintstone'

    def test_min_length(self):
        assert_raises(TranslationError, self.min_length_field.translate,
                      MD(name='a'))
        assert_raises(TranslationError, self.min_length_field.translate,
                      MD(name='  abc    '))
        assert self.min_length_field.translate(MD(name='abcde')) == 'abcde'

    def test_extra_params(self):
        assert self.field.translate(MD([('name', 'Nick'), ('name', 'Desi')])) \
            == 'Desi'


class TestInteger(object):
    def setup(self):
        self.field = fields.Integer('num')
        self.required_field = fields.Integer('num', False)
        self.optional_field = fields.Integer('num', True)
        self.bounded_field = fields.Integer('num', True, -5, 5)

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate,
                      MD(foobar='3'))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(num=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(num='    '))
        assert self.required_field.translate(MD(num='0')) == 0

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD(num='')) == None
        assert self.optional_field.translate(MD(num='   ')) == None
        assert self.optional_field.translate(MD()) == None

    def test_has_value(self):
        assert not self.field.has_value(MD(foobar='3'))
        assert not self.field.has_value(MD(num=''))
        assert not self.field.has_value(MD(num='    '))
        assert self.field.has_value(MD(num='0'))

    def test_format(self):
        assert_raises(TranslationError, self.field.translate, MD(num='one'))
        assert_raises(TranslationError, self.field.translate, MD(num='50j'))
        assert self.field.translate(MD(num='50')) == 50
        assert self.field.translate(MD(num='-1')) == -1
        assert self.field.translate(MD(num='   234  ')) == 234

    def test_bounds(self):
        assert_raises(TranslationError, self.bounded_field.translate,
                      MD(num='-6'))
        assert_raises(TranslationError, self.bounded_field.translate,
                      MD(num='6'))
        assert self.bounded_field.translate(MD(num='5')) == 5
        assert self.bounded_field.translate(MD(num='-5')) == -5

    def test_long_promotion(self):
        assert self.field.translate(MD(num='5000000000')) == 5000000000L

    def test_extra_params(self):
        assert self.field.translate(MD([('num', '3'), ('num', '4')])) == 4


class TestFloat(object):
    def setup(self):
        self.field = fields.Float('x')
        self.required_field = fields.Float('x', False)
        self.optional_field = fields.Float('x', True)
        self.closed_field = fields.Float('x', False, -1, 1)
        self.open_field = fields.Float('x', False, -1, 1, True, True)

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate,
                      MD(foo='3'))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(x=''))
        assert_almost_equal(self.required_field.translate(MD(x='1.0')), 1.0)

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD(x='')) == None
        assert self.optional_field.translate(MD(x='     ')) == None
        assert self.optional_field.translate(MD()) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(foo='3'))
        assert not self.field.has_value(MD(x=''))
        assert self.field.has_value(MD(x='1.0'))

    def test_format(self):
        assert_raises(TranslationError, self.field.translate, MD(x='asdf'))
        assert_raises(TranslationError, self.field.translate, MD(x='- 0.5'))
        assert_almost_equal(self.field.translate(MD(x='  0.1 ')), 0.1)
        assert_almost_equal(self.field.translate(MD(x='  -.2 ')), -0.2)

    def test_closed_bounds(self):
        assert_raises(TranslationError, self.closed_field.translate,
                      MD(x='-1.1'))
        assert_raises(TranslationError, self.closed_field.translate,
                      MD(x='1.1'))
        assert_almost_equal(self.closed_field.translate(MD(x='1')), 1.0)
        assert_almost_equal(self.closed_field.translate(MD(x='-1')), -1.0)

    def test_open_bounds(self):
        assert_raises(TranslationError, self.open_field.translate, MD(x='1'))
        assert_raises(TranslationError, self.open_field.translate, MD(x='-1'))
        assert self.open_field.translate(MD(x='0.9999')) == 0.9999
        assert self.open_field.translate(MD(x='-0.9999')) == -0.9999

    def test_extra_params(self):
        assert_almost_equal(self.field.translate(MD([('x', '3'), ('x', '5')])),
                            5.0)


class TestDecimal(object):
    def setup(self):
        self.field = fields.Decimal('x', 6, 2)
        self.required_field = fields.Decimal('x', 6, 2, False)
        self.optional_field = fields.Decimal('x', 6, 2, True)
        self.closed_field = fields.Decimal('x', 6, 2, False, -1, 1)
        self.open_field = fields.Decimal('x', 6, 2, False, Decimal('-1'),
                                         Decimal('1'), True, True)
        self.precision_field = fields.Decimal('x', 6, 2)

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate, MD())
        assert_raises(TranslationError, self.required_field.translate,
                      MD(x=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(x='    '))
        assert self.required_field.translate(MD(x='5.3')) == Decimal('5.3')

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD()) == None
        assert self.optional_field.translate(MD(x='')) == None
        assert self.optional_field.translate(MD(x='     ')) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(x=''))
        assert not self.field.has_value(MD(x='    '))
        assert self.field.has_value(MD(x='5.3'))

    def test_format(self):
        assert_raises(TranslationError, self.field.translate, MD(x='asdf'))
        assert_raises(TranslationError, self.field.translate, MD(x='- 0.5'))
        assert self.field.translate(MD(x='  0.1 ')) == Decimal('0.1')

    def test_closed_bounds(self):
        assert_raises(TranslationError, self.closed_field.translate,
                      MD(x='-1.1'))
        assert_raises(TranslationError, self.closed_field.translate,
                      MD(x='1.1'))
        assert self.closed_field.translate(MD(x='-1')) == Decimal('-1')
        assert self.closed_field.translate(MD(x='0.5')) == Decimal('0.5')
        assert self.closed_field.translate(MD(x='1')) == Decimal('1')

    def test_open_bounds(self):
        assert_raises(TranslationError, self.open_field.translate, MD(x='-1'))
        assert_raises(TranslationError, self.open_field.translate, MD(x='1'))
        assert_raises(TranslationError, self.open_field.translate,
                      MD(x='- 0.5'))
        assert self.open_field.translate(MD(x='-0.5')) == Decimal('-0.5')
        assert self.open_field.translate(MD(x='0.2')) == Decimal('0.2')

    def test_precision(self):
        assert_raises(TranslationError, self.precision_field.translate,
                      MD(price='5.253'))
        assert_raises(TranslationError, self.precision_field.translate,
                      MD(price='51232.53'))
        assert_raises(TranslationError, self.precision_field.translate,
                      MD(price='512325.3'))
        assert_raises(TranslationError, self.precision_field.translate,
                      MD(price='5123253'))
        assert_raises(TranslationError, self.precision_field.translate,
                      MD(price='512325325.2345'))

    def test_extra_params(self):
        assert self.field.translate(MD([('x', '3'), ('x', '5')])) \
            == Decimal('5')


class TestPassword(object):
    def setup(self):
        self.field = fields.Password('pw', '23F7spLb')
        self.required_field = fields.Password('pw', '23F7spLb', False)
        self.optional_field = fields.Password('pw', '23F7spLb', True)
        self.min_length_field = fields.Password('pw', '23F7spLb', min_length=5)
        self.max_length_field = fields.Password('pw', '23F7spLb',
                                                max_length=10)
        self.salted_field_1 = fields.Password('pw', '23F7spLb')
        self.salted_field_2 = fields.Password('pw', 'GkwEcm3T')
        self.sha1_field = fields.Password('pw', '23F7spLb',
                                          algorithm=hashlib.sha1)

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate, MD())
        assert_raises(TranslationError, self.required_field.translate,
                      MD(pw=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(pw='    '))
        assert self.required_field.translate(MD(pw='asdfasdfasdf')) \
            == 'b812a9ef39ffac87c7dfd1e0171c6149'

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD()) == None
        assert self.optional_field.translate(MD(pw='')) == None
        assert self.optional_field.translate(MD(pw='   ')) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(pw=''))
        assert not self.field.has_value(MD(pw='    '))
        assert self.field.has_value(MD(pw='asdfasdfasdf'))

    def test_min_length(self):
        assert_raises(TranslationError, self.min_length_field.translate,
                      MD(pw='asdf'))
        assert_raises(TranslationError, self.min_length_field.translate,
                      MD(pw='    asdf   '))
        assert self.min_length_field.translate(MD(pw='abcde')) \
            == '7ff0596853f0e107d30d1adae243c004'

    def test_max_length(self):
        assert_raises(TranslationError, self.max_length_field.translate,
                      MD(pw='12345678901'))
        assert self.max_length_field.translate(MD(pw='1234567890')) \
            == '5fab417c84515cbcb525aa98e07e8b5b'
        assert self.max_length_field.translate(MD(pw='   abcdef    ')) \
            == '109b26eb65626a26a24c928ca5d549b5'

    def test_salt(self):
        assert self.salted_field_1.translate(MD(pw='foobarbaz')) \
            == 'fe3f75cd4791b61e7c80c2d2f0358e57'
        assert self.salted_field_2.translate(MD(pw='foobarbaz')) \
            == 'eb8973e89ae67e8311d7c465610c8a21'

    def test_algorithm(self):
        assert self.sha1_field.translate(MD(pw='foobarbaz')) \
            == '405c3c4c9871218bae7e27a26c577d1a8240bc65'


class TestRegex(object):
    def setup(self):
        self.field = fields.Regex('code', '^AZ-\d{4}$')
        self.required_field = fields.Regex('code', '^AZ-\d{4}$', False)
        self.optional_field = fields.Regex('code', '^AZ-\d{4}$', True)
        self.max_length_field = fields.Regex('value', '^a*$', max_length=10)
        self.min_length_field = fields.Regex('value', '^a*$', min_length=5)
        self.pattern_field = fields.Regex('code', '^AZ-\d{4}$')
        self.compiled_field = fields.Regex('code', re.compile('^AZ-\d{4}$'))

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate, MD())
        assert_raises(TranslationError, self.required_field.translate,
                      MD(code=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(code='   '))
        assert self.required_field.translate(MD(code='AZ-1234')) == 'AZ-1234'

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD()) == None
        assert self.optional_field.translate(MD(code='')) == None
        assert self.optional_field.translate(MD(code='    ')) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(code=''))
        assert not self.field.has_value(MD(code='   '))
        assert self.field.has_value(MD(code='AZ-1234'))

    def test_max_length(self):
        assert_raises(TranslationError, self.max_length_field.translate,
                      MD(value='aaaaaaaaaaa'))
        assert self.max_length_field.translate(MD(value='   aaaaaaaaaa     ')) \
            == 'aaaaaaaaaa'

    def test_min_length(self):
        assert_raises(TranslationError, self.min_length_field.translate,
                      MD(value='a'))
        assert_raises(TranslationError, self.min_length_field.translate,
                      MD(value='  aaaa    '))
        assert self.min_length_field.translate(MD(value=' aaaaa ')) == 'aaaaa'

    def test_pattern(self):
        assert_raises(TranslationError, self.pattern_field.translate,
                      MD(foo='bar'))
        assert_raises(TranslationError, self.pattern_field.translate,
                      MD(code='AZ-532'))
        assert self.pattern_field.translate(MD(code='  AZ-0532 ')) == 'AZ-0532'

    def test_compiled_pattern(self):
        assert self.compiled_field.translate(MD(code='AZ-0532')) == 'AZ-0532'

    def test_extra_params(self):
        assert self.field.translate(MD([('code', 'AZ-1234'),
                                        ('code', 'AZ-2345')])) \
            == 'AZ-2345'


class TestEmail(object):
    def setup(self):
        self.field = fields.Email('email')
        self.required_field = fields.Email('email', False)
        self.optional_field = fields.Email('email', True)
        self.max_length_field = fields.Email('email', True, 20)

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate, MD())
        assert_raises(TranslationError, self.required_field.translate,
                      MD(email=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(email='   '))
        assert self.required_field.translate(MD(email='foo@bar.com')) \
            == 'foo@bar.com'

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD()) == None
        assert self.optional_field.translate(MD(email='')) == None
        assert self.optional_field.translate(MD(email='   ')) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(email=''))
        assert not self.field.has_value(MD(email='   '))
        assert self.field.has_value(MD(email='foo@bar.com'))

    def test_max_length(self):
        assert_raises(TranslationError, self.max_length_field.translate,
                      MD(email='a.very.very.long@email.address.com'))
        assert self.max_length_field.translate(MD(email='abcdefgh@example.com')) \
            == 'abcdefgh@example.com'

    def test_format(self):
        assert self.field.translate(MD(email='njm@test.com')) == 'njm@test.com'
        assert self.field.translate(MD(email='nick.murphy-test@test.com')) \
            == 'nick.murphy-test@test.com'
        assert self.field.translate(MD(email='njm+test@test.com')) \
            == 'njm+test@test.com'
        assert self.field.translate(MD(email='nick.murphy+test-mania@email.test.example.co.uk')) \
            == 'nick.murphy+test-mania@email.test.example.co.uk'

    def test_extra_params(self):
        assert self.field.translate(MD([('email', 'njm@foo.com'),
                                        ('email', 'njm@bar.com')])) \
            == 'njm@bar.com'


class TestDate(object):
    def setup(self):
        self.field = fields.Date('date')
        self.required_field = fields.Date('date', False)
        self.optional_field = fields.Date('date', True)
        self.range_field = fields.Date('date', False, datetime.date(2007, 1, 1),
                                       datetime.date(2007, 12, 31))

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate, MD())
        assert_raises(TranslationError, self.required_field.translate,
                      MD(date=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(date='    '))
        assert self.required_field.translate(MD(date='4/29/1999')) \
            == datetime.date(1999, 4, 29)

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD()) == None
        assert self.optional_field.translate(MD(date='')) == None
        assert self.optional_field.translate(MD(date='    ')) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(date=''))
        assert not self.field.has_value(MD(date='    '))
        assert self.field.has_value(MD(date='4/29/1999'))

    def test_format(self):
        assert_raises(TranslationError, self.field.translate,
                      MD(date='12/23/asdf'))
        assert_raises(TranslationError, self.field.translate,
                      MD(date='asdfasdf'))
        assert self.field.translate(MD(date='12/23/2005')) \
            == datetime.date(2005, 12, 23)
        assert self.field._format_date(datetime.date(2007, 6, 3)) \
            == 'June 3, 2007'

    def test_valid_dates(self):
        assert_raises(TranslationError, self.field.translate,
                      MD(date='2/29/2007'))
        assert_raises(TranslationError, self.field.translate,
                      MD(date='2/30/2008'))
        assert_raises(TranslationError, self.field.translate,
                      MD(date='14/12/2008'))
        assert_raises(TranslationError, self.field.translate,
                      MD(date='1/0/2008'))
        assert_raises(TranslationError, self.field.translate,
                      MD(date='0/5/2008'))

    def test_range(self):
        assert_raises(TranslationError, self.range_field.translate,
                      MD(date='12/21/2006'))
        assert_raises(TranslationError, self.range_field.translate,
                      MD(date='1/1/2008'))
        assert self.range_field.translate(MD(date='6/4/2007')) \
            == datetime.date(2007, 6, 4)

    def test_extra_params(self):
        assert self.field.translate(MD([('date', '6/30/2007'),
                                        ('date', '8/12/2008')])) \
            == datetime.date(2008, 8, 12)


class TestTime(object):
    def setup(self):
        self.field = fields.Time('h', 'm', 'ampm', True)
        self.required_field = fields.Time('h', 'm', 'ampm', False)
        self.optional_field = fields.Time('h', 'm', 'ampm', True)
        self.range_field = fields.Time('h', 'm', 'ampm', True,
                                       datetime.time(9, 0),
                                       datetime.time(17, 0))
        self.tzinfo_field = fields.Time('h', 'm', 'ampm',
                                        tzinfo=pytz.timezone('US/Arizona'))

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate,
                      MD(h=' ', m=' ', ampm=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(h='12', m='', ampm=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(h='', m='30', ampm=''))
        assert_raises(TranslationError, self.required_field.translate, MD())
        assert self.required_field.translate(MD(h='12', m='30', ampm='PM')) \
            == datetime.time(12, 30)

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD()) == None
        assert self.optional_field.translate(MD(h='', m='', ampm='')) == None
        assert self.optional_field.translate(MD(h=' ', m=' ', ampm=' ')) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(h=' ', m=' ', ampm=''))
        assert self.field.has_value(MD(h='12', m='', ampm=''))
        assert self.field.has_value(MD(h='', m='30', ampm=''))
        assert self.field.has_value(MD(h='12', m='30', ampm='PM'))

    def test_completeness(self):
        assert_raises(TranslationError, self.field.translate,
                      MD(h='12', m='', ampm=''))
        assert_raises(TranslationError, self.field.translate,
                      MD(h='', m='30', ampm=''))
        assert_raises(TranslationError, self.field.translate,
                      MD(h='', m='', ampm='PM'))
        assert_raises(TranslationError, self.field.translate,
                      MD(h='12', m='30', ampm=''))
        assert_raises(TranslationError, self.field.translate,
                      MD(h='12', m='', ampm='PM'))
        assert_raises(TranslationError, self.field.translate,
                      MD(h='', m='30', ampm='PM'))
        assert self.field.translate(MD(h='12', m='30', ampm='PM')) \
            == datetime.time(12, 30)

    def test_format(self):
        assert_raises(TranslationError, self.field.translate,
                      MD(h='asdf', m='30', ampm='AM'))
        assert_raises(TranslationError, self.field.translate,
                      MD(h='12', m='asdfasdf', ampm='AM'))
        assert_raises(TranslationError, self.field.translate,
                      MD(h='12', m='30', ampm='adsdfasdf'))
        assert self.field._format_time(datetime.time(15, 8)) == '3:08 PM'

    def test_valid_times(self):
        assert_raises(TranslationError, self.field.translate,
                      MD(h='15', m='30', ampm='PM'))
        assert_raises(TranslationError, self.field.translate,
                      MD(h='12', m='99', ampm='PM'))
        assert_raises(TranslationError, self.field.translate,
                      MD(h='12', m='30', ampm='ZM'))

    def test_range(self):
        assert_raises(TranslationError, self.range_field.translate,
                      MD(h='8', m='30', ampm='AM'))
        assert_raises(TranslationError, self.range_field.translate,
                      MD(h='5', m='02', ampm='PM'))

    def test_tzinfo(self):
        value = self.tzinfo_field.translate(MD(h='12', m='30', ampm='PM'))
        assert value.tzinfo == pytz.timezone('US/Arizona')

    def test_extra_params(self):
        assert self.field.translate(MD([('h', '12'), ('h', '1'), ('m', '30'),
                                        ('m', '51'), ('ampm', 'AM'),
                                        ('ampm', 'PM')])) \
            == datetime.time(13, 51)


class TestChoice(object):
    def setup(self):
        rgb = set(['Red', 'Green', 'Blue'])
        self.field = fields.Choice('color', rgb)
        self.required_field = fields.Choice('color', rgb, False)
        self.optional_field = fields.Choice('color', rgb, True)

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate, MD())
        assert_raises(TranslationError, self.required_field.translate,
                      MD(color=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(color='   '))

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD()) == None
        assert self.optional_field.translate(MD(color='')) == None
        assert self.optional_field.translate(MD(color='     ')) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(color=''))
        assert not self.field.has_value(MD(color='   '))
        assert self.field.has_value(MD(color='asdf'))
        assert self.field.has_value(MD(color='Red'))

    def test_valid_choices(self):
        assert_raises(TranslationError, self.field.translate,
                      MD(color='Brown'))
        assert_raises(TranslationError, self.field.translate, MD(color='red'))
        assert self.field.translate(MD(color='Red')) == 'Red'
        assert self.field.translate(MD(color='  Blue ')) == 'Blue'

    def test_extra_params(self):
        assert self.field.translate(MD([('color', 'Red'),
                                        ('color', 'Blue')])) \
            == 'Blue'


class TestMultiChoice(object):
    def setup(self):
        rgb = set(['Red', 'Green', 'Blue'])
        self.field = fields.MultiChoice('color', rgb)
        self.required_field = fields.MultiChoice('color', rgb, 1)
        self.optional_field = fields.MultiChoice('color', rgb, 0)
        self.bounded_field = fields.MultiChoice('color', rgb, 1, 2)

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate, MD())
        assert_raises(TranslationError, self.required_field.translate,
                      MD(color=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(color='   '))

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD()) == set()
        assert self.optional_field.translate(MD(color='')) == set()
        assert self.optional_field.translate(MD(color='     ')) == set()

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(color=''))
        assert not self.field.has_value(MD(color='   '))
        assert self.field.has_value(MD(color='asdfasdfasdf'))
        assert self.field.has_value(MD(color='Blue'))
        assert self.field.has_value(MD([('color', ''), ('color', 'foo')]))

    def test_valid_choices(self):
        assert_raises(TranslationError, self.field.translate, MD(color='Brown'))
        assert_raises(TranslationError, self.field.translate, MD(color='red'))
        assert self.field.translate(MD(color='Red')) == set(['Red'])
        assert self.field.translate(MD(color='  Blue ')) == set(['Blue'])

    def test_bounds(self):
        assert_raises(TranslationError, self.bounded_field.translate, MD())
        assert_raises(TranslationError, self.bounded_field.translate,
                      MD([('color', 'Red'), ('color', 'Blue'),
                          ('color', 'Green')]))
        assert self.bounded_field.translate(MD([('color', 'Blue')])) \
            == set(['Blue'])
        assert self.bounded_field.translate(MD([('color', 'Red'),
                                                ('color', 'Green')])) \
            == set(['Red', 'Green'])


class TestBoolean(object):
    def setup(self):
        self.field = fields.Boolean('selected')
        self.required_field = fields.Boolean('selected', False)
        self.optional_field = fields.Boolean('selected', True)

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate, MD())
        assert_raises(TranslationError, self.required_field.translate,
                      MD(selected=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(selected='    '))

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD()) == None
        assert self.optional_field.translate(MD(selected='')) == None
        assert self.optional_field.translate(MD(selected='    ')) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(selected=''))
        assert not self.field.has_value(MD(selected='    '))
        assert self.field.has_value(MD(selected='asdf'))
        assert self.field.has_value(MD(selected='true'))

    def test_true_values(self):
        assert self.field.translate(MD(selected='true')) == True
        for i in ('True', 't', 'on', 'ON', '1', 'yes', 'y', 'Y'):
            assert_raises(TranslationError, self.field.translate,
                          MD(selected=i))

    def test_false_values(self):
        assert self.field.translate(MD(selected='false')) == False
        for i in ('False', 'f', 'off', 'OFF', '0', 'no', 'n', 'N'):
            assert_raises(TranslationError, self.field.translate,
                          MD(selected=i))

    def test_extra_params(self):
        assert self.field.translate(MD([('selected', 'true'),
                                        ('selected', 'false')])) \
            == False


class TestToggle(object):
    def setup(self):
        self.field = fields.Toggle('value')

    def test_required(self):
        assert self.field.required()

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(value=''))
        assert not self.field.has_value(MD(value='    '))
        assert self.field.has_value(MD(value=' asdfasdf'))

    def test_true_values(self):
        for i in ['on', 'off', 'afskljdfasdf', 'true', 'false', '0']:
            assert self.field.translate(MD(value=i)) == True

    def test_false_values(self):
        for i in [MD(), MD(value=''), MD(value='  ')]:
            assert self.field.translate(i) == False


class MockFile(object):
    def __init__(self, filename, value):
        self.filename = filename
        self.value = value


class TestFile(object):
    def setup(self):
        self.field = fields.File('file')
        self.required_field = fields.File('file', False)
        self.optional_field = fields.File('file', True)
        self.mime_type_field = fields.File('pic', True,
                                           set(['image/jpeg', 'image/gif']))
        self.max_size_field = fields.File('file', max_size=10)

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate, MD())
        assert_raises(TranslationError, self.required_field.translate,
                      MD(value=''))

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD()) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(file=''))
        assert self.field.has_value(MD(file='   '))
        myfile = MockFile('foo.txt', 'This is my file')
        assert self.field.has_value(MD(file=myfile))

    def test_translate(self):
        myfile = MockFile('foo.txt', 'This is my file')
        assert self.field.translate(MD(file=myfile)) == myfile

    def test_mime_types(self):
        assert_raises(TranslationError, self.mime_type_field.translate,
                      MD(pic=MockFile('foo.xpm', '')))
        assert_raises(TranslationError, self.mime_type_field.translate,
                      MD(pic=MockFile('foo.png', '')))
        myfile = MockFile('test.jpg', '')
        assert self.mime_type_field.translate(MD(pic=myfile)) == myfile

    def test_max_size(self):
        assert_raises(TranslationError, self.max_size_field.translate,
                      MD(file=MockFile('asdf', 'longerthanten')))
        myfile = MockFile('foo.txt', 'shorter')
        assert self.max_size_field.translate(MD(file=myfile)) == myfile

    def test_extra_params(self):
        file1 = MockFile('foo.txt', 'Foo!')
        file2 = MockFile('bar.txt', 'Bar!')
        assert self.field.translate(MD([('file', file1), ('file', file2)])) \
            == file2


class TestZipCode(object):
    def setup(self):
        self.field = fields.ZipCode('zip')
        self.required_field = fields.ZipCode('zip', False)
        self.optional_field = fields.ZipCode('zip', True)

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate, MD())
        assert_raises(TranslationError, self.required_field.translate,
                      MD(zip=''))
        assert_raises(TranslationError, self.required_field.translate,
                      MD(zip='  '))
        assert self.required_field.translate(MD(zip='56303')) == '56303'

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD()) == None
        assert self.optional_field.translate(MD(zip='')) == None
        assert self.optional_field.translate(MD(zip='    ')) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(zip=''))
        assert not self.field.has_value(MD(zip='  '))
        assert self.field.has_value(MD(zip='56303'))

    def test_pattern(self):
        assert_raises(TranslationError, self.field.translate, MD(zip='5630'))
        assert_raises(TranslationError, self.field.translate,
                      MD(zip='563031234'))
        assert self.field.translate(MD(zip='56303')) == '56303'
        assert self.field.translate(MD(zip='   56303  ')) == '56303'
        assert self.field.translate(MD(zip=' 56303-1234 ')) == '56303-1234'

    def test_extra_params(self):
        assert self.field.translate(MD([('zip', '56303'), ('zip', '85712')])) \
            == '85712'


class TestPhone(object):
    def setup(self):
        self.field = fields.Phone('phone')
        self.required_field = fields.Phone('phone', False)
        self.optional_field = fields.Phone('phone', True)

    def test_required(self):
        assert self.required_field.required()
        assert_raises(TranslationError, self.required_field.translate,
                      MD(phone=''))
        assert self.required_field.translate(MD(phone='320-867-5309')) \
            == '320-867-5309'

    def test_optional(self):
        assert not self.optional_field.required()
        assert self.optional_field.translate(MD()) == None
        assert self.optional_field.translate(MD(phone='')) == None
        assert self.optional_field.translate(MD(phone='    ')) == None

    def test_has_value(self):
        assert not self.field.has_value(MD())
        assert not self.field.has_value(MD(phone=''))
        assert not self.field.has_value(MD(phone='   '))
        assert self.field.has_value(MD(phone='320-867-5309'))

    def test_pattern(self):
        assert_raises(TranslationError, self.field.translate,
                      MD(phone='3208675309'))
        assert_raises(TranslationError, self.field.translate,
                      MD(phone='867-5309'))
        assert self.field.translate(MD(phone='320-867-5309')) == '320-867-5309'
        assert self.field.translate(MD(phone='   320-867-5309  ')) \
            == '320-867-5309'

    def test_extra_params(self):
        assert self.field.translate(MD([('phone', '320-867-5309'),
                                        ('phone', '320-857-5310')])) \
            == '320-857-5310'


class TestSubForm(object):
    def setup(self):
        self.fields = {'street1': fields.String('street1', False, 30),
                       'street2': fields.String('street2', True, 30),
                       'city': fields.String('city', False, 20),
                       'state': fields.String('state', False, 2, 2),
                       'zip': fields.ZipCode('zip')}
        self.field = fields.SubForm(self.fields)
        self.required_field = fields.SubForm(self.fields, allow_empty=False)
        self.optional_field = fields.SubForm(self.fields, allow_empty=True)

    def test_required(self):
        assert_raises(TranslationError, self.required_field.translate, MD())

    def test_optional(self):
        assert self.optional_field.translate(MD()) == None

    def test_child_validation(self):
        assert_raises(ErrorSet, self.field.translate,
                      MD(street1='123 Test St.'))
        value = self.field.translate(MD(street1='123 Test St.',
                                        street2='Apt. 1234',
                                        city='Testville',
                                        state='MN',
                                        zip='56301'))
        assert value == {'street1': '123 Test St.', 'street2': 'Apt. 1234',
                         'city': 'Testville', 'state': 'MN', 'zip': '56301'}
