# -*- coding: utf-8 -


from __future__ import with_statement

import errno
import os
import select
import signal
import sys
import time
import traceback
import argparse
import logging
import pwd
import grp
import os.path
import runpy
from gevent.server import _tcp_listener
import gevent
import util

__all__ = []
__author__ = "Wei Zhuo <zeaphoo@gmail.com>"
__license__ = "MIT"
__version__ = ".".join(map(str, (0, 3, 2)))

SERVER_SOFTWARE = "grunner/%s" % __version__


STOP_SIGNAL = signal.SIGTERM
RELOAD_SIGNAL = signal.SIGHUP

sys.path.insert(0, os.getcwd())

class HaltServer(BaseException):
    def __init__(self, reason, exit_status=1):
        self.reason = reason
        self.exit_status = exit_status

    def __str__(self):
        return "<HaltServer %r %d>" % (self.reason, self.exit_status)

def init_logging(logpath = None):
    from logging.handlers import TimedRotatingFileHandler
    from logging import StreamHandler
    if logpath is not None:
        if os.path.isabs(logpath):
            logfile = logpath
        else:
            logfile = os.path.join(os.getcwd(), logpath)
        logdir = os.path.dirname(logfile)
        if not os.path.exists(logdir):
            os.makedirs(logdir)
        handler = TimedRotatingFileHandler(filename=logfile, when='MIDNIGHT', interval=1, backupCount=14) 
    else:
        handler = StreamHandler() 
    handler.setFormatter(logging.Formatter('%(asctime)s  %(levelname)-8s %(message)s')) 
    handler.setLevel(logging.INFO)
    logging.getLogger().addHandler(handler) 
    logging.getLogger().setLevel(logging.INFO)


def run():
    parser = argparse.ArgumentParser(prog="grunner", add_help=False)
    parser.add_argument("-v", "--version",
        action="version", version="%(prog)s {}".format(__version__))
    parser.add_argument("-h", "--help", action="store_true", help="""
        show program's help text and exit
        """.strip())
    parser.add_argument("-d", "--daemon", action="store_true", help="""
        Daemonize the grunner process.

        Detaches the server from the controlling terminal and enters the
        background.
        """.strip())
    parser.add_argument("-w", "--workers", type=int, default=1, help="""
        number of workers
        """.strip())
    parser.add_argument("-b", "--bind", help="""
        The socket to bind.

        A string of the form: 'HOST', 'HOST:PORT'. An IP is a valid
        HOST.
        """.strip())
    parser.add_argument("-p", "--pid", help="""
        pid or pidfile to use instead of target
        """.strip())
    parser.add_argument("--pythonpath", help="""
        add python path to python environ.
        """.strip())
    parser.add_argument("--log", help="""
        log file path when daemonized.
        """.strip())
    parser.add_argument("target", nargs='?', help="""
        service class path to run (modulename:serve_forever)
        """.strip())
    args = parser.parse_args()
    if args.help:
        parser.print_help()
    if args.target:
        print # blank line
        try:
            if args.pythonpath:
                paths = args.pythonpath.split(',')
                for path in reversed(paths):
                    if os.path.exists(path) and os.path.isabs(path):
                        sys.path.insert(0, path)
            if args.daemon:
                util.prevent_core_dump()
                util.daemonize()
            init_logging(args.log if args.daemon else None)
            runner = GeventRunner(daemon=args.daemon, 
                num_workers = args.workers, bind=args.bind or '0.0.0.0',
                pid = args.pid, target = args.target)
            runner.run()
        except RuntimeError, e:
            parser.error(e)
    else:
        parser.print_usage()

class GeventRunner(object):
    """
    GeventRunner maintain the workers processes alive. It launches or
    kills them if needed. It also manages application reloading
    via SIGHUP/USR2.
    """

    # A flag indicating if a worker failed to
    # to boot. If a worker process exist with
    # this error code, the arbiter will terminate.
    WORKER_BOOT_ERROR = 3

    START_CTX = {}

    LISTENERS = []
    WORKERS = {}
    PIPE = []

    # I love dynamic languages
    SIG_QUEUE = []
    SIGNALS = [getattr(signal, "SIG%s" % x) \
            for x in "HUP QUIT INT TERM TTIN TTOU USR1 USR2 WINCH".split()]
    SIG_NAMES = dict(
        (getattr(signal, name), name[3:].lower()) for name in dir(signal)
        if name[:3] == "SIG" and name[3] != "_"
    )

    def __init__(self, daemon=False, num_workers = 1, bind='0.0.0.0:8000',
                    pid = None, target = None):
        os.environ["SERVER_SOFTWARE"] = SERVER_SOFTWARE

        
        self.address = util.parse_address(bind)
        self.num_workers = num_workers
        self.daemon = daemon
        self.pid = os.getpid()
        self.pidfile = None
        if pid is not None:
            self.pidfile = util.Pidfile(pid)
            self.pidfile.create(self.pid)
        self.target = target
        self.proc_name = self.target
        self.worker_class = Worker
        if hasattr(self.worker_class, 'setup'):
            self.worker_class.setup()

        self.worker_age = 0
        self.reexec_pid = 0
        self.master_name = "Master"

        self.log = logging.getLogger('grunner')

        # get current path, try to use PWD env first
        try:
            a = os.stat(os.environ['PWD'])
            b = os.stat(os.getcwd())
            if a.st_ino == b.st_ino and a.st_dev == b.st_dev:
                cwd = os.environ['PWD']
            else:
                cwd = os.getcwd()
        except:
            cwd = os.getcwd()

        args = sys.argv[:]
        args.insert(0, sys.executable)

        # init start context
        self.START_CTX = {
            "args": args,
            "cwd": cwd,
            0: sys.executable
        }        

    def start(self):
        """\
        Initialize the arbiter. Start listening and set pidfile if needed.
        """
        self.log.info("Starting grunner %s", __version__)
        self.init_signals()
        if not self.LISTENERS:
            self.LISTENERS.append(_tcp_listener(self.address), reuse_addr = 1)
        self.log.info("Listening at: %s:%d (%s)", self.address[0], self.address[1], self.pid)

    def init_signals(self):
        """\
        Initialize master signal handling. Most of the signals
        are queued. Child signals only wake up the master.
        """
        # close old PIPE
        if self.PIPE:
            [os.close(p) for p in self.PIPE]

        # initialize the pipe
        self.PIPE = pair = os.pipe()
        for p in pair:
            util.set_non_blocking(p)
            util.close_on_exec(p)

        # intialiatze all signals
        [signal.signal(s, self.signal) for s in self.SIGNALS]
        signal.signal(signal.SIGCHLD, self.handle_chld)

    def signal(self, sig, frame):
        if len(self.SIG_QUEUE) < 5:
            self.SIG_QUEUE.append(sig)
            self.wakeup()

    def run(self):
        "Main master loop."
        self.start()
        util._setproctitle("master [%s]" % self.proc_name)

        self.manage_workers()
        while True:
            try:
                self.reap_workers()
                sig = self.SIG_QUEUE.pop(0) if len(self.SIG_QUEUE) else None
                if sig is None:
                    self.sleep()
                    self.manage_workers()
                    continue

                if sig not in self.SIG_NAMES:
                    self.log.info("Ignoring unknown signal: %s", sig)
                    continue

                signame = self.SIG_NAMES.get(sig)
                handler = getattr(self, "handle_%s" % signame, None)
                if not handler:
                    self.log.error("Unhandled signal: %s", signame)
                    continue
                self.log.info("Handling signal: %s", signame)
                handler()
                self.wakeup()
            except StopIteration:
                self.halt()
            except KeyboardInterrupt:
                self.halt()
            except HaltServer as inst:
                self.halt(reason=inst.reason, exit_status=inst.exit_status)
            except SystemExit:
                raise
            except Exception:
                self.log.info("Unhandled exception in main loop:\n%s",
                            traceback.format_exc())
                self.stop()
                if self.pidfile is not None:
                    self.pidfile.unlink()
                sys.exit(-1)

    def handle_chld(self, sig, frame):
        "SIGCHLD handling"
        self.wakeup()

    def handle_hup(self):
        """\
        HUP handling.
        - Reload configuration
        - Start the new worker processes with a new configuration
        - Gracefully shutdown the old worker processes
        """
        self.log.info("Hang up: %s", self.master_name)
        self.reload()

    def handle_quit(self):
        "SIGQUIT handling"
        self.stop()
        raise StopIteration

    def handle_int(self):
        "SIGINT handling"
        self.stop()
        raise StopIteration

    def handle_term(self):
        "SIGTERM handling"
        self.stop()
        raise StopIteration

    def handle_ttin(self):
        """\
        SIGTTIN handling.
        Increases the number of workers by one.
        """
        self.num_workers += 1
        self.manage_workers()

    def handle_ttou(self):
        """\
        SIGTTOU handling.
        Decreases the number of workers by one.
        """
        if self.num_workers <= 1:
            return
        self.num_workers -= 1
        self.manage_workers()

    def handle_usr1(self):
        """\
        SIGUSR1 handling.
        Kill all workers by sending them a SIGUSR1
        """
        self.kill_workers(signal.SIGUSR1)

    def handle_usr2(self):
        """\
        SIGUSR2 handling.
        Creates a new master/worker set as a slave of the current
        master without affecting old workers. Use this to do live
        deployment with the ability to backout a change.
        """
        self.reexec()

    def handle_winch(self):
        "SIGWINCH handling"
        if os.getppid() == 1 or os.getpgrp() != os.getpid():
            self.log.info("graceful stop of workers")
            self.num_workers = 0
            self.kill_workers(signal.SIGQUIT)
        else:
            self.log.info("SIGWINCH ignored. Not daemonized")

    def wakeup(self):
        """\
        Wake up the arbiter by writing to the PIPE
        """
        try:
            os.write(self.PIPE[1], b'.')
        except IOError as e:
            if e.errno not in [errno.EAGAIN, errno.EINTR]:
                raise

    def halt(self, reason=None, exit_status=0):
        """ halt arbiter """
        self.stop()
        self.log.info("Shutting down: %s", self.master_name)
        if reason is not None:
            self.log.info("Reason: %s", reason)
        if self.pidfile is not None:
            self.pidfile.unlink()
        sys.exit(exit_status)

    def sleep(self):
        """\
        Sleep until PIPE is readable or we timeout.
        A readable PIPE means a signal occurred.
        """
        try:
            ready = select.select([self.PIPE[0]], [], [], 1.0)
            if not ready[0]:
                return
            while os.read(self.PIPE[0], 1):
                pass
        except select.error as e:
            if e.args[0] not in [errno.EAGAIN, errno.EINTR]:
                raise
        except OSError as e:
            if e.errno not in [errno.EAGAIN, errno.EINTR]:
                raise
        except KeyboardInterrupt:
            sys.exit()

    def stop(self):
        """\
        Stop workers
        """
        for l in self.LISTENERS:
            try:
                l.shutdown()
                l.close()
            except Exception:
                pass

        self.LISTENERS = []
        sig = signal.SIGTERM
        while self.WORKERS:
            self.kill_workers(sig)
            time.sleep(0.1)
            self.reap_workers()
        self.kill_workers(signal.SIGKILL)

    def reexec(self):
        """\
        Relaunch the master and workers.
        """
        if self.pidfile is not None:
            self.pidfile.rename("%s.oldbin" % self.pidfile.fname)

        self.reexec_pid = os.fork()
        if self.reexec_pid != 0:
            self.master_name = "Old Master"
            return

        fds = [l.fileno() for l in self.LISTENERS]
        os.environ['GRUNNER_FD'] = ",".join([str(fd) for fd in fds])

        os.chdir(self.START_CTX['cwd'])

        # close all file descriptors except bound sockets
        util.closerange(3, fds[0])
        util.closerange(fds[-1] + 1, util.get_maxfd())

        os.execvpe(self.START_CTX[0], self.START_CTX['args'], os.environ)

    def reload(self):
        old_address = self.address
        # TODO reload 

        # manage workers
        self.manage_workers()

    def reap_workers(self):
        """\
        Reap workers to avoid zombie processes
        """
        try:
            while True:
                wpid, status = os.waitpid(-1, os.WNOHANG)
                if not wpid:
                    break
                if self.reexec_pid == wpid:
                    self.reexec_pid = 0
                else:
                    # A worker said it cannot boot. We'll shutdown
                    # to avoid infinite start/stop cycles.
                    exitcode = status >> 8
                    if exitcode == self.WORKER_BOOT_ERROR:
                        reason = "Worker failed to boot."
                        raise HaltServer(reason, self.WORKER_BOOT_ERROR)
                    worker = self.WORKERS.pop(wpid, None)
                    if not worker:
                        continue
        except OSError as e:
            if e.errno == errno.ECHILD:
                pass

    def manage_workers(self):
        """\
        Maintain the number of workers by spawning or killing
        as required.
        """
        if len(self.WORKERS.keys()) < self.num_workers:
            self.spawn_workers()

        workers = self.WORKERS.items()
        workers = sorted(workers, key=lambda w: w[1].age)
        while len(workers) > self.num_workers:
            (pid, _) = workers.pop(0)
            self.kill_worker(pid, signal.SIGQUIT)

    def spawn_worker(self):
        self.worker_age += 1
        worker = self.worker_class(self.worker_age, self.pid, self.LISTENERS,
                                    self.target, 1.0, self.log)
        pid = os.fork()
        if pid != 0:
            self.WORKERS[pid] = worker
            return pid

        # Process Child
        worker_pid = os.getpid()
        try:
            util._setproctitle("worker [%s]" % self.proc_name)
            self.log.info("Booting worker with pid: %s", worker_pid)
            worker.init_process()
            sys.exit(0)
        except SystemExit:
            raise
        except:
            self.log.exception("Exception in worker process:\n%s",
                    traceback.format_exc())
            self.log.exception('Exception in worker:%s'%(worker.booted))
            if not worker.booted:
                print 'boot error'
                sys.exit(self.WORKER_BOOT_ERROR)
            sys.exit(-1)
        finally:
            self.log.info("Worker exiting (pid: %s)", worker_pid)


    def spawn_workers(self):
        """\
        Spawn new workers as needed.

        This is where a worker process leaves the main loop
        of the master process.
        """

        for i in range(self.num_workers - len(self.WORKERS.keys())):
            self.spawn_worker()

    def kill_workers(self, sig):
        """\
        Kill all workers with the signal `sig`
        :attr sig: `signal.SIG*` value
        """
        for pid in self.WORKERS.keys():
            self.kill_worker(pid, sig)

    def kill_worker(self, pid, sig):
        """\
        Kill a worker

        :attr pid: int, worker pid
        :attr sig: `signal.SIG*` value
         """
        os.kill(pid, sig)
        '''
        try:
            os.kill(pid, sig)
        except OSError as e:
            if e.errno == errno.ESRCH:
                try:
                    worker = self.WORKERS.pop(pid)
                    return
                except (KeyError, OSError):
                    return
            raise
        '''

class Worker(object):

    SIGNALS = [getattr(signal, "SIG%s" % x) \
            for x in "HUP QUIT INT TERM USR1 USR2 WINCH CHLD".split()]

    def __init__(self, age, ppid, sockets, target, timeout, log):
        """\
        This is called pre-fork so it shouldn't do anything to the
        current process. If there's a need to make process wide
        changes you'll want to do that in ``self.init_process()``.
        """
        self.age = age
        self.ppid = ppid
        self.sockets = sockets
        self.target = target
        self.timeout = timeout
        self.booted = False
        self.alive = True
        self.log = log

    def __str__(self):
        return "<Worker %s>" % self.pid

    @property
    def pid(self):
        return os.getpid()

    def run(self):
        """\
        This is the mainloop of a worker process.
        """
        try:
            func = util.import_app(self.target)
        except Exception, e:
            self.booted = False
            raise(e)
        if callable(func):
            func(self.sockets)
        else:
            self.booted = False
            self.log.error('target function not available.')
            raise

    def init_process(self):
        #util.set_owner_process(self.uid, self.gid)
        # Reseed the random number generator
        util.seed()

        # Prevent fd inherientence
        [util.close_on_exec(s) for s in self.sockets]

        self.init_signals()

        # Enter main run loop
        self.booted = True
        self.run()

    def init_signals(self):
        # reset signaling
        [signal.signal(s, signal.SIG_DFL) for s in self.SIGNALS]
        # init new signaling
        signal.signal(signal.SIGQUIT, self.handle_quit)
        signal.signal(signal.SIGTERM, self.handle_exit)
        signal.signal(signal.SIGINT, self.handle_exit)
        signal.signal(signal.SIGWINCH, self.handle_winch)
        signal.signal(signal.SIGUSR1, self.handle_usr1)
        # Don't let SIGQUIT and SIGUSR1 disturb active requests
        # by interrupting system calls
        if hasattr(signal, 'siginterrupt'):  # python >= 2.6
            signal.siginterrupt(signal.SIGQUIT, False)
            signal.siginterrupt(signal.SIGUSR1, False)

    @classmethod
    def setup(cls):
        from gevent import monkey
        monkey.noisy = False
        monkey.patch_all()

    def handle_usr1(self, sig, frame):
        pass

    def handle_quit(self, sig, frame):
        sys.exit(0)

    def handle_exit(self, sig, frame):
        sys.exit(0)

    def handle_error(self, req, client, addr, exc):
        pass

    def handle_winch(self, sig, fname):
        # Ignore SIGWINCH in worker. Fixes a crash on OpenBSD.
        return

if __name__ == '__main__':
    run()
