#!/usr/bin/env
"""Pipeline utilities"""
import gem
import json
import logging
import os
import errno
import signal
import traceback

from gem.utils import Timer
import gem.gemtools as gt


class dotdict(dict):
    def __getattr__(self, attr):
        return self.get(attr, None)
    __setattr__ = dict.__setitem__
    __delattr__ = dict.__delitem__


class PipelineError(Exception):
    """Exception thrown by the mapping pipeline"""
    def __init__(self, message):
        self.message = message


class PipelineStep(object):
    """General mapping pipeline step"""
    def __init__(self, name, dependencies=None, final=False, description=""):
        self.id = None
        self.name = name
        self.description = description
        self.pipeline = None
        self.dependencies = []
        self._files = None
        self.configuration = None
        self.final = final
        if dependencies is not None:
            self.dependencies.extend(dependencies)

    def prepare(self, id, pipeline, configuration):
        """Implement this to prepare the step"""
        self.pipeline = pipeline
        self.id = id
        self.configuration = configuration
        # initialize files
        self.files()

    def run(self, final=False):
        """Implement this method to execute the step"""
        pass

    def cleanup(self, force=False):
        if force or (not self.final and self.pipeline.remove_temp):
            for f in self.files():
                if os.path.exists(f):
                    logging.gemtools.debug("Remove temporary file %s" % f)
                    os.remove(f)

    def files(self):
        """Return the output files generated by this step.
        By default one .map output file is generated
        """
        if self._files is None:
            self._files = []
            self._files.append(self.pipeline.create_file_name(self.name, final=self.final))
        return self._files

    def _compress(self):
        """Returns true if this step needs compression for
        all output
        """
        return self.pipeline.compress_all

    def _output(self):
        """Return the output file if its not final
        step, otherwise return none
        """
        if self.final:
            return None
        else:
            return self._final_output()

    def _final_output(self):
        """Return the last file created by this step"""
        return self.files()[-1]

    def _input(self, raw=False):
        """Return pipeline input if this step
        has no dependencies or the
        output of the last dependency
        """
        if self.dependencies is None or len(self.dependencies) == 0:
            return self.pipeline.open_input()
        return self.pipeline.open_step(self.dependencies[-1], raw=raw)

    def open(self, raw=False):
        """Open the steps output. The default implementation
        opnes the last file"""
        fs = self.files()
        if len(fs) > 0:
            if raw:
                logging.gemtools.debug("Returning raw step %s output : %s", self.name, fs[-1])
                return fs[-1]
            else:
                logging.gemtools.debug("Opening step %s output : %s", self.name, fs[-1])
                return gem.files.open(fs[-1])
        else:
            logging.error("Step does not produce output files! Unable to open output")
            return None

    def is_done(self):
        """Return true if this step is done and
        does not need execution

        The basic implementation checks if all files exists
        """
        for f in self.files():
            if not os.path.exists(f):
                return False
        return True


class PrepareInputStep(PipelineStep):
    """Prepare multiple input files,
    clean ids and write uncompressd file
    """
    def files(self):
        if self._files is None:
            self._files = []
            self._files.append(self.pipeline.create_file_name(self.name, file_suffix="gt.fastq", final=self.final))
        return self._files

    def __write(self):
        outfile = gt.OutputFile(self._final_output(), clean_id=True, append_extra=False)
        infile = self._input()
        infile.write_stream(outfile, write_map=False)
        infile.close()
        outfile.close()

    def run(self):
        """Merge current set of mappings and delete last ones"""
        # p = mp.Process(target=PrepareInputStep.__write, args=(self,))
        # p.start()
        # p.join()
        outfile = gt.OutputFile(self._final_output(), clean_id=True, append_extra=False)
        infile = self._input()
        infile.write_stream(outfile, write_map=False)
        infile.close()
        outfile.close()


class MergeStep(PipelineStep):
    """Merge up to the current step"""

    def run(self):
        """Merge current set of mappings and delete last ones"""
        same_content = self.configuration.get("same_content", True)
        inputs = self._input()
        master = inputs[0]
        slaves = inputs[1:]
        mapping = gem.merger(master, slaves).merge(self._output(), self.pipeline.threads, same_content=same_content, compress=self._compress())

        if self.final:
            gem.score(mapping, self.configuration["index"], self._final_output(),
                filter=self.pipeline.filter,
                threads=max(2, self.pipeline.threads / 2),
                quality=self.pipeline.quality,
                compress=self.pipeline.compress
            )

    def _input(self):
        """Return the output of all
        dependencies"""
        if not self.dependencies:
            raise PipelineError("You have to specify what to merge!")
        return [self.pipeline.open_step(i) for i in self.dependencies if i >= 0]


class MergeAndPairStep(PipelineStep):
    """Do merging and pairing in one single step"""
    def run(self):
        cfg = self.configuration
        same_content = self.configuration.get("same_content", True)
        inputs = self._input()
        master = inputs[0]
        slaves = inputs[1:]

        mapping = gem.merger(master, slaves).merge(None, threads=self.pipeline.threads, same_content=same_content, compress=False)

        pair_mapping = gem.pairalign(
            mapping,
            cfg["index"],
            self._output(),
          quality_threshold=cfg["quality_threshold"],
          max_decoded_matches=cfg["max_decoded_matches"],
          min_decoded_strata=cfg["min_decoded_strata"],
          min_insert_size=cfg["min_insert_size"],
          max_insert_size=cfg["max_insert_size"],
          max_edit_distance=cfg["max_edit_distance"],
          min_matched_bases=cfg["min_matched_bases"],
          max_extendable_matches=cfg["max_extendable_matches"],
          max_matches_per_extension=cfg["max_matches_per_extension"],
          threads=max(2, self.pipeline.threads / 2),  # self.pipeline.threads,
          quality=self.pipeline.quality,
          compress=self._compress())

        if self.final:
            gem.score(pair_mapping, cfg["index"], self._final_output(),
                filter=self.pipeline.filter,
                threads=self.pipeline.threads,  # max(2, self.pipeline.threads / 2),
                quality=self.pipeline.quality,
                compress=self.pipeline.compress,
                raw=True)

    def _input(self):
        """Return the output of all
        dependencies"""
        if not self.dependencies:
            raise PipelineError("You have to specify what to merge!")
        return [self.pipeline.open_step(i) for i in self.dependencies if i >= 0]


class CreateStatsStep(PipelineStep):
    """Create stats file"""

    def files(self):
        if self._files is None:
            self._files = []
            self._files.append(self.pipeline.create_file_name(self.name, name_suffix=".stats.all", file_suffix="txt", final=self.final))
            self._files.append(self.pipeline.create_file_name(self.name, name_suffix=".stats.all", file_suffix="json", final=self.final))
            self._files.append(self.pipeline.create_file_name(self.name, name_suffix=".stats.best", file_suffix="txt", final=self.final))
            self._files.append(self.pipeline.create_file_name(self.name, name_suffix=".stats.best", file_suffix="json", final=self.final))
        return self._files

    def run(self):
        def __read__write_stats(cfg, all_stats, best_stats, threads, out, infile):
            gt.read_stats(infile, all_stats, best_stats, threads=threads)
            # out = self.files()
            if cfg['stats_json'] is not None:
                with open(out[1], 'w') as f:
                    json.dump(all_stats.__dict__, f)
                with open(out[3], 'w') as f:
                    json.dump(best_stats.__dict__, f)

            with open(out[0], 'w') as f:
                all_stats.write(f)

            with open(out[2], 'w') as f:
                best_stats.write(f)

        cfg = self.configuration
        infile = self._input()
        best_stats = gt.Stats(True, cfg["stats_paired"])
        all_stats = gt.Stats(False, cfg["stats_paired"])

        import gem.utils
        import multiprocessing

        process = multiprocessing.Process(target=__read__write_stats, args=(cfg, all_stats, best_stats, self.pipeline.threads, self.files(), infile))
        gem.utils.register_process(process)
        process.start()
        process.join()





class CreateBamStep(PipelineStep):
    """Create BAM file"""

    def files(self):
        if self._files is None:
            self._files = []
            self._files.append(self.pipeline.create_file_name(self.name, file_suffix="bam", final=self.final))
        return self._files

    def run(self):
        cfg = self.configuration
        sam = gem.gem2sam(self._input(), cfg["index"], threads=self.pipeline.threads, quality=self.pipeline.quality, consensus=cfg['consensus'], exclude_header=cfg['sam_no_seq_header'], compact=cfg['sam_compact'])
        gem.sam2bam(sam, self._final_output(), sorted=cfg["sort"], mapq=cfg["mapq"], threads=self.pipeline.threads, sort_memory=self.pipeline.sort_memory)


class IndexBamStep(PipelineStep):
    """Index BAM file"""

    def files(self):
        if self._files is None:
            self._files = []
            self._files.append(self.pipeline.create_file_name(self.name, file_suffix="bam.bai", final=self.final))
        return self._files

    def run(self):
        #cfg = self.configuration
        gem.bamIndex(self._input(raw=True), self._final_output())


class MapStep(PipelineStep):
    """Mapping step"""
    def run(self):
        cfg = self.configuration
        mapping = gem.mapper(
            self._input(),
            cfg["index"],
            self._output(),
            mismatches=cfg["mismatches"],
            quality_threshold=cfg["quality_threshold"],
            max_decoded_matches=cfg["max_decoded_matches"],
            min_decoded_strata=cfg["min_decoded_strata"],
            min_matched_bases=cfg["min_matched_bases"],
            max_big_indel_length=cfg["max_big_indel_length"],
            max_edit_distance=cfg["max_edit_distance"],
            mismatch_alphabet=cfg["mismatch_alphabet"],
            delta=cfg["strata_after_best"],
            trim=cfg["trim"],
            quality=self.pipeline.quality,
            threads=self.pipeline.threads,
            compress=self._compress()
        )
        if self.final:
            gem.score(mapping, cfg["index"], self._final_output(),
                filter=self.pipeline.filter,
                threads=max(2, self.pipeline.threads / 2),
                quality=self.pipeline.quality,
                compress=self.pipeline.compress
            )


class PairalignStep(PipelineStep):
    """Pairalign"""
    def run(self):
        cfg = self.configuration
        mapping = gem.pairalign(
            self._input(),
            cfg["index"],
            self._output(),
          quality_threshold=cfg["quality_threshold"],
          max_decoded_matches=cfg["max_decoded_matches"],
          min_decoded_strata=cfg["min_decoded_strata"],
          min_insert_size=cfg["min_insert_size"],
          max_insert_size=cfg["max_insert_size"],
          max_edit_distance=cfg["max_edit_distance"],
          min_matched_bases=cfg["min_matched_bases"],
          max_extendable_matches=cfg["max_extendable_matches"],
          max_matches_per_extension=cfg["max_matches_per_extension"],
          threads=self.pipeline.threads,
          quality=self.pipeline.quality,
          compress=self._compress())

        if self.final:
            gem.score(mapping, cfg["index"], self._final_output(),
                filter=self.pipeline.filter,
                threads=max(2, self.pipeline.threads / 2),
                quality=self.pipeline.quality,
                compress=self.pipeline.compress)


class CreateDenovoTranscriptomeStep(PipelineStep):
    """Create denovo transcriptome"""

    def files(self):
        """Return the output files generated by this step.
        By default one .map output file is generated
        """
        if self._files is None:
            self._files = []
            index_denovo_out = self.pipeline.create_file_name(self.name, file_suffix="gem")
            junctions_out = self.pipeline.create_file_name(self.name, file_suffix="junctions")
            denovo_out = self.pipeline.create_file_name("", file_suffix="junctions")
            denovo_keys = junctions_out + ".keys"
            self._files.append(index_denovo_out)
            self._files.append(junctions_out)
            self._files.append(junctions_out + ".fa")
            self._files.append(denovo_keys)
            self._files.append(denovo_out)
            self._files.append(index_denovo_out[:-4] + ".log")
            self.index_denovo_out = index_denovo_out
            self.junctions_out = junctions_out
            self.denovo_keys = denovo_keys
            self.denovo_out = denovo_out
        return self._files

    def run(self):
        """Create the denovo transcriptome"""
        cfg = self.configuration
        (gtf_junctions, junctions_gtf_out) = self.pipeline.gtf_junctions()
        denovo_junctions = gem.extract_junctions(
            self._input(),
            cfg["index"],
            filter=cfg["filter"],
            splice_consensus=cfg["junctions_consensus"],
            mismatches=cfg["mismatches"],
            threads=self.pipeline.threads,
            strata_after_first=cfg["strata_after_best"],
            coverage=cfg["coverage"],
            min_split=cfg["min_split_length"],
            max_split=cfg["max_split_length"],
            refinement_step_size=cfg["refinement_step_size"],
            min_split_size=cfg["min_split_size"],
            matches_threshold=cfg["matches_threshold"],
            max_junction_matches=cfg["max_junction_matches"],
            annotation=cfg['annotation']

        )

        logging.gemtools.gt("Found Denovo Junctions %d with coverage >= %s" % (len(denovo_junctions), str(cfg["coverage"])))

        filtered_denovo_junctions = set(gem.junctions.filter_by_distance(denovo_junctions, cfg["min_split_length"], cfg["max_split_length"]))
        logging.gemtools.gt("Denovo junction passing distance filter (min: %d max: %d): %d (%d removed)" % (cfg["min_split_length"], cfg["max_split_length"],
            len(filtered_denovo_junctions), (len(denovo_junctions) - len(filtered_denovo_junctions))))

        gem.junctions.write_junctions(filtered_denovo_junctions, self.denovo_out, cfg["index"])

        if gtf_junctions is not None:
            logging.gemtools.gt("Joining with Annotation - denovo: %d annotation: %d" % (len(filtered_denovo_junctions), len(gtf_junctions)))
            junctions = gtf_junctions.union(filtered_denovo_junctions)
            logging.gemtools.gt("Joined Junctions %d" % (len(junctions)))
            gem.junctions.write_junctions(junctions, self.junctions_out, cfg["index"])
        else:
            logging.gemtools.gt("Skipped merging with annotation, denovo junctions: %d" % (len(filtered_denovo_junctions)))
            gem.junctions.write_junctions(filtered_denovo_junctions, self.junctions_out, cfg["index"])

        logging.gemtools.gt("Computing denovo transcriptome")
        (denovo_transcriptome, denovo_keys) = gem.compute_transcriptome(self.pipeline.max_read_length, cfg["index"], self.junctions_out, junctions_gtf_out)

        logging.gemtools.gt("Indexing denovo transcriptome")
        gem.index(denovo_transcriptome, self.index_denovo_out, threads=self.pipeline.threads)
        return (self.index_denovo_out, self.denovo_keys)

    def cleanup(self, force=False):
        if force or (not self.final and self.pipeline.remove_temp):
            for f in self.files():
                if os.path.exists(f) and f != self.denovo_out:
                    logging.gemtools.debug("Remove temporary file %s" % f)
                    os.remove(f)


class ExtractJunctionsStep(PipelineStep):
    """Extract denovo junctions"""

    def files(self):
        if self._files is None:
            self._files = []
            junctions_out = self.pipeline.create_file_name(self.name, file_suffix="junctions", final=self.final)
            self._files.append(junctions_out)
            self.junctions_out = junctions_out
        return self._files

    def run(self):
        """Extract denovo junctions"""
        cfg = self.configuration
        denovo_junctions = gem.extract_junctions(
            self._input(),
            cfg["index"],
            filter=cfg["filter"],
            splice_consensus=cfg["junctions_consensus"],
            mismatches=cfg["mismatches"],
            threads=self.pipeline.threads,
            strata_after_first=cfg["strata_after_best"],
            coverage=cfg["coverage"],
            min_split=cfg["min_split_length"],
            max_split=cfg["max_split_length"],
            refinement_step_size=cfg["refinement_step_size"],
            min_split_size=cfg["min_split_size"],
            matches_threshold=cfg["matches_threshold"],
            max_junction_matches=cfg["max_junction_matches"],
            annotation=cfg["annotation"],
        )

        logging.gemtools.gt("Found de-novo Junctions %d with coverage >= %s" % (len(denovo_junctions), str(cfg["coverage"])))
        filtered_denovo_junctions = set(gem.junctions.filter_by_distance(denovo_junctions, cfg["min_split_length"], cfg["max_split_length"]))

        logging.gemtools.gt("de-novo junction passing distance filter (min: %s max: %s): %d (%s removed)" % (str(cfg["min_split_length"]), str(cfg["max_split_length"]),
            len(filtered_denovo_junctions), (len(denovo_junctions) - len(filtered_denovo_junctions))))

        gem.junctions.write_junctions(filtered_denovo_junctions, self.junctions_out, cfg["index"])
        return self.junctions_out

class SplitMapStep(PipelineStep):
    """Call the split mapper"""

    def run(self):
        """Extract denovo junctions"""
        cfg = self.configuration
        splitmap = gem.splitmapper(self._input(),
                cfg["index"],
                output=self._final_output(),
                mismatches=cfg["mismatches"],
                splice_consensus=cfg["junctions_consensus"],
                filter=cfg["filter"],
                refinement_step_size=cfg["refinement_step_size"],
                min_split_size=cfg["min_split_size"],
                matches_threshold=cfg["matches_threshold"],
                strata_after_first=cfg["strata_after_best"],
                mismatch_alphabet=cfg["mismatch_alphabet"],
                quality=self.pipeline.quality,
                trim=cfg["trim"],
                filter_splitmaps=True,
                post_validate=True,
                threads=self.pipeline.threads,
                extra=None)
        return splitmap


class TranscriptMapStep(PipelineStep):
    """Transcript Mapping step"""

    def prepare(self, id, pipeline, config):
        PipelineStep.prepare(self, id, pipeline, config)
        if config["denovo"]:
            # denovo transcript mapping
            if config["index"] is None or not os.path.exists(config["index"]):
                # add denovo transript step
                create_step_id = pipeline.create_transcriptome("denovo-index", dependencies=self.dependencies)
                self.dependencies.append(create_step_id)
                self.configuration["create_index"] = create_step_id
                if config["index"] is None:
                    # update the configuration
                    config["index"] = pipeline.steps[create_step_id].index_denovo_out
                    config["keys"] = pipeline.steps[create_step_id].denovo_keys
                # this is ugly but we have to increase the id here as we squeezed in another job
                self.id = create_step_id + 1

    def run(self):
        cfg = self.configuration
        if cfg["denovo"] and cfg["create_index"]:
            step = self.pipeline.steps[cfg["create_index"]]
            cfg["index"] = step.index_denovo_out
            cfg["keys"] = step.denovo_keys

        gem.mapper(
            self._input(),
            cfg["index"],
            self.files()[0],
            mismatches=cfg["mismatches"],
            quality_threshold=cfg["quality_threshold"],
            max_decoded_matches=cfg["max_decoded_matches"],
            min_decoded_strata=cfg["min_decoded_strata"],
            min_matched_bases=cfg["min_matched_bases"],
            max_big_indel_length=cfg["max_big_indel_length"],
            max_edit_distance=cfg["max_edit_distance"],
            mismatch_alphabet=cfg["mismatch_alphabet"],
            delta=cfg["strata_after_best"],
            trim=cfg["trim"],
            key_file=cfg["keys"],
            quality=self.pipeline.quality,
            threads=self.pipeline.threads
        )

    def _input(self, raw=False):
        """Return pipeline input if this step
        has no dependencies or the
        output of the last dependency
        """
        if self.configuration["denovo"]:
            if self.dependencies is None or len(self.dependencies) == 1:
                return self.pipeline.open_input()
            return self.pipeline.open_step(self.dependencies[0], raw=raw)
        else:
            return PipelineStep._input(self, raw=raw)


class MappingPipeline(object):
    """General mapping pipeline class."""

    def __init__(self, args=None):
        self.steps = []  # pipeline steps
        self.run_steps = []  # steps to run

        # general parameter
        self.input = None  # input files
        self.name = None  # target name
        self.index = None  # genome index
        self.output_dir = None  # Output directory
        self.annotation = None  # GTF annotation to use
        self.threads = 1  # number of threads
        self.max_read_length = 150  # max read length
        self.transcript_index = None  # transcriptome index
        self.transcript_keys = None  # transcriptome keys file
        self.denovo_index = None  # the denovo index to use
        self.denovo_keys = None  # the denovo keys to use
        self.quality = None  # quality offset
        self.junctions_file = None  # file with both denovo and GTF junctions
        self.junctions_annotation = None  # file with the annotation junctions
        self.scoring_scheme = "+U,+u,-s,-t,+1,-i,-a"  # scoring scheme
        self.compress = True  # compress final output
        self.compress_all = False  # also compress intermediate output
        self.remove_temp = True  # remove temporary
        self.bam_mapq = 0  # filter bam content mapq
        self.bam_create = True  # create bam
        self.bam_sort = True  # sort bam
        self.bam_index = True  # index bam
        self.sam_no_seq_header = False  # exlude seq header
        self.sam_compact = False  # sam compact format
        self.single_end = False  # single end alignments
        self.write_config = None  # write configuration
        self.dry = False  # only dry run
        self.sort_memory = "768M"  # samtools sort memory
        self.direct_input = False  # if true, skip the preparation step
        self.force = False  # force computation of all steps

        self.filter_max_matches = 25
        self.filter_min_strata = 1
        self.filter_max_strata = 2
        self.filter = None

        # genome mapping parameter
        self.genome_mismatches = 0.06
        self.genome_quality_threshold = 26
        self.genome_max_decoded_matches = 25
        self.genome_min_decoded_strata = 1
        self.genome_min_matched_bases = 0.80
        self.genome_max_big_indel_length = 15
        self.genome_max_edit_distance = 0.20
        self.genome_mismatch_alphabet = "ACGT"
        self.genome_strata_after_best = 1

        # transcript mapping parameter
        self.transcript_mismatches = None  # initialize from genom
        self.transcript_quality_threshold = None  # initialize from genome
        self.transcript_max_decoded_matches = 150  # this need to be custom
        self.transcript_min_decoded_strata = None  # initialize from genome
        self.transcript_min_matched_bases = None  # initialize from genome
        self.transcript_max_big_indel_length = None  # initialize from genome
        self.transcript_max_edit_distance = None  # initialize from genome
        self.transcript_mismatch_alphabet = None  # initialize from genome
        self.transcript_strata_after_best = None  # initialize from genome

        # junction detection parameter
        self.junction_mismatches = 0.04
        self.junctions_max_junction_matches = 5
        self.junctions_min_intron_size = 4
        self.junctions_max_intron_size = 500000
        self.junctions_refinement_step_size = 2
        self.junctions_min_split_size = 15
        self.junctions_matches_threshold = 75
        self.junctions_coverage = 2
        self.junctions_filtering = "ordered,non-zero-distance"
        self.junctions_consensus = gem.extended_splice_consensus
        self.junctions_strata_after_best = 0

        # pair alignment parameter
        self.pairing_quality_threshold = None
        self.pairing_max_decoded_matches = 25
        self.pairing_min_decoded_strata = 1
        self.pairing_min_insert_size = 0
        self.pairing_max_insert_size = None
        self.pairing_max_edit_distance = 0.30
        self.pairing_min_matched_bases = 0.80
        self.pairing_max_extendable_matches = 0
        self.pairing_max_matches_per_extension = 0

        # stats parameter
        self.stats_create = True
        self.stats_json = False

        if args is not None:
            # initialize from arguments
            # load configuration
            try:
                if args.load_configuration is not None:
                    self.load(args.load_configuration)
            except AttributeError:
                pass
            ## update parameter
            self.update(vars(args))
            ## initialize pipeline and check values
            self.initialize()

    def update(self, configuration):
        """Update configuration from given map

        configuration -- the input configuration
        """
        for k, v in configuration.items():
            try:
                if v is not None:
                    setattr(self, k, v)
            except AttributeError:
                pass

    def __update_dict(self, target, source):
        if source is None:
            return
        for k, v in source.items():
            #if v is not None:
            target[k] = v

    def map(self, name, configuration=None, dependencies=None, final=False, description=""):
        """Add mapping step"""
        step = MapStep(name, final=final, dependencies=dependencies, description=description)
        config = dotdict()

        config.index = self.index
        config.mismatches = self.genome_mismatches
        config.quality_threshold = self.genome_quality_threshold
        config.max_decoded_matches = self.genome_max_decoded_matches
        config.min_decoded_strata = self.genome_min_decoded_strata
        config.min_matched_bases = self.genome_min_matched_bases
        config.max_big_indel_length = self.genome_max_big_indel_length
        config.max_edit_distance = self.genome_max_edit_distance
        config.mismatch_alphabet = self.genome_mismatch_alphabet
        config.strata_after_best = self.genome_strata_after_best
        config.trim = None

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id


    def splitmap(self, name, configuration=None, dependencies=None, final=False, description=""):
        """Add split-mapping step"""
        step = SplitMapStep(name, final=final, dependencies=dependencies, description=description)
        config = dotdict()

        config.index = self.index
        config.mismatches = self.junction_mismatches
        config.junctions_consensus = self.junctions_consensus
        config.filter = self.junctions_filtering
        config.refinement_step_size = self.junctions_refinement_step_size
        config.min_split_size = self.junctions_min_split_size
        config.matches_threshold = self.junctions_matches_threshold
        config.strata_after_best = self.junctions_strata_after_best
        config.mismatch_alphabet = self.genome_mismatch_alphabet
        config.max_edit_distance = self.genome_max_edit_distance
        config.mismatch_alphabet = self.genome_mismatch_alphabet
        config.strata_after_best = self.genome_strata_after_best
        config.trim = None

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id


    def pair(self, name, configuration=None, dependencies=None, final=False, description=""):
        """Add mapping step"""
        step = PairalignStep(name, dependencies=dependencies, final=final, description=description)
        config = dotdict()

        config.index = self.index
        config.quality_threshold = self.pairing_quality_threshold
        config.max_decoded_matches = self.pairing_max_decoded_matches
        config.min_decoded_strata = self.pairing_min_decoded_strata
        config.min_insert_size = self.pairing_min_insert_size
        config.max_insert_size = self.pairing_max_insert_size
        config.max_edit_distance = self.pairing_max_edit_distance
        config.min_matched_bases = self.pairing_min_matched_bases
        config.max_extendable_matches = self.pairing_max_extendable_matches
        config.max_matches_per_extension = self.pairing_max_matches_per_extension

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id

    def transcripts_annotation(self, name=None, configuration=None, dependencies=None, final=False, description=""):
        """Create annotation based transcriptom and map"""
        if self.annotation is None:
            logging.gemtools.info("No annotation specified, skipping annotation mapping")
            return -1
        step = TranscriptMapStep(name, dependencies=dependencies, final=final, description=description)
        config = dotdict()
        config.denovo = False
        config.annotation = self.annotation
        config.index = self.transcript_index
        config.keys = self.transcript_keys
        config.mismatches = self.transcript_mismatches
        config.quality_threshold = self.transcript_quality_threshold
        config.max_decoded_matches = self.transcript_max_decoded_matches
        config.min_decoded_strata = self.transcript_min_decoded_strata
        config.min_matched_bases = self.transcript_min_matched_bases
        config.max_big_indel_length = self.transcript_max_big_indel_length
        config.max_edit_distance = self.transcript_max_edit_distance
        config.mismatch_alphabet = self.transcript_mismatch_alphabet
        config.strata_after_best = self.transcript_strata_after_best
        config.trim = None

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id

    def transcripts_denovo(self, name=None, configuration=None, dependencies=None, final=False, description=""):
        """Create annotation based transcriptom and map"""
        step = TranscriptMapStep(name, dependencies=dependencies, final=final, description=description)

        config = dotdict()
        config.denovo = True
        config.index = self.denovo_index
        config.keys = self.denovo_keys
        config.mismatches = self.transcript_mismatches
        config.quality_threshold = self.transcript_quality_threshold
        config.max_decoded_matches = self.transcript_max_decoded_matches
        config.min_decoded_strata = self.transcript_min_decoded_strata
        config.min_matched_bases = self.transcript_min_matched_bases
        config.max_big_indel_length = self.transcript_max_big_indel_length
        config.max_edit_distance = self.transcript_max_edit_distance
        config.mismatch_alphabet = self.transcript_mismatch_alphabet
        config.strata_after_best = self.transcript_strata_after_best
        config.trim = None

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id

    def create_transcriptome(self, name, configuration=None, dependencies=None, final=False, description="Create denovo transcript index"):
        step = CreateDenovoTranscriptomeStep(name, dependencies=dependencies, final=final, description=description)
        config = dotdict()

        config.index = self.index
        config.filter = self.junctions_filtering
        config.junctions_consensus = self.junctions_consensus
        config.mismatches = self.junction_mismatches
        config.max_junction_matches = self.junctions_max_junction_matches
        config.min_split_length = self.junctions_min_intron_size
        config.max_split_length = self.junctions_max_intron_size
        config.strata_after_best = self.junctions_strata_after_best
        config.refinement_step_size = self.junctions_refinement_step_size
        config.min_split_size = self.junctions_min_split_size
        config.matches_threshold = self.junctions_matches_threshold
        config.coverage = self.junctions_coverage
        config.annotation = self.annotation

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id

    def extract_junctions(self, name, configuration=None, dependencies=None, final=False, description="Extract de-novo junctions"):
        step = ExtractJunctionsStep(name, dependencies=dependencies, final=final, description=description)
        config = dotdict()

        config.index = self.index
        config.annotation = self.annotation
        config.filter = self.junctions_filtering
        config.junctions_consensus = self.junctions_consensus
        config.mismatches = self.junction_mismatches
        config.max_junction_matches = self.junctions_max_junction_matches
        config.min_split_length = self.junctions_min_intron_size
        config.max_split_length = self.junctions_max_intron_size
        config.strata_after_best = self.junctions_strata_after_best
        config.refinement_step_size = self.junctions_refinement_step_size
        config.min_split_size = self.junctions_min_split_size
        config.matches_threshold = self.junctions_matches_threshold
        config.coverage = self.junctions_coverage

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id

    def create_stats(self, name, configuration=None, dependencies=None, final=False, description="Create stats"):
        step = CreateStatsStep(name, dependencies=dependencies, final=final, description=description)
        config = dotdict()

        config.stats_json = self.stats_json
        config.stats_paired = not self.single_end

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id

    def bam(self, name, configuration=None, dependencies=None, final=False, description="Create BAM file"):
        step = CreateBamStep(name, dependencies=dependencies, final=final, description=description)
        config = dotdict()

        config.index = self.index
        config.mapq = self.bam_mapq
        config.sort = self.bam_sort
        config.consensus = self.junctions_consensus
        config.sam_no_seq_header = self.sam_no_seq_header
        config.sam_compact = self.sam_compact

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id

    def prepare_input(self, name, configuration=None, dependencies=None, final=False, description="Prepare input"):
        step = PrepareInputStep(name, dependencies=dependencies, final=final, description=description)
        config = dotdict()

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id

    def index_bam(self, name, configuration=None, dependencies=None, final=False, description="Index BAM file"):
        step = IndexBamStep(name, dependencies=dependencies, final=final, description=description)
        config = dotdict()

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id

    def merge(self, name, configuration=None, dependencies=None, final=False, description="Merge alignments"):
        step = MergeStep(name, dependencies=dependencies, final=final, description=description)
        config = dotdict()

        config.same_content = True
        config.index = self.index

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id

    def merge_and_pair(self, name, configuration=None, dependencies=None, final=False, description="Merge and Pair alignments"):
        step = MergeAndPairStep(name, dependencies=dependencies, final=final, description=description)
        config = dotdict()

        config.same_content = True
        config.index = self.index
        config.quality_threshold = self.pairing_quality_threshold
        config.max_decoded_matches = self.pairing_max_decoded_matches
        config.min_decoded_strata = self.pairing_min_decoded_strata
        config.min_insert_size = self.pairing_min_insert_size
        config.max_insert_size = self.pairing_max_insert_size
        config.max_edit_distance = self.pairing_max_edit_distance
        config.min_matched_bases = self.pairing_min_matched_bases
        config.max_extendable_matches = self.pairing_max_extendable_matches
        config.max_matches_per_extension = self.pairing_max_matches_per_extension

        if configuration is not None:
            self.__update_dict(config, configuration)

        step.prepare(len(self.steps), self, config)
        self.steps.append(step)
        return step.id

    def open_input(self):
        """Open the original input files"""
        if len(self.input) == 1:
            return gem.files.open(self.input[0])
        else:
            return gem.filter.interleave([gem.files.open(f) for f in self.input], threads=max(1, self.threads / 2))

    def open_step(self, id, raw=False):
        """Open the original input files"""
        return self.steps[id].open(raw=raw)

    def initialize(self, silent=False):
        # check general parameter
        errors = []
        if self.input is None:
            errors.append("No input file specified")
        else:
            if len(self.input) == 1 and not self.single_end:
                # search for second file
                (n, p) = gem.utils.find_pair(self.input[0])
                if p is None:
                    #errors.append("Unable to deduce second pair input file from %s " % self.input[0])
                    logging.gemtools.warning("No second input file specified, assuming interleaved paird end reads!")
                else:
                    logging.gemtools.warning("Second pair input file found: %s " % p)
                    if self.name is None:
                        self.name = n
                    self.input.append(p)

            # check file counts
            if self.single_end and len(self.input) != 1:
                errors.append("Specify exactly one input file in single end mode")
            elif not self.single_end and len(self.input) > 2:
                errors.append("Paired end mode takes up to 2 input files, you specified %d" % (len(self.input)))
            else:
                # check input files
                input_abs = []
                for f in self.input:
                    if f is None or not os.path.exists(f):
                        errors.append("Input file not found: %s" % (f))
                    else:
                        # make aboslute path
                        input_abs.append(os.path.abspath(f))
                self.input = input_abs

        if self.name is None and self.input is not None and len(self.input) > 0:
            # get name from input files
            name = os.path.basename(self.input[0])
            if name.endswith(".gz"):
                name = name[:-3]
            idx = name.rfind(".")
            if idx > 0:
                self.name = name[:idx]

        if self.name is None or len(self.name) == 0:
            errors.append("No name specified and unable to guess one. Please use --name to set a name explicitly.")

        if self.index is None:
            errors.append("No index specified")
        else:
            if not os.path.exists(self.index):
                errors.append("Index not found: %s" % (self.index))
            else:
                self.index = os.path.abspath(self.index)

        if self.quality is None:
            errors.append("You have to specify a quality offset (33, 64, or 'ignore' to disable)")
        elif str(self.quality) not in ["33", "64", "ignore", "offset-33", "offset-64"]:
            errors.append("Unknown quality offset: %s, please use 33, 64 or ignore" % (str(self.quality)))

        if self.output_dir is None:
            self.output_dir = os.getcwd()

        self.output_dir = os.path.abspath(self.output_dir)

        if self.annotation is not None:
            if not os.path.exists(self.annotation):
                errors.append("Annotaiton not found : %s" % self.annotation)
            else:
                self.annotation = os.path.abspath(self.annotation)

        if self.threads <= 0:
            self.threads = 1

        if self.transcript_index is None and self.annotation is not None:
            # guess the transcript index
            self.transcript_index = self.annotation + ".gem"
            transcript_index_found = os.path.exists(self.transcript_index)
            if not transcript_index_found:
                self.transcript_index = self.annotation + ".junctions.gem"
                transcript_index_found = os.path.exists(self.transcript_index)
            if not transcript_index_found:
                errors.append("Deduced transcript index not found: %s" % (self.transcript_index))
                errors.append("""We look for the transcriptome index just next to your annotation, but
could not find it there. Try to specify a path to the transcriptome index using
[-r|--transcript-index] <index>, where index is the path to the transcriptome
index generated from your annotation.""")
            else:
                self.transcript_index = os.path.abspath(self.transcript_index)
        elif self.annotation is not None and not os.path.exists(self.transcript_index):
            errors.append("Transcript index not found : %s")
        elif self.transcript_index is not None and os.path.exists(self.transcript_index):
            self.transcript_index = os.path.abspath(self.transcript_index)

        if self.transcript_keys is None and self.transcript_index is not None:
            self.transcript_keys = self.transcript_index[:-4] + ".junctions.keys"
            transcript_keys_found = os.path.exists(self.transcript_keys)
            if not transcript_keys_found:
                self.transcript_keys = self.transcript_index[:-14] + ".keys"
                transcript_keys_found = os.path.exists(self.transcript_keys)
                if not transcript_keys_found:
                    self.transcript_keys = self.transcript_index[:-4] + ".keys"
                    transcript_keys_found = os.path.exists(self.transcript_keys)

            if not transcript_keys_found:
                errors.append("Deduced transcript keys not found: %s" % (self.transcript_keys))
            else:
                self.transcript_keys = os.path.abspath(self.transcript_keys)
        elif self.transcript_keys is not None and not os.path.exists(self.transcript_keys):
            errors.append("Transcript keys not found : %s")
        elif self.transcript_keys is not None and os.path.exists(self.transcript_keys):
            self.transcript_keys = os.path.abspath(self.transcript_keys)

        # check inpuf compression
        if self.compress_all and not self.direct_input:
            logging.gemtools.warning("Enabeling direct input for compressed temporay files")
            self.direct_input = True

        # annotaiton junctons should be generated if not found
        #self.junctions_annotation = None  # file with the annotation junctions

        # todo : can we check for a valid scoring scheme ?
        #self.scoring_scheme = "+U,+u,-s,-t,+1,-i,-a"  # scoring scheme

        if self.filter_min_strata >= self.filter_max_strata:
            errors.append("Invalid filtering configuration, min-strata >= max-strata!")
        else:
            if self.filter_max_matches <= 0:
                errors.append("Invalid filtering configuration, max-matches <= 0!")
            else:
                # compose filter
                self.filter = (self.filter_min_strata, (self.filter_max_strata - self.filter_min_strata), self.filter_max_matches)

        if self.bam_mapq > 254:
            errors.append("Invalid mapq filter: %s" % (self.bam_mapq))

        # transcript mapping parameter
        if self.transcript_mismatches is None:
            self.transcript_mismatches = self.genome_mismatches
        if self.transcript_quality_threshold is None:
            self.transcript_quality_threshold = self.genome_quality_threshold

        self.transcript_min_decoded_strata = self.genome_min_decoded_strata
        self.transcript_min_matched_bases = self.genome_min_matched_bases
        self.transcript_max_big_indel_length = self.genome_max_big_indel_length
        self.transcript_max_edit_distance = self.genome_max_edit_distance

        if self.transcript_mismatch_alphabet is None:
            self.transcript_mismatch_alphabet = self.genome_mismatch_alphabet
        if self.transcript_strata_after_best is None:
            self.transcript_strata_after_best = self.genome_strata_after_best

        # pair alignment parameter
        if self.pairing_quality_threshold is None:
            self.pairing_quality_threshold = self.genome_quality_threshold

        if self.pairing_max_insert_size is None:
            self.pairing_max_insert_size = self.junctions_max_intron_size

        if not self.single_end and len(errors) == 0:
            # check pairing information
            p1 = None
            p2 = None
            c = 0
            inp = self.open_input()
            for template in inp:
                if template.num_alignments == 2:
                    ## paired alignment
                    p1 = 1
                    p2 = 2
                    inp.close()
                    break
                else:
                    if c == 0:
                        p1 = template.get_pair()
                    elif c == 1:
                        p2 = template.get_pair()
                    c += 1
                    if c >= 2:
                        inp.close()
                        break
            inp.close()
            if p1 == 0 or p2 == 0 or (p1 == 1 and p2 != 2) or (p2 == 1 and p1 != 2):
                errors.append("""Unable to get pairing information from input.
                    Please check your read id's and make sure its either in casava >= 1.8 format or the
                    ids end with /1 and /2""")

        if not silent and len(errors) > 0 and self.write_config is None:
            raise PipelineError("Failed to initialize neccessary parameters:\n\n%s" % ("\n".join(errors)))
        if self.write_config is not None:
            # log configuration errors
            logging.gemtools.warning("---------------------------------------------")
            logging.gemtools.warning("Writing configuration")
            logging.gemtools.warning("")
            logging.gemtools.warning("Note that some of the parameters are missing:\n")
            for e in errors:
                logging.gemtools.warning("\t" + str(e))
            logging.gemtools.warning("---------------------------------------------")

    def log_parameter(self):
        """Print selected parameters"""
        printer = logging.gemtools.gt
        run_step = len(self.run_steps) > 0

        printer("------------ Input Parameter ------------")
        printer("Input File(s)    : %s", self.input)
        printer("Index            : %s", self.index)
        printer("Annotation       : %s", self.annotation)
        printer("Transcript Index : %s", self.transcript_index)
        printer("Max read length  : %s", self.max_read_length)
        printer("")
        printer("Compress output  : %s", self.compress)
        printer("Compress all     : %s", self.compress_all)
        printer("Create BAM       : %s", self.bam_create)
        printer("SAM/BAM compact  : %s", self.sam_compact)
        printer("Sort BAM         : %s", self.bam_sort)
        printer("Index BAM        : %s", self.bam_index)
        printer("Keep Temporary   : %s", not self.remove_temp)
        printer("")

        if not run_step:
            printer("------------ Pipeline Steps  ------------")
            for i, s in enumerate(self.steps):
                printer("%-2d - %20s : %s", i, s.name, s.description)
        else:
            printer("------------ Single Step execution  ------------")

        for i, s in enumerate(self.steps):
            if run_step and s.id not in self.run_steps:
                continue
            printer("")
            if len(s.dependencies) > 0:
                printer("------------ [ID:{0:-3} -- '{1}'] [Depends On: {2}] ------------".format(i, s.name, ", ".join([self.steps[j].name for j in s.dependencies])))
            else:
                printer("------------ [ID:{0:-3} -- '{1}'] ------------".format(i, s.name))

            for k, v in s.configuration.items():
                printer("%25s : %s", k, str(v))

            for i, f in enumerate(s.files()):
                if i == 0:
                    printer("%25s : %s", "Temporary Outputs", not s.final)
                    printer("%25s : %s", "Outputs", f)
                else:
                    printer("%25s : %s", "", f)
        printer("--------------------------------------------------------------")
        printer("")

    def load(self, file):
        """Load pipeline configuration from file"""
        if file is None or not os.path.exists(file):
            raise PipelineError("Configuration file not found: %s" % file)
        fd = open(file, "r")
        logging.gemtools.info("Loading configuraiton from %s", file)
        data = json.load(fd)
        for k, v in data.items():
            if hasattr(self, k):
                setattr(self, k, v)
        fd.close()

    def write_pipeline(self, file_name):
        """Write the pipeline and its configuration to a file
        based on the name
        """

        json_container = dict(self.__dict__)
        # skip the steps here, we convert them manually
        del json_container["steps"]
        del json_container["run_steps"]
        del json_container["write_config"]
        # remove non default values
        default_pipeline = MappingPipeline()
        default_pipeline.initialize(silent=True)
        for k, v in json_container.items():
            if hasattr(default_pipeline, k) and getattr(default_pipeline, k) == v:
                del json_container[k]

        # json_container['piepline_steps'] = json_steps

        fd = open(file_name, "w")
        json.dump(json_container, fd, indent=2, sort_keys=True)
        fd.close()
        logging.gemtools.gt("Configuration saved to %s\n", file_name)

    def run(self):
        run_step = len(self.run_steps) > 0

        if self.write_config is not None:
            self.write_pipeline(self.write_config)
            return
        if self.dry:
            return

        error = False

        all_done = True
        final_files = []
        # check final steps if we are not running a set of steps
        if not run_step and not self.force:
            for step in self.steps:
                if step.final:
                    final_files.extend(step.files())
                    all_done = all_done & step.is_done()
            if all_done:
                logging.gemtools.warning("The following files already exist. Nothing to be run!\n\n%s\n" % ("\n".join(final_files)))
                return

        time = Timer()
        times = {}

        if run_step:
            # sort by id
            self.run_steps.sort()
        ids = [s.id for s in self.steps]
        if run_step:
            ids = self.run_steps

        step = None

        # register signal handler to catch
        # interruptions and perform cleanup
         # register cleanup signal handler
        def cleanup_in_signal(signal, frame):
            logging.gemtools.warning("Job step canceled, forcing cleanup!")
            step.cleanup(force=True)

        signal.signal(signal.SIGINT, cleanup_in_signal)
        signal.signal(signal.SIGQUIT, cleanup_in_signal)
        signal.signal(signal.SIGHUP, cleanup_in_signal)
        signal.signal(signal.SIGTERM, cleanup_in_signal)

        for step_id in ids:
            step = self.steps[step_id]

            if run_step:
                # check dependencies are done
                for d in step.dependencies:
                    if not self.steps[d].is_done():
                        logging.gemtools.error("Step dependency is not completed : %s", self.steps[d].name)
                        error = True
                        break
            if run_step or self.force or not step.is_done():
                logging.gemtools.gt("Running step: %s" % step.name)
                t = Timer()

                if not os.path.exists(self.output_dir):
                    # make sure we create the ouput folder
                    logging.gemtools.warn("Creating output folder %s", self.output_dir)
                    try:
                        os.makedirs(self.output_dir)
                    except OSError as exc: # Python >2.5
                        if exc.errno == errno.EEXIST and os.path.isdir(path):
                            pass
                        else:
                            logging.gemtools.error("unable to create output folder %s", self.output_dir)
                            error = True
                            break

                try:
                    step.run()
                except KeyboardInterrupt:
                    logging.gemtools.warning("Job step canceled, forcing cleanup!")
                    error = True
                    step.cleanup(force=True)
                    break
                except PipelineError, e:
                    logging.gemtools.error("Error while executing step %s : %s" % (step.name, str(e)))
                    logging.gemtools.warning("Cleaning up after failed step : %s", step.name)
                    step.cleanup(force=True)
                    error = True
                    break
                except gem.utils.ProcessError, e:
                    logging.gemtools.error("Error while executing step %s : %s" % (step.name, str(e)))
                    logging.gemtools.warning("Cleaning up after failed step : %s", step.name)
                    step.cleanup(force=True)
                    error = True
                    break
                except Exception, e:
                    traceback.print_exc()
                    logging.gemtools.error("Error while executing step %s : %s" % (step.name, str(e)))
                    logging.gemtools.warning("Cleaning up after failed step : %s", step.name)
                    step.cleanup(force=True)
                    error = True
                    break
                finally:
                    t.stop(step.name + " completed in %s", loglevel=None)
                    times[step.id] = t.end
                    if not error:
                        logging.gemtools.gt("Step %s finished in : %s", step.name, t.end)
                    else:
                        logging.gemtools.gt("Step %s failed after : %s", step.name, t.end)
            else:
                logging.gemtools.warning("Skipping step %s, output already exists" % (step.name))

        # do celanup if not in error state
        if not error:
            logging.gemtools.debug("Cleanup after run")
            for step in self.steps:
                step.cleanup()

            time.stop("Completed in %s", loglevel=None)
            logging.gemtools.gt("Step Times")
            logging.gemtools.gt("-------------------------------------")
            for s in self.steps:
                if s.id in times:
                    logging.gemtools.gt("{0:>25} : {1}".format(s.name, times[s.id]))
                else:
                    logging.gemtools.gt("{0:>25} : skipped".format(s.name))
            logging.gemtools.gt("-------------------------------------")
            logging.gemtools.gt("Pipeline run finshed in %s", time.end)

    def cleanup(self):
        """Delete all remaining temporary and intermediate files
        """
        pass

    def create_file_name(self, suffix, name_suffix=None, file_suffix="map", final=False):
        """Create a result file name"""
        file = ""
        if final:
            suffix = None
        if name_suffix is None:
            name_suffix = ""

        if suffix is not None and len(suffix) > 0:
            file = "%s/%s%s_%s.%s" % (self.output_dir, self.name, name_suffix, suffix, file_suffix)
        else:
            file = "%s/%s%s.%s" % (self.output_dir, self.name, name_suffix, file_suffix)
        if (self.compress_all or final and self.compress) and file_suffix in ["map", "fastq"]:
            file += ".gz"
        return file

    def gtf_junctions(self):
        """check if there is a .junctions file for the given annotation, if not,
        create it. Returns a tuple of the set of junctions and the output file.
        """
        if self.annotation is None:
            return (None, None)

        timer = Timer()
        gtf_junctions = self.annotation + ".junctions"
        out = None
        junctions = None
        if os.path.exists(gtf_junctions):
            logging.gemtools.info("Loading existing junctions from %s" % (gtf_junctions))
            out = gtf_junctions
            junctions = set(gem.junctions.from_junctions(gtf_junctions))
        else:
            out = self.create_file_name("gtf", file_suffix="junctions")
            if os.path.exists(out):
                logging.gemtools.info("Loading existing junctions from %s" % (out))
                junctions = set(gem.junctions.from_junctions(out))
            else:
                logging.gemtools.info("Extracting junctions from %s" % (self.annotation))
                junctions = set(gem.junctions.from_gtf(self.annotation))
                gem.junctions.write_junctions(junctions, out, self.index)

        logging.gemtools.info("%d Junctions from GTF" % (len(junctions)))
        timer.stop("GTF-Junctions prepared in %s")
        return (junctions, out)

    def register_parameter(self, parser):
        """Register all parameters with the given
        arparse parser"""
        self.register_general(parser)
        self.register_output(parser)
        self.register_filtering(parser)
        self.register_mapping(parser)
        self.register_transcript_mapping(parser)
        self.register_junctions(parser)
        self.register_pairing(parser)
        self.register_bam(parser)
        self.register_stats(parser)
        self.register_execution(parser)

    def register_filtering(self, parser):
        """Register all filtering parameters with given
        argparse parser

        parser -- the argparse parser
        """
        filtering_group = parser.add_argument_group('Filtering and Scoring')
        filtering_group.add_argument('-S', '--scoring', dest="scoring_scheme", metavar="scheme", help='''The scoring scheme to use. Default %s''' % str(self.scoring_scheme))
        filtering_group.add_argument('--filter-max-matches', dest="filter_max_matches", metavar="max_matches", type=int, help='''Maximum number of printed matches. Default %d''' % self.filter_max_matches)
        filtering_group.add_argument('--filter-min-strata', dest="filter_min_strata", metavar="min_strata", type=int, help='''Minimum number of printed strata. Default %d''' % self.filter_min_strata)
        filtering_group.add_argument('--filter-max-strata', dest="filter_max_strata", metavar="max_strata", type=int, help='''Maximum number of printed strata. Default %d''' % self.filter_max_strata)

    def register_bam(self, parser):
        """Register all bam parameters with given
        argparse parser

        parser -- the argparse parser
        """
        bam_group = parser.add_argument_group('BAM conversion')
        bam_group.add_argument('--map-quality', dest="bam_mapq", default=self.bam_mapq, type=int, help="Filter resulting bam for minimum map quality, Default %d" % self.bam_mapq)
        bam_group.add_argument('--no-bam', dest="bam_create", action="store_false", default=None, help="Do not create bam file")
        bam_group.add_argument('--no-bam-sort', dest="bam_sort", action="store_false", default=None, help="Do not sort bam file")
        bam_group.add_argument('--no-bam-index', dest="bam_index", action="store_false", default=None, help="Do not index the bam file")
        bam_group.add_argument('--no-sequence-header', dest="sam_no_seq_header", action="store_true", default=None, help="Do not add the reference sequence header to the sam/bam file")
        bam_group.add_argument('--compact', dest="sam_compact", action="store_true", default=None, help="Create sam/bam compact format where each read is represented as a single line and any multi-maps are encoded in extra fields. The selection is based on the score.")
        bam_group.add_argument('--sort-memory', dest="sort_memory", default=self.sort_memory, metavar="mem", help="Memory used for samtools sort per thread. Suffix K/M/G recognized. Default %s" % (str(self.sort_memory)))

    def register_general(self, parser):
        """Register all general parameters with the given
        argparse parser

        parser -- the argparse parser
        """
        input_group = parser.add_argument_group('Input')
        ## general pipeline paramters
        input_group.add_argument('-f', '--files', dest="input", nargs="+", metavar="input",
            help='''Single fastq input file or both files for a paired-end run separated by space.
            Note that if you specify only one file, we will look for the file containing the other pairs
            automatically and start a paired-end run. Add the --single-end parameter to disable
            pairing and file search. The file search for the second pair detects pairs
            ending in [_|.|-][0|1|2].[fq|fastq|txt][.gz].''')
        input_group.add_argument('--single-end', dest="single_end", action="store_true", default=None, help="Single end reads")
        input_group.add_argument('-q', '--quality', dest="quality", metavar="quality",
            default=self.quality, help='Quality offset. 33, 64 or "ignore" to disable qualities.')
        input_group.add_argument('-i', '--index', dest="index", metavar="index", help='Path to the .gem genome index')
        input_group.add_argument('--direct-input', dest="direct_input", default=None, action="store_true", help="Skip preparation step and pipe the input directly into the first mapping step")

    def register_output(self, parser):
        """Register all output parameters with the given
        argparse parser

        parser -- the argparse parser
        """
        output_group = parser.add_argument_group('Output')
        output_group.add_argument('-n', '--name', dest="name", metavar="name", help="""Name used for the results. If not specified, the name is inferred from
            the input files""")
        output_group.add_argument('-o', '--output-dir', dest="output_dir", metavar="dir", help='Optional output folder. If not specified the current working directory is used.')
        output_group.add_argument('-g', '--no-gzip', dest="compress", action="store_false", default=None, help="Do not compress final mapping file")
        output_group.add_argument('--compress-all', dest="compress_all", action="store_true", default=None, help="Compress all intermediate output")
        output_group.add_argument('--keep-temp', dest="remove_temp", action="store_false", default=None, help="Keep temporary files")

    def register_execution(self, parser):
        """Register the execution mapping parameters with the
        given arparse parser

        parser -- the argparse parser
        """
        execution_group = parser.add_argument_group('Execution')
        execution_group.add_argument('--save', dest="write_config", nargs="?", const=None, help="Write the given configuration to disk")
        execution_group.add_argument('--dry', dest="dry", action="store_true", default=None, help="Print and write configuration but do not start the pipeline")
        execution_group.add_argument('--load', dest="load_configuration", default=None, metavar="cfg", help="Load pipeline configuration from file")
        execution_group.add_argument('--run', dest="run_steps", type=int, default=None, nargs="+", metavar="cfg", help="Run given pipeline steps idenfified by the step id")
        execution_group.add_argument('--force', dest="force", default=None, action="store_true", help="Force running all steps and skip checking for completed steps")
        execution_group.add_argument('-t', '--threads', dest="threads", metavar="threads", type=int, help="Number of threads to use. Default %d" % self.threads)

    def register_mapping(self, parser):
        """Register the genome mapping parameters with the
        given arparse parser

        parser -- the argparse parser
        """
        # genome mapping parameter
        mapping_group = parser.add_argument_group('General mapping parameters')
        mapping_group.add_argument('-s', '--strata-after-best', dest="genome_strata_after_best", type=int, metavar="strata", help='The number of strata examined after the best one. Default %d' % (self.genome_strata_after_best))
        mapping_group.add_argument('-m', '--mismatches', dest="genome_mismatches", metavar="mm", help='Set the allowed mismatch ratio as 0 < mm < 1. Default %s' % (str(self.genome_mismatches)))
        mapping_group.add_argument('--quality-threshold', dest="genome_quality_threshold", type=int, metavar="qth", help='Good quality threshold. Bases with a quality score >= threshold are considered good. Default %d' % self.genome_quality_threshold)
        mapping_group.add_argument('--max-decoded-matches', dest="genome_max_decoded_matches", metavar="mdm", help='Maximum decoded matches. Default %d' % (self.genome_max_decoded_matches))
        mapping_group.add_argument('--min-decoded-strata', dest="genome_min_decoded_strata", metavar="mds", help='Minimum decoded strata. Default to %d' % self.genome_min_decoded_strata)
        mapping_group.add_argument('--min-matched-bases', dest="genome_min_matched_bases", metavar="mmb", help='Minimum ratio of bases that must be matched. Default %d' % (self.genome_min_matched_bases))
        mapping_group.add_argument('--max-edit-distance', dest="genome_max_edit_distance", metavar="med", help='Maximum edit distance (ratio) allowed for an alignment. Default %s' % (str(self.genome_max_edit_distance)))
        mapping_group.add_argument('--mismatch-alphabet', dest="genome_mismatch_alphabet", metavar="alphabet", help='The mismatch alphabet. Default "%s"' % (self.genome_mismatch_alphabet))

    def register_transcript_mapping(self, parser):
        """Register the transcript mapping parameters with the
        given arparse parser

        parser -- the argparse parser
        """
        # transcript mapping parameter
        transcript_mapping_group = parser.add_argument_group('Transcript mapping parameters')
        transcript_mapping_group.add_argument('-a', '--annotation', dest="annotation", metavar="gtf", help='''Path to the GTF annotation. If specified the transcriptome generated from the annotation is
            used in addition to de-novo junctions.''')
        transcript_mapping_group.add_argument('-r', '--transcript-index', dest="transcript_index", help='''GTF Transcriptome index. If not specified and an annotation is given,
            it is assumed to be <gtf>.junctions.gem''')
        transcript_mapping_group.add_argument('-k', '--transcript-keys', dest="transcript_keys", help='''Transcriptome .keys file. If not specified and an annotation is given,
            it is assumed to be <gtf>.junctions.keys''')

        transcript_mapping_group.add_argument('-tm', '--transcript-mismatches', dest="transcript_mismatches", metavar="mm", help='Set the allowed mismatch ratio as 0 < mm < 1. Default to genome setting.')
        transcript_mapping_group.add_argument('--transcript-quality-threshold', dest="transcript_quality_threshold", metavar="qth", help='Good quality threshold. Bases with a quality score >= threshold are considered good. Default to genome setting.')
        transcript_mapping_group.add_argument('--transcript-max-decoded-matches', dest="transcript_max_decoded_matches", metavar="mdm", help='Maximum decoded matches. Default %d' % (self.transcript_max_decoded_matches))
        transcript_mapping_group.add_argument('--transcript-min-decoded-strata', dest="transcript_min_decoded_strata", metavar="mds", help='Minimum decoded strata. Default to genome setting.')
        transcript_mapping_group.add_argument('--transcript-min-matched-bases', dest="transcript_min_matched_bases", metavar="mmb", help='Minimum ratio of bases that must be matched. Default to genome setting.')
        transcript_mapping_group.add_argument('--transcript-max-edit-distance', dest="transcript_max_edit_distance", metavar="med", help='Maximum edit distance (ratio) allowed for an alignment. Default to genome setting.')
        transcript_mapping_group.add_argument('--transcript-mismatch-alphabet', dest="transcript_mismatch_alphabet", metavar="alphabet", help='The mismatch alphabet. Default to genome setting.')
        transcript_mapping_group.add_argument('--transcript-strata-after-best', dest="transcript_strata_after_best", metavar="strata", help='The number of strata examined after the best one. Default to genome setting.')
        transcript_mapping_group.add_argument('--max-read-length', dest="max_read_length", type=int, help='''The maximum read length. This is used to create the de-novo
            transcriptome and acts as an upper bound. Default %d''' % (self.max_read_length))

    def register_junctions(self, parser):
        """Register the junction detection parameter with the
        given arparse parser

        parser -- the argparse parser
        """
        # junction detection parameter
        junctions_group = parser.add_argument_group('De-novo junction detection parameters')
        junctions_group.add_argument('-jm', '--junction-mismatches', dest="junction_mismatches", metavar="jmm",
            help='Set the allowed mismatch ratio for junction detection as 0 < mm < 1. Default %s' % (str(self.junction_mismatches)))
        junctions_group.add_argument('--junction-max-matches', dest="junctions_max_junction_matches", metavar="mm",
            help='Maximum number of multi-maps allowed for a junction. Default %d' % (self.junctions_max_junction_matches))
        junctions_group.add_argument('--min-intron-size', dest="junctions_min_intron_size", type=int, metavar="mil", help='Minimum intron length. Default %d' % self.junctions_min_intron_size)
        junctions_group.add_argument('--max-intron-length', dest="junctions_max_intron_size", type=int, metavar="mil", help='Maximum intron length. Default %d' % self.junctions_max_intron_size)
        junctions_group.add_argument('--refinement-step', dest="junctions_refinement_step_size", metavar="r", help='Refine the minimum split size when constraints on number of candidates are not met. Default %d' % self.junctions_refinement_step_size)
        junctions_group.add_argument('--min-split-size', dest="junctions_min_split_size", type=int, metavar="mss", help='Minimum split length. Default 15')
        junctions_group.add_argument('--matches-threshold', dest="junctions_matches_threshold", metavar="mt", help='Maximum number canidates considered when splitting the read. Default 75')
        junctions_group.add_argument('--junction-coverage', dest="junctions_coverage", type=int, metavar="jc", help='Minimum allowed junction converage. Default %d' % self.junctions_coverage)
        junctions_group.add_argument('--junction-consensus', dest="junctions_consensus", metavar="jc", help='Consensus used to detect junction sites. Default \'%s\'' % (",".join(["(%s,%s)" % (c[0], c[1]) for c in self.junctions_consensus])))
        junctions_group.add_argument('--junction-strata-after-best', dest="junctions_strata_after_best", metavar="s", help='Maximum number of strata to examin after best. Default %d' % (self.junctions_strata_after_best))

    def register_pairing(self, parser):
        """Register the pairing parameter with the
        given arparse parser

        parser -- the argparse parser
        """
        pairing_group = parser.add_argument_group('Pairing parameters')
        pairing_group.add_argument('--pairing-quality-threshold', dest="pairing_quality_threshold", metavar="pq", help='Good quality threshold. Bases with a quality score >= threshold are considered good. Defaults to genome setting.')
        pairing_group.add_argument('--pairing-max-decoded-matches', dest="pairing_max_decoded_matches", metavar="pdm", help='Maximum decoded matches. Default %d' % self.pairing_max_decoded_matches)
        pairing_group.add_argument('--pairing-min-decoded-strata', dest="pairing_min_decoded_strata", metavar="pds", help='Minimum decoded strata. Default to %d' % self.pairing_min_decoded_strata)
        pairing_group.add_argument('--pairing-min-insert-size', dest="pairing_min_insert_size", metavar="is", help='Minimum insert size allowed for pairing. Default %d' % self.pairing_min_insert_size)
        pairing_group.add_argument('--pairing-max-insert-size', dest="pairing_max_insert_size", metavar="is", help='Maximum insert size allowed for pairing. Default to max intron size')

    def register_stats(self, parser):
        """Register stats parameter with the
        given arparse parser

        parser -- the argparse parser
        """
        stats_group = parser.add_argument_group('Stats')

        stats_group.add_argument('--no-stats', dest="stats_create", default=None, action="store_false", help='Skip creating stats')
        stats_group.add_argument('--stats-json', dest="stats_json", default=None, action="store_true", help='Write a json file with the statistics in addition to the normal stats output.')

